"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autobind = autobind;
exports.extractProps = extractProps;
exports.unusedProps = unusedProps;
exports.getPackageRoot = getPackageRoot;
exports.getAtomHelperPath = getAtomHelperPath;
exports.getDugitePath = getDugitePath;
exports.getSharedModulePath = getSharedModulePath;
exports.isBinary = isBinary;
exports.firstImplementer = firstImplementer;
exports.isValidWorkdir = isValidWorkdir;
exports.fileExists = fileExists;
exports.getTempDir = getTempDir;
exports.isFileExecutable = isFileExecutable;
exports.isFileSymlink = isFileSymlink;
exports.shortenSha = shortenSha;
exports.normalizeGitHelperPath = normalizeGitHelperPath;
exports.toNativePathSep = toNativePathSep;
exports.toGitPathSep = toGitPathSep;
exports.filePathEndsWith = filePathEndsWith;
exports.toSentence = toSentence;
exports.pushAtKey = pushAtKey;
exports.getCommitMessagePath = getCommitMessagePath;
exports.getCommitMessageEditors = getCommitMessageEditors;
exports.getFilePatchPaneItems = getFilePatchPaneItems;
exports.destroyFilePatchPaneItems = destroyFilePatchPaneItems;
exports.destroyEmptyFilePatchPaneItems = destroyEmptyFilePatchPaneItems;
exports.extractCoAuthorsAndRawCommitMessage = extractCoAuthorsAndRawCommitMessage;
exports.createItem = createItem;
exports.equalSets = equalSets;
exports.classNameForStatus = exports.PAGINATION_WAIT_TIME_MS = exports.PAGE_SIZE = exports.CO_AUTHOR_REGEX = exports.LINE_ENDING_REGEX = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _os = _interopRequireDefault(require("os"));

var _temp = _interopRequireDefault(require("temp"));

var _multiFilePatchController = _interopRequireDefault(require("./controllers/multi-file-patch-controller"));

var _refHolder = _interopRequireDefault(require("./models/ref-holder"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LINE_ENDING_REGEX = /\r?\n/;
exports.LINE_ENDING_REGEX = LINE_ENDING_REGEX;
const CO_AUTHOR_REGEX = /^co-authored-by. (.+?) <(.+?)>$/i;
exports.CO_AUTHOR_REGEX = CO_AUTHOR_REGEX;
const PAGE_SIZE = 50;
exports.PAGE_SIZE = PAGE_SIZE;
const PAGINATION_WAIT_TIME_MS = 100;
exports.PAGINATION_WAIT_TIME_MS = PAGINATION_WAIT_TIME_MS;

function autobind(self, ...methods) {
  for (const method of methods) {
    if (typeof self[method] !== 'function') {
      throw new Error(`Unable to autobind method ${method}`);
    }

    self[method] = self[method].bind(self);
  }
} // Extract a subset of props chosen from a propTypes object from a component's props to pass to a different API.
//
// Usage:
//
// ```js
// const apiProps = {
//   zero: PropTypes.number.isRequired,
//   one: PropTypes.string,
//   two: PropTypes.object,
// };
//
// class Component extends React.Component {
//   static propTypes = {
//     ...apiProps,
//     extra: PropTypes.func,
//   }
//
//   action() {
//     const options = extractProps(this.props, apiProps);
//     // options contains zero, one, and two, but not extra
//   }
// }
// ```


function extractProps(props, propTypes, nameMap = {}) {
  return Object.keys(propTypes).reduce((opts, propName) => {
    if (props[propName] !== undefined) {
      const destPropName = nameMap[propName] || propName;
      opts[destPropName] = props[propName];
    }

    return opts;
  }, {});
} // The opposite of extractProps. Return a subset of props that do *not* appear in a component's prop types.


function unusedProps(props, propTypes) {
  return Object.keys(props).reduce((opts, propName) => {
    if (propTypes[propName] === undefined) {
      opts[propName] = props[propName];
    }

    return opts;
  }, {});
}

function getPackageRoot() {
  const {
    resourcePath
  } = atom.getLoadSettings();
  const currentFileWasRequiredFromSnapshot = !_path.default.isAbsolute(__dirname);

  if (currentFileWasRequiredFromSnapshot) {
    return _path.default.join(resourcePath, 'node_modules', 'github');
  } else {
    const packageRoot = _path.default.resolve(__dirname, '..');

    if (_path.default.extname(resourcePath) === '.asar') {
      if (packageRoot.indexOf(resourcePath) === 0) {
        return _path.default.join(`${resourcePath}.unpacked`, 'node_modules', 'github');
      }
    }

    return packageRoot;
  }
}

function getAtomAppName() {
  const match = atom.getVersion().match(/-([A-Za-z]+)(\d+|-)/);

  if (match) {
    const channel = match[1];
    return `Atom ${channel.charAt(0).toUpperCase() + channel.slice(1)} Helper`;
  }

  return 'Atom Helper';
}

function getAtomHelperPath() {
  if (process.platform === 'darwin') {
    const appName = getAtomAppName();
    return _path.default.resolve(process.resourcesPath, '..', 'Frameworks', `${appName}.app`, 'Contents', 'MacOS', appName);
  } else {
    return process.execPath;
  }
}

let DUGITE_PATH;

function getDugitePath() {
  if (!DUGITE_PATH) {
    DUGITE_PATH = require.resolve('dugite');

    if (!_path.default.isAbsolute(DUGITE_PATH)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();

      if (_path.default.extname(resourcePath) === '.asar') {
        DUGITE_PATH = _path.default.join(`${resourcePath}.unpacked`, 'node_modules', 'dugite');
      } else {
        DUGITE_PATH = _path.default.join(resourcePath, 'node_modules', 'dugite');
      }
    }
  }

  return DUGITE_PATH;
}

const SHARED_MODULE_PATHS = new Map();

function getSharedModulePath(relPath) {
  let modulePath = SHARED_MODULE_PATHS.get(relPath);

  if (!modulePath) {
    modulePath = require.resolve(_path.default.join(__dirname, 'shared', relPath));

    if (!_path.default.isAbsolute(modulePath)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();
      modulePath = _path.default.join(resourcePath, modulePath);
    }

    SHARED_MODULE_PATHS.set(relPath, modulePath);
  }

  return modulePath;
}

function isBinary(data) {
  for (let i = 0; i < 50; i++) {
    const code = data.charCodeAt(i); // Char code 65533 is the "replacement character";
    // 8 and below are control characters.

    if (code === 65533 || code < 9) {
      return true;
    }
  }

  return false;
}

function descriptorsFromProto(proto) {
  return Object.getOwnPropertyNames(proto).reduce((acc, name) => {
    Object.assign(acc, {
      [name]: Reflect.getOwnPropertyDescriptor(proto, name)
    });
    return acc;
  }, {});
}
/**
 * Takes an array of targets and returns a proxy. The proxy intercepts property accessor calls and
 * returns the value of that property on the first object in `targets` where the target implements that property.
 */


function firstImplementer(...targets) {
  return new Proxy({
    __implementations: targets
  }, {
    get(target, name) {
      if (name === 'getImplementers') {
        return () => targets;
      }

      if (Reflect.has(target, name)) {
        return target[name];
      }

      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        return firstValidTarget[name];
      } else {
        return undefined;
      }
    },

    set(target, name, value) {
      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        // eslint-disable-next-line no-return-assign
        return firstValidTarget[name] = value;
      } else {
        // eslint-disable-next-line no-return-assign
        return target[name] = value;
      }
    },

    // Used by sinon
    has(target, name) {
      if (name === 'getImplementers') {
        return true;
      }

      return targets.some(t => Reflect.has(t, name));
    },

    // Used by sinon
    getOwnPropertyDescriptor(target, name) {
      const firstValidTarget = targets.find(t => Reflect.getOwnPropertyDescriptor(t, name));
      const compositeOwnPropertyDescriptor = Reflect.getOwnPropertyDescriptor(target, name);

      if (firstValidTarget) {
        return Reflect.getOwnPropertyDescriptor(firstValidTarget, name);
      } else if (compositeOwnPropertyDescriptor) {
        return compositeOwnPropertyDescriptor;
      } else {
        return undefined;
      }
    },

    // Used by sinon
    getPrototypeOf(target) {
      return targets.reduceRight((acc, t) => {
        return Object.create(acc, descriptorsFromProto(Object.getPrototypeOf(t)));
      }, Object.prototype);
    }

  });
}

function isRoot(dir) {
  return _path.default.resolve(dir, '..') === dir;
}

function isValidWorkdir(dir) {
  return dir !== _os.default.homedir() && !isRoot(dir);
}

async function fileExists(absoluteFilePath) {
  try {
    await _fsExtra.default.access(absoluteFilePath);
    return true;
  } catch (e) {
    if (e.code === 'ENOENT') {
      return false;
    }

    throw e;
  }
}

function getTempDir(options = {}) {
  _temp.default.track();

  return new Promise((resolve, reject) => {
    _temp.default.mkdir(options, (tempError, folder) => {
      if (tempError) {
        reject(tempError);
        return;
      }

      if (options.symlinkOk) {
        resolve(folder);
      } else {
        _fsExtra.default.realpath(folder, (realError, rpath) => realError ? reject(realError) : resolve(rpath));
      }
    });
  });
}

async function isFileExecutable(absoluteFilePath) {
  const stat = await _fsExtra.default.stat(absoluteFilePath);
  return stat.mode & _fsExtra.default.constants.S_IXUSR; // eslint-disable-line no-bitwise
}

async function isFileSymlink(absoluteFilePath) {
  const stat = await _fsExtra.default.lstat(absoluteFilePath);
  return stat.isSymbolicLink();
}

function shortenSha(sha) {
  return sha.slice(0, 8);
}

const classNameForStatus = {
  added: 'added',
  deleted: 'removed',
  modified: 'modified',
  typechange: 'modified',
  equivalent: 'ignored'
};
/*
 * Apply any platform-specific munging to a path before presenting it as
 * a git environment variable or option.
 *
 * Convert a Windows-style "C:\foo\bar\baz" path to a "/c/foo/bar/baz" UNIX-y
 * path that the sh.exe used to execute git's credential helpers will
 * understand.
 */

exports.classNameForStatus = classNameForStatus;

function normalizeGitHelperPath(inPath) {
  if (process.platform === 'win32') {
    return inPath.replace(/\\/g, '/').replace(/^([^:]+):/, '/$1');
  } else {
    return inPath;
  }
}
/*
 * On Windows, git commands report paths with / delimiters. Convert them to \-delimited paths
 * so that Atom unifromly treats paths with native path separators.
 */


function toNativePathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split('/').join(_path.default.sep);
  }
}
/*
 * Convert Windows paths back to /-delimited paths to be presented to git.
 */


function toGitPathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split(_path.default.sep).join('/');
  }
}

function filePathEndsWith(filePath, ...segments) {
  return filePath.endsWith(_path.default.join(...segments));
}
/**
 * Turns an array of things @kuychaco cannot eat
 * into a sentence containing things @kuychaco cannot eat
 *
 * ['toast'] => 'toast'
 * ['toast', 'eggs'] => 'toast and eggs'
 * ['toast', 'eggs', 'cheese'] => 'toast, eggs, and cheese'
 *
 * Oxford comma included because you're wrong, shut up.
 */


function toSentence(array) {
  const len = array.length;

  if (len === 1) {
    return `${array[0]}`;
  } else if (len === 2) {
    return `${array[0]} and ${array[1]}`;
  }

  return array.reduce((acc, item, idx) => {
    if (idx === 0) {
      return `${item}`;
    } else if (idx === len - 1) {
      return `${acc}, and ${item}`;
    } else {
      return `${acc}, ${item}`;
    }
  }, '');
}

function pushAtKey(map, key, value) {
  let existing = map.get(key);

  if (!existing) {
    existing = [];
    map.set(key, existing);
  }

  existing.push(value);
} // Repository and workspace helpers


function getCommitMessagePath(repository) {
  return _path.default.join(repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
}

function getCommitMessageEditors(repository, workspace) {
  if (!repository.isPresent()) {
    return [];
  }

  return workspace.getTextEditors().filter(editor => editor.getPath() === getCommitMessagePath(repository));
}

function getFilePatchPaneItems({
  onlyStaged,
  empty
} = {}, workspace) {
  return workspace.getPaneItems().filter(item => {
    const isFilePatchItem = item && item.getRealItem && item.getRealItem() instanceof _multiFilePatchController.default;

    if (onlyStaged) {
      return isFilePatchItem && item.stagingStatus === 'staged';
    } else if (empty) {
      return isFilePatchItem ? item.isEmpty() : false;
    } else {
      return isFilePatchItem;
    }
  });
}

function destroyFilePatchPaneItems({
  onlyStaged
} = {}, workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    onlyStaged
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function destroyEmptyFilePatchPaneItems(workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    empty: true
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function extractCoAuthorsAndRawCommitMessage(commitMessage) {
  const messageLines = [];
  const coAuthors = [];

  for (const line of commitMessage.split(LINE_ENDING_REGEX)) {
    const match = line.match(CO_AUTHOR_REGEX);

    if (match) {
      // eslint-disable-next-line no-unused-vars
      const [_, name, email] = match;
      coAuthors.push({
        name,
        email
      });
    } else {
      messageLines.push(line);
    }
  }

  return {
    message: messageLines.join('\n'),
    coAuthors
  };
} // Atom API pane item manipulation


function createItem(node, componentHolder = null, uri = null, extra = {}) {
  const holder = componentHolder || new _refHolder.default();

  const override = _objectSpread({
    getElement: () => node,
    getRealItem: () => holder.getOr(null),
    getRealItemPromise: () => holder.getPromise()
  }, extra);

  if (uri) {
    override.getURI = () => uri;
  }

  if (componentHolder) {
    return new Proxy(override, {
      get(target, name) {
        if (Reflect.has(target, name)) {
          return target[name];
        } // The {value: ...} wrapper prevents .map() from flattening a returned RefHolder.
        // If component[name] is a RefHolder, we want to return that RefHolder as-is.


        const {
          value
        } = holder.map(component => ({
          value: component[name]
        })).getOr({
          value: undefined
        });
        return value;
      },

      set(target, name, value) {
        return holder.map(component => {
          component[name] = value;
          return true;
        }).getOr(true);
      },

      has(target, name) {
        return holder.map(component => Reflect.has(component, name)).getOr(false) || Reflect.has(target, name);
      }

    });
  } else {
    return override;
  }
} // Set functions


function equalSets(left, right) {
  if (left.size !== right.size) {
    return false;
  }

  for (const each of left) {
    if (!right.has(each)) {
      return false;
    }
  }

  return true;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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