"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _statusBar = _interopRequireDefault(require("../atom/status-bar"));

var _panel = _interopRequireDefault(require("../atom/panel"));

var _paneItem = _interopRequireDefault(require("../atom/pane-item"));

var _cloneDialog = _interopRequireDefault(require("../views/clone-dialog"));

var _openIssueishDialog = _interopRequireDefault(require("../views/open-issueish-dialog"));

var _openCommitDialog = _interopRequireDefault(require("../views/open-commit-dialog"));

var _initDialog = _interopRequireDefault(require("../views/init-dialog"));

var _credentialDialog = _interopRequireDefault(require("../views/credential-dialog"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _gitTimingsView = _interopRequireDefault(require("../views/git-timings-view"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _gitTabItem = _interopRequireDefault(require("../items/git-tab-item"));

var _githubTabItem = _interopRequireDefault(require("../items/github-tab-item"));

var _statusBarTileController = _interopRequireDefault(require("./status-bar-tile-controller"));

var _repositoryConflictController = _interopRequireDefault(require("./repository-conflict-controller"));

var _gitCacheView = _interopRequireDefault(require("../views/git-cache-view"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _switchboard = _interopRequireDefault(require("../switchboard"));

var _propTypes2 = require("../prop-types");

var _helpers = require("../helpers");

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RootController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "toggleCommitPreviewItem", () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem.default.buildURI(workdir));
    });

    _defineProperty(this, "showOpenCommitDialog", () => {
      this.setState({
        openCommitDialogActive: true
      });
    });

    _defineProperty(this, "isValidCommit", async ref => {
      try {
        await this.props.repository.getCommit(ref);
        return true;
      } catch (error) {
        if (error instanceof _gitShellOutStrategy.GitError && error.code === 128) {
          return false;
        } else {
          throw error;
        }
      }
    });

    _defineProperty(this, "acceptOpenCommit", ({
      ref
    }) => {
      const workdir = this.props.repository.getWorkingDirectoryPath();

      const uri = _commitDetailItem.default.buildURI(workdir, ref);

      this.setState({
        openCommitDialogActive: false
      });
      this.props.workspace.open(uri).then(() => {
        (0, _reporterProxy.addEvent)('open-commit-in-pane', {
          package: 'github',
          from: _openCommitDialog.default.name
        });
      });
    });

    _defineProperty(this, "cancelOpenCommit", () => {
      this.setState({
        openCommitDialogActive: false
      });
    });

    _defineProperty(this, "surfaceFromFileAtPath", (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    });

    _defineProperty(this, "surfaceToCommitPreviewButton", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    });

    _defineProperty(this, "surfaceToRecentCommit", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    });

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'initializeRepo', 'showOpenIssueishDialog', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'acceptClone', 'cancelClone', 'acceptInit', 'cancelInit', 'acceptOpenIssueish', 'cancelOpenIssueish', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'openCloneDialog', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');
    this.state = {
      cloneDialogActive: false,
      cloneDialogInProgress: false,
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null,
      credentialDialogQuery: null
    };
    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));
    this.props.commandRegistry.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          package: 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react.default.createElement(_react.Fragment, null, this.renderCommands(), this.renderStatusBarTile(), this.renderPaneItems(), this.renderDialogs(), this.renderConflictResolver());
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();
    return _react.default.createElement(_commands.default, {
      registry: this.props.commandRegistry,
      target: "atom-workspace"
    }, devMode && _react.default.createElement(_commands.Command, {
      command: "github:install-react-dev-tools",
      callback: this.installReactDevTools
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-commit-preview",
      callback: this.toggleCommitPreviewItem
    }), _react.default.createElement(_commands.Command, {
      command: "github:logout",
      callback: this.clearGithubToken
    }), _react.default.createElement(_commands.Command, {
      command: "github:show-waterfall-diagnostics",
      callback: this.showWaterfallDiagnostics
    }), _react.default.createElement(_commands.Command, {
      command: "github:show-cache-diagnostics",
      callback: this.showCacheDiagnostics
    }), _react.default.createElement(_commands.Command, {
      command: "github:open-issue-or-pull-request",
      callback: this.showOpenIssueishDialog
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-git-tab",
      callback: this.gitTabTracker.toggle
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-git-tab-focus",
      callback: this.gitTabTracker.toggleFocus
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-github-tab",
      callback: this.githubTabTracker.toggle
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-github-tab-focus",
      callback: this.githubTabTracker.toggleFocus
    }), _react.default.createElement(_commands.Command, {
      command: "github:clone",
      callback: this.openCloneDialog
    }), _react.default.createElement(_commands.Command, {
      command: "github:open-commit",
      callback: this.showOpenCommitDialog
    }), _react.default.createElement(_commands.Command, {
      command: "github:view-unstaged-changes-for-current-file",
      callback: this.viewUnstagedChangesForCurrentFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:view-staged-changes-for-current-file",
      callback: this.viewStagedChangesForCurrentFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:close-all-diff-views",
      callback: this.destroyFilePatchPaneItems
    }), _react.default.createElement(_commands.Command, {
      command: "github:close-empty-diff-views",
      callback: this.destroyEmptyFilePatchPaneItems
    }));
  }

  renderStatusBarTile() {
    return _react.default.createElement(_statusBar.default, {
      statusBar: this.props.statusBar,
      onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
      className: "github-StatusBarTileController"
    }, _react.default.createElement(_statusBarTileController.default, {
      pipelineManager: this.props.pipelineManager,
      workspace: this.props.workspace,
      repository: this.props.repository,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      confirm: this.props.confirm,
      toggleGitTab: this.gitTabTracker.toggle,
      toggleGithubTab: this.githubTabTracker.toggle
    }));
  }

  renderDialogs() {
    return _react.default.createElement(_react.Fragment, null, this.renderInitDialog(), this.renderCloneDialog(), this.renderCredentialDialog(), this.renderOpenIssueishDialog(), this.renderOpenCommitDialog());
  }

  renderInitDialog() {
    if (!this.state.initDialogActive) {
      return null;
    }

    return _react.default.createElement(_panel.default, {
      workspace: this.props.workspace,
      location: "modal"
    }, _react.default.createElement(_initDialog.default, {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      initPath: this.state.initDialogPath,
      didAccept: this.acceptInit,
      didCancel: this.cancelInit
    }));
  }

  renderCloneDialog() {
    if (!this.state.cloneDialogActive) {
      return null;
    }

    return _react.default.createElement(_panel.default, {
      workspace: this.props.workspace,
      location: "modal"
    }, _react.default.createElement(_cloneDialog.default, {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptClone,
      didCancel: this.cancelClone,
      inProgress: this.state.cloneDialogInProgress
    }));
  }

  renderOpenIssueishDialog() {
    if (!this.state.openIssueishDialogActive) {
      return null;
    }

    return _react.default.createElement(_panel.default, {
      workspace: this.props.workspace,
      location: "modal"
    }, _react.default.createElement(_openIssueishDialog.default, {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenIssueish,
      didCancel: this.cancelOpenIssueish
    }));
  }

  renderOpenCommitDialog() {
    if (!this.state.openCommitDialogActive) {
      return null;
    }

    return _react.default.createElement(_panel.default, {
      workspace: this.props.workspace,
      location: "modal"
    }, _react.default.createElement(_openCommitDialog.default, {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenCommit,
      didCancel: this.cancelOpenCommit,
      isValidEntry: this.isValidCommit
    }));
  }

  renderCredentialDialog() {
    if (this.state.credentialDialogQuery === null) {
      return null;
    }

    return _react.default.createElement(_panel.default, {
      workspace: this.props.workspace,
      location: "modal"
    }, _react.default.createElement(_credentialDialog.default, _extends({
      commandRegistry: this.props.commandRegistry
    }, this.state.credentialDialogQuery)));
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react.default.createElement(_repositoryConflictController.default, {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commandRegistry: this.props.commandRegistry
    });
  }

  renderPaneItems() {
    return _react.default.createElement(_react.Fragment, null, _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitTabItem.default.uriPattern,
      className: "github-Git-root"
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitTabItem.default, {
      ref: itemHolder.setter,
      workspace: this.props.workspace,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      grammars: this.props.grammars,
      project: this.props.project,
      confirm: this.props.confirm,
      config: this.props.config,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      initializeRepo: this.initializeRepo,
      resolutionProgress: this.props.resolutionProgress,
      ensureGitTab: this.gitTabTracker.ensureVisible,
      openFiles: this.openFiles,
      discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
      undoLastDiscard: this.undoLastDiscard,
      refreshResolutionProgress: this.refreshResolutionProgress
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _githubTabItem.default.uriPattern,
      className: "github-GitHub-root"
    }, ({
      itemHolder
    }) => _react.default.createElement(_githubTabItem.default, {
      ref: itemHolder.setter,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _changedFileItem.default.uriPattern
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_changedFileItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      relPath: _path.default.join(...params.relPath),
      workingDirectory: params.workingDirectory,
      stagingStatus: params.stagingStatus,
      tooltips: this.props.tooltips,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      workspace: this.props.workspace,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceFileAtPath: this.surfaceFromFileAtPath
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _commitPreviewItem.default.uriPattern,
      className: "github-CommitPreview-root"
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_commitPreviewItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _commitDetailItem.default.uriPattern,
      className: "github-CommitDetail-root"
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_commitDetailItem.default, {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      sha: params.sha,
      surfaceCommit: this.surfaceToRecentCommit
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _issueishDetailItem.default.uriPattern
    }, ({
      itemHolder,
      params
    }) => _react.default.createElement(_issueishDetailItem.default, {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      issueishNumber: parseInt(params.issueishNumber, 10),
      workingDirectory: params.workingDirectory,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitTimingsView.default.uriPattern
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitTimingsView.default, {
      ref: itemHolder.setter
    })), _react.default.createElement(_paneItem.default, {
      workspace: this.props.workspace,
      uriPattern: _gitCacheView.default.uriPattern
    }, ({
      itemHolder
    }) => _react.default.createElement(_gitCacheView.default, {
      ref: itemHolder.setter,
      repository: this.props.repository
    })));
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem.default.buildURI(), _githubTabItem.default.buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';

    const unzip = require(crossUnzipName);

    const reactId = devTools.REACT_DEVELOPER_TOOLS.id;
    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${reactId}%26uc&prodversion=32`;

    const extensionFolder = _path.default.resolve(_electron.remote.app.getPath('userData'), `extensions/${reactId}`);

    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra.default.ensureDir(_path.default.dirname(extensionFile));
    const response = await fetch(url, {
      method: 'GET'
    });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra.default.writeFile(extensionFile, body);
    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra.default.exists(_path.default.join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });
    await _fsExtra.default.ensureDir(extensionFolder, 0o755);
    await devTools.default(devTools.REACT_DEVELOPER_TOOLS);
    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React dev tools!');
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  initializeRepo(initDialogPath) {
    if (this.state.initDialogActive) {
      return null;
    }

    if (!initDialogPath) {
      initDialogPath = this.props.repository.getWorkingDirectoryPath();
    }

    return new Promise(resolve => {
      this.setState({
        initDialogActive: true,
        initDialogPath,
        initDialogResolve: resolve
      });
    });
  }

  showOpenIssueishDialog() {
    this.setState({
      openIssueishDialogActive: true
    });
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView.default.buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView.default.buildURI());
  }

  async acceptClone(remoteUrl, projectPath) {
    this.setState({
      cloneDialogInProgress: true
    });

    try {
      await this.props.cloneRepositoryForProjectPath(remoteUrl, projectPath);
      (0, _reporterProxy.addEvent)('clone-repo', {
        package: 'github'
      });
    } catch (e) {
      this.props.notificationManager.addError(`Unable to clone ${remoteUrl}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        cloneDialogInProgress: false,
        cloneDialogActive: false
      });
    }
  }

  cancelClone() {
    this.setState({
      cloneDialogActive: false
    });
  }

  async acceptInit(projectPath) {
    try {
      await this.props.createRepositoryForProjectPath(projectPath);

      if (this.state.initDialogResolve) {
        this.state.initDialogResolve(projectPath);
      }
    } catch (e) {
      this.props.notificationManager.addError(`Unable to initialize git repository in ${projectPath}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        initDialogActive: false,
        initDialogPath: null,
        initDialogResolve: null
      });
    }
  }

  cancelInit() {
    if (this.state.initDialogResolve) {
      this.state.initDialogResolve(false);
    }

    this.setState({
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null
    });
  }

  acceptOpenIssueish({
    repoOwner,
    repoName,
    issueishNumber
  }) {
    const uri = _issueishDetailItem.default.buildURI('github.com', repoOwner, repoName, issueishNumber);

    this.setState({
      openIssueishDialogActive: false
    });
    this.props.workspace.open(uri).then(() => {
      (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
        package: 'github',
        from: 'dialog'
      });
    });
  }

  cancelOpenIssueish() {
    this.setState({
      openIssueishDialogActive: false
    });
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({
      onlyStaged: false
    }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  openCloneDialog() {
    this.setState({
      cloneDialogActive: true
    });
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();

    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra.default.realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();

    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath); // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place

            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }

    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();

      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }

      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem.default.buildURI(filePath, repoPath, stagingStatus), {
        pending: true,
        activatePane: true,
        activateItem: true
      });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path.default.join(repository.getWorkingDirectoryPath(), filePath);

      return this.props.workspace.open(absolutePath, {
        pending: filePaths.length === 1
      });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path.default.join(workdirPath, filePath);

      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');

    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };

    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();

    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };

    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }

    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);

    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);

      if (results.length === 0) {
        return;
      }

      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({
      conflict
    }) => conflict);

    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({
      filePath
    }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });

    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({
        resultPath
      }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const {
        filePath,
        resultPath,
        deleted,
        conflict,
        theirsSha,
        commonBaseSha,
        currentSha
      } = result;

      const absFilePath = _path.default.join(this.props.repository.getWorkingDirectoryPath(), filePath);

      if (deleted && resultPath === null) {
        await _fsExtra.default.remove(absFilePath);
      } else {
        await _fsExtra.default.copy(resultPath, absFilePath);
      }

      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }
  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */


  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra.default.createReadStream(fullPath, {
      encoding: 'utf8'
    });

    return new Promise(resolve => {
      _conflict.default.countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }
  /*
   * Display the credential entry dialog. Return a Promise that will resolve with the provided credentials on accept
   * or reject on cancel.
   */


  promptForCredentials(query) {
    return new Promise((resolve, reject) => {
      this.setState({
        credentialDialogQuery: _objectSpread({}, query, {
          onSubmit: response => this.setState({
            credentialDialogQuery: null
          }, () => resolve(response)),
          onCancel: () => this.setState({
            credentialDialogQuery: null
          }, reject)
        })
      });
    });
  }

}

exports.default = RootController;

_defineProperty(RootController, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  commandRegistry: _propTypes.default.object.isRequired,
  deserializers: _propTypes.default.object.isRequired,
  notificationManager: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  grammars: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  project: _propTypes.default.object.isRequired,
  loginModel: _propTypes.default.object.isRequired,
  confirm: _propTypes.default.func.isRequired,
  createRepositoryForProjectPath: _propTypes.default.func,
  cloneRepositoryForProjectPath: _propTypes.default.func,
  repository: _propTypes.default.object.isRequired,
  resolutionProgress: _propTypes.default.object.isRequired,
  statusBar: _propTypes.default.object,
  switchboard: _propTypes.default.instanceOf(_switchboard.default),
  startOpen: _propTypes.default.bool,
  startRevealed: _propTypes.default.bool,
  pipelineManager: _propTypes.default.object,
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired
});

_defineProperty(RootController, "defaultProps", {
  switchboard: new _switchboard.default(),
  startOpen: false,
  startRevealed: false
});

class TabTracker {
  constructor(name, {
    getWorkspace,
    uri
  }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;
    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false; // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.

    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();

      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }

      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }

    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: false,
      activatePane: false
    });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: true,
      activatePane: true
    });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);

    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);

    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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