"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _commit = require("../commit");

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _operationStates = require("../operation-states");

var _multiFilePatch = _interopRequireDefault(require("../patch/multi-file-patch"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();
/**
 * Base class for Repository states. Implements default "null" behavior.
 */

class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  } // This state has just been entered. Perform any asynchronous initialization that needs to occur.


  start() {
    return Promise.resolve();
  } // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.


  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  } // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.


  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  } // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.


  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }
  /* istanbul ignore next */


  refresh() {} // No-op

  /* istanbul ignore next */


  observeFilesystemChange(events) {
    this.repository.refresh();
  }
  /* istanbul ignore next */


  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    this.repository.refresh();
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.
  // Staging and unstaging


  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  } // Committing


  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  } // Merging


  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  } // Checkout


  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  } // Reset


  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  } // Remote interactions


  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  } // Configuration


  setConfig(option, value, {
    replaceAll
  } = {}) {
    return unsupportedOperationPromise(this, 'setConfig');
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  } // Direct blob interactions


  createBlob({
    filePath,
    stdin
  } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  } // Discard history


  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.
  // Index queries


  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: {
          ahead: null,
          behind: null
        }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(_multiFilePatch.default.createNull());
  }

  getStagedChangesPatch() {
    return Promise.resolve(_multiFilePatch.default.createNull());
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  } // Commit access


  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  isCommitPushed(sha) {
    return false;
  } // Author information


  getAuthors() {
    return Promise.resolve([]);
  } // Branches


  getBranches() {
    return Promise.resolve(new _branchSet.default());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  } // Merging and rebasing status


  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  } // Remotes


  getRemotes() {
    return Promise.resolve(new _remoteSet.default([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(null);
  }

  getBehindCount(branchName) {
    return Promise.resolve(null);
  }

  getConfig(option, {
    local
  } = {}) {
    return Promise.resolve(null);
  } // Direct blob access


  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  } // Atom repo state


  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  } // Cache


  getCache() {
    return null;
  } // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.


  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  } // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.


  current() {
    return this.repository.state;
  } // pipeline


  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  } // Return a Promise that will resolve once the state transitions from Loading.


  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  } // Initiate a transition to another state.


  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    /* istanbul ignore if */

    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }

    return this.repository.transition(this, StateConstructor, ...payload);
  } // Event broadcast


  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  } // Direct git access
  // Non-delegated git operations for internal use within states.

  /* istanbul ignore next */


  directResolveDotGitDir() {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }
  /* istanbul ignore next */


  directInit() {
    return Promise.resolve();
  }
  /* istanbul ignore next */


  directClone(remoteUrl, options) {
    return Promise.resolve();
  } // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.


  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  } // Parse a DiscardHistory payload from the SHA recorded in config.


  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');

    if (!historySha) {
      return {};
    }

    let blob;

    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  } // Debugging assistance.


  toString() {
    return this.constructor.name;
  }

}

exports.default = State;

function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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