"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _actionPipeline = _interopRequireDefault(require("./action-pipeline"));

var _gitShellOutStrategy = require("./git-shell-out-strategy");

var _helpers = require("./helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Note: Middleware that catches errors should re-throw the errors so that they propogate
// and other middleware in the pipeline can be made aware of the errors.
// Ultimately, the views are responsible for catching the errors and handling them accordingly
function _default({
  confirm,
  notificationManager,
  workspace
}) {
  const pipelineManager = new _actionPipeline.default({
    actionNames: ['PUSH', 'PULL', 'FETCH', 'COMMIT', 'CHECKOUT', 'ADDREMOTE']
  });
  const pushPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.PUSH);
  pushPipeline.addMiddleware('confirm-force-push', async (next, repository, branchName, options) => {
    if (options.force) {
      const choice = confirm({
        message: 'Are you sure you want to force push?',
        detailedMessage: 'This operation could result in losing data on the remote.',
        buttons: ['Force Push', 'Cancel']
      });

      if (choice !== 0) {
        /* do nothing */
      } else {
        await next();
      }
    } else {
      await next();
    }
  });
  pushPipeline.addMiddleware('set-push-in-progress', async (next, repository, branchName, options) => {
    repository.getOperationStates().setPushInProgress(true);

    try {
      await next();
    } finally {
      repository.getOperationStates().setPushInProgress(false);
    }
  });
  pushPipeline.addMiddleware('failed-to-push-error', async (next, repository, branchName, options) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        if (/rejected[\s\S]*failed to push/.test(error.stdErr)) {
          notificationManager.addError('Push rejected', {
            description: 'The tip of your current branch is behind its remote counterpart.' + ' Try pulling before pushing.<br />To force push, hold `cmd` or `ctrl` while clicking.',
            dismissable: true
          });
        } else {
          notificationManager.addError('Unable to push', {
            detail: error.stdErr,
            dismissable: true
          });
        }
      }

      throw error;
    }
  });
  const pullPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.PULL);
  pullPipeline.addMiddleware('set-pull-in-progress', async (next, repository, branchName) => {
    repository.getOperationStates().setPullInProgress(true);

    try {
      await next();
    } finally {
      repository.getOperationStates().setPullInProgress(false);
    }
  });
  pullPipeline.addMiddleware('failed-to-pull-error', async (next, repository, branchName) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        repository.didPullError();

        if (/error: Your local changes to the following files would be overwritten by merge/.test(error.stdErr)) {
          const lines = error.stdErr.split('\n');
          const files = lines.slice(3, lines.length - 3).map(l => `\`${l.trim()}\``).join('\n');
          notificationManager.addError('Pull aborted', {
            description: 'Local changes to the following would be overwritten by merge:<br/>' + files + '<br/>Please commit your changes or stash them before you merge.',
            dismissable: true
          });
        } else if (/Automatic merge failed; fix conflicts and then commit the result./.test(error.stdOut)) {
          notificationManager.addWarning('Merge conflicts', {
            description: `Your local changes conflicted with changes made on the remote branch. Resolve the conflicts
              with the Git panel and commit to continue.`,
            dismissable: true
          });
        } else if (/fatal: Not possible to fast-forward, aborting./.test(error.stdErr)) {
          notificationManager.addWarning('Unmerged changes', {
            description: 'Your local branch has diverged from its remote counterpart.<br/>' + 'Merge or rebase your local work to continue.',
            dismissable: true
          });
        } else {
          notificationManager.addError('Unable to pull', {
            detail: error.stdErr,
            dismissable: true
          });
        }
      }

      throw error;
    }
  });
  const fetchPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.FETCH);
  fetchPipeline.addMiddleware('set-fetch-in-progress', async (next, repository) => {
    repository.getOperationStates().setFetchInProgress(true);

    try {
      await next();
    } finally {
      repository.getOperationStates().setFetchInProgress(false);
    }
  });
  fetchPipeline.addMiddleware('failed-to-fetch-error', async (next, repository) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        notificationManager.addError('Unable to fetch', {
          detail: error.stdErr,
          dismissable: true
        });
      }

      throw error;
    }
  });
  const checkoutPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.CHECKOUT);
  checkoutPipeline.addMiddleware('set-checkout-in-progress', async (next, repository, branchName) => {
    repository.getOperationStates().setCheckoutInProgress(true);

    try {
      await next();
    } finally {
      repository.getOperationStates().setCheckoutInProgress(false);
    }
  });
  checkoutPipeline.addMiddleware('failed-to-checkout-error', async (next, repository, branchName, options) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        const message = options.createNew ? 'Cannot create branch' : 'Checkout aborted';
        let detail = undefined;
        let description = undefined;

        if (error.stdErr.match(/local changes.*would be overwritten/)) {
          const files = error.stdErr.split(/\r?\n/).filter(l => l.startsWith('\t')).map(l => `\`${l.trim()}\``).join('<br/>');
          description = 'Local changes to the following would be overwritten:<br/>' + files + '<br/>Please commit your changes or stash them.';
        } else if (error.stdErr.match(/branch.*already exists/)) {
          description = `\`${branchName}\` already exists. Choose another branch name.`;
        } else if (error.stdErr.match(/error: you need to resolve your current index first/)) {
          description = 'You must first resolve merge conflicts.';
        }

        if (description === undefined && detail === undefined) {
          detail = error.stdErr;
        }

        notificationManager.addError(message, {
          description,
          detail,
          dismissable: true
        });
      }

      throw error;
    }
  });
  const commitPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.COMMIT);
  commitPipeline.addMiddleware('confirm-commit', async (next, repository) => {
    function confirmCommit() {
      const choice = confirm({
        message: 'One or more text editors for the commit message are unsaved.',
        detailedMessage: 'Do you want to commit and close all open commit message editors?',
        buttons: ['Commit', 'Cancel']
      });
      return choice === 0;
    }

    const commitMessageEditors = (0, _helpers.getCommitMessageEditors)(repository, workspace);

    if (commitMessageEditors.length > 0) {
      if (!commitMessageEditors.some(e => e.isModified()) || confirmCommit()) {
        await next();
        commitMessageEditors.forEach(editor => editor.destroy());
      }
    } else {
      await next();
    }
  });
  commitPipeline.addMiddleware('clean-up-disk-commit-msg', async (next, repository) => {
    await next();

    try {
      await _fsExtra.default.remove((0, _helpers.getCommitMessagePath)(repository));
    } catch (error) {// do nothing
    }
  });
  commitPipeline.addMiddleware('set-commit-in-progress', async (next, repository) => {
    repository.getOperationStates().setCommitInProgress(true);

    try {
      await next();
    } finally {
      repository.getOperationStates().setCommitInProgress(false);
    }
  });
  commitPipeline.addMiddleware('failed-to-commit-error', async (next, repository) => {
    try {
      const result = await next();
      const template = await repository.fetchCommitMessageTemplate();
      repository.setCommitMessage(template || '');
      (0, _helpers.destroyFilePatchPaneItems)({
        onlyStaged: true
      }, workspace);
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        notificationManager.addError('Unable to commit', {
          detail: error.stdErr,
          dismissable: true
        });
      }

      throw error;
    }
  });
  const addRemotePipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.ADDREMOTE);
  addRemotePipeline.addMiddleware('failed-to-add-remote', async (next, repository, remoteName) => {
    try {
      return await next();
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        let detail = error.stdErr;

        if (error.stdErr.match(/^fatal: remote .* already exists\./)) {
          detail = `The repository already contains a remote named ${remoteName}.`;
        }

        notificationManager.addError('Cannot create remote', {
          detail,
          dismissable: true
        });
      }

      throw error;
    }
  });
  return pipelineManager;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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