(function() {
  var CSON, Disposable, MenuHelpers, MenuManager, _, fs, ipcRenderer, path, platformMenu, ref, ref1;

  path = require('path');

  _ = require('underscore-plus');

  ipcRenderer = require('electron').ipcRenderer;

  CSON = require('season');

  fs = require('fs-plus');

  Disposable = require('event-kit').Disposable;

  MenuHelpers = require('./menu-helpers');

  platformMenu = (ref = require('../package.json')) != null ? (ref1 = ref._atomMenu) != null ? ref1.menu : void 0 : void 0;

  module.exports = MenuManager = (function() {
    function MenuManager(arg) {
      this.resourcePath = arg.resourcePath, this.keymapManager = arg.keymapManager, this.packageManager = arg.packageManager;
      this.initialized = false;
      this.pendingUpdateOperation = null;
      this.template = [];
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
      this.packageManager.onDidActivateInitialPackages((function(_this) {
        return function() {
          return _this.sortPackagesMenu();
        };
      })(this));
    }

    MenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath;
      this.keymapManager.onDidReloadKeymap((function(_this) {
        return function() {
          return _this.update();
        };
      })(this));
      this.update();
      return this.initialized = true;
    };

    MenuManager.prototype.add = function(items) {
      var i, item, len;
      items = _.deepClone(items);
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.merge(this.template, item);
      }
      this.update();
      return new Disposable((function(_this) {
        return function() {
          return _this.remove(items);
        };
      })(this));
    };

    MenuManager.prototype.remove = function(items) {
      var i, item, len;
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.unmerge(this.template, item);
      }
      return this.update();
    };

    MenuManager.prototype.clear = function() {
      this.template = [];
      return this.update();
    };

    MenuManager.prototype.includeSelector = function(selector) {
      var element, error, ref2, ref3, testBody, testDocument, testWorkspace, workspaceClasses;
      try {
        if (document.body.webkitMatchesSelector(selector)) {
          return true;
        }
      } catch (error1) {
        error = error1;
        return false;
      }
      if (this.testEditor == null) {
        testDocument = document.implementation.createDocument(document.namespaceURI, 'html');
        testBody = testDocument.createElement('body');
        (ref2 = testBody.classList).add.apply(ref2, this.classesForElement(document.body));
        testWorkspace = testDocument.createElement('atom-workspace');
        workspaceClasses = this.classesForElement(document.body.querySelector('atom-workspace'));
        if (workspaceClasses.length === 0) {
          workspaceClasses = ['workspace'];
        }
        (ref3 = testWorkspace.classList).add.apply(ref3, workspaceClasses);
        testBody.appendChild(testWorkspace);
        this.testEditor = testDocument.createElement('atom-text-editor');
        this.testEditor.classList.add('editor');
        testWorkspace.appendChild(this.testEditor);
      }
      element = this.testEditor;
      while (element) {
        if (element.webkitMatchesSelector(selector)) {
          return true;
        }
        element = element.parentElement;
      }
      return false;
    };

    MenuManager.prototype.update = function() {
      if (!this.initialized) {
        return;
      }
      if (this.pendingUpdateOperation != null) {
        clearTimeout(this.pendingUpdateOperation);
      }
      return this.pendingUpdateOperation = setTimeout((function(_this) {
        return function() {
          var binding, i, j, keystrokesByCommand, len, len1, name, ref2, ref3, unsetKeystrokes;
          unsetKeystrokes = new Set;
          ref2 = _this.keymapManager.getKeyBindings();
          for (i = 0, len = ref2.length; i < len; i++) {
            binding = ref2[i];
            if (binding.command === 'unset!') {
              unsetKeystrokes.add(binding.keystrokes);
            }
          }
          keystrokesByCommand = {};
          ref3 = _this.keymapManager.getKeyBindings();
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            binding = ref3[j];
            if (!_this.includeSelector(binding.selector)) {
              continue;
            }
            if (unsetKeystrokes.has(binding.keystrokes)) {
              continue;
            }
            if (binding.keystrokes.includes(' ')) {
              continue;
            }
            if (process.platform === 'darwin' && /^alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'win32' && /^ctrl-alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (keystrokesByCommand[name = binding.command] == null) {
              keystrokesByCommand[name] = [];
            }
            keystrokesByCommand[binding.command].unshift(binding.keystrokes);
          }
          return _this.sendToBrowserProcess(_this.template, keystrokesByCommand);
        };
      })(this), 1);
    };

    MenuManager.prototype.loadPlatformItems = function() {
      var menu, menusDirPath, platformMenuPath;
      if (platformMenu != null) {
        return this.add(platformMenu);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        menu = CSON.readFileSync(platformMenuPath).menu;
        return this.add(menu);
      }
    };

    MenuManager.prototype.merge = function(menu, item) {
      return MenuHelpers.merge(menu, item);
    };

    MenuManager.prototype.unmerge = function(menu, item) {
      return MenuHelpers.unmerge(menu, item);
    };

    MenuManager.prototype.sendToBrowserProcess = function(template, keystrokesByCommand) {
      return ipcRenderer.send('update-application-menu', template, keystrokesByCommand);
    };

    MenuManager.prototype.classesForElement = function(element) {
      var classList;
      if (classList = element != null ? element.classList : void 0) {
        return Array.prototype.slice.apply(classList);
      } else {
        return [];
      }
    };

    MenuManager.prototype.sortPackagesMenu = function() {
      var packagesMenu;
      packagesMenu = _.find(this.template, function(arg) {
        var label;
        label = arg.label;
        return MenuHelpers.normalizeLabel(label) === 'Packages';
      });
      if ((packagesMenu != null ? packagesMenu.submenu : void 0) == null) {
        return;
      }
      packagesMenu.submenu.sort(function(item1, item2) {
        if (item1.label && item2.label) {
          return MenuHelpers.normalizeLabel(item1.label).localeCompare(MenuHelpers.normalizeLabel(item2.label));
        } else {
          return 0;
        }
      });
      return this.update();
    };

    return MenuManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
