(function() {
  var CSON, CompositeDisposable, Disposable, EditorStore, Emitter, File, ScopedPropertyStore, Snippet, SnippetExpansion, _, async, fs, getPackageRoot, path, ref,
    slice = [].slice;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  async = require('async');

  CSON = require('season');

  fs = require('fs-plus');

  ScopedPropertyStore = require('scoped-property-store');

  Snippet = require('./snippet');

  SnippetExpansion = require('./snippet-expansion');

  EditorStore = require('./editor-store');

  getPackageRoot = require('./helpers').getPackageRoot;

  module.exports = {
    activate: function() {
      var snippets;
      this.loaded = false;
      this.userSnippetsPath = null;
      this.snippetIdCounter = 0;
      this.snippetsByPackage = new Map;
      this.parsedSnippetsById = new Map;
      this.editorMarkerLayers = new WeakMap;
      this.scopedPropertyStore = new ScopedPropertyStore;
      this.disabledSnippetsScopedPropertyStore = new ScopedPropertyStore;
      this.subscriptions = new CompositeDisposable;
      this.subscriptions.add(atom.workspace.addOpener((function(_this) {
        return function(uri) {
          if (uri === 'atom://.atom/snippets') {
            return atom.workspace.openTextFile(_this.getUserSnippetsPath());
          }
        };
      })(this)));
      this.loadAll();
      this.watchUserSnippets((function(_this) {
        return function(watchDisposable) {
          return _this.subscriptions.add(watchDisposable);
        };
      })(this));
      this.subscriptions.add(atom.config.onDidChange('core.packagesWithSnippetsDisabled', (function(_this) {
        return function(arg) {
          var newValue, oldValue;
          newValue = arg.newValue, oldValue = arg.oldValue;
          return _this.handleDisabledPackagesDidChange(newValue, oldValue);
        };
      })(this)));
      snippets = this;
      return this.subscriptions.add(atom.commands.add('atom-text-editor', {
        'snippets:expand': function(event) {
          var editor;
          editor = this.getModel();
          if (snippets.snippetToExpandUnderCursor(editor)) {
            snippets.clearExpansions(editor);
            return snippets.expandSnippetsUnderCursors(editor);
          } else {
            return event.abortKeyBinding();
          }
        },
        'snippets:next-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToNextTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:previous-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToPreviousTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:available': function(event) {
          var SnippetsAvailable, editor;
          editor = this.getModel();
          SnippetsAvailable = require('./snippets-available');
          if (snippets.availableSnippetsView == null) {
            snippets.availableSnippetsView = new SnippetsAvailable(snippets);
          }
          return snippets.availableSnippetsView.toggle(editor);
        }
      }));
    },
    deactivate: function() {
      var ref1;
      if ((ref1 = this.emitter) != null) {
        ref1.dispose();
      }
      this.emitter = null;
      this.editorSnippetExpansions = null;
      return atom.config.transact((function(_this) {
        return function() {
          return _this.subscriptions.dispose();
        };
      })(this));
    },
    getUserSnippetsPath: function() {
      if (this.userSnippetsPath != null) {
        return this.userSnippetsPath;
      }
      this.userSnippetsPath = CSON.resolve(path.join(atom.getConfigDirPath(), 'snippets'));
      if (this.userSnippetsPath == null) {
        this.userSnippetsPath = path.join(atom.getConfigDirPath(), 'snippets.cson');
      }
      return this.userSnippetsPath;
    },
    loadAll: function() {
      return this.loadBundledSnippets((function(_this) {
        return function(bundledSnippets) {
          return _this.loadPackageSnippets(function(packageSnippets) {
            return _this.loadUserSnippets(function(userSnippets) {
              atom.config.transact(function() {
                var filepath, i, len, ref1, snippetSet, snippetsBySelector;
                ref1 = [bundledSnippets, packageSnippets, userSnippets];
                for (i = 0, len = ref1.length; i < len; i++) {
                  snippetSet = ref1[i];
                  for (filepath in snippetSet) {
                    snippetsBySelector = snippetSet[filepath];
                    _this.add(filepath, snippetsBySelector);
                  }
                }
              });
              return _this.doneLoading();
            });
          });
        };
      })(this));
    },
    loadBundledSnippets: function(callback) {
      var bundledSnippetsPath;
      bundledSnippetsPath = CSON.resolve(path.join(getPackageRoot(), 'lib', 'snippets'));
      return this.loadSnippetsFile(bundledSnippetsPath, function(snippets) {
        var snippetsByPath;
        snippetsByPath = {};
        snippetsByPath[bundledSnippetsPath] = snippets;
        return callback(snippetsByPath);
      });
    },
    loadUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          if (stat != null ? stat.isFile() : void 0) {
            return _this.loadSnippetsFile(userSnippetsPath, function(snippets) {
              var result;
              result = {};
              result[userSnippetsPath] = snippets;
              return callback(result);
            });
          } else {
            return callback({});
          }
        };
      })(this));
    },
    watchUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          var e, message, userSnippetsFile, userSnippetsFileDisposable;
          if (stat != null ? stat.isFile() : void 0) {
            userSnippetsFileDisposable = new CompositeDisposable();
            userSnippetsFile = new File(userSnippetsPath);
            try {
              userSnippetsFileDisposable.add(userSnippetsFile.onDidChange(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidDelete(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidRename(function() {
                return _this.handleUserSnippetsDidChange();
              }));
            } catch (error1) {
              e = error1;
              message = "Unable to watch path: `snippets.cson`. Make sure you have permissions\nto the `~/.atom` directory and `" + userSnippetsPath + "`.\n\nOn linux there are currently problems with watch sizes. See\n[this document][watches] for more info.\n[watches]:https://github.com/atom/atom/blob/master/docs/build-instructions/linux.md#typeerror-unable-to-watch-path";
              atom.notifications.addError(message, {
                dismissable: true
              });
            }
            return callback(userSnippetsFileDisposable);
          } else {
            return callback(new Disposable(function() {}));
          }
        };
      })(this));
    },
    handleUserSnippetsDidChange: function() {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return atom.config.transact((function(_this) {
        return function() {
          _this.clearSnippetsForPath(userSnippetsPath);
          return _this.loadSnippetsFile(userSnippetsPath, function(result) {
            return _this.add(userSnippetsPath, result);
          });
        };
      })(this));
    },
    handleDisabledPackagesDidChange: function(newDisabledPackages, oldDisabledPackages) {
      var i, j, len, len1, p, packagesToAdd, packagesToRemove;
      packagesToAdd = [];
      packagesToRemove = [];
      if (oldDisabledPackages == null) {
        oldDisabledPackages = [];
      }
      if (newDisabledPackages == null) {
        newDisabledPackages = [];
      }
      for (i = 0, len = oldDisabledPackages.length; i < len; i++) {
        p = oldDisabledPackages[i];
        if (!newDisabledPackages.includes(p)) {
          packagesToAdd.push(p);
        }
      }
      for (j = 0, len1 = newDisabledPackages.length; j < len1; j++) {
        p = newDisabledPackages[j];
        if (!oldDisabledPackages.includes(p)) {
          packagesToRemove.push(p);
        }
      }
      return atom.config.transact((function(_this) {
        return function() {
          var k, l, len2, len3, results1;
          for (k = 0, len2 = packagesToRemove.length; k < len2; k++) {
            p = packagesToRemove[k];
            _this.removeSnippetsForPackage(p);
          }
          results1 = [];
          for (l = 0, len3 = packagesToAdd.length; l < len3; l++) {
            p = packagesToAdd[l];
            results1.push(_this.addSnippetsForPackage(p));
          }
          return results1;
        };
      })(this));
    },
    addSnippetsForPackage: function(packageName) {
      var filePath, results1, snippetSet, snippetsBySelector;
      snippetSet = this.snippetsByPackage.get(packageName);
      results1 = [];
      for (filePath in snippetSet) {
        snippetsBySelector = snippetSet[filePath];
        results1.push(this.add(filePath, snippetsBySelector));
      }
      return results1;
    },
    removeSnippetsForPackage: function(packageName) {
      var filePath, results1, snippetSet, snippetsBySelector;
      snippetSet = this.snippetsByPackage.get(packageName);
      this.addSnippetsInDisabledPackage(snippetSet);
      results1 = [];
      for (filePath in snippetSet) {
        snippetsBySelector = snippetSet[filePath];
        results1.push(this.clearSnippetsForPath(filePath));
      }
      return results1;
    },
    loadPackageSnippets: function(callback) {
      var disabledPackageNames, pack, packages, snippetsDirPaths;
      disabledPackageNames = atom.config.get('core.packagesWithSnippetsDisabled') || [];
      packages = atom.packages.getLoadedPackages().sort(function(pack, b) {
        if (/\/app\.asar\/node_modules\//.test(pack.path)) {
          return -1;
        } else {
          return 1;
        }
      });
      snippetsDirPaths = (function() {
        var i, len, results1;
        results1 = [];
        for (i = 0, len = packages.length; i < len; i++) {
          pack = packages[i];
          results1.push(path.join(pack.path, 'snippets'));
        }
        return results1;
      })();
      return async.map(snippetsDirPaths, this.loadSnippetsDirectory.bind(this), (function(_this) {
        return function(error, results) {
          var enabledPackages, i, key, len, o, result, zipped;
          zipped = (function() {
            var results1;
            results1 = [];
            for (key in results) {
              result = results[key];
              results1.push({
                result: result,
                pack: packages[key]
              });
            }
            return results1;
          })();
          enabledPackages = [];
          for (i = 0, len = zipped.length; i < len; i++) {
            o = zipped[i];
            if (Object.keys(o.result).length === 0) {
              continue;
            }
            _this.snippetsByPackage.set(o.pack.name, o.result);
            if (disabledPackageNames.includes(o.pack.name)) {
              _this.addSnippetsInDisabledPackage(o.result);
            } else {
              enabledPackages.push(o.result);
            }
          }
          return callback(_.extend.apply(_, [{}].concat(slice.call(enabledPackages))));
        };
      })(this));
    },
    doneLoading: function() {
      this.loaded = true;
      return this.getEmitter().emit('did-load-snippets');
    },
    onDidLoadSnippets: function(callback) {
      return this.getEmitter().on('did-load-snippets', callback);
    },
    getEmitter: function() {
      return this.emitter != null ? this.emitter : this.emitter = new Emitter;
    },
    loadSnippetsDirectory: function(snippetsDirPath, callback) {
      return fs.isDirectory(snippetsDirPath, (function(_this) {
        return function(isDirectory) {
          if (!isDirectory) {
            return callback(null, {});
          }
          return fs.readdir(snippetsDirPath, function(error, entries) {
            if (error) {
              console.warn("Error reading snippets directory " + snippetsDirPath, error);
              return callback(null, {});
            }
            return async.map(entries, function(entry, done) {
              var filePath;
              filePath = path.join(snippetsDirPath, entry);
              return _this.loadSnippetsFile(filePath, function(snippets) {
                return done(null, {
                  filePath: filePath,
                  snippets: snippets
                });
              });
            }, function(error, results) {
              var filePath, i, len, ref1, snippets, snippetsByPath;
              snippetsByPath = {};
              for (i = 0, len = results.length; i < len; i++) {
                ref1 = results[i], filePath = ref1.filePath, snippets = ref1.snippets;
                snippetsByPath[filePath] = snippets;
              }
              return callback(null, snippetsByPath);
            });
          });
        };
      })(this));
    },
    loadSnippetsFile: function(filePath, callback) {
      if (!CSON.isObjectPath(filePath)) {
        return callback({});
      }
      return CSON.readFile(filePath, {
        allowDuplicateKeys: false
      }, function(error, object) {
        var ref1;
        if (object == null) {
          object = {};
        }
        if (error != null) {
          console.warn("Error reading snippets file '" + filePath + "': " + ((ref1 = error.stack) != null ? ref1 : error));
          atom.notifications.addError("Failed to load snippets from '" + filePath + "'", {
            detail: error.message,
            dismissable: true
          });
        }
        return callback(object);
      });
    },
    add: function(filePath, snippetsBySelector, isDisabled) {
      var attributes, body, name, prefix, selector, snippetsByName, unparsedSnippetsByPrefix;
      if (isDisabled == null) {
        isDisabled = false;
      }
      for (selector in snippetsBySelector) {
        snippetsByName = snippetsBySelector[selector];
        unparsedSnippetsByPrefix = {};
        for (name in snippetsByName) {
          attributes = snippetsByName[name];
          prefix = attributes.prefix, body = attributes.body;
          attributes.name = name;
          attributes.id = this.snippetIdCounter++;
          if (typeof body === 'string') {
            unparsedSnippetsByPrefix[prefix] = attributes;
          } else if (body == null) {
            unparsedSnippetsByPrefix[prefix] = null;
          }
        }
        this.storeUnparsedSnippets(unparsedSnippetsByPrefix, filePath, selector, isDisabled);
      }
    },
    addSnippetsInDisabledPackage: function(bundle) {
      var filePath, results1, snippetsBySelector;
      results1 = [];
      for (filePath in bundle) {
        snippetsBySelector = bundle[filePath];
        results1.push(this.add(filePath, snippetsBySelector, true));
      }
      return results1;
    },
    getScopeChain: function(object) {
      var scopesArray;
      scopesArray = object != null ? typeof object.getScopesArray === "function" ? object.getScopesArray() : void 0 : void 0;
      if (scopesArray == null) {
        scopesArray = object;
      }
      return scopesArray.map(function(scope) {
        if (scope[0] !== '.') {
          scope = "." + scope;
        }
        return scope;
      }).join(' ');
    },
    storeUnparsedSnippets: function(value, path, selector, isDisabled) {
      var store, unparsedSnippets;
      if (isDisabled == null) {
        isDisabled = false;
      }
      unparsedSnippets = {};
      unparsedSnippets[selector] = {
        "snippets": value
      };
      store = isDisabled ? this.disabledSnippetsScopedPropertyStore : this.scopedPropertyStore;
      return store.addProperties(path, unparsedSnippets, {
        priority: this.priorityForSource(path)
      });
    },
    clearSnippetsForPath: function(path) {
      var attributes, prefix, ref1, scopeSelector;
      for (scopeSelector in this.scopedPropertyStore.propertiesForSource(path)) {
        ref1 = this.scopedPropertyStore.propertiesForSourceAndSelector(path, scopeSelector);
        for (prefix in ref1) {
          attributes = ref1[prefix];
          this.parsedSnippetsById["delete"](attributes.id);
        }
        this.scopedPropertyStore.removePropertiesForSourceAndSelector(path, scopeSelector);
      }
    },
    parsedSnippetsForScopes: function(scopeDescriptor) {
      var attributes, base, legacyScopeDescriptor, prefix, snippets, unparsedLegacySnippetsByPrefix, unparsedSnippetsByPrefix;
      unparsedSnippetsByPrefix = this.scopedPropertyStore.getPropertyValue(this.getScopeChain(scopeDescriptor), "snippets");
      legacyScopeDescriptor = typeof (base = atom.config).getLegacyScopeDescriptorForNewScopeDescriptor === "function" ? base.getLegacyScopeDescriptorForNewScopeDescriptor(scopeDescriptor) : void 0;
      if (legacyScopeDescriptor != null) {
        unparsedLegacySnippetsByPrefix = this.scopedPropertyStore.getPropertyValue(this.getScopeChain(legacyScopeDescriptor), "snippets");
      }
      snippets = {};
      if (unparsedSnippetsByPrefix != null) {
        for (prefix in unparsedSnippetsByPrefix) {
          attributes = unparsedSnippetsByPrefix[prefix];
          if (typeof (attributes != null ? attributes.body : void 0) !== 'string') {
            continue;
          }
          snippets[prefix] = this.getParsedSnippet(attributes);
        }
      }
      if (unparsedLegacySnippetsByPrefix != null) {
        for (prefix in unparsedLegacySnippetsByPrefix) {
          attributes = unparsedLegacySnippetsByPrefix[prefix];
          if (snippets[prefix] != null) {
            continue;
          }
          if (typeof (attributes != null ? attributes.body : void 0) !== 'string') {
            continue;
          }
          snippets[prefix] = this.getParsedSnippet(attributes);
        }
      }
      return snippets;
    },
    getParsedSnippet: function(attributes) {
      var body, bodyTree, description, descriptionMoreURL, id, leftLabel, leftLabelHTML, name, prefix, rightLabelHTML, snippet;
      snippet = this.parsedSnippetsById.get(attributes.id);
      if (snippet == null) {
        id = attributes.id, prefix = attributes.prefix, name = attributes.name, body = attributes.body, bodyTree = attributes.bodyTree, description = attributes.description, descriptionMoreURL = attributes.descriptionMoreURL, rightLabelHTML = attributes.rightLabelHTML, leftLabel = attributes.leftLabel, leftLabelHTML = attributes.leftLabelHTML;
        if (bodyTree == null) {
          bodyTree = this.getBodyParser().parse(body);
        }
        snippet = new Snippet({
          id: id,
          name: name,
          prefix: prefix,
          bodyTree: bodyTree,
          description: description,
          descriptionMoreURL: descriptionMoreURL,
          rightLabelHTML: rightLabelHTML,
          leftLabel: leftLabel,
          leftLabelHTML: leftLabelHTML,
          bodyText: body
        });
        this.parsedSnippetsById.set(attributes.id, snippet);
      }
      return snippet;
    },
    priorityForSource: function(source) {
      if (source === this.getUserSnippetsPath()) {
        return 1000;
      } else {
        return 0;
      }
    },
    getBodyParser: function() {
      return this.bodyParser != null ? this.bodyParser : this.bodyParser = require('./snippet-body-parser');
    },
    getPrefixText: function(snippets, editor) {
      var cursor, cursorSnippetPrefix, cursorWordPrefix, i, len, position, prefixStart, ref1, ref2, snippetPrefix, wordPrefix, wordRegex, wordStart;
      wordRegex = this.wordRegexForSnippets(snippets);
      ref1 = [], snippetPrefix = ref1[0], wordPrefix = ref1[1];
      ref2 = editor.getCursors();
      for (i = 0, len = ref2.length; i < len; i++) {
        cursor = ref2[i];
        position = cursor.getBufferPosition();
        prefixStart = cursor.getBeginningOfCurrentWordBufferPosition({
          wordRegex: wordRegex
        });
        cursorSnippetPrefix = editor.getTextInRange([prefixStart, position]);
        if ((snippetPrefix != null) && cursorSnippetPrefix !== snippetPrefix) {
          return null;
        }
        snippetPrefix = cursorSnippetPrefix;
        wordStart = cursor.getBeginningOfCurrentWordBufferPosition();
        cursorWordPrefix = editor.getTextInRange([wordStart, position]);
        if ((wordPrefix != null) && cursorWordPrefix !== wordPrefix) {
          return null;
        }
        wordPrefix = cursorWordPrefix;
      }
      return {
        snippetPrefix: snippetPrefix,
        wordPrefix: wordPrefix
      };
    },
    wordRegexForSnippets: function(snippets) {
      var character, i, len, prefix, prefixCharacters, prefixes;
      prefixes = {};
      for (prefix in snippets) {
        for (i = 0, len = prefix.length; i < len; i++) {
          character = prefix[i];
          prefixes[character] = true;
        }
      }
      prefixCharacters = Object.keys(prefixes).join('');
      return new RegExp("[" + (_.escapeRegExp(prefixCharacters)) + "]+");
    },
    snippetForPrefix: function(snippets, prefix, wordPrefix) {
      var longestPrefixMatch, snippet, snippetPrefix;
      longestPrefixMatch = null;
      for (snippetPrefix in snippets) {
        snippet = snippets[snippetPrefix];
        if (prefix.endsWith(snippetPrefix) && wordPrefix.length <= snippetPrefix.length) {
          if ((longestPrefixMatch == null) || snippetPrefix.length > longestPrefixMatch.prefix.length) {
            longestPrefixMatch = snippet;
          }
        }
      }
      return longestPrefixMatch;
    },
    getSnippets: function(editor) {
      return this.parsedSnippetsForScopes(editor.getLastCursor().getScopeDescriptor());
    },
    snippetToExpandUnderCursor: function(editor) {
      var prefixData, snippets;
      if (!editor.getLastSelection().isEmpty()) {
        return false;
      }
      snippets = this.getSnippets(editor);
      if (_.isEmpty(snippets)) {
        return false;
      }
      if (prefixData = this.getPrefixText(snippets, editor)) {
        return this.snippetForPrefix(snippets, prefixData.snippetPrefix, prefixData.wordPrefix);
      }
    },
    expandSnippetsUnderCursors: function(editor) {
      var snippet;
      if (!(snippet = this.snippetToExpandUnderCursor(editor))) {
        return false;
      }
      this.getStore(editor).observeHistory({
        undo: (function(_this) {
          return function(event) {
            return _this.onUndoOrRedo(editor, event, true);
          };
        })(this),
        redo: (function(_this) {
          return function(event) {
            return _this.onUndoOrRedo(editor, event, false);
          };
        })(this)
      });
      this.findOrCreateMarkerLayer(editor);
      editor.transact((function(_this) {
        return function() {
          var cursor, cursorPosition, cursors, i, len, startPoint;
          cursors = editor.getCursors();
          for (i = 0, len = cursors.length; i < len; i++) {
            cursor = cursors[i];
            cursorPosition = cursor.getBufferPosition();
            startPoint = cursorPosition.translate([0, -snippet.prefix.length], [0, 0]);
            cursor.selection.setBufferRange([startPoint, cursorPosition]);
            _this.insert(snippet, editor, cursor);
          }
        };
      })(this));
      return true;
    },
    goToNextTabStop: function(editor) {
      var expansion, i, len, nextTabStopVisited, ref1;
      nextTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToNextTabStop() : void 0) {
          nextTabStopVisited = true;
        }
      }
      return nextTabStopVisited;
    },
    goToPreviousTabStop: function(editor) {
      var expansion, i, len, previousTabStopVisited, ref1;
      previousTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToPreviousTabStop() : void 0) {
          previousTabStopVisited = true;
        }
      }
      return previousTabStopVisited;
    },
    getStore: function(editor) {
      return EditorStore.findOrCreate(editor);
    },
    createMarkerLayer: function(editor) {
      return this.editorMarkerLayers.set(editor, editor.addMarkerLayer({
        maintainHistory: true
      }));
    },
    findOrCreateMarkerLayer: function(editor) {
      var layer;
      layer = this.editorMarkerLayers.get(editor);
      if (layer == null) {
        layer = editor.addMarkerLayer({
          maintainHistory: true
        });
        this.editorMarkerLayers.set(editor, layer);
      }
      return layer;
    },
    getExpansions: function(editor) {
      return this.getStore(editor).getExpansions();
    },
    clearExpansions: function(editor) {
      var store;
      store = this.getStore(editor);
      store.clearExpansions();
      store.stopObserving();
      return store.stopObservingHistory();
    },
    addExpansion: function(editor, snippetExpansion) {
      return this.getStore(editor).addExpansion(snippetExpansion);
    },
    textChanged: function(editor, event) {
      var activeExpansions, store;
      store = this.getStore(editor);
      activeExpansions = store.getExpansions();
      if (activeExpansions.length === 0 || activeExpansions[0].isIgnoringBufferChanges) {
        return;
      }
      this.ignoringTextChangesForEditor(editor, function() {
        return editor.transact(function() {
          var expansion, i, len, results1;
          results1 = [];
          for (i = 0, len = activeExpansions.length; i < len; i++) {
            expansion = activeExpansions[i];
            results1.push(expansion.textChanged(event));
          }
          return results1;
        });
      });
      return this.makeCheckpoint(editor);
    },
    ignoringTextChangesForEditor: function(editor, callback) {
      this.stopObservingEditor(editor);
      callback();
      return this.observeEditor(editor);
    },
    observeEditor: function(editor) {
      return this.getStore(editor).observe((function(_this) {
        return function(event) {
          return _this.textChanged(editor, event);
        };
      })(this));
    },
    stopObservingEditor: function(editor) {
      return this.getStore(editor).stopObserving();
    },
    makeCheckpoint: function(editor) {
      return this.getStore(editor).makeCheckpoint();
    },
    insert: function(snippet, editor, cursor) {
      var bodyTree;
      if (editor == null) {
        editor = atom.workspace.getActiveTextEditor();
      }
      if (cursor == null) {
        cursor = editor.getLastCursor();
      }
      if (typeof snippet === 'string') {
        bodyTree = this.getBodyParser().parse(snippet);
        snippet = new Snippet({
          name: '__anonymous',
          prefix: '',
          bodyTree: bodyTree,
          bodyText: snippet
        });
      }
      return new SnippetExpansion(snippet, editor, cursor, this);
    },
    getUnparsedSnippets: function() {
      var iterate, results;
      results = [];
      iterate = function(sets) {
        var i, item, len, newItem, results1;
        results1 = [];
        for (i = 0, len = sets.length; i < len; i++) {
          item = sets[i];
          newItem = _.deepClone(item);
          newItem.selectorString = item.selector.toString();
          results1.push(results.push(newItem));
        }
        return results1;
      };
      iterate(this.scopedPropertyStore.propertySets);
      iterate(this.disabledSnippetsScopedPropertyStore.propertySets);
      return results;
    },
    provideSnippets: function() {
      return {
        bundledSnippetsLoaded: (function(_this) {
          return function() {
            return _this.loaded;
          };
        })(this),
        insertSnippet: this.insert.bind(this),
        snippetsForScopes: this.parsedSnippetsForScopes.bind(this),
        getUnparsedSnippets: this.getUnparsedSnippets.bind(this),
        getUserSnippetsPath: this.getUserSnippetsPath.bind(this)
      };
    },
    onUndoOrRedo: function(editor, isUndo) {
      var activeExpansions, expansion, i, len, results1;
      activeExpansions = this.getExpansions(editor);
      results1 = [];
      for (i = 0, len = activeExpansions.length; i < len; i++) {
        expansion = activeExpansions[i];
        results1.push(expansion.onUndoOrRedo(isUndo));
      }
      return results1;
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
