"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _atom = require("atom");

var _eventKit = require("event-kit");

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _lodash = _interopRequireDefault(require("lodash.memoize"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _octicon = _interopRequireDefault(require("../atom/octicon"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  dialog
} = _electron.remote;
const genArray = (0, _lodash.default)(function genArray(interval, count) {
  const arr = [];

  for (let i = 1; i <= count; i++) {
    arr.push(interval * i);
  }

  return arr;
}, (interval, count) => `${interval}:${count}`);

class Marker {
  static deserialize(data) {
    const marker = new Marker(data.label, () => {});
    marker.end = data.end;
    marker.markers = data.markers;
    return marker;
  }

  constructor(label, didUpdate) {
    this.label = label;
    this.didUpdate = didUpdate;
    this.end = null;
    this.markers = [];
  }

  getStart() {
    return this.markers.length ? this.markers[0].start : null;
  }

  getEnd() {
    return this.end;
  }

  mark(sectionName, start) {
    this.markers.push({
      name: sectionName,
      start: start || performance.now()
    });
  }

  finalize() {
    this.end = performance.now();
    this.didUpdate();
  }

  getTimings() {
    return this.markers.map((timing, idx, ary) => {
      const next = ary[idx + 1];
      const end = next ? next.start : this.getEnd();
      return _objectSpread({}, timing, {
        end
      });
    });
  }

  serialize() {
    return {
      label: this.label,
      end: this.end,
      markers: this.markers.slice()
    };
  }

}

class MarkerTooltip extends _react.default.Component {
  render() {
    const {
      marker
    } = this.props;
    const timings = marker.getTimings();
    return _react.default.createElement("div", {
      style: {
        textAlign: 'left',
        maxWidth: 300,
        whiteSpace: 'initial'
      }
    }, _react.default.createElement("strong", null, _react.default.createElement("tt", null, marker.label)), _react.default.createElement("ul", {
      style: {
        paddingLeft: 20,
        marginTop: 10
      }
    }, timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return _react.default.createElement("li", {
        key: name
      }, name, ": ", Math.floor(duration * 100) / 100, "ms");
    })));
  }

}

_defineProperty(MarkerTooltip, "propTypes", {
  marker: _propTypes.default.instanceOf(Marker).isRequired
});

const COLORS = {
  queued: 'red',
  prepare: 'cyan',
  nexttick: 'yellow',
  execute: 'green',
  ipc: 'pink'
};

class MarkerSpan extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleMouseOver', 'handleMouseOut');
  }

  render() {
    const _this$props = this.props,
          {
      marker
    } = _this$props,
          others = _objectWithoutProperties(_this$props, ["marker"]);

    const timings = marker.getTimings();
    const totalTime = marker.getEnd() - marker.getStart();
    const percentages = timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return {
        color: COLORS[name],
        percent: duration / totalTime * 100
      };
    });
    return _react.default.createElement("span", _extends({}, others, {
      ref: c => {
        this.element = c;
      },
      onMouseOver: this.handleMouseOver,
      onMouseOut: this.handleMouseOut
    }), percentages.map(({
      color,
      percent
    }, i) => {
      const style = {
        width: `${percent}%`,
        background: color
      };
      return _react.default.createElement("span", {
        className: "waterfall-marker-section",
        key: i,
        style: style
      });
    }));
  }

  handleMouseOver(e) {
    const elem = document.createElement('div');

    _reactDom.default.render(_react.default.createElement(MarkerTooltip, {
      marker: this.props.marker
    }), elem);

    this.tooltipDisposable = atom.tooltips.add(this.element, {
      item: elem,
      placement: 'auto bottom',
      trigger: 'manual'
    });
  }

  closeTooltip() {
    this.tooltipDisposable && this.tooltipDisposable.dispose();
    this.tooltipDisposable = null;
  }

  handleMouseOut(e) {
    this.closeTooltip();
  }

  componentWillUnmount() {
    this.closeTooltip();
  }

}

_defineProperty(MarkerSpan, "propTypes", {
  marker: _propTypes.default.instanceOf(Marker).isRequired
});

class Waterfall extends _react.default.Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'renderMarker');
    this.state = this.getNextState(props);
  }

  componentWillReceiveProps(nextProps) {
    this.setState(this.getNextState(nextProps));
  }

  getNextState(props) {
    const {
      markers
    } = props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const totalDuration = endTime - startTime;
    let timelineMarkInterval = null;

    if (props.zoomFactor <= 0.15) {
      timelineMarkInterval = 1000;
    } else if (props.zoomFactor <= 0.3) {
      timelineMarkInterval = 500;
    } else if (props.zoomFactor <= 0.6) {
      timelineMarkInterval = 250;
    } else {
      timelineMarkInterval = 100;
    }

    const timelineMarks = genArray(timelineMarkInterval, Math.ceil(totalDuration / timelineMarkInterval));
    return {
      firstMarker,
      lastMarker,
      startTime,
      endTime,
      totalDuration,
      timelineMarks
    };
  }

  render() {
    return _react.default.createElement("div", {
      className: "waterfall-scroller"
    }, _react.default.createElement("div", {
      className: "waterfall-container"
    }, this.renderTimeMarkers(), this.renderTimeline(), this.props.markers.map(this.renderMarker)));
  }

  renderTimeline() {
    return _react.default.createElement("div", {
      className: "waterfall-timeline"
    }, "\xA0", this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react.default.createElement("span", {
        className: "waterfall-timeline-label",
        style: style,
        key: `tl:${time}`
      }, time, "ms");
    }));
  }

  renderTimeMarkers() {
    return _react.default.createElement("div", {
      className: "waterfall-time-markers"
    }, this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react.default.createElement("span", {
        className: "waterfall-time-marker",
        style: style,
        key: `tm:${time}`
      });
    }));
  }

  renderMarker(marker, i) {
    if (marker.getStart() === null || marker.getEnd() === null) {
      return _react.default.createElement("div", {
        key: i
      });
    }

    const startOffset = marker.getStart() - this.state.startTime;
    const duration = marker.getEnd() - marker.getStart();
    const markerStyle = {
      left: startOffset * this.props.zoomFactor,
      width: duration * this.props.zoomFactor
    };
    return _react.default.createElement("div", {
      className: "waterfall-row",
      key: i
    }, _react.default.createElement("span", {
      className: "waterfall-row-label",
      style: {
        paddingLeft: markerStyle.left + markerStyle.width
      }
    }, marker.label), _react.default.createElement(MarkerSpan, {
      className: "waterfall-marker",
      style: markerStyle,
      marker: marker
    }));
  }

}

_defineProperty(Waterfall, "propTypes", {
  markers: _propTypes.default.arrayOf(_propTypes.default.instanceOf(Marker)).isRequired,
  zoomFactor: _propTypes.default.number.isRequired
});

class WaterfallWidget extends _react.default.Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'handleZoomFactorChange', 'handleCollapseClick', 'handleExportClick');
    this.state = {
      zoomFactor: 0.3,
      collapsed: false
    };
  }

  render() {
    const {
      markers
    } = this.props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const duration = endTime - startTime;
    return _react.default.createElement("div", {
      className: "waterfall-widget inset-pannel"
    }, _react.default.createElement("div", {
      className: "waterfall-header"
    }, _react.default.createElement("div", {
      className: "waterfall-header-text"
    }, _react.default.createElement("span", {
      onClick: this.handleCollapseClick,
      className: "collapse-toggle"
    }, this.state.collapsed ? '\u25b6' : '\u25bc'), this.props.markers.length, " event(s) over ", Math.floor(duration), "ms"), _react.default.createElement("div", {
      className: "waterfall-header-controls"
    }, _react.default.createElement("button", {
      className: "waterfall-export-button btn btn-sm",
      onClick: this.handleExportClick
    }, "Export"), _react.default.createElement(_octicon.default, {
      icon: "search"
    }), _react.default.createElement("input", {
      type: "range",
      className: "input-range",
      min: 0.1,
      max: 1,
      step: 0.01,
      value: this.state.zoomFactor,
      onChange: this.handleZoomFactorChange
    }))), this.state.collapsed ? null : _react.default.createElement(Waterfall, {
      markers: this.props.markers,
      zoomFactor: this.state.zoomFactor
    }));
  }

  handleZoomFactorChange(e) {
    this.setState({
      zoomFactor: parseFloat(e.target.value)
    });
  }

  handleCollapseClick(e) {
    this.setState(s => ({
      collapsed: !s.collapsed
    }));
  }

  handleExportClick(e) {
    e.preventDefault();
    const json = JSON.stringify(this.props.markers.map(m => m.serialize()), null, '  ');
    const buffer = new _atom.TextBuffer({
      text: json
    });
    dialog.showSaveDialog({
      defaultPath: 'git-timings.json'
    }, filename => {
      if (!filename) {
        return;
      }

      buffer.saveAs(filename);
    });
  }

}

_defineProperty(WaterfallWidget, "propTypes", {
  markers: _propTypes.default.arrayOf(_propTypes.default.instanceOf(Marker)).isRequired
});

let markers = null;
let groupId = 0;
const groups = [];
let lastMarkerTime = null;
let updateTimer = null;

class GitTimingsView extends _react.default.Component {
  static buildURI() {
    return this.uriPattern;
  }

  static generateMarker(label) {
    const marker = new Marker(label, () => {
      GitTimingsView.scheduleUpdate();
    });
    const now = performance.now();

    if (!markers || lastMarkerTime && Math.abs(now - lastMarkerTime) >= 5000) {
      groupId++;
      markers = [];
      groups.unshift({
        id: groupId,
        markers
      });

      if (groups.length > 100) {
        groups.pop();
      }
    }

    lastMarkerTime = now;
    markers.push(marker);
    GitTimingsView.scheduleUpdate();
    return marker;
  }

  static restoreGroup(group) {
    groupId++;
    groups.unshift({
      id: groupId,
      markers: group
    });
    GitTimingsView.scheduleUpdate(true);
  }

  static scheduleUpdate(immediate = false) {
    if (updateTimer) {
      clearTimeout(updateTimer);
    }

    updateTimer = setTimeout(() => {
      GitTimingsView.emitter.emit('did-update');
    }, immediate ? 0 : 1000);
  }

  static onDidUpdate(callback) {
    return GitTimingsView.emitter.on('did-update', callback);
  }

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleImportClick');
  }

  componentDidMount() {
    this.subscriptions = new _eventKit.CompositeDisposable(GitTimingsView.onDidUpdate(() => this.forceUpdate()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  render() {
    return _react.default.createElement("div", {
      className: "github-GitTimingsView"
    }, _react.default.createElement("div", {
      className: "github-GitTimingsView-header"
    }, _react.default.createElement("button", {
      className: "import-button btn",
      onClick: this.handleImportClick
    }, "Import")), groups.map((group, idx) => _react.default.createElement(WaterfallWidget, {
      key: group.id,
      markers: group.markers
    })));
  }

  handleImportClick(e) {
    e.preventDefault();
    dialog.showOpenDialog({
      properties: ['openFile']
    }, async filenames => {
      if (!filenames) {
        return;
      }

      const filename = filenames[0];

      try {
        const contents = await _fsExtra.default.readFile(filename, {
          encoding: 'utf8'
        });
        const data = JSON.parse(contents);
        const restoredMarkers = data.map(item => Marker.deserialize(item));
        GitTimingsView.restoreGroup(restoredMarkers);
      } catch (_err) {
        atom.notifications.addError(`Could not import timings from ${filename}`);
      }
    });
  }

  serialize() {
    return {
      deserializer: 'GitTimingsView'
    };
  }

  getURI() {
    return this.constructor.buildURI();
  }

  getTitle() {
    return 'GitHub Package Timings View';
  }

}

exports.default = GitTimingsView;

_defineProperty(GitTimingsView, "uriPattern", 'atom-github://debug/timings');

_defineProperty(GitTimingsView, "emitter", new _eventKit.Emitter());
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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