"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);

    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }

    this.known.set(startPath, workDir);
    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra.default.stat(startPath)).isDirectory() ? startPath : _path.default.dirname(startPath); // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Within a gitdir or outside of a worktree, return an empty string.
      // Throw if startDir does not exist.

      const topLevel = await _compositeGitStrategy.default.create(startDir).exec(['rev-parse', '--show-toplevel']);

      if (/\S/.test(topLevel)) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      } // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.


      const gitDir = await _compositeGitStrategy.default.create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path.default.resolve(gitDir, '..'));
    } catch (e) {
      /* istanbul ignore if */
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }

      return null;
    }
  }

}

exports.default = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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