"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _eventKit = require("event-kit");

var _keytarStrategy = require("../shared/keytar-strategy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

let instance = null;

class GithubLoginModel {
  // Be sure that we're requesting at least this many scopes on the token we grant through github.atom.io or we'll
  // give everyone a really frustrating experience ;-)
  static get() {
    if (!instance) {
      instance = new GithubLoginModel();
    }

    return instance;
  }

  constructor(Strategy) {
    this._Strategy = Strategy;
    this._strategy = null;
    this.emitter = new _eventKit.Emitter();
    this.checked = new Map();
  }

  async getStrategy() {
    if (this._strategy) {
      return this._strategy;
    }

    if (this._Strategy) {
      this._strategy = new this._Strategy();
      return this._strategy;
    }

    this._strategy = await (0, _keytarStrategy.createStrategy)();
    return this._strategy;
  }

  async getToken(account) {
    const strategy = await this.getStrategy();
    const password = await strategy.getPassword('atom-github', account);

    if (!password || password === _keytarStrategy.UNAUTHENTICATED) {
      // User is not logged in
      return _keytarStrategy.UNAUTHENTICATED;
    }

    if (/^https?:\/\//.test(account)) {
      // Avoid storing tokens in memory longer than necessary. Let's cache token scope checks by storing a set of
      // checksums instead.
      const hash = _crypto.default.createHash('md5');

      hash.update(password);
      const fingerprint = hash.digest('base64');
      const outcome = this.checked.get(fingerprint);

      if (outcome === _keytarStrategy.UNAUTHENTICATED || outcome === _keytarStrategy.INSUFFICIENT) {
        // Cached failure
        return outcome;
      } else if (!outcome) {
        // No cached outcome. Query for scopes.
        try {
          const scopes = await this.getScopes(account, password);

          if (scopes === _keytarStrategy.UNAUTHORIZED) {
            // Password is incorrect. Treat it as though you aren't authenticated at all.
            this.checked.set(fingerprint, _keytarStrategy.UNAUTHENTICATED);
            return _keytarStrategy.UNAUTHENTICATED;
          }

          const scopeSet = new Set(scopes);

          for (const scope of this.constructor.REQUIRED_SCOPES) {
            if (!scopeSet.has(scope)) {
              // Token doesn't have enough OAuth scopes, need to reauthenticate
              this.checked.set(fingerprint, _keytarStrategy.INSUFFICIENT);
              return _keytarStrategy.INSUFFICIENT;
            }
          } // Successfully authenticated and had all required scopes.


          this.checked.set(fingerprint, true);
        } catch (e) {
          // Most likely a network error. Do not cache the failure.
          // eslint-disable-next-line no-console
          console.error(`Unable to validate token scopes against ${account}`, e);
          return _keytarStrategy.UNAUTHENTICATED;
        }
      }
    }

    return password;
  }

  async setToken(account, token) {
    const strategy = await this.getStrategy();
    await strategy.replacePassword('atom-github', account, token);
    this.didUpdate();
  }

  async removeToken(account) {
    const strategy = await this.getStrategy();
    await strategy.deletePassword('atom-github', account);
    this.didUpdate();
  }
  /* istanbul ignore next */


  async getScopes(host, token) {
    if (atom.inSpecMode()) {
      if (token === 'good-token') {
        return this.constructor.REQUIRED_SCOPES;
      }

      throw new Error('Attempt to check token scopes in specs');
    }

    const response = await fetch(host, {
      method: 'HEAD',
      headers: {
        Authorization: `bearer ${token}`
      }
    });

    if (response.status === 401) {
      return _keytarStrategy.UNAUTHORIZED;
    }

    if (response.status !== 200) {
      throw new Error(`Unable to check token for OAuth scopes against ${host}: ${await response.text()}`);
    }

    return response.headers.get('X-OAuth-Scopes').split(/\s*,\s*/);
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }

  onDidUpdate(cb) {
    return this.emitter.on('did-update', cb);
  }

  destroy() {
    this.emitter.dispose();
  }

}

exports.default = GithubLoginModel;

_defineProperty(GithubLoginModel, "REQUIRED_SCOPES", ['repo', 'read:org', 'user:email']);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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