"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _helpers = require("./helpers");

var _workdirCache = _interopRequireDefault(require("./models/workdir-cache"));

var _workdirContext = _interopRequireDefault(require("./models/workdir-context"));

var _workdirContextPool = _interopRequireDefault(require("./models/workdir-context-pool"));

var _repository = _interopRequireDefault(require("./models/repository"));

var _styleCalculator = _interopRequireDefault(require("./models/style-calculator"));

var _githubLoginModel = _interopRequireDefault(require("./models/github-login-model"));

var _rootController = _interopRequireDefault(require("./controllers/root-controller"));

var _stubItem = _interopRequireDefault(require("./items/stub-item"));

var _switchboard = _interopRequireDefault(require("./switchboard"));

var _yardstick = _interopRequireDefault(require("./yardstick"));

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _contextMenuInterceptor = _interopRequireDefault(require("./context-menu-interceptor"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

var _getRepoPipelineManager = _interopRequireDefault(require("./get-repo-pipeline-manager"));

var _reporterProxy = require("./reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const defaultState = {
  newProject: true
};

class GithubPackage {
  constructor({
    workspace,
    project,
    commandRegistry,
    notificationManager,
    tooltips,
    styles,
    grammars,
    keymaps,
    config,
    deserializers,
    confirm,
    getLoadSettings,
    configDirPath,
    renderFn,
    loginModel
  }) {
    (0, _helpers.autobind)(this, 'consumeStatusBar', 'createGitTimingsView', 'createIssueishPaneItemStub', 'createDockItemStub', 'createFilePatchControllerStub', 'destroyGitTabItem', 'destroyGithubTabItem', 'createRepositoryForProjectPath', 'cloneRepositoryForProjectPath', 'getRepositoryForWorkdir', 'scheduleActiveContextUpdate');
    this.workspace = workspace;
    this.project = project;
    this.commandRegistry = commandRegistry;
    this.deserializers = deserializers;
    this.notificationManager = notificationManager;
    this.tooltips = tooltips;
    this.config = config;
    this.styles = styles;
    this.grammars = grammars;
    this.keymaps = keymaps;
    this.configPath = _path.default.join(configDirPath, 'github.cson');
    this.styleCalculator = new _styleCalculator.default(this.styles, this.config);
    this.confirm = confirm;
    this.startOpen = false;
    this.activated = false;
    const criteria = {
      projectPathCount: this.project.getPaths().length,
      initPathCount: (getLoadSettings().initialPaths || []).length
    };
    this.pipelineManager = (0, _getRepoPipelineManager.default)({
      confirm,
      notificationManager,
      workspace
    });
    this.activeContextQueue = new _asyncQueue.default();
    this.guessedContext = _workdirContext.default.guess(criteria, this.pipelineManager);
    this.activeContext = this.guessedContext;
    this.workdirCache = new _workdirCache.default();
    this.contextPool = new _workdirContextPool.default({
      window,
      workspace,
      promptCallback: query => this.controller.promptForCredentials(query),
      pipelineManager: this.pipelineManager
    });
    this.switchboard = new _switchboard.default();
    this.loginModel = loginModel || new _githubLoginModel.default();

    this.renderFn = renderFn || ((component, node, callback) => {
      return _reactDom.default.render(component, node, callback);
    }); // Handle events from all resident contexts.


    this.subscriptions = new _eventKit.CompositeDisposable(this.contextPool.onDidChangeWorkdirOrHead(context => {
      this.refreshAtomGitRepository(context.getWorkingDirectory());
    }), this.contextPool.onDidUpdateRepository(context => {
      this.switchboard.didUpdateRepository(context.getRepository());
    }), this.contextPool.onDidDestroyRepository(context => {
      if (context === this.activeContext) {
        this.setActiveContext(_workdirContext.default.absent({
          pipelineManager: this.pipelineManager
        }));
      }
    }), _contextMenuInterceptor.default);
    this.setupYardstick();
  }

  setupYardstick() {
    const stagingSeries = ['stageLine', 'stageHunk', 'unstageLine', 'unstageHunk'];
    this.subscriptions.add( // Staging and unstaging operations
    this.switchboard.onDidBeginStageOperation(payload => {
      if (payload.stage && payload.line) {
        _yardstick.default.begin('stageLine');
      } else if (payload.stage && payload.hunk) {
        _yardstick.default.begin('stageHunk');
      } else if (payload.stage && payload.file) {
        _yardstick.default.begin('stageFile');
      } else if (payload.stage && payload.mode) {
        _yardstick.default.begin('stageMode');
      } else if (payload.stage && payload.symlink) {
        _yardstick.default.begin('stageSymlink');
      } else if (payload.unstage && payload.line) {
        _yardstick.default.begin('unstageLine');
      } else if (payload.unstage && payload.hunk) {
        _yardstick.default.begin('unstageHunk');
      } else if (payload.unstage && payload.file) {
        _yardstick.default.begin('unstageFile');
      } else if (payload.unstage && payload.mode) {
        _yardstick.default.begin('unstageMode');
      } else if (payload.unstage && payload.symlink) {
        _yardstick.default.begin('unstageSymlink');
      }
    }), this.switchboard.onDidUpdateRepository(() => {
      _yardstick.default.mark(stagingSeries, 'update-repository');
    }), this.switchboard.onDidFinishRender(context => {
      if (context === 'RootController.showFilePatchForPath') {
        _yardstick.default.finish(stagingSeries);
      }
    }), // Active context changes
    this.switchboard.onDidScheduleActiveContextUpdate(() => {
      _yardstick.default.begin('activeContextChange');
    }), this.switchboard.onDidBeginActiveContextUpdate(() => {
      _yardstick.default.mark('activeContextChange', 'queue-wait');
    }), this.switchboard.onDidFinishContextChangeRender(() => {
      _yardstick.default.mark('activeContextChange', 'render');
    }), this.switchboard.onDidFinishActiveContextUpdate(() => {
      _yardstick.default.finish('activeContextChange');
    }));
  }

  async activate(state = {}) {
    this.savedState = _objectSpread({}, defaultState, state);
    const firstRun = !(await (0, _helpers.fileExists)(this.configPath));
    const newProject = this.savedState.firstRun !== undefined ? this.savedState.firstRun : this.savedState.newProject;
    this.startOpen = firstRun || newProject;
    this.startRevealed = firstRun && !this.config.get('welcome.showOnStartup');

    if (firstRun) {
      await _fsExtra.default.writeFile(this.configPath, '# Store non-visible GitHub package state.\n', {
        encoding: 'utf8'
      });
    }

    const hasSelectedFiles = event => {
      return !!event.target.closest('.github-FilePatchListView').querySelector('.is-selected');
    };

    this.subscriptions.add(this.project.onDidChangePaths(this.scheduleActiveContextUpdate), this.workspace.getCenter().onDidChangeActivePaneItem(this.scheduleActiveContextUpdate), this.styleCalculator.startWatching('github-package-styles', ['editor.fontSize', 'editor.fontFamily', 'editor.lineHeight', 'editor.tabLength'], config => `
          .github-HunkView-line {
            font-family: ${config.get('editor.fontFamily')};
            line-height: ${config.get('editor.lineHeight')};
            tab-size: ${config.get('editor.tabLength')}
          }
        `), atom.contextMenu.add({
      '.github-UnstagedChanges .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Discard Changes',
        command: 'github:discard-changes-in-selected-files',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-StagedChanges .github-FilePatchListView': [{
        label: 'Unstage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-MergeConflictPaths .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Ours',
        command: 'github:resolve-file-as-ours',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Theirs',
        command: 'github:resolve-file-as-theirs',
        shouldDisplay: hasSelectedFiles
      }]
    }));
    this.activated = true;
    this.scheduleActiveContextUpdate(this.savedState);
    this.rerender();
  }

  serialize() {
    const activeRepository = this.getActiveRepository();
    const activeRepositoryPath = activeRepository ? activeRepository.getWorkingDirectoryPath() : null;
    return {
      activeRepositoryPath,
      newProject: false
    };
  }

  rerender(callback) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    if (!this.activated) {
      return;
    }

    if (!this.element) {
      this.element = document.createElement('div');
      this.subscriptions.add(new _eventKit.Disposable(() => {
        _reactDom.default.unmountComponentAtNode(this.element);

        delete this.element;
      }));
    }

    this.renderFn(_react.default.createElement(_rootController.default, {
      ref: c => {
        this.controller = c;
      },
      workspace: this.workspace,
      deserializers: this.deserializers,
      commandRegistry: this.commandRegistry,
      notificationManager: this.notificationManager,
      tooltips: this.tooltips,
      grammars: this.grammars,
      keymaps: this.keymaps,
      config: this.config,
      project: this.project,
      confirm: this.confirm,
      workdirContextPool: this.contextPool,
      loginModel: this.loginModel,
      repository: this.getActiveRepository(),
      resolutionProgress: this.getActiveResolutionProgress(),
      statusBar: this.statusBar,
      createRepositoryForProjectPath: this.createRepositoryForProjectPath,
      cloneRepositoryForProjectPath: this.cloneRepositoryForProjectPath,
      switchboard: this.switchboard,
      startOpen: this.startOpen,
      startRevealed: this.startRevealed,
      removeFilePatchItem: this.removeFilePatchItem
    }), this.element, callback);
  }

  async deactivate() {
    this.subscriptions.dispose();
    this.contextPool.clear();

    _workerManager.default.reset(false);

    if (this.guessedContext) {
      this.guessedContext.destroy();
      this.guessedContext = null;
    }

    await _yardstick.default.flush();
  }

  consumeStatusBar(statusBar) {
    this.statusBar = statusBar;
    this.rerender();
  }

  consumeReporter(reporter) {
    _reporterProxy.reporterProxy.setReporter(reporter);
  }

  createGitTimingsView() {
    return _stubItem.default.create('git-timings-view', {
      title: 'GitHub Package Timings View'
    }, _gitTimingsView.default.buildURI());
  }

  createIssueishPaneItemStub({
    uri
  }) {
    return _stubItem.default.create('issueish-detail-item', {
      title: 'Issueish'
    }, uri);
  }

  createDockItemStub({
    uri
  }) {
    let item;

    switch (uri) {
      // always return an empty stub
      // but only set it as the active item for a tab type
      // if it doesn't already exist
      case 'atom-github://dock-item/git':
        item = this.createGitStub(uri);
        this.gitTabStubItem = this.gitTabStubItem || item;
        break;

      case 'atom-github://dock-item/github':
        item = this.createGitHubStub(uri);
        this.githubTabStubItem = this.githubTabStubItem || item;
        break;

      default:
        throw new Error(`Invalid DockItem stub URI: ${uri}`);
    }

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createGitStub(uri) {
    return _stubItem.default.create('git', {
      title: 'Git'
    }, uri);
  }

  createGitHubStub(uri) {
    return _stubItem.default.create('github', {
      title: 'GitHub'
    }, uri);
  }

  createFilePatchControllerStub({
    uri
  } = {}) {
    const item = _stubItem.default.create('git-file-patch-controller', {
      title: 'Diff'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitPreviewStub({
    uri
  }) {
    const item = _stubItem.default.create('git-commit-preview', {
      title: 'Commit preview'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  createCommitDetailStub({
    uri
  }) {
    const item = _stubItem.default.create('git-commit-detail', {
      title: 'Commit'
    }, uri);

    if (this.controller) {
      this.rerender();
    }

    return item;
  }

  destroyGitTabItem() {
    if (this.gitTabStubItem) {
      this.gitTabStubItem.destroy();
      this.gitTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  destroyGithubTabItem() {
    if (this.githubTabStubItem) {
      this.githubTabStubItem.destroy();
      this.githubTabStubItem = null;

      if (this.controller) {
        this.rerender();
      }
    }
  }

  async createRepositoryForProjectPath(projectPath) {
    await _fsExtra.default.mkdirs(projectPath);
    const repository = this.contextPool.add(projectPath).getRepository();
    await repository.init();
    this.workdirCache.invalidate();

    if (!this.project.contains(projectPath)) {
      this.project.addPath(projectPath);
    }

    await this.scheduleActiveContextUpdate();
  }

  async cloneRepositoryForProjectPath(remoteUrl, projectPath) {
    const context = this.contextPool.getContext(projectPath);
    let repository;

    if (context.isPresent()) {
      repository = context.getRepository();
      await repository.clone(remoteUrl);
      repository.destroy();
    } else {
      repository = new _repository.default(projectPath, null, {
        pipelineManager: this.pipelineManager
      });
      await repository.clone(remoteUrl);
    }

    this.workdirCache.invalidate();
    this.project.addPath(projectPath);
    await this.scheduleActiveContextUpdate();
  }

  getRepositoryForWorkdir(projectPath) {
    const loadingGuessRepo = _repository.default.loadingGuess({
      pipelineManager: this.pipelineManager
    });

    return this.guessedContext ? loadingGuessRepo : this.contextPool.getContext(projectPath).getRepository();
  }

  getActiveWorkdir() {
    return this.activeContext.getWorkingDirectory();
  }

  getActiveRepository() {
    return this.activeContext.getRepository();
  }

  getActiveResolutionProgress() {
    return this.activeContext.getResolutionProgress();
  }

  getContextPool() {
    return this.contextPool;
  }

  getSwitchboard() {
    return this.switchboard;
  }

  async scheduleActiveContextUpdate(savedState = {}) {
    this.switchboard.didScheduleActiveContextUpdate();
    await this.activeContextQueue.push(this.updateActiveContext.bind(this, savedState), {
      parallel: false
    });
  }
  /**
   * Derive the git working directory context that should be used for the package's git operations based on the current
   * state of the Atom workspace. In priority, this prefers:
   *
   * - A git working directory that contains the active pane item in the workspace's center.
   * - A git working directory corresponding to a single Project.
   * - When initially activating the package, the working directory that was active when the package was last
   *   serialized.
   * - The current context, unchanged, which may be a `NullWorkdirContext`.
   *
   * First updates the pool of resident contexts to match all git working directories that correspond to open
   * projects and pane items.
   */


  async getNextContext(savedState) {
    const workdirs = new Set((await Promise.all(this.project.getPaths().map(async projectPath => {
      const workdir = await this.workdirCache.find(projectPath);
      return workdir || projectPath;
    }))));

    const fromPaneItem = async maybeItem => {
      const itemPath = pathForPaneItem(maybeItem);

      if (!itemPath) {
        return {};
      }

      const itemWorkdir = await this.workdirCache.find(itemPath);

      if (itemWorkdir && !this.project.contains(itemPath)) {
        workdirs.add(itemWorkdir);
      }

      return {
        itemPath,
        itemWorkdir
      };
    };

    const active = await fromPaneItem(this.workspace.getCenter().getActivePaneItem());
    this.contextPool.set(workdirs, savedState);

    if (active.itemPath) {
      // Prefer an active item
      return this.contextPool.getContext(active.itemWorkdir || active.itemPath);
    }

    if (this.project.getPaths().length === 1) {
      // Single project
      const projectPath = this.project.getPaths()[0];
      const activeWorkingDir = await this.workdirCache.find(projectPath);
      return this.contextPool.getContext(activeWorkingDir || projectPath);
    }

    if (this.project.getPaths().length === 0 && !this.activeContext.getRepository().isUndetermined()) {
      // No projects. Revert to the absent context unless we've guessed that more projects are on the way.
      return _workdirContext.default.absent({
        pipelineManager: this.pipelineManager
      });
    } // Restore models from saved state. Will return a NullWorkdirContext if this path is not presently
    // resident in the pool.


    const savedWorkingDir = savedState.activeRepositoryPath;

    if (savedWorkingDir) {
      return this.contextPool.getContext(savedWorkingDir);
    }

    return this.activeContext;
  }

  setActiveContext(nextActiveContext) {
    if (nextActiveContext !== this.activeContext) {
      if (this.activeContext === this.guessedContext) {
        this.guessedContext.destroy();
        this.guessedContext = null;
      }

      this.activeContext = nextActiveContext;
      this.rerender(() => {
        this.switchboard.didFinishContextChangeRender();
        this.switchboard.didFinishActiveContextUpdate();
      });
    } else {
      this.switchboard.didFinishActiveContextUpdate();
    }
  }

  async updateActiveContext(savedState = {}) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    this.switchboard.didBeginActiveContextUpdate();
    const nextActiveContext = await this.getNextContext(savedState);
    this.setActiveContext(nextActiveContext);
  }

  refreshAtomGitRepository(workdir) {
    const atomGitRepo = this.project.getRepositories().find(repo => {
      return repo && _path.default.normalize(repo.getWorkingDirectory()) === workdir;
    });
    return atomGitRepo ? atomGitRepo.refreshStatus() : Promise.resolve();
  }

}

exports.default = GithubPackage;

function pathForPaneItem(paneItem) {
  if (!paneItem) {
    return null;
  } // Likely GitHub package provided pane item


  if (typeof paneItem.getWorkingDirectory === 'function') {
    return paneItem.getWorkingDirectory();
  } // TextEditor-like


  if (typeof paneItem.getPath === 'function') {
    return paneItem.getPath();
  } // Oh well


  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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