# Examples for ANATPH

# This should be a constructor so that one
# can give "IsPregroupPresentation" for method selection

#! @Chapter Overview
#! @Section Examples

#! @Arguments p,q,r
#! @Returns a pregroup presentation
#! @Description
#! Returns a pregroup presentation for the <M>(<A>p</A>,<A>q</A>,<A>r</A>)</M>-triangle group,
#! the pregroup is the pregroup of the free product of a cyclic group of order <A>p</A> generated
#! by <M>x</M> and a cyclic group of order <A>q</A> generated by <M>y</M> together with the relation
#! <M>(xy)^r</M>.
#! @BeginExample
#! gap> T := TriangleGroup(2,3,7);
#! <pregroup presentation with 3 generators and 1 relators>
#! gap> Pregroup(T);
#! <pregroup with 4 elements in table rep>
#! gap> Relators(T);
#! [ <pregroup relator ([ "2", "3" ])^7> ]
#! gap> T := TriangleGroup(17,22,100);
#! <pregroup presentation with 37 generators and 1 relators>
#! gap> Pregroup(T);
#! <pregroup with 38 elements in table rep>
#! gap> Relators(T);
#! [ <pregroup relator ([ "2", "18" ])^100> ]
#! gap> IsHyperbolic(T, 1/6);
#! true
#! @EndExample
DeclareGlobalFunction("TriangleGroup");

#! @BeginGroup
#! @Arguments m,n
#! @Returns a pregroup presentation
#! @Description
#! Returns a pregroup presentation of the quotient of the <M>(2,3,<A>m</A>)</M>-triangle group
#! by the normal subgroup generated by <M>n</M>th power of the commutator of
#! <M>x</M> and <M>y</M>.
#! The name <Ref Func="TriSH" /> is a synonym for <Ref Func="TriangleCommutatorQuotient" /> provided
#! for backwards compatibility and might be removed in the future.
DeclareGlobalFunction("TriangleCommutatorQuotient");
DeclareGlobalFunction("TriSH");
#! @EndGroup

InstallGlobalFunction(TriSH, {m,n} -> TriangleCommutatorQuotient(m,n));

#! @Arguments p,q,r,len
#! @Returns a pregroup presentation
#! @Description
#! Returns the quotient of the <M>(<A>p</A>,<A>q</A>,<A>r</A>)</M>-triangle group
#! by a random relator of length <A>len</A>.
DeclareGlobalFunction("RandomTriangleQuotient");

#! @Arguments
#! @Returns a pregroup presentation
#! @Description
#! The Jack-Button group, as suggested to me by Alan Logan. It is known 
#! to be hyperbolic, but the tester fails for it. The pregroup is the pregroup
#! of the free group of rank 3 with generators <M>a</M>,<M>b</M>, and <M>t</M>
#! and two
#! relators <M>t^{-1}atb^{-1}a^{-1}</M> and <M>t^{-1}ata^{-1}b^{-1}</M>.
#! @BeginExample
#! gap> J := JackButtonGroup();
#! <pregroup presentation with 6 generators and 2 relators>
#! gap> Pregroup(J);
#! <pregroup of free group of rank 3>
#! gap> Relators(J);
#! [ <pregroup relator TatBA>, <pregroup relator TatAB> ]
#! @EndExample
DeclareGlobalFunction("JackButtonGroup");

#! @Arguments pg, nrel, lrel
#! @Returns a pregroup presentation
#! @Description
#! Returns a pregroup presentation over the given pregroup <A>pg</A> with
#! <A>nrel</A> randomly chosen relators of length <A>lrel</A>.
DeclareGlobalFunction("RandomPregroupPresentation");

#! @Arguments pg, len
#! @Returns a list of integers
#! @Description
#! A random list of pregroup element numbers of the pregroup <A>pregroup</A> of
#! length <A>len</A>. When interpreted as a pregroup word this is cyclically
#! reduced.
DeclareGlobalFunction("RandomPregroupWord");



