"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _helpers = require("./helpers");

const refreshMapPerUniqueId = new WeakMap();

class PeriodicRefresher {
  static getRefreshMap(uniqueId) {
    let refreshMap = refreshMapPerUniqueId.get(uniqueId);

    if (!refreshMap) {
      refreshMap = new Map();
      refreshMapPerUniqueId.set(uniqueId, refreshMap);
    }

    return refreshMap;
  }

  constructor(uniqueId, options) {
    (0, _helpers.autobind)(this, 'refreshNow');
    this.options = options;
    this._refreshesPerId = PeriodicRefresher.getRefreshMap(uniqueId);
  }

  start() {
    if (!this._timer) {
      this._scheduleNext();
    }
  }

  _clearTimer() {
    if (this._timer) {
      clearTimeout(this._timer);
      delete this._timer;
    }
  }

  _scheduleNext() {
    this._timer = setTimeout(this.refreshNow, this.options.interval());
  }

  refreshNow(force = false) {
    const currentId = this.options.getCurrentId();
    const lastRefreshForId = this._refreshesPerId.get(currentId) || 0;
    const delta = performance.now() - lastRefreshForId;

    if (force || delta > this.options.minimumIntervalPerId) {
      this._refreshesPerId.set(currentId, performance.now());

      this.options.refresh();
    } // Schedule another refresh if one is already scheduled


    if (this._timer) {
      this._clearTimer();

      this._scheduleNext();
    }
  }

  destroy() {
    this._clearTimer();
  }

}

exports.default = PeriodicRefresher;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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