"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _atom = require("atom");

var _util = require("util");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LAYER_NAMES = ['unchanged', 'addition', 'deletion', 'nonewline', 'hunk', 'patch'];

class PatchBuffer {
  constructor() {
    this.buffer = new _atom.TextBuffer();
    this.buffer.retain();
    this.layers = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.buffer.addMarkerLayer();
      return map;
    }, {});
  }

  getBuffer() {
    return this.buffer;
  }

  getInsertionPoint() {
    return this.buffer.getEndPosition();
  }

  getLayer(layerName) {
    return this.layers[layerName];
  }

  findMarkers(layerName, ...args) {
    return this.layers[layerName].findMarkers(...args);
  }

  findAllMarkers(...args) {
    return LAYER_NAMES.reduce((arr, layerName) => {
      arr.push(...this.findMarkers(layerName, ...args));
      return arr;
    }, []);
  }

  markPosition(layerName, ...args) {
    return this.layers[layerName].markPosition(...args);
  }

  markRange(layerName, ...args) {
    return this.layers[layerName].markRange(...args);
  }

  clearAllLayers() {
    for (const layerName of LAYER_NAMES) {
      this.layers[layerName].clear();
    }
  }

  createInserterAt(insertionPoint) {
    return new Inserter(this, _atom.Point.fromObject(insertionPoint));
  }

  createInserterAtEnd() {
    return this.createInserterAt(this.getInsertionPoint());
  }

  extractPatchBuffer(rangeLike, options = {}) {
    const opts = _objectSpread({
      exclude: new Set()
    }, options);

    const range = _atom.Range.fromObject(rangeLike);

    const baseOffset = range.start.negate();
    const movedMarkersByLayer = LAYER_NAMES.reduce((map, layerName) => {
      map[layerName] = this.layers[layerName].findMarkers({
        containedInRange: range
      }).filter(m => !opts.exclude.has(m));
      return map;
    }, {});
    const markerMap = new Map();
    const subBuffer = new PatchBuffer();
    subBuffer.getBuffer().setText(this.buffer.getTextInRange(range));

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of movedMarkersByLayer[layerName]) {
        const startOffset = oldMarker.getRange().start.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const endOffset = oldMarker.getRange().end.row === range.start.row ? baseOffset : [baseOffset.row, 0];
        const newMarker = subBuffer.markRange(layerName, oldMarker.getRange().translate(startOffset, endOffset), oldMarker.getProperties());
        markerMap.set(oldMarker, newMarker);
        oldMarker.destroy();
      }
    }

    this.buffer.setTextInRange(range, '');
    return {
      patchBuffer: subBuffer,
      markerMap
    };
  }

  deleteLastNewline() {
    if (this.buffer.getLastLine() === '') {
      this.buffer.deleteRow(this.buffer.getLastRow());
    }

    return this;
  }

  adopt(original) {
    this.clearAllLayers();
    this.buffer.setText(original.getBuffer().getText());
    const markerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const originalMarker of original.getLayer(layerName).getMarkers()) {
        const newMarker = this.markRange(layerName, originalMarker.getRange(), originalMarker.getProperties());
        markerMap.set(originalMarker, newMarker);
      }
    }

    return markerMap;
  }
  /* istanbul ignore next */


  inspect(opts = {}) {
    /* istanbul ignore next */
    const options = _objectSpread({
      layerNames: LAYER_NAMES
    }, opts);

    let inspectString = '';
    const increasingMarkers = [];

    for (const layerName of options.layerNames) {
      for (const marker of this.findMarkers(layerName, {})) {
        increasingMarkers.push({
          layerName,
          point: marker.getRange().start,
          start: true,
          id: marker.id
        });
        increasingMarkers.push({
          layerName,
          point: marker.getRange().end,
          end: true,
          id: marker.id
        });
      }
    }

    increasingMarkers.sort((a, b) => {
      const cmp = a.point.compare(b.point);

      if (cmp !== 0) {
        return cmp;
      } else if (a.start && b.start) {
        return 0;
      } else if (a.start && !b.start) {
        return -1;
      } else if (!a.start && b.start) {
        return 1;
      } else {
        return 0;
      }
    });

    let inspectPoint = _atom.Point.fromObject([0, 0]);

    for (const marker of increasingMarkers) {
      if (!marker.point.isEqual(inspectPoint)) {
        inspectString += (0, _util.inspect)(this.buffer.getTextInRange([inspectPoint, marker.point])) + '\n';
      }

      if (marker.start) {
        inspectString += `  start ${marker.layerName}@${marker.id}\n`;
      } else if (marker.end) {
        inspectString += `  end ${marker.layerName}@${marker.id}\n`;
      }

      inspectPoint = marker.point;
    }

    return inspectString;
  }

}

exports.default = PatchBuffer;

class Inserter {
  constructor(patchBuffer, insertionPoint) {
    const clipped = patchBuffer.getBuffer().clipPosition(insertionPoint);
    this.patchBuffer = patchBuffer;
    this.startPoint = clipped.copy();
    this.insertionPoint = clipped.copy();
    this.markerBlueprints = [];
    this.markerMapCallbacks = [];
    this.markersBefore = new Set();
    this.markersAfter = new Set();
  }

  keepBefore(markers) {
    for (const marker of markers) {
      if (marker.getRange().end.isEqual(this.startPoint)) {
        this.markersBefore.add(marker);
      }
    }

    return this;
  }

  keepAfter(markers) {
    for (const marker of markers) {
      if (marker.getRange().start.isEqual(this.startPoint)) {
        this.markersAfter.add(marker);
      }
    }

    return this;
  }

  markWhile(layerName, block, markerOpts) {
    const start = this.insertionPoint.copy();
    block();
    const end = this.insertionPoint.copy();
    this.markerBlueprints.push({
      layerName,
      range: new _atom.Range(start, end),
      markerOpts
    });
    return this;
  }

  insert(text) {
    const insertedRange = this.patchBuffer.getBuffer().insert(this.insertionPoint, text);
    this.insertionPoint = insertedRange.end;
    return this;
  }

  insertMarked(text, layerName, markerOpts) {
    return this.markWhile(layerName, () => this.insert(text), markerOpts);
  }

  insertPatchBuffer(subPatchBuffer, opts) {
    const baseOffset = this.insertionPoint.copy();
    this.insert(subPatchBuffer.getBuffer().getText());
    const subMarkerMap = new Map();

    for (const layerName of LAYER_NAMES) {
      for (const oldMarker of subPatchBuffer.findMarkers(layerName, {})) {
        const startOffset = oldMarker.getRange().start.row === 0 ? baseOffset : [baseOffset.row, 0];
        const endOffset = oldMarker.getRange().end.row === 0 ? baseOffset : [baseOffset.row, 0];
        const range = oldMarker.getRange().translate(startOffset, endOffset);

        const markerOpts = _objectSpread({}, oldMarker.getProperties(), {
          callback: newMarker => {
            subMarkerMap.set(oldMarker, newMarker);
          }
        });

        this.markerBlueprints.push({
          layerName,
          range,
          markerOpts
        });
      }
    }

    this.markerMapCallbacks.push({
      markerMap: subMarkerMap,
      callback: opts.callback
    });
    return this;
  }

  apply() {
    for (const _ref of this.markerBlueprints) {
      const {
        layerName,
        range,
        markerOpts
      } = _ref;
      const callback = markerOpts.callback;
      delete markerOpts.callback;
      const marker = this.patchBuffer.markRange(layerName, range, markerOpts);

      if (callback) {
        callback(marker);
      }
    }

    for (const _ref2 of this.markerMapCallbacks) {
      const {
        markerMap,
        callback
      } = _ref2;
      callback(markerMap);
    }

    for (const beforeMarker of this.markersBefore) {
      const isEmpty = beforeMarker.getRange().isEmpty();

      if (!beforeMarker.isReversed()) {
        beforeMarker.setHeadPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setTailPosition(this.startPoint);
        }
      } else {
        beforeMarker.setTailPosition(this.startPoint);

        if (isEmpty) {
          beforeMarker.setHeadPosition(this.startPoint);
        }
      }
    }

    for (const afterMarker of this.markersAfter) {
      const isEmpty = afterMarker.getRange().isEmpty();

      if (!afterMarker.isReversed()) {
        afterMarker.setTailPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setHeadPosition(this.insertionPoint);
        }
      } else {
        afterMarker.setHeadPosition(this.insertionPoint);

        if (isEmpty) {
          afterMarker.setTailPosition(this.insertionPoint);
        }
      }
    }
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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