"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _util = _interopRequireDefault(require("util"));

var _branch = require("./branch");

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Store and index a set of Branches in a repository.
class BranchSet {
  constructor(all = []) {
    this.all = [];
    this.head = _branch.nullBranch;
    this.byUpstreamRef = new Map();
    this.byPushRef = new Map();

    for (const branch of all) {
      this.add(branch);
    }
  }

  add(branch) {
    this.all.push(branch);

    if (branch.isHead()) {
      this.head = branch;
    }

    const u = branch.getUpstream();

    if (u.isPresent() && u.isRemoteTracking()) {
      const k = `${u.getRemoteName()}\0${u.getRemoteRef()}`;
      (0, _helpers.pushAtKey)(this.byUpstreamRef, k, branch);
    }

    const p = branch.getPush();

    if (p.isPresent() && p.isRemoteTracking()) {
      const k = `${p.getRemoteName()}\0${p.getRemoteRef()}`;
      (0, _helpers.pushAtKey)(this.byPushRef, k, branch);
    }
  }

  getNames() {
    return this.all.map(branch => branch.getName());
  } // Return the HEAD branch, or `nullBranch` if HEAD is not a branch. This can happen if HEAD is unborn (the repository
  // was just initialized) or if HEAD is detached.


  getHeadBranch() {
    return this.head;
  } // Return an Array of Branches that would be updated from a given remote ref with a `git pull`. This corresponds with
  // git's notion of an _upstream_ and takes into account the current `branch.remote` setting and `remote.<name>.fetch`
  // refspec.


  getPullTargets(remoteName, remoteRefName) {
    return this.byUpstreamRef.get(`${remoteName}\0${remoteRefName}`) || [];
  } // Return an Array of Branches that will update a given remote ref on an unqualified `git push`. This accounts for
  // the current `branch.pushRemote` setting and `remote.<name>.push` refspec.


  getPushSources(remoteName, remoteRefName) {
    return this.byPushRef.get(`${remoteName}\0${remoteRefName}`) || [];
  }

  inspect(depth, options) {
    return `BranchSet {${_util.default.inspect(this.all)}}`;
  }

}

exports.default = BranchSet;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImJyYW5jaC1zZXQuanMiXSwibmFtZXMiOlsiQnJhbmNoU2V0IiwiY29uc3RydWN0b3IiLCJhbGwiLCJoZWFkIiwibnVsbEJyYW5jaCIsImJ5VXBzdHJlYW1SZWYiLCJNYXAiLCJieVB1c2hSZWYiLCJicmFuY2giLCJhZGQiLCJwdXNoIiwiaXNIZWFkIiwidSIsImdldFVwc3RyZWFtIiwiaXNQcmVzZW50IiwiaXNSZW1vdGVUcmFja2luZyIsImsiLCJnZXRSZW1vdGVOYW1lIiwiZ2V0UmVtb3RlUmVmIiwicCIsImdldFB1c2giLCJnZXROYW1lcyIsIm1hcCIsImdldE5hbWUiLCJnZXRIZWFkQnJhbmNoIiwiZ2V0UHVsbFRhcmdldHMiLCJyZW1vdGVOYW1lIiwicmVtb3RlUmVmTmFtZSIsImdldCIsImdldFB1c2hTb3VyY2VzIiwiaW5zcGVjdCIsImRlcHRoIiwib3B0aW9ucyIsInV0aWwiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFBQTs7QUFFQTs7QUFDQTs7OztBQUVBO0FBQ2UsTUFBTUEsU0FBTixDQUFnQjtBQUM3QkMsRUFBQUEsV0FBVyxDQUFDQyxHQUFHLEdBQUcsRUFBUCxFQUFXO0FBQ3BCLFNBQUtBLEdBQUwsR0FBVyxFQUFYO0FBQ0EsU0FBS0MsSUFBTCxHQUFZQyxrQkFBWjtBQUNBLFNBQUtDLGFBQUwsR0FBcUIsSUFBSUMsR0FBSixFQUFyQjtBQUNBLFNBQUtDLFNBQUwsR0FBaUIsSUFBSUQsR0FBSixFQUFqQjs7QUFDQSxTQUFLLE1BQU1FLE1BQVgsSUFBcUJOLEdBQXJCLEVBQTBCO0FBQ3hCLFdBQUtPLEdBQUwsQ0FBU0QsTUFBVDtBQUNEO0FBQ0Y7O0FBRURDLEVBQUFBLEdBQUcsQ0FBQ0QsTUFBRCxFQUFTO0FBQ1YsU0FBS04sR0FBTCxDQUFTUSxJQUFULENBQWNGLE1BQWQ7O0FBRUEsUUFBSUEsTUFBTSxDQUFDRyxNQUFQLEVBQUosRUFBcUI7QUFDbkIsV0FBS1IsSUFBTCxHQUFZSyxNQUFaO0FBQ0Q7O0FBRUQsVUFBTUksQ0FBQyxHQUFHSixNQUFNLENBQUNLLFdBQVAsRUFBVjs7QUFDQSxRQUFJRCxDQUFDLENBQUNFLFNBQUYsTUFBaUJGLENBQUMsQ0FBQ0csZ0JBQUYsRUFBckIsRUFBMkM7QUFDekMsWUFBTUMsQ0FBQyxHQUFJLEdBQUVKLENBQUMsQ0FBQ0ssYUFBRixFQUFrQixLQUFJTCxDQUFDLENBQUNNLFlBQUYsRUFBaUIsRUFBcEQ7QUFDQSw4QkFBVSxLQUFLYixhQUFmLEVBQThCVyxDQUE5QixFQUFpQ1IsTUFBakM7QUFDRDs7QUFFRCxVQUFNVyxDQUFDLEdBQUdYLE1BQU0sQ0FBQ1ksT0FBUCxFQUFWOztBQUNBLFFBQUlELENBQUMsQ0FBQ0wsU0FBRixNQUFpQkssQ0FBQyxDQUFDSixnQkFBRixFQUFyQixFQUEyQztBQUN6QyxZQUFNQyxDQUFDLEdBQUksR0FBRUcsQ0FBQyxDQUFDRixhQUFGLEVBQWtCLEtBQUlFLENBQUMsQ0FBQ0QsWUFBRixFQUFpQixFQUFwRDtBQUNBLDhCQUFVLEtBQUtYLFNBQWYsRUFBMEJTLENBQTFCLEVBQTZCUixNQUE3QjtBQUNEO0FBQ0Y7O0FBRURhLEVBQUFBLFFBQVEsR0FBRztBQUNULFdBQU8sS0FBS25CLEdBQUwsQ0FBU29CLEdBQVQsQ0FBYWQsTUFBTSxJQUFJQSxNQUFNLENBQUNlLE9BQVAsRUFBdkIsQ0FBUDtBQUNELEdBakM0QixDQW1DN0I7QUFDQTs7O0FBQ0FDLEVBQUFBLGFBQWEsR0FBRztBQUNkLFdBQU8sS0FBS3JCLElBQVo7QUFDRCxHQXZDNEIsQ0F5QzdCO0FBQ0E7QUFDQTs7O0FBQ0FzQixFQUFBQSxjQUFjLENBQUNDLFVBQUQsRUFBYUMsYUFBYixFQUE0QjtBQUN4QyxXQUFPLEtBQUt0QixhQUFMLENBQW1CdUIsR0FBbkIsQ0FBd0IsR0FBRUYsVUFBVyxLQUFJQyxhQUFjLEVBQXZELEtBQTZELEVBQXBFO0FBQ0QsR0E5QzRCLENBZ0Q3QjtBQUNBOzs7QUFDQUUsRUFBQUEsY0FBYyxDQUFDSCxVQUFELEVBQWFDLGFBQWIsRUFBNEI7QUFDeEMsV0FBTyxLQUFLcEIsU0FBTCxDQUFlcUIsR0FBZixDQUFvQixHQUFFRixVQUFXLEtBQUlDLGFBQWMsRUFBbkQsS0FBeUQsRUFBaEU7QUFDRDs7QUFFREcsRUFBQUEsT0FBTyxDQUFDQyxLQUFELEVBQVFDLE9BQVIsRUFBaUI7QUFDdEIsV0FBUSxjQUFhQyxjQUFLSCxPQUFMLENBQWEsS0FBSzVCLEdBQWxCLENBQXVCLEdBQTVDO0FBQ0Q7O0FBeEQ0QiIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM2LjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHV0aWwgZnJvbSAndXRpbCc7XG5cbmltcG9ydCB7bnVsbEJyYW5jaH0gZnJvbSAnLi9icmFuY2gnO1xuaW1wb3J0IHtwdXNoQXRLZXl9IGZyb20gJy4uL2hlbHBlcnMnO1xuXG4vLyBTdG9yZSBhbmQgaW5kZXggYSBzZXQgb2YgQnJhbmNoZXMgaW4gYSByZXBvc2l0b3J5LlxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgQnJhbmNoU2V0IHtcbiAgY29uc3RydWN0b3IoYWxsID0gW10pIHtcbiAgICB0aGlzLmFsbCA9IFtdO1xuICAgIHRoaXMuaGVhZCA9IG51bGxCcmFuY2g7XG4gICAgdGhpcy5ieVVwc3RyZWFtUmVmID0gbmV3IE1hcCgpO1xuICAgIHRoaXMuYnlQdXNoUmVmID0gbmV3IE1hcCgpO1xuICAgIGZvciAoY29uc3QgYnJhbmNoIG9mIGFsbCkge1xuICAgICAgdGhpcy5hZGQoYnJhbmNoKTtcbiAgICB9XG4gIH1cblxuICBhZGQoYnJhbmNoKSB7XG4gICAgdGhpcy5hbGwucHVzaChicmFuY2gpO1xuXG4gICAgaWYgKGJyYW5jaC5pc0hlYWQoKSkge1xuICAgICAgdGhpcy5oZWFkID0gYnJhbmNoO1xuICAgIH1cblxuICAgIGNvbnN0IHUgPSBicmFuY2guZ2V0VXBzdHJlYW0oKTtcbiAgICBpZiAodS5pc1ByZXNlbnQoKSAmJiB1LmlzUmVtb3RlVHJhY2tpbmcoKSkge1xuICAgICAgY29uc3QgayA9IGAke3UuZ2V0UmVtb3RlTmFtZSgpfVxcMCR7dS5nZXRSZW1vdGVSZWYoKX1gO1xuICAgICAgcHVzaEF0S2V5KHRoaXMuYnlVcHN0cmVhbVJlZiwgaywgYnJhbmNoKTtcbiAgICB9XG5cbiAgICBjb25zdCBwID0gYnJhbmNoLmdldFB1c2goKTtcbiAgICBpZiAocC5pc1ByZXNlbnQoKSAmJiBwLmlzUmVtb3RlVHJhY2tpbmcoKSkge1xuICAgICAgY29uc3QgayA9IGAke3AuZ2V0UmVtb3RlTmFtZSgpfVxcMCR7cC5nZXRSZW1vdGVSZWYoKX1gO1xuICAgICAgcHVzaEF0S2V5KHRoaXMuYnlQdXNoUmVmLCBrLCBicmFuY2gpO1xuICAgIH1cbiAgfVxuXG4gIGdldE5hbWVzKCkge1xuICAgIHJldHVybiB0aGlzLmFsbC5tYXAoYnJhbmNoID0+IGJyYW5jaC5nZXROYW1lKCkpO1xuICB9XG5cbiAgLy8gUmV0dXJuIHRoZSBIRUFEIGJyYW5jaCwgb3IgYG51bGxCcmFuY2hgIGlmIEhFQUQgaXMgbm90IGEgYnJhbmNoLiBUaGlzIGNhbiBoYXBwZW4gaWYgSEVBRCBpcyB1bmJvcm4gKHRoZSByZXBvc2l0b3J5XG4gIC8vIHdhcyBqdXN0IGluaXRpYWxpemVkKSBvciBpZiBIRUFEIGlzIGRldGFjaGVkLlxuICBnZXRIZWFkQnJhbmNoKCkge1xuICAgIHJldHVybiB0aGlzLmhlYWQ7XG4gIH1cblxuICAvLyBSZXR1cm4gYW4gQXJyYXkgb2YgQnJhbmNoZXMgdGhhdCB3b3VsZCBiZSB1cGRhdGVkIGZyb20gYSBnaXZlbiByZW1vdGUgcmVmIHdpdGggYSBgZ2l0IHB1bGxgLiBUaGlzIGNvcnJlc3BvbmRzIHdpdGhcbiAgLy8gZ2l0J3Mgbm90aW9uIG9mIGFuIF91cHN0cmVhbV8gYW5kIHRha2VzIGludG8gYWNjb3VudCB0aGUgY3VycmVudCBgYnJhbmNoLnJlbW90ZWAgc2V0dGluZyBhbmQgYHJlbW90ZS48bmFtZT4uZmV0Y2hgXG4gIC8vIHJlZnNwZWMuXG4gIGdldFB1bGxUYXJnZXRzKHJlbW90ZU5hbWUsIHJlbW90ZVJlZk5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5ieVVwc3RyZWFtUmVmLmdldChgJHtyZW1vdGVOYW1lfVxcMCR7cmVtb3RlUmVmTmFtZX1gKSB8fCBbXTtcbiAgfVxuXG4gIC8vIFJldHVybiBhbiBBcnJheSBvZiBCcmFuY2hlcyB0aGF0IHdpbGwgdXBkYXRlIGEgZ2l2ZW4gcmVtb3RlIHJlZiBvbiBhbiB1bnF1YWxpZmllZCBgZ2l0IHB1c2hgLiBUaGlzIGFjY291bnRzIGZvclxuICAvLyB0aGUgY3VycmVudCBgYnJhbmNoLnB1c2hSZW1vdGVgIHNldHRpbmcgYW5kIGByZW1vdGUuPG5hbWU+LnB1c2hgIHJlZnNwZWMuXG4gIGdldFB1c2hTb3VyY2VzKHJlbW90ZU5hbWUsIHJlbW90ZVJlZk5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5ieVB1c2hSZWYuZ2V0KGAke3JlbW90ZU5hbWV9XFwwJHtyZW1vdGVSZWZOYW1lfWApIHx8IFtdO1xuICB9XG5cbiAgaW5zcGVjdChkZXB0aCwgb3B0aW9ucykge1xuICAgIHJldHVybiBgQnJhbmNoU2V0IHske3V0aWwuaW5zcGVjdCh0aGlzLmFsbCl9fWA7XG4gIH1cbn1cbiJdfQ==