"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _helpers = require("../helpers");

var _propTypes2 = require("../prop-types");

var _reporterProxy = require("../reporter-proxy");

var _repository = _interopRequireDefault(require("../models/repository"));

var _endpoint = require("../models/endpoint");

var _issueishDetailContainer = _interopRequireDefault(require("../containers/issueish-detail-container"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IssueishDetailItem extends _react.Component {
  static buildURI(host, owner, repo, number, workdir = null) {
    const encodedWorkdir = workdir ? encodeURIComponent(workdir) : '';
    return 'atom-github://issueish/' + encodeURIComponent(host) + '/' + encodeURIComponent(owner) + '/' + encodeURIComponent(repo) + '/' + encodeURIComponent(number) + '?workdir=' + encodedWorkdir;
  }

  constructor(props) {
    super(props);

    _defineProperty(this, "destroy", () => {
      /* istanbul ignore else */
      if (!this.isDestroyed) {
        this.emitter.emit('did-destroy');
        this.isDestroyed = true;
      }
    });

    (0, _helpers.autobind)(this, 'switchToIssueish', 'handleTitleChanged');
    this.emitter = new _eventKit.Emitter();
    this.title = `${this.props.owner}/${this.props.repo}#${this.props.issueishNumber}`;
    this.hasTerminatedPendingState = false;
    const repository = this.props.workingDirectory === '' ? _repository.default.absent() : this.props.workdirContextPool.add(this.props.workingDirectory).getRepository();
    this.state = {
      host: this.props.host,
      owner: this.props.owner,
      repo: this.props.repo,
      issueishNumber: this.props.issueishNumber,
      repository
    };

    if (repository.isAbsent()) {
      this.switchToIssueish(this.props.owner, this.props.repo, this.props.issueishNumber);
    }

    this.refEditor = new _refHolder.default();
    this.refEditor.observe(editor => {
      this.emitter.emit('did-change-embedded-text-editor', editor);
    });
  }

  render() {
    return _react.default.createElement(_issueishDetailContainer.default, {
      endpoint: (0, _endpoint.getEndpoint)(this.state.host),
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber,
      repository: this.state.repository,
      workspace: this.props.workspace,
      loginModel: this.props.loginModel,
      onTitleChange: this.handleTitleChanged,
      switchToIssueish: this.switchToIssueish,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      destroy: this.destroy,
      itemType: this.constructor,
      refEditor: this.refEditor
    });
  }

  async switchToIssueish(owner, repo, issueishNumber) {
    const pool = this.props.workdirContextPool;
    const prev = {
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber
    };
    const matchingRepositories = (await Promise.all(pool.withResidentContexts((workdir, context) => {
      const repository = context.getRepository();
      return repository.hasGitHubRemote(this.state.host, owner, repo).then(hasRemote => hasRemote ? repository : null);
    }))).filter(Boolean);
    const nextRepository = matchingRepositories.length === 1 ? matchingRepositories[0] : _repository.default.absent();
    await new Promise(resolve => {
      this.setState((prevState, props) => {
        if (pool === props.workdirContextPool && prevState.owner === prev.owner && prevState.repo === prev.repo && prevState.issueishNumber === prev.issueishNumber) {
          (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
            package: 'github',
            from: 'issueish-link',
            target: 'current-tab'
          });
          return {
            owner,
            repo,
            issueishNumber,
            repository: nextRepository
          };
        }

        return {};
      }, resolve);
    });
  }

  handleTitleChanged(title) {
    if (this.title !== title) {
      this.title = title;
      this.emitter.emit('did-change-title', title);
    }
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  terminatePendingState() {
    if (!this.hasTerminatedPendingState) {
      this.emitter.emit('did-terminate-pending-state');
      this.hasTerminatedPendingState = true;
    }
  }

  onDidTerminatePendingState(callback) {
    return this.emitter.on('did-terminate-pending-state', callback);
  }

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  serialize() {
    return {
      uri: this.getURI(),
      deserializer: 'IssueishDetailItem'
    };
  }

  getTitle() {
    return this.title;
  }

  observeEmbeddedTextEditor(cb) {
    this.refEditor.map(editor => cb(editor));
    return this.emitter.on('did-change-embedded-text-editor', cb);
  }

}

exports.default = IssueishDetailItem;

_defineProperty(IssueishDetailItem, "propTypes", {
  // Issueish selection criteria
  // Parsed from item URI
  host: _propTypes.default.string.isRequired,
  owner: _propTypes.default.string.isRequired,
  repo: _propTypes.default.string.isRequired,
  issueishNumber: _propTypes.default.number.isRequired,
  workingDirectory: _propTypes.default.string.isRequired,
  // Package models
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired,
  loginModel: _propTypes2.GithubLoginModelPropType.isRequired,
  // Atom environment
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired
});

_defineProperty(IssueishDetailItem, "uriPattern", 'atom-github://issueish/{host}/{owner}/{repo}/{issueishNumber}?workdir={workingDirectory}');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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