"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.LargeRepoError = exports.GitError = void 0;

var _path = _interopRequireDefault(require("path"));

var _os = _interopRequireDefault(require("os"));

var _child_process = _interopRequireDefault(require("child_process"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _util = _interopRequireDefault(require("util"));

var _electron = require("electron");

var _eventKit = require("event-kit");

var _dugite = require("dugite");

var _whatTheDiff = require("what-the-diff");

var _whatTheStatus = require("what-the-status");

var _gitPromptServer = _interopRequireDefault(require("./git-prompt-server"));

var _gitTempDir = _interopRequireDefault(require("./git-temp-dir"));

var _asyncQueue = _interopRequireDefault(require("./async-queue"));

var _reporterProxy = require("./reporter-proxy");

var _helpers = require("./helpers");

var _gitTimingsView = _interopRequireDefault(require("./views/git-timings-view"));

var _file = _interopRequireDefault(require("./models/patch/file"));

var _workerManager = _interopRequireDefault(require("./worker-manager"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;
let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

}

exports.GitError = GitError;

class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }

} // ignored for the purposes of usage metrics tracking because they're noisy


exports.LargeRepoError = LargeRepoError;
const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];
const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);
/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^\\\\/]*)[\\\\/] captures non-slash characters before first slash
 */

const EXPAND_TILDE_REGEX = new RegExp('^~([^\\\\/]*)[\\\\/]');

class GitShellOutStrategy {
  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;

    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os.default.cpus().length);
      this.commandQueue = new _asyncQueue.default({
        parallelism
      });
    }

    this.prompt = options.prompt || (query => Promise.reject());

    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }
  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */


  setPromptCallback(prompt) {
    this.prompt = prompt;
  } // Execute a command and read the output using the embedded Git environment


  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const {
      stdin,
      useGitPromptServer,
      useGpgWrapper,
      useGpgAtomPrompt,
      writeOperation
    } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');
    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;

    const timingMarker = _gitTimingsView.default.generateMarker(`git ${args.join(' ')}`);

    timingMarker.mark('queued');
    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise((resolve, reject) => {
        _child_process.default.exec('git --exec-path', (error, stdout, stderr) => {
          /* istanbul ignore if */
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }

    const execPath = await execPathPromise;
    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;
      const pathParts = [];

      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }

      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _objectSpread({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path.default.delimiter)
      });

      const gitTempDir = new _gitTempDir.default();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer.default(gitTempDir);
        await gitPromptServer.start(this.prompt);
        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getSocketPath());
        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy'); // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.

        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';
        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }
      /* istanbul ignore if */


      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = {
        env
      };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }
      /* istanbul ignore if */


      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }

      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({
            args,
            opts
          });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }

        const {
          promise,
          cancel
        } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;

        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({
            handlerPid
          }) => {
            expectCancel = true;
            await cancel(); // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.

            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                /* istanbul ignore if */
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const {
          stdout,
          stderr,
          exitCode,
          signal,
          timing
        } = await promise.catch(err => {
          if (err.signal) {
            return {
              signal: err.signal
            };
          }

          reject(err);
          return {};
        });

        if (timing) {
          const {
            execTime,
            spawnTime,
            ipcTime
          } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }

        timingMarker.finalize();
        /* istanbul ignore if */

        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }

        if (gitPromptServer) {
          gitPromptServer.terminate();
        }

        subscriptions.dispose();
        /* istanbul ignore if */

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;

            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }

            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }

            summary += 'stdout:';

            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }

            summary += 'stderr:';

            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';
            console.groupCollapsed(`git:${formattedArgs}`);

            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }

            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util.default.inspect(args, {
              breakLength: Infinity
            }));

            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }

            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }

        resolve(stdout);
      });
    }, {
      parallel: !writeOperation
    });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _objectSpread({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _objectSpread({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');
      let childPid;

      options.processCallback = child => {
        childPid = child.pid;
        /* istanbul ignore next */

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);

      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          /* istanbul ignore if */
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              /* istanbul ignore if */
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager.default.getInstance();

      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra.default.stat(this.workingDir); // fails if folder doesn't exist

      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path.default.join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      return (0, _helpers.toNativePathSep)(dotGitDir);
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }
  /**
   * Staging/Unstaging files and patches and committing
   */


  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');

    if (!templatePath) {
      return null;
    }

    const homeDir = _os.default.homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path.default.join(_path.default.dirname(homeDir), user) : homeDir}/`;
    });
    templatePath = (0, _helpers.toNativePathSep)(templatePath);

    if (!_path.default.isAbsolute(templatePath)) {
      templatePath = _path.default.join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }

    return await _fsExtra.default.readFile(templatePath, {
      encoding: 'utf8'
    });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }

    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, {
      writeOperation: true
    });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({
        args,
        opts
      }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], {
      writeOperation: true
    });
  }

  applyPatch(patch, {
    index
  } = {}) {
    const args = ['apply', '-'];

    if (index) {
      args.splice(1, 0, '--cached');
    }

    return this.exec(args, {
      stdin: patch,
      writeOperation: true
    });
  }

  async commit(rawMessage, {
    allowEmpty,
    amend,
    coAuthors,
    verbatim
  } = {}) {
    const args = ['commit'];
    let msg; // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.

    if (amend && rawMessage.length === 0) {
      const {
        unbornRef,
        messageBody,
        messageSubject
      } = await this.getHeadCommit();

      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    } // if commit template is used, strip commented lines from commit
    // to be consistent with command line git.


    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      // respecting the comment character from user settings or fall back to # as default.
      // https://git-scm.com/docs/git-config#git-config-corecommentChar
      let commentChar = await this.getConfig('core.commentChar');

      if (!commentChar) {
        commentChar = '#';
      }

      msg = msg.split('\n').filter(line => !line.startsWith(commentChar)).join('\n');
    } // Determine the cleanup mode.


    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    } // add co-author commit trailers if necessary


    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }

    if (allowEmpty) {
      args.push('--allow-empty');
    }

    return this.gpgExec(args, {
      writeOperation: true
    });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    }); // Ensure that message ends with newline for git-interpret trailers to work

    const msg = `${message.trim()}\n`;
    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }
  /**
   * File Status and Diffs
   */


  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);

    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }

      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];

    if (options.staged) {
      args.push('--staged');
    }

    if (options.target) {
      args.push(options.target);
    }

    const output = await this.exec(args);
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };
    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });

    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }

    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);

    if (output.trim() === '') {
      return [];
    }

    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, {
    staged,
    baseCommit
  } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];

    if (staged) {
      args.push('--staged');
    }

    if (baseCommit) {
      args.push(baseCommit);
    }

    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);
    let rawDiffs = [];

    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];

        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }

        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path.default.join(this.workingDir, filePath);

      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra.default.readFile(absPath, {
        encoding: 'utf8'
      });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;

      if (executable) {
        mode = _file.default.modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file.default.modes.SYMLINK;
        realpath = await _fsExtra.default.realpath(absPath);
      } else {
        mode = _file.default.modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }

    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }

    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);

    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }

      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }

    return diffs;
  }
  /**
   * Miscellaneous getters
   */


  async getCommit(ref) {
    const [commit] = await this.getCommits({
      max: 1,
      ref,
      includeUnborn: true
    });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({
      max: 1,
      ref: 'HEAD',
      includeUnborn: true
    });
    return headCommit;
  }

  async getCommits(options = {}) {
    const {
      max,
      ref,
      includeUnborn,
      includePatch
    } = _objectSpread({
      max: 1,
      ref: 'HEAD',
      includeUnborn: false,
      includePatch: false
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %an = author full name
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body


    const args = ['log', '--pretty=format:%H%x00%ae%x00%an%x00%at%x00%s%x00%b%x00', '--no-abbrev-commit', '--no-prefix', '--no-ext-diff', '--no-renames', '-z', '-n', max, ref];

    if (includePatch) {
      args.push('--patch', '-m', '--first-parent');
    }

    const output = await this.exec(args.concat('--')).catch(err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{
        sha: '',
        message: '',
        unbornRef: true
      }] : [];
    }

    const fields = output.trim().split('\0');
    const commits = [];

    for (let i = 0; i < fields.length; i += 7) {
      const body = fields[i + 5].trim();
      let patch = [];

      if (includePatch) {
        const diffs = fields[i + 6];
        patch = (0, _whatTheDiff.parse)(diffs.trim());
      }

      const {
        message: messageBody,
        coAuthors
      } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);
      commits.push({
        sha: fields[i] && fields[i].trim(),
        authorEmail: fields[i + 1] && fields[i + 1].trim(),
        authorName: fields[i + 2] && fields[i + 2].trim(),
        authorDate: parseInt(fields[i + 3], 10),
        messageSubject: fields[i + 4],
        messageBody,
        coAuthors,
        unbornRef: false,
        patch
      });
    }

    return commits;
  }

  async getAuthors(options = {}) {
    const {
      max,
      ref
    } = _objectSpread({
      max: 1,
      ref: 'HEAD'
    }, options); // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).


    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);
      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });
        acc[ae] = an;
        acc[ce] = cn;
        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers) {
    const args = ['interpret-trailers'];

    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }

    return this.exec(args, {
      stdin: commitMessage
    });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }
  /**
   * Merge
   */


  merge(branchName) {
    return this.gpgExec(['merge', branchName], {
      writeOperation: true
    });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path.default.join(dotGitDir, 'MERGE_HEAD')).catch(() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], {
      writeOperation: true
    });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }
  /**
   * Rebase
   */


  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path.default.join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }
  /**
   * Remote interactions
   */


  clone(remoteUrl, options = {}) {
    const args = ['clone'];

    if (options.noLocal) {
      args.push('--no-local');
    }

    if (options.bare) {
      args.push('--bare');
    }

    if (options.recursive) {
      args.push('--recursive');
    }

    args.push(remoteUrl, this.workingDir);
    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];

    if (options.ffOnly) {
      args.push('--ff-only');
    }

    return this.gpgExec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];

    if (options.setUpstream) {
      args.push('--set-upstream');
    }

    if (options.force) {
      args.push('--force');
    }

    return this.exec(args, {
      useGitPromptServer: true,
      writeOperation: true
    });
  }
  /**
   * Undo Operations
   */


  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];

    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }

    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }
  /**
   * Branches
   */


  checkout(branchName, options = {}) {
    const args = ['checkout'];

    if (options.createNew) {
      args.push('-b');
    }

    args.push(branchName);

    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }

      args.push(options.startPoint);
    }

    return this.exec(args, {
      writeOperation: true
    });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');
    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');
      const branch = {
        name,
        sha,
        head: head === '*'
      };

      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }

      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }

      return branch;
    });
  }

  async getBranchesWithCommit(sha, option = {}) {
    const args = ['branch', '--format=%(refname)', '--contains', sha];

    if (option.showLocal && option.showRemote) {
      args.splice(1, 0, '--all');
    } else if (option.showRemote) {
      args.splice(1, 0, '--remotes');
    }

    return (await this.exec(args)).trim().split(_helpers.LINE_ENDING_REGEX);
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }

    const args = ['checkout'];

    if (revision) {
      args.push(revision);
    }

    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), {
      writeOperation: true
    });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, {
    local
  } = {}) {
    let output;

    try {
      let args = ['config'];

      if (local || atom.inSpecMode()) {
        args.push('--local');
      }

      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1) {
        // No matching config found
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, {
    replaceAll
  } = {}) {
    let args = ['config'];

    if (replaceAll) {
      args.push('--replace-all');
    }

    args = args.concat(option, value);
    return this.exec(args, {
      writeOperation: true
    });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], {
      writeOperation: true
    });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], {
      local: true
    });

    if (output) {
      output = output.trim();

      if (!output.length) {
        return [];
      }

      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({
    filePath,
    stdin
  } = {}) {
    let output;

    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], {
          writeOperation: true
        })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], {
        stdin,
        writeOperation: true
      })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }

    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra.default.writeFile(absFilePath, output, {
      encoding: 'utf8'
    });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;

    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    } // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.


    const resolvedResultPath = _path.default.resolve(this.workingDir, resultPath);

    await _fsExtra.default.writeFile(resolvedResultPath, output, {
      encoding: 'utf8'
    });
    return {
      filePath: oursPath,
      resultPath,
      conflict
    };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;

    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }

    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }

    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }

    return this.exec(['update-index', '--index-info'], {
      stdin: indexInfo,
      writeOperation: true
    });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);

    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path.default.join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path.default.join(this.workingDir, filePath));

      if (symlink) {
        return _file.default.modes.SYMLINK;
      } else if (executable) {
        return _file.default.modes.EXECUTABLE;
      } else {
        return _file.default.modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }

}

exports.default = GitShellOutStrategy;

_defineProperty(GitShellOutStrategy, "defaultExecArgs", {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
});

function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];

  if (contents) {
    let noNewLine;
    let lines;

    if (mode === _file.default.modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }

    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }

    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }

  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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