"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _observeModel = _interopRequireDefault(require("../views/observe-model"));

var _resolutionProgress = _interopRequireDefault(require("../models/conflicts/resolution-progress"));

var _editorConflictController = _interopRequireDefault(require("./editor-conflict-controller"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_REPO_DATA = {
  mergeConflictPaths: [],
  isRebasing: false
};
/**
 * Render an `EditorConflictController` for each `TextEditor` open on a file that contains git conflict markers.
 */

class RepositoryConflictController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "fetchData", repository => {
      return (0, _yubikiri.default)({
        workingDirectoryPath: repository.getWorkingDirectoryPath(),
        mergeConflictPaths: repository.getMergeConflicts().then(conflicts => {
          return conflicts.map(conflict => conflict.filePath);
        }),
        isRebasing: repository.isRebasing()
      });
    });

    this.state = {
      openEditors: this.props.workspace.getTextEditors()
    };
    this.subscriptions = new _eventKit.CompositeDisposable();
  }

  componentDidMount() {
    const updateState = () => {
      this.setState({
        openEditors: this.props.workspace.getTextEditors()
      });
    };

    this.subscriptions.add(this.props.workspace.observeTextEditors(updateState), this.props.workspace.onDidDestroyPaneItem(updateState), this.props.config.observe('github.graphicalConflictResolution', () => this.forceUpdate()));
  }

  render() {
    return _react.default.createElement(_observeModel.default, {
      model: this.props.repository,
      fetchData: this.fetchData
    }, data => this.renderWithData(data || DEFAULT_REPO_DATA));
  }

  renderWithData(repoData) {
    const conflictingEditors = this.getConflictingEditors(repoData);
    return _react.default.createElement("div", null, conflictingEditors.map(editor => _react.default.createElement(_editorConflictController.default, {
      key: editor.id,
      commandRegistry: this.props.commandRegistry,
      resolutionProgress: this.props.resolutionProgress,
      editor: editor,
      isRebase: repoData.isRebasing,
      refreshResolutionProgress: this.props.refreshResolutionProgress
    })));
  }

  getConflictingEditors(repoData) {
    if (repoData.mergeConflictPaths.length === 0 || this.state.openEditors.length === 0 || !this.props.config.get('github.graphicalConflictResolution')) {
      return [];
    }

    const commonBasePath = this.props.repository.getWorkingDirectoryPath();
    const fullMergeConflictPaths = new Set(repoData.mergeConflictPaths.map(relativePath => _path.default.join(commonBasePath, relativePath)));
    return this.state.openEditors.filter(editor => fullMergeConflictPaths.has(editor.getPath()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

}

exports.default = RepositoryConflictController;

_defineProperty(RepositoryConflictController, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  commandRegistry: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  resolutionProgress: _propTypes.default.object.isRequired,
  repository: _propTypes.default.object.isRequired,
  refreshResolutionProgress: _propTypes.default.func
});

_defineProperty(RepositoryConflictController, "defaultProps", {
  refreshResolutionProgress: () => {},
  resolutionProgress: new _resolutionProgress.default()
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlcG9zaXRvcnktY29uZmxpY3QtY29udHJvbGxlci5qcyJdLCJuYW1lcyI6WyJERUZBVUxUX1JFUE9fREFUQSIsIm1lcmdlQ29uZmxpY3RQYXRocyIsImlzUmViYXNpbmciLCJSZXBvc2l0b3J5Q29uZmxpY3RDb250cm9sbGVyIiwiUmVhY3QiLCJDb21wb25lbnQiLCJjb25zdHJ1Y3RvciIsInByb3BzIiwiY29udGV4dCIsInJlcG9zaXRvcnkiLCJ3b3JraW5nRGlyZWN0b3J5UGF0aCIsImdldFdvcmtpbmdEaXJlY3RvcnlQYXRoIiwiZ2V0TWVyZ2VDb25mbGljdHMiLCJ0aGVuIiwiY29uZmxpY3RzIiwibWFwIiwiY29uZmxpY3QiLCJmaWxlUGF0aCIsInN0YXRlIiwib3BlbkVkaXRvcnMiLCJ3b3Jrc3BhY2UiLCJnZXRUZXh0RWRpdG9ycyIsInN1YnNjcmlwdGlvbnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiY29tcG9uZW50RGlkTW91bnQiLCJ1cGRhdGVTdGF0ZSIsInNldFN0YXRlIiwiYWRkIiwib2JzZXJ2ZVRleHRFZGl0b3JzIiwib25EaWREZXN0cm95UGFuZUl0ZW0iLCJjb25maWciLCJvYnNlcnZlIiwiZm9yY2VVcGRhdGUiLCJyZW5kZXIiLCJmZXRjaERhdGEiLCJkYXRhIiwicmVuZGVyV2l0aERhdGEiLCJyZXBvRGF0YSIsImNvbmZsaWN0aW5nRWRpdG9ycyIsImdldENvbmZsaWN0aW5nRWRpdG9ycyIsImVkaXRvciIsImlkIiwiY29tbWFuZFJlZ2lzdHJ5IiwicmVzb2x1dGlvblByb2dyZXNzIiwicmVmcmVzaFJlc29sdXRpb25Qcm9ncmVzcyIsImxlbmd0aCIsImdldCIsImNvbW1vbkJhc2VQYXRoIiwiZnVsbE1lcmdlQ29uZmxpY3RQYXRocyIsIlNldCIsInJlbGF0aXZlUGF0aCIsInBhdGgiLCJqb2luIiwiZmlsdGVyIiwiaGFzIiwiZ2V0UGF0aCIsImNvbXBvbmVudFdpbGxVbm1vdW50IiwiZGlzcG9zZSIsIlByb3BUeXBlcyIsIm9iamVjdCIsImlzUmVxdWlyZWQiLCJmdW5jIiwiUmVzb2x1dGlvblByb2dyZXNzIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7Ozs7OztBQUVBLE1BQU1BLGlCQUFpQixHQUFHO0FBQ3hCQyxFQUFBQSxrQkFBa0IsRUFBRSxFQURJO0FBRXhCQyxFQUFBQSxVQUFVLEVBQUU7QUFGWSxDQUExQjtBQUtBOzs7O0FBR2UsTUFBTUMsNEJBQU4sU0FBMkNDLGVBQU1DLFNBQWpELENBQTJEO0FBZXhFQyxFQUFBQSxXQUFXLENBQUNDLEtBQUQsRUFBUUMsT0FBUixFQUFpQjtBQUMxQixVQUFNRCxLQUFOLEVBQWFDLE9BQWI7O0FBRDBCLHVDQXFCaEJDLFVBQVUsSUFBSTtBQUN4QixhQUFPLHVCQUFTO0FBQ2RDLFFBQUFBLG9CQUFvQixFQUFFRCxVQUFVLENBQUNFLHVCQUFYLEVBRFI7QUFFZFYsUUFBQUEsa0JBQWtCLEVBQUVRLFVBQVUsQ0FBQ0csaUJBQVgsR0FBK0JDLElBQS9CLENBQW9DQyxTQUFTLElBQUk7QUFDbkUsaUJBQU9BLFNBQVMsQ0FBQ0MsR0FBVixDQUFjQyxRQUFRLElBQUlBLFFBQVEsQ0FBQ0MsUUFBbkMsQ0FBUDtBQUNELFNBRm1CLENBRk47QUFLZGYsUUFBQUEsVUFBVSxFQUFFTyxVQUFVLENBQUNQLFVBQVg7QUFMRSxPQUFULENBQVA7QUFPRCxLQTdCMkI7O0FBRzFCLFNBQUtnQixLQUFMLEdBQWE7QUFBQ0MsTUFBQUEsV0FBVyxFQUFFLEtBQUtaLEtBQUwsQ0FBV2EsU0FBWCxDQUFxQkMsY0FBckI7QUFBZCxLQUFiO0FBQ0EsU0FBS0MsYUFBTCxHQUFxQixJQUFJQyw2QkFBSixFQUFyQjtBQUNEOztBQUVEQyxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixVQUFNQyxXQUFXLEdBQUcsTUFBTTtBQUN4QixXQUFLQyxRQUFMLENBQWM7QUFDWlAsUUFBQUEsV0FBVyxFQUFFLEtBQUtaLEtBQUwsQ0FBV2EsU0FBWCxDQUFxQkMsY0FBckI7QUFERCxPQUFkO0FBR0QsS0FKRDs7QUFNQSxTQUFLQyxhQUFMLENBQW1CSyxHQUFuQixDQUNFLEtBQUtwQixLQUFMLENBQVdhLFNBQVgsQ0FBcUJRLGtCQUFyQixDQUF3Q0gsV0FBeEMsQ0FERixFQUVFLEtBQUtsQixLQUFMLENBQVdhLFNBQVgsQ0FBcUJTLG9CQUFyQixDQUEwQ0osV0FBMUMsQ0FGRixFQUdFLEtBQUtsQixLQUFMLENBQVd1QixNQUFYLENBQWtCQyxPQUFsQixDQUEwQixvQ0FBMUIsRUFBZ0UsTUFBTSxLQUFLQyxXQUFMLEVBQXRFLENBSEY7QUFLRDs7QUFZREMsRUFBQUEsTUFBTSxHQUFHO0FBQ1AsV0FDRSw2QkFBQyxxQkFBRDtBQUFjLE1BQUEsS0FBSyxFQUFFLEtBQUsxQixLQUFMLENBQVdFLFVBQWhDO0FBQTRDLE1BQUEsU0FBUyxFQUFFLEtBQUt5QjtBQUE1RCxPQUNHQyxJQUFJLElBQUksS0FBS0MsY0FBTCxDQUFvQkQsSUFBSSxJQUFJbkMsaUJBQTVCLENBRFgsQ0FERjtBQUtEOztBQUVEb0MsRUFBQUEsY0FBYyxDQUFDQyxRQUFELEVBQVc7QUFDdkIsVUFBTUMsa0JBQWtCLEdBQUcsS0FBS0MscUJBQUwsQ0FBMkJGLFFBQTNCLENBQTNCO0FBRUEsV0FDRSwwQ0FDR0Msa0JBQWtCLENBQUN2QixHQUFuQixDQUF1QnlCLE1BQU0sSUFDNUIsNkJBQUMsaUNBQUQ7QUFDRSxNQUFBLEdBQUcsRUFBRUEsTUFBTSxDQUFDQyxFQURkO0FBRUUsTUFBQSxlQUFlLEVBQUUsS0FBS2xDLEtBQUwsQ0FBV21DLGVBRjlCO0FBR0UsTUFBQSxrQkFBa0IsRUFBRSxLQUFLbkMsS0FBTCxDQUFXb0Msa0JBSGpDO0FBSUUsTUFBQSxNQUFNLEVBQUVILE1BSlY7QUFLRSxNQUFBLFFBQVEsRUFBRUgsUUFBUSxDQUFDbkMsVUFMckI7QUFNRSxNQUFBLHlCQUF5QixFQUFFLEtBQUtLLEtBQUwsQ0FBV3FDO0FBTnhDLE1BREQsQ0FESCxDQURGO0FBY0Q7O0FBRURMLEVBQUFBLHFCQUFxQixDQUFDRixRQUFELEVBQVc7QUFDOUIsUUFDRUEsUUFBUSxDQUFDcEMsa0JBQVQsQ0FBNEI0QyxNQUE1QixLQUF1QyxDQUF2QyxJQUNBLEtBQUszQixLQUFMLENBQVdDLFdBQVgsQ0FBdUIwQixNQUF2QixLQUFrQyxDQURsQyxJQUVBLENBQUMsS0FBS3RDLEtBQUwsQ0FBV3VCLE1BQVgsQ0FBa0JnQixHQUFsQixDQUFzQixvQ0FBdEIsQ0FISCxFQUlFO0FBQ0EsYUFBTyxFQUFQO0FBQ0Q7O0FBRUQsVUFBTUMsY0FBYyxHQUFHLEtBQUt4QyxLQUFMLENBQVdFLFVBQVgsQ0FBc0JFLHVCQUF0QixFQUF2QjtBQUNBLFVBQU1xQyxzQkFBc0IsR0FBRyxJQUFJQyxHQUFKLENBQzdCWixRQUFRLENBQUNwQyxrQkFBVCxDQUE0QmMsR0FBNUIsQ0FBZ0NtQyxZQUFZLElBQUlDLGNBQUtDLElBQUwsQ0FBVUwsY0FBVixFQUEwQkcsWUFBMUIsQ0FBaEQsQ0FENkIsQ0FBL0I7QUFJQSxXQUFPLEtBQUtoQyxLQUFMLENBQVdDLFdBQVgsQ0FBdUJrQyxNQUF2QixDQUE4QmIsTUFBTSxJQUFJUSxzQkFBc0IsQ0FBQ00sR0FBdkIsQ0FBMkJkLE1BQU0sQ0FBQ2UsT0FBUCxFQUEzQixDQUF4QyxDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLG9CQUFvQixHQUFHO0FBQ3JCLFNBQUtsQyxhQUFMLENBQW1CbUMsT0FBbkI7QUFDRDs7QUE1RnVFOzs7O2dCQUFyRHRELDRCLGVBQ0E7QUFDakJpQixFQUFBQSxTQUFTLEVBQUVzQyxtQkFBVUMsTUFBVixDQUFpQkMsVUFEWDtBQUVqQmxCLEVBQUFBLGVBQWUsRUFBRWdCLG1CQUFVQyxNQUFWLENBQWlCQyxVQUZqQjtBQUdqQjlCLEVBQUFBLE1BQU0sRUFBRTRCLG1CQUFVQyxNQUFWLENBQWlCQyxVQUhSO0FBSWpCakIsRUFBQUEsa0JBQWtCLEVBQUVlLG1CQUFVQyxNQUFWLENBQWlCQyxVQUpwQjtBQUtqQm5ELEVBQUFBLFVBQVUsRUFBRWlELG1CQUFVQyxNQUFWLENBQWlCQyxVQUxaO0FBTWpCaEIsRUFBQUEseUJBQXlCLEVBQUVjLG1CQUFVRztBQU5wQixDOztnQkFEQTFELDRCLGtCQVVHO0FBQ3BCeUMsRUFBQUEseUJBQXlCLEVBQUUsTUFBTSxDQUFFLENBRGY7QUFFcEJELEVBQUFBLGtCQUFrQixFQUFFLElBQUltQiwyQkFBSjtBQUZBLEMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNi4xL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IFJlYWN0IGZyb20gJ3JlYWN0JztcbmltcG9ydCBQcm9wVHlwZXMgZnJvbSAncHJvcC10eXBlcyc7XG5pbXBvcnQgeXViaWtpcmkgZnJvbSAneXViaWtpcmknO1xuaW1wb3J0IHtDb21wb3NpdGVEaXNwb3NhYmxlfSBmcm9tICdldmVudC1raXQnO1xuXG5pbXBvcnQgT2JzZXJ2ZU1vZGVsIGZyb20gJy4uL3ZpZXdzL29ic2VydmUtbW9kZWwnO1xuaW1wb3J0IFJlc29sdXRpb25Qcm9ncmVzcyBmcm9tICcuLi9tb2RlbHMvY29uZmxpY3RzL3Jlc29sdXRpb24tcHJvZ3Jlc3MnO1xuaW1wb3J0IEVkaXRvckNvbmZsaWN0Q29udHJvbGxlciBmcm9tICcuL2VkaXRvci1jb25mbGljdC1jb250cm9sbGVyJztcblxuY29uc3QgREVGQVVMVF9SRVBPX0RBVEEgPSB7XG4gIG1lcmdlQ29uZmxpY3RQYXRoczogW10sXG4gIGlzUmViYXNpbmc6IGZhbHNlLFxufTtcblxuLyoqXG4gKiBSZW5kZXIgYW4gYEVkaXRvckNvbmZsaWN0Q29udHJvbGxlcmAgZm9yIGVhY2ggYFRleHRFZGl0b3JgIG9wZW4gb24gYSBmaWxlIHRoYXQgY29udGFpbnMgZ2l0IGNvbmZsaWN0IG1hcmtlcnMuXG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIFJlcG9zaXRvcnlDb25mbGljdENvbnRyb2xsZXIgZXh0ZW5kcyBSZWFjdC5Db21wb25lbnQge1xuICBzdGF0aWMgcHJvcFR5cGVzID0ge1xuICAgIHdvcmtzcGFjZTogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIGNvbW1hbmRSZWdpc3RyeTogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIGNvbmZpZzogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIHJlc29sdXRpb25Qcm9ncmVzczogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIHJlcG9zaXRvcnk6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICByZWZyZXNoUmVzb2x1dGlvblByb2dyZXNzOiBQcm9wVHlwZXMuZnVuYyxcbiAgfTtcblxuICBzdGF0aWMgZGVmYXVsdFByb3BzID0ge1xuICAgIHJlZnJlc2hSZXNvbHV0aW9uUHJvZ3Jlc3M6ICgpID0+IHt9LFxuICAgIHJlc29sdXRpb25Qcm9ncmVzczogbmV3IFJlc29sdXRpb25Qcm9ncmVzcygpLFxuICB9O1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzLCBjb250ZXh0KSB7XG4gICAgc3VwZXIocHJvcHMsIGNvbnRleHQpO1xuXG4gICAgdGhpcy5zdGF0ZSA9IHtvcGVuRWRpdG9yczogdGhpcy5wcm9wcy53b3Jrc3BhY2UuZ2V0VGV4dEVkaXRvcnMoKX07XG4gICAgdGhpcy5zdWJzY3JpcHRpb25zID0gbmV3IENvbXBvc2l0ZURpc3Bvc2FibGUoKTtcbiAgfVxuXG4gIGNvbXBvbmVudERpZE1vdW50KCkge1xuICAgIGNvbnN0IHVwZGF0ZVN0YXRlID0gKCkgPT4ge1xuICAgICAgdGhpcy5zZXRTdGF0ZSh7XG4gICAgICAgIG9wZW5FZGl0b3JzOiB0aGlzLnByb3BzLndvcmtzcGFjZS5nZXRUZXh0RWRpdG9ycygpLFxuICAgICAgfSk7XG4gICAgfTtcblxuICAgIHRoaXMuc3Vic2NyaXB0aW9ucy5hZGQoXG4gICAgICB0aGlzLnByb3BzLndvcmtzcGFjZS5vYnNlcnZlVGV4dEVkaXRvcnModXBkYXRlU3RhdGUpLFxuICAgICAgdGhpcy5wcm9wcy53b3Jrc3BhY2Uub25EaWREZXN0cm95UGFuZUl0ZW0odXBkYXRlU3RhdGUpLFxuICAgICAgdGhpcy5wcm9wcy5jb25maWcub2JzZXJ2ZSgnZ2l0aHViLmdyYXBoaWNhbENvbmZsaWN0UmVzb2x1dGlvbicsICgpID0+IHRoaXMuZm9yY2VVcGRhdGUoKSksXG4gICAgKTtcbiAgfVxuXG4gIGZldGNoRGF0YSA9IHJlcG9zaXRvcnkgPT4ge1xuICAgIHJldHVybiB5dWJpa2lyaSh7XG4gICAgICB3b3JraW5nRGlyZWN0b3J5UGF0aDogcmVwb3NpdG9yeS5nZXRXb3JraW5nRGlyZWN0b3J5UGF0aCgpLFxuICAgICAgbWVyZ2VDb25mbGljdFBhdGhzOiByZXBvc2l0b3J5LmdldE1lcmdlQ29uZmxpY3RzKCkudGhlbihjb25mbGljdHMgPT4ge1xuICAgICAgICByZXR1cm4gY29uZmxpY3RzLm1hcChjb25mbGljdCA9PiBjb25mbGljdC5maWxlUGF0aCk7XG4gICAgICB9KSxcbiAgICAgIGlzUmViYXNpbmc6IHJlcG9zaXRvcnkuaXNSZWJhc2luZygpLFxuICAgIH0pO1xuICB9XG5cbiAgcmVuZGVyKCkge1xuICAgIHJldHVybiAoXG4gICAgICA8T2JzZXJ2ZU1vZGVsIG1vZGVsPXt0aGlzLnByb3BzLnJlcG9zaXRvcnl9IGZldGNoRGF0YT17dGhpcy5mZXRjaERhdGF9PlxuICAgICAgICB7ZGF0YSA9PiB0aGlzLnJlbmRlcldpdGhEYXRhKGRhdGEgfHwgREVGQVVMVF9SRVBPX0RBVEEpfVxuICAgICAgPC9PYnNlcnZlTW9kZWw+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlcldpdGhEYXRhKHJlcG9EYXRhKSB7XG4gICAgY29uc3QgY29uZmxpY3RpbmdFZGl0b3JzID0gdGhpcy5nZXRDb25mbGljdGluZ0VkaXRvcnMocmVwb0RhdGEpO1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxkaXY+XG4gICAgICAgIHtjb25mbGljdGluZ0VkaXRvcnMubWFwKGVkaXRvciA9PiAoXG4gICAgICAgICAgPEVkaXRvckNvbmZsaWN0Q29udHJvbGxlclxuICAgICAgICAgICAga2V5PXtlZGl0b3IuaWR9XG4gICAgICAgICAgICBjb21tYW5kUmVnaXN0cnk9e3RoaXMucHJvcHMuY29tbWFuZFJlZ2lzdHJ5fVxuICAgICAgICAgICAgcmVzb2x1dGlvblByb2dyZXNzPXt0aGlzLnByb3BzLnJlc29sdXRpb25Qcm9ncmVzc31cbiAgICAgICAgICAgIGVkaXRvcj17ZWRpdG9yfVxuICAgICAgICAgICAgaXNSZWJhc2U9e3JlcG9EYXRhLmlzUmViYXNpbmd9XG4gICAgICAgICAgICByZWZyZXNoUmVzb2x1dGlvblByb2dyZXNzPXt0aGlzLnByb3BzLnJlZnJlc2hSZXNvbHV0aW9uUHJvZ3Jlc3N9XG4gICAgICAgICAgLz5cbiAgICAgICAgKSl9XG4gICAgICA8L2Rpdj5cbiAgICApO1xuICB9XG5cbiAgZ2V0Q29uZmxpY3RpbmdFZGl0b3JzKHJlcG9EYXRhKSB7XG4gICAgaWYgKFxuICAgICAgcmVwb0RhdGEubWVyZ2VDb25mbGljdFBhdGhzLmxlbmd0aCA9PT0gMCB8fFxuICAgICAgdGhpcy5zdGF0ZS5vcGVuRWRpdG9ycy5sZW5ndGggPT09IDAgfHxcbiAgICAgICF0aGlzLnByb3BzLmNvbmZpZy5nZXQoJ2dpdGh1Yi5ncmFwaGljYWxDb25mbGljdFJlc29sdXRpb24nKVxuICAgICkge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cblxuICAgIGNvbnN0IGNvbW1vbkJhc2VQYXRoID0gdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmdldFdvcmtpbmdEaXJlY3RvcnlQYXRoKCk7XG4gICAgY29uc3QgZnVsbE1lcmdlQ29uZmxpY3RQYXRocyA9IG5ldyBTZXQoXG4gICAgICByZXBvRGF0YS5tZXJnZUNvbmZsaWN0UGF0aHMubWFwKHJlbGF0aXZlUGF0aCA9PiBwYXRoLmpvaW4oY29tbW9uQmFzZVBhdGgsIHJlbGF0aXZlUGF0aCkpLFxuICAgICk7XG5cbiAgICByZXR1cm4gdGhpcy5zdGF0ZS5vcGVuRWRpdG9ycy5maWx0ZXIoZWRpdG9yID0+IGZ1bGxNZXJnZUNvbmZsaWN0UGF0aHMuaGFzKGVkaXRvci5nZXRQYXRoKCkpKTtcbiAgfVxuXG4gIGNvbXBvbmVudFdpbGxVbm1vdW50KCkge1xuICAgIHRoaXMuc3Vic2NyaXB0aW9ucy5kaXNwb3NlKCk7XG4gIH1cbn1cbiJdfQ==