"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _path = _interopRequireDefault(require("path"));

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

var _propTypes2 = require("../prop-types");

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _multiFilePatchView = _interopRequireDefault(require("../views/multi-file-patch-view"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultiFilePatchController extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'selectedRowsChanged', 'undoLastDiscard', 'diveIntoMirrorPatch', 'openFile', 'toggleFile', 'toggleRows', 'toggleModeChange', 'toggleSymlinkChange', 'discardRows');
    this.state = {
      selectionMode: 'hunk',
      selectedRows: new Set(),
      hasMultipleFileSelections: false
    };
    this.mouseSelectionInProgress = false;
    this.stagingOperationInProgress = false;
    this.lastPatchString = null;
    this.patchChangePromise = new Promise(resolve => {
      this.resolvePatchChangePromise = resolve;
    });
  }

  componentDidUpdate(prevProps) {
    if (this.lastPatchString !== null && this.lastPatchString !== this.props.multiFilePatch.toString()) {
      this.resolvePatchChangePromise();
      this.patchChangePromise = new Promise(resolve => {
        this.resolvePatchChangePromise = resolve;
      });
    }
  }

  render() {
    return _react.default.createElement(_multiFilePatchView.default, _extends({}, this.props, {
      selectedRows: this.state.selectedRows,
      selectionMode: this.state.selectionMode,
      hasMultipleFileSelections: this.state.hasMultipleFileSelections,
      selectedRowsChanged: this.selectedRowsChanged,
      diveIntoMirrorPatch: this.diveIntoMirrorPatch,
      openFile: this.openFile,
      toggleFile: this.toggleFile,
      toggleRows: this.toggleRows,
      toggleModeChange: this.toggleModeChange,
      toggleSymlinkChange: this.toggleSymlinkChange,
      undoLastDiscard: this.undoLastDiscard,
      discardRows: this.discardRows,
      selectNextHunk: this.selectNextHunk,
      selectPreviousHunk: this.selectPreviousHunk,
      switchToIssueish: this.props.switchToIssueish
    }));
  }

  undoLastDiscard(filePatch, {
    eventSource
  } = {}) {
    (0, _reporterProxy.addEvent)('undo-last-discard', {
      package: 'github',
      component: this.constructor.name,
      eventSource
    });
    return this.props.undoLastDiscard(filePatch.getPath(), this.props.repository);
  }

  diveIntoMirrorPatch(filePatch) {
    const mirrorStatus = this.withStagingStatus({
      staged: 'unstaged',
      unstaged: 'staged'
    });
    const workingDirectory = this.props.repository.getWorkingDirectoryPath();

    const uri = _changedFileItem.default.buildURI(filePatch.getPath(), workingDirectory, mirrorStatus);

    this.props.destroy();
    return this.props.workspace.open(uri);
  }

  async openFile(filePatch, positions, pending) {
    const absolutePath = _path.default.join(this.props.repository.getWorkingDirectoryPath(), filePatch.getPath());

    const editor = await this.props.workspace.open(absolutePath, {
      pending
    });

    if (positions.length > 0) {
      editor.setCursorBufferPosition(positions[0], {
        autoscroll: false
      });

      for (const position of positions.slice(1)) {
        editor.addCursorAtBufferPosition(position);
      }

      editor.scrollToBufferPosition(positions[positions.length - 1], {
        center: true
      });
    }

    return editor;
  }

  toggleFile(filePatch) {
    return this.stagingOperation(() => {
      const methodName = this.withStagingStatus({
        staged: 'unstageFiles',
        unstaged: 'stageFiles'
      });
      return this.props.repository[methodName]([filePatch.getPath()]);
    });
  }

  async toggleRows(rowSet, nextSelectionMode) {
    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    if (chosenRows.size === 0) {
      return Promise.resolve();
    }

    return this.stagingOperation(() => {
      const patch = this.withStagingStatus({
        staged: () => this.props.multiFilePatch.getUnstagePatchForLines(chosenRows),
        unstaged: () => this.props.multiFilePatch.getStagePatchForLines(chosenRows)
      });
      return this.props.repository.applyPatchToIndex(patch);
    });
  }

  toggleModeChange(filePatch) {
    return this.stagingOperation(() => {
      const targetMode = this.withStagingStatus({
        unstaged: filePatch.getNewMode(),
        staged: filePatch.getOldMode()
      });
      return this.props.repository.stageFileModeChange(filePatch.getPath(), targetMode);
    });
  }

  toggleSymlinkChange(filePatch) {
    return this.stagingOperation(() => {
      const relPath = filePatch.getPath();
      const repository = this.props.repository;
      return this.withStagingStatus({
        unstaged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'added') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.stageFiles([relPath]);
        },
        staged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'deleted') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.unstageFiles([relPath]);
        }
      });
    });
  }

  async discardRows(rowSet, nextSelectionMode, {
    eventSource
  } = {}) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    // This check is duplicated in RootController#discardLines. We also want it here to prevent us from sending metrics
    // unnecessarily
    if (this.props.multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: this.constructor.name,
      lineCount: chosenRows.size,
      eventSource
    });
    return this.props.discardLines(this.props.multiFilePatch, chosenRows, this.props.repository);
  }

  selectedRowsChanged(rows, nextSelectionMode, nextMultipleFileSelections) {
    if ((0, _helpers.equalSets)(this.state.selectedRows, rows) && this.state.selectionMode === nextSelectionMode && this.state.hasMultipleFileSelections === nextMultipleFileSelections) {
      return Promise.resolve();
    }

    return new Promise(resolve => {
      this.setState({
        selectedRows: rows,
        selectionMode: nextSelectionMode,
        hasMultipleFileSelections: nextMultipleFileSelections
      }, resolve);
    });
  }

  withStagingStatus(callbacks) {
    const callback = callbacks[this.props.stagingStatus];
    /* istanbul ignore if */

    if (!callback) {
      throw new Error(`Unknown staging status: ${this.props.stagingStatus}`);
    }

    return callback instanceof Function ? callback() : callback;
  }

  stagingOperation(fn) {
    if (this.stagingOperationInProgress) {
      return null;
    }

    this.stagingOperationInProgress = true;
    this.lastPatchString = this.props.multiFilePatch.toString();
    const operationPromise = fn();
    operationPromise.then(() => this.patchChangePromise).then(() => {
      this.stagingOperationInProgress = false;
      this.lastPatchString = null;
    });
    return operationPromise;
  }

}

exports.default = MultiFilePatchController;

_defineProperty(MultiFilePatchController, "propTypes", {
  repository: _propTypes.default.object.isRequired,
  stagingStatus: _propTypes.default.oneOf(['staged', 'unstaged']),
  multiFilePatch: _propTypes2.MultiFilePatchPropType.isRequired,
  hasUndoHistory: _propTypes.default.bool,
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  destroy: _propTypes.default.func.isRequired,
  discardLines: _propTypes.default.func,
  undoLastDiscard: _propTypes.default.func,
  surface: _propTypes.default.func,
  switchToIssueish: _propTypes.default.func
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm11bHRpLWZpbGUtcGF0Y2gtY29udHJvbGxlci5qcyJdLCJuYW1lcyI6WyJNdWx0aUZpbGVQYXRjaENvbnRyb2xsZXIiLCJSZWFjdCIsIkNvbXBvbmVudCIsImNvbnN0cnVjdG9yIiwicHJvcHMiLCJzdGF0ZSIsInNlbGVjdGlvbk1vZGUiLCJzZWxlY3RlZFJvd3MiLCJTZXQiLCJoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zIiwibW91c2VTZWxlY3Rpb25JblByb2dyZXNzIiwic3RhZ2luZ09wZXJhdGlvbkluUHJvZ3Jlc3MiLCJsYXN0UGF0Y2hTdHJpbmciLCJwYXRjaENoYW5nZVByb21pc2UiLCJQcm9taXNlIiwicmVzb2x2ZSIsInJlc29sdmVQYXRjaENoYW5nZVByb21pc2UiLCJjb21wb25lbnREaWRVcGRhdGUiLCJwcmV2UHJvcHMiLCJtdWx0aUZpbGVQYXRjaCIsInRvU3RyaW5nIiwicmVuZGVyIiwic2VsZWN0ZWRSb3dzQ2hhbmdlZCIsImRpdmVJbnRvTWlycm9yUGF0Y2giLCJvcGVuRmlsZSIsInRvZ2dsZUZpbGUiLCJ0b2dnbGVSb3dzIiwidG9nZ2xlTW9kZUNoYW5nZSIsInRvZ2dsZVN5bWxpbmtDaGFuZ2UiLCJ1bmRvTGFzdERpc2NhcmQiLCJkaXNjYXJkUm93cyIsInNlbGVjdE5leHRIdW5rIiwic2VsZWN0UHJldmlvdXNIdW5rIiwic3dpdGNoVG9Jc3N1ZWlzaCIsImZpbGVQYXRjaCIsImV2ZW50U291cmNlIiwicGFja2FnZSIsImNvbXBvbmVudCIsIm5hbWUiLCJnZXRQYXRoIiwicmVwb3NpdG9yeSIsIm1pcnJvclN0YXR1cyIsIndpdGhTdGFnaW5nU3RhdHVzIiwic3RhZ2VkIiwidW5zdGFnZWQiLCJ3b3JraW5nRGlyZWN0b3J5IiwiZ2V0V29ya2luZ0RpcmVjdG9yeVBhdGgiLCJ1cmkiLCJDaGFuZ2VkRmlsZUl0ZW0iLCJidWlsZFVSSSIsImRlc3Ryb3kiLCJ3b3Jrc3BhY2UiLCJvcGVuIiwicG9zaXRpb25zIiwicGVuZGluZyIsImFic29sdXRlUGF0aCIsInBhdGgiLCJqb2luIiwiZWRpdG9yIiwibGVuZ3RoIiwic2V0Q3Vyc29yQnVmZmVyUG9zaXRpb24iLCJhdXRvc2Nyb2xsIiwicG9zaXRpb24iLCJzbGljZSIsImFkZEN1cnNvckF0QnVmZmVyUG9zaXRpb24iLCJzY3JvbGxUb0J1ZmZlclBvc2l0aW9uIiwiY2VudGVyIiwic3RhZ2luZ09wZXJhdGlvbiIsIm1ldGhvZE5hbWUiLCJyb3dTZXQiLCJuZXh0U2VsZWN0aW9uTW9kZSIsImNob3NlblJvd3MiLCJuZXh0TXVsdGlwbGVGaWxlU2VsZWN0aW9ucyIsInNwYW5zTXVsdGlwbGVGaWxlcyIsInNpemUiLCJwYXRjaCIsImdldFVuc3RhZ2VQYXRjaEZvckxpbmVzIiwiZ2V0U3RhZ2VQYXRjaEZvckxpbmVzIiwiYXBwbHlQYXRjaFRvSW5kZXgiLCJ0YXJnZXRNb2RlIiwiZ2V0TmV3TW9kZSIsImdldE9sZE1vZGUiLCJzdGFnZUZpbGVNb2RlQ2hhbmdlIiwicmVsUGF0aCIsImhhc1R5cGVjaGFuZ2UiLCJnZXRTdGF0dXMiLCJzdGFnZUZpbGVTeW1saW5rQ2hhbmdlIiwic3RhZ2VGaWxlcyIsInVuc3RhZ2VGaWxlcyIsImdldEZpbGVQYXRjaGVzIiwibGluZUNvdW50IiwiZGlzY2FyZExpbmVzIiwicm93cyIsInNldFN0YXRlIiwiY2FsbGJhY2tzIiwiY2FsbGJhY2siLCJzdGFnaW5nU3RhdHVzIiwiRXJyb3IiLCJGdW5jdGlvbiIsImZuIiwib3BlcmF0aW9uUHJvbWlzZSIsInRoZW4iLCJQcm9wVHlwZXMiLCJvYmplY3QiLCJpc1JlcXVpcmVkIiwib25lT2YiLCJNdWx0aUZpbGVQYXRjaFByb3BUeXBlIiwiaGFzVW5kb0hpc3RvcnkiLCJib29sIiwiY29tbWFuZHMiLCJrZXltYXBzIiwidG9vbHRpcHMiLCJjb25maWciLCJmdW5jIiwic3VyZmFjZSJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOzs7Ozs7OztBQUVlLE1BQU1BLHdCQUFOLFNBQXVDQyxlQUFNQyxTQUE3QyxDQUF1RDtBQXFCcEVDLEVBQUFBLFdBQVcsQ0FBQ0MsS0FBRCxFQUFRO0FBQ2pCLFVBQU1BLEtBQU47QUFDQSwyQkFDRSxJQURGLEVBRUUscUJBRkYsRUFHRSxpQkFIRixFQUdxQixxQkFIckIsRUFHNEMsVUFINUMsRUFJRSxZQUpGLEVBSWdCLFlBSmhCLEVBSThCLGtCQUo5QixFQUlrRCxxQkFKbEQsRUFJeUUsYUFKekU7QUFPQSxTQUFLQyxLQUFMLEdBQWE7QUFDWEMsTUFBQUEsYUFBYSxFQUFFLE1BREo7QUFFWEMsTUFBQUEsWUFBWSxFQUFFLElBQUlDLEdBQUosRUFGSDtBQUdYQyxNQUFBQSx5QkFBeUIsRUFBRTtBQUhoQixLQUFiO0FBTUEsU0FBS0Msd0JBQUwsR0FBZ0MsS0FBaEM7QUFDQSxTQUFLQywwQkFBTCxHQUFrQyxLQUFsQztBQUVBLFNBQUtDLGVBQUwsR0FBdUIsSUFBdkI7QUFDQSxTQUFLQyxrQkFBTCxHQUEwQixJQUFJQyxPQUFKLENBQVlDLE9BQU8sSUFBSTtBQUMvQyxXQUFLQyx5QkFBTCxHQUFpQ0QsT0FBakM7QUFDRCxLQUZ5QixDQUExQjtBQUdEOztBQUVERSxFQUFBQSxrQkFBa0IsQ0FBQ0MsU0FBRCxFQUFZO0FBQzVCLFFBQ0UsS0FBS04sZUFBTCxLQUF5QixJQUF6QixJQUNBLEtBQUtBLGVBQUwsS0FBeUIsS0FBS1IsS0FBTCxDQUFXZSxjQUFYLENBQTBCQyxRQUExQixFQUYzQixFQUdFO0FBQ0EsV0FBS0oseUJBQUw7QUFDQSxXQUFLSCxrQkFBTCxHQUEwQixJQUFJQyxPQUFKLENBQVlDLE9BQU8sSUFBSTtBQUMvQyxhQUFLQyx5QkFBTCxHQUFpQ0QsT0FBakM7QUFDRCxPQUZ5QixDQUExQjtBQUdEO0FBQ0Y7O0FBRURNLEVBQUFBLE1BQU0sR0FBRztBQUNQLFdBQ0UsNkJBQUMsMkJBQUQsZUFDTSxLQUFLakIsS0FEWDtBQUdFLE1BQUEsWUFBWSxFQUFFLEtBQUtDLEtBQUwsQ0FBV0UsWUFIM0I7QUFJRSxNQUFBLGFBQWEsRUFBRSxLQUFLRixLQUFMLENBQVdDLGFBSjVCO0FBS0UsTUFBQSx5QkFBeUIsRUFBRSxLQUFLRCxLQUFMLENBQVdJLHlCQUx4QztBQU1FLE1BQUEsbUJBQW1CLEVBQUUsS0FBS2EsbUJBTjVCO0FBUUUsTUFBQSxtQkFBbUIsRUFBRSxLQUFLQyxtQkFSNUI7QUFTRSxNQUFBLFFBQVEsRUFBRSxLQUFLQyxRQVRqQjtBQVVFLE1BQUEsVUFBVSxFQUFFLEtBQUtDLFVBVm5CO0FBV0UsTUFBQSxVQUFVLEVBQUUsS0FBS0MsVUFYbkI7QUFZRSxNQUFBLGdCQUFnQixFQUFFLEtBQUtDLGdCQVp6QjtBQWFFLE1BQUEsbUJBQW1CLEVBQUUsS0FBS0MsbUJBYjVCO0FBY0UsTUFBQSxlQUFlLEVBQUUsS0FBS0MsZUFkeEI7QUFlRSxNQUFBLFdBQVcsRUFBRSxLQUFLQyxXQWZwQjtBQWdCRSxNQUFBLGNBQWMsRUFBRSxLQUFLQyxjQWhCdkI7QUFpQkUsTUFBQSxrQkFBa0IsRUFBRSxLQUFLQyxrQkFqQjNCO0FBa0JFLE1BQUEsZ0JBQWdCLEVBQUUsS0FBSzVCLEtBQUwsQ0FBVzZCO0FBbEIvQixPQURGO0FBc0JEOztBQUVESixFQUFBQSxlQUFlLENBQUNLLFNBQUQsRUFBWTtBQUFDQyxJQUFBQTtBQUFELE1BQWdCLEVBQTVCLEVBQWdDO0FBQzdDLGlDQUFTLG1CQUFULEVBQThCO0FBQzVCQyxNQUFBQSxPQUFPLEVBQUUsUUFEbUI7QUFFNUJDLE1BQUFBLFNBQVMsRUFBRSxLQUFLbEMsV0FBTCxDQUFpQm1DLElBRkE7QUFHNUJILE1BQUFBO0FBSDRCLEtBQTlCO0FBTUEsV0FBTyxLQUFLL0IsS0FBTCxDQUFXeUIsZUFBWCxDQUEyQkssU0FBUyxDQUFDSyxPQUFWLEVBQTNCLEVBQWdELEtBQUtuQyxLQUFMLENBQVdvQyxVQUEzRCxDQUFQO0FBQ0Q7O0FBRURqQixFQUFBQSxtQkFBbUIsQ0FBQ1csU0FBRCxFQUFZO0FBQzdCLFVBQU1PLFlBQVksR0FBRyxLQUFLQyxpQkFBTCxDQUF1QjtBQUFDQyxNQUFBQSxNQUFNLEVBQUUsVUFBVDtBQUFxQkMsTUFBQUEsUUFBUSxFQUFFO0FBQS9CLEtBQXZCLENBQXJCO0FBQ0EsVUFBTUMsZ0JBQWdCLEdBQUcsS0FBS3pDLEtBQUwsQ0FBV29DLFVBQVgsQ0FBc0JNLHVCQUF0QixFQUF6Qjs7QUFDQSxVQUFNQyxHQUFHLEdBQUdDLHlCQUFnQkMsUUFBaEIsQ0FBeUJmLFNBQVMsQ0FBQ0ssT0FBVixFQUF6QixFQUE4Q00sZ0JBQTlDLEVBQWdFSixZQUFoRSxDQUFaOztBQUVBLFNBQUtyQyxLQUFMLENBQVc4QyxPQUFYO0FBQ0EsV0FBTyxLQUFLOUMsS0FBTCxDQUFXK0MsU0FBWCxDQUFxQkMsSUFBckIsQ0FBMEJMLEdBQTFCLENBQVA7QUFDRDs7QUFFRCxRQUFNdkIsUUFBTixDQUFlVSxTQUFmLEVBQTBCbUIsU0FBMUIsRUFBcUNDLE9BQXJDLEVBQThDO0FBQzVDLFVBQU1DLFlBQVksR0FBR0MsY0FBS0MsSUFBTCxDQUFVLEtBQUtyRCxLQUFMLENBQVdvQyxVQUFYLENBQXNCTSx1QkFBdEIsRUFBVixFQUEyRFosU0FBUyxDQUFDSyxPQUFWLEVBQTNELENBQXJCOztBQUNBLFVBQU1tQixNQUFNLEdBQUcsTUFBTSxLQUFLdEQsS0FBTCxDQUFXK0MsU0FBWCxDQUFxQkMsSUFBckIsQ0FBMEJHLFlBQTFCLEVBQXdDO0FBQUNELE1BQUFBO0FBQUQsS0FBeEMsQ0FBckI7O0FBQ0EsUUFBSUQsU0FBUyxDQUFDTSxNQUFWLEdBQW1CLENBQXZCLEVBQTBCO0FBQ3hCRCxNQUFBQSxNQUFNLENBQUNFLHVCQUFQLENBQStCUCxTQUFTLENBQUMsQ0FBRCxDQUF4QyxFQUE2QztBQUFDUSxRQUFBQSxVQUFVLEVBQUU7QUFBYixPQUE3Qzs7QUFDQSxXQUFLLE1BQU1DLFFBQVgsSUFBdUJULFNBQVMsQ0FBQ1UsS0FBVixDQUFnQixDQUFoQixDQUF2QixFQUEyQztBQUN6Q0wsUUFBQUEsTUFBTSxDQUFDTSx5QkFBUCxDQUFpQ0YsUUFBakM7QUFDRDs7QUFDREosTUFBQUEsTUFBTSxDQUFDTyxzQkFBUCxDQUE4QlosU0FBUyxDQUFDQSxTQUFTLENBQUNNLE1BQVYsR0FBbUIsQ0FBcEIsQ0FBdkMsRUFBK0Q7QUFBQ08sUUFBQUEsTUFBTSxFQUFFO0FBQVQsT0FBL0Q7QUFDRDs7QUFDRCxXQUFPUixNQUFQO0FBQ0Q7O0FBRURqQyxFQUFBQSxVQUFVLENBQUNTLFNBQUQsRUFBWTtBQUNwQixXQUFPLEtBQUtpQyxnQkFBTCxDQUFzQixNQUFNO0FBQ2pDLFlBQU1DLFVBQVUsR0FBRyxLQUFLMUIsaUJBQUwsQ0FBdUI7QUFBQ0MsUUFBQUEsTUFBTSxFQUFFLGNBQVQ7QUFBeUJDLFFBQUFBLFFBQVEsRUFBRTtBQUFuQyxPQUF2QixDQUFuQjtBQUNBLGFBQU8sS0FBS3hDLEtBQUwsQ0FBV29DLFVBQVgsQ0FBc0I0QixVQUF0QixFQUFrQyxDQUFDbEMsU0FBUyxDQUFDSyxPQUFWLEVBQUQsQ0FBbEMsQ0FBUDtBQUNELEtBSE0sQ0FBUDtBQUlEOztBQUVELFFBQU1iLFVBQU4sQ0FBaUIyQyxNQUFqQixFQUF5QkMsaUJBQXpCLEVBQTRDO0FBQzFDLFFBQUlDLFVBQVUsR0FBR0YsTUFBakI7O0FBQ0EsUUFBSUUsVUFBSixFQUFnQjtBQUNkLFlBQU1DLDBCQUEwQixHQUFHLEtBQUtwRSxLQUFMLENBQVdlLGNBQVgsQ0FBMEJzRCxrQkFBMUIsQ0FBNkNGLFVBQTdDLENBQW5DO0FBQ0EsWUFBTSxLQUFLakQsbUJBQUwsQ0FBeUJpRCxVQUF6QixFQUFxQ0QsaUJBQXJDLEVBQXdERSwwQkFBeEQsQ0FBTjtBQUNELEtBSEQsTUFHTztBQUNMRCxNQUFBQSxVQUFVLEdBQUcsS0FBS2xFLEtBQUwsQ0FBV0UsWUFBeEI7QUFDRDs7QUFFRCxRQUFJZ0UsVUFBVSxDQUFDRyxJQUFYLEtBQW9CLENBQXhCLEVBQTJCO0FBQ3pCLGFBQU81RCxPQUFPLENBQUNDLE9BQVIsRUFBUDtBQUNEOztBQUVELFdBQU8sS0FBS29ELGdCQUFMLENBQXNCLE1BQU07QUFDakMsWUFBTVEsS0FBSyxHQUFHLEtBQUtqQyxpQkFBTCxDQUF1QjtBQUNuQ0MsUUFBQUEsTUFBTSxFQUFFLE1BQU0sS0FBS3ZDLEtBQUwsQ0FBV2UsY0FBWCxDQUEwQnlELHVCQUExQixDQUFrREwsVUFBbEQsQ0FEcUI7QUFFbkMzQixRQUFBQSxRQUFRLEVBQUUsTUFBTSxLQUFLeEMsS0FBTCxDQUFXZSxjQUFYLENBQTBCMEQscUJBQTFCLENBQWdETixVQUFoRDtBQUZtQixPQUF2QixDQUFkO0FBSUEsYUFBTyxLQUFLbkUsS0FBTCxDQUFXb0MsVUFBWCxDQUFzQnNDLGlCQUF0QixDQUF3Q0gsS0FBeEMsQ0FBUDtBQUNELEtBTk0sQ0FBUDtBQU9EOztBQUVEaEQsRUFBQUEsZ0JBQWdCLENBQUNPLFNBQUQsRUFBWTtBQUMxQixXQUFPLEtBQUtpQyxnQkFBTCxDQUFzQixNQUFNO0FBQ2pDLFlBQU1ZLFVBQVUsR0FBRyxLQUFLckMsaUJBQUwsQ0FBdUI7QUFDeENFLFFBQUFBLFFBQVEsRUFBRVYsU0FBUyxDQUFDOEMsVUFBVixFQUQ4QjtBQUV4Q3JDLFFBQUFBLE1BQU0sRUFBRVQsU0FBUyxDQUFDK0MsVUFBVjtBQUZnQyxPQUF2QixDQUFuQjtBQUlBLGFBQU8sS0FBSzdFLEtBQUwsQ0FBV29DLFVBQVgsQ0FBc0IwQyxtQkFBdEIsQ0FBMENoRCxTQUFTLENBQUNLLE9BQVYsRUFBMUMsRUFBK0R3QyxVQUEvRCxDQUFQO0FBQ0QsS0FOTSxDQUFQO0FBT0Q7O0FBRURuRCxFQUFBQSxtQkFBbUIsQ0FBQ00sU0FBRCxFQUFZO0FBQzdCLFdBQU8sS0FBS2lDLGdCQUFMLENBQXNCLE1BQU07QUFDakMsWUFBTWdCLE9BQU8sR0FBR2pELFNBQVMsQ0FBQ0ssT0FBVixFQUFoQjtBQUNBLFlBQU1DLFVBQVUsR0FBRyxLQUFLcEMsS0FBTCxDQUFXb0MsVUFBOUI7QUFDQSxhQUFPLEtBQUtFLGlCQUFMLENBQXVCO0FBQzVCRSxRQUFBQSxRQUFRLEVBQUUsTUFBTTtBQUNkLGNBQUlWLFNBQVMsQ0FBQ2tELGFBQVYsTUFBNkJsRCxTQUFTLENBQUNtRCxTQUFWLE9BQTBCLE9BQTNELEVBQW9FO0FBQ2xFLG1CQUFPN0MsVUFBVSxDQUFDOEMsc0JBQVgsQ0FBa0NILE9BQWxDLENBQVA7QUFDRDs7QUFFRCxpQkFBTzNDLFVBQVUsQ0FBQytDLFVBQVgsQ0FBc0IsQ0FBQ0osT0FBRCxDQUF0QixDQUFQO0FBQ0QsU0FQMkI7QUFRNUJ4QyxRQUFBQSxNQUFNLEVBQUUsTUFBTTtBQUNaLGNBQUlULFNBQVMsQ0FBQ2tELGFBQVYsTUFBNkJsRCxTQUFTLENBQUNtRCxTQUFWLE9BQTBCLFNBQTNELEVBQXNFO0FBQ3BFLG1CQUFPN0MsVUFBVSxDQUFDOEMsc0JBQVgsQ0FBa0NILE9BQWxDLENBQVA7QUFDRDs7QUFFRCxpQkFBTzNDLFVBQVUsQ0FBQ2dELFlBQVgsQ0FBd0IsQ0FBQ0wsT0FBRCxDQUF4QixDQUFQO0FBQ0Q7QUFkMkIsT0FBdkIsQ0FBUDtBQWdCRCxLQW5CTSxDQUFQO0FBb0JEOztBQUVELFFBQU1yRCxXQUFOLENBQWtCdUMsTUFBbEIsRUFBMEJDLGlCQUExQixFQUE2QztBQUFDbkMsSUFBQUE7QUFBRCxNQUFnQixFQUE3RCxFQUFpRTtBQUMvRDtBQUNBO0FBQ0E7QUFDQTtBQUNBLFFBQUksS0FBSy9CLEtBQUwsQ0FBV2UsY0FBWCxDQUEwQnNFLGNBQTFCLEdBQTJDOUIsTUFBM0MsS0FBc0QsQ0FBMUQsRUFBNkQ7QUFDM0QsYUFBTzdDLE9BQU8sQ0FBQ0MsT0FBUixDQUFnQixJQUFoQixDQUFQO0FBQ0Q7O0FBRUQsUUFBSXdELFVBQVUsR0FBR0YsTUFBakI7O0FBQ0EsUUFBSUUsVUFBSixFQUFnQjtBQUNkLFlBQU1DLDBCQUEwQixHQUFHLEtBQUtwRSxLQUFMLENBQVdlLGNBQVgsQ0FBMEJzRCxrQkFBMUIsQ0FBNkNGLFVBQTdDLENBQW5DO0FBQ0EsWUFBTSxLQUFLakQsbUJBQUwsQ0FBeUJpRCxVQUF6QixFQUFxQ0QsaUJBQXJDLEVBQXdERSwwQkFBeEQsQ0FBTjtBQUNELEtBSEQsTUFHTztBQUNMRCxNQUFBQSxVQUFVLEdBQUcsS0FBS2xFLEtBQUwsQ0FBV0UsWUFBeEI7QUFDRDs7QUFFRCxpQ0FBUywwQkFBVCxFQUFxQztBQUNuQzZCLE1BQUFBLE9BQU8sRUFBRSxRQUQwQjtBQUVuQ0MsTUFBQUEsU0FBUyxFQUFFLEtBQUtsQyxXQUFMLENBQWlCbUMsSUFGTztBQUduQ29ELE1BQUFBLFNBQVMsRUFBRW5CLFVBQVUsQ0FBQ0csSUFIYTtBQUluQ3ZDLE1BQUFBO0FBSm1DLEtBQXJDO0FBT0EsV0FBTyxLQUFLL0IsS0FBTCxDQUFXdUYsWUFBWCxDQUF3QixLQUFLdkYsS0FBTCxDQUFXZSxjQUFuQyxFQUFtRG9ELFVBQW5ELEVBQStELEtBQUtuRSxLQUFMLENBQVdvQyxVQUExRSxDQUFQO0FBQ0Q7O0FBRURsQixFQUFBQSxtQkFBbUIsQ0FBQ3NFLElBQUQsRUFBT3RCLGlCQUFQLEVBQTBCRSwwQkFBMUIsRUFBc0Q7QUFDdkUsUUFDRSx3QkFBVSxLQUFLbkUsS0FBTCxDQUFXRSxZQUFyQixFQUFtQ3FGLElBQW5DLEtBQ0EsS0FBS3ZGLEtBQUwsQ0FBV0MsYUFBWCxLQUE2QmdFLGlCQUQ3QixJQUVBLEtBQUtqRSxLQUFMLENBQVdJLHlCQUFYLEtBQXlDK0QsMEJBSDNDLEVBSUU7QUFDQSxhQUFPMUQsT0FBTyxDQUFDQyxPQUFSLEVBQVA7QUFDRDs7QUFFRCxXQUFPLElBQUlELE9BQUosQ0FBWUMsT0FBTyxJQUFJO0FBQzVCLFdBQUs4RSxRQUFMLENBQWM7QUFDWnRGLFFBQUFBLFlBQVksRUFBRXFGLElBREY7QUFFWnRGLFFBQUFBLGFBQWEsRUFBRWdFLGlCQUZIO0FBR1o3RCxRQUFBQSx5QkFBeUIsRUFBRStEO0FBSGYsT0FBZCxFQUlHekQsT0FKSDtBQUtELEtBTk0sQ0FBUDtBQU9EOztBQUVEMkIsRUFBQUEsaUJBQWlCLENBQUNvRCxTQUFELEVBQVk7QUFDM0IsVUFBTUMsUUFBUSxHQUFHRCxTQUFTLENBQUMsS0FBSzFGLEtBQUwsQ0FBVzRGLGFBQVosQ0FBMUI7QUFDQTs7QUFDQSxRQUFJLENBQUNELFFBQUwsRUFBZTtBQUNiLFlBQU0sSUFBSUUsS0FBSixDQUFXLDJCQUEwQixLQUFLN0YsS0FBTCxDQUFXNEYsYUFBYyxFQUE5RCxDQUFOO0FBQ0Q7O0FBQ0QsV0FBT0QsUUFBUSxZQUFZRyxRQUFwQixHQUErQkgsUUFBUSxFQUF2QyxHQUE0Q0EsUUFBbkQ7QUFDRDs7QUFFRDVCLEVBQUFBLGdCQUFnQixDQUFDZ0MsRUFBRCxFQUFLO0FBQ25CLFFBQUksS0FBS3hGLDBCQUFULEVBQXFDO0FBQ25DLGFBQU8sSUFBUDtBQUNEOztBQUVELFNBQUtBLDBCQUFMLEdBQWtDLElBQWxDO0FBQ0EsU0FBS0MsZUFBTCxHQUF1QixLQUFLUixLQUFMLENBQVdlLGNBQVgsQ0FBMEJDLFFBQTFCLEVBQXZCO0FBQ0EsVUFBTWdGLGdCQUFnQixHQUFHRCxFQUFFLEVBQTNCO0FBRUFDLElBQUFBLGdCQUFnQixDQUNiQyxJQURILENBQ1EsTUFBTSxLQUFLeEYsa0JBRG5CLEVBRUd3RixJQUZILENBRVEsTUFBTTtBQUNWLFdBQUsxRiwwQkFBTCxHQUFrQyxLQUFsQztBQUNBLFdBQUtDLGVBQUwsR0FBdUIsSUFBdkI7QUFDRCxLQUxIO0FBT0EsV0FBT3dGLGdCQUFQO0FBQ0Q7O0FBdlBtRTs7OztnQkFBakRwRyx3QixlQUNBO0FBQ2pCd0MsRUFBQUEsVUFBVSxFQUFFOEQsbUJBQVVDLE1BQVYsQ0FBaUJDLFVBRFo7QUFFakJSLEVBQUFBLGFBQWEsRUFBRU0sbUJBQVVHLEtBQVYsQ0FBZ0IsQ0FBQyxRQUFELEVBQVcsVUFBWCxDQUFoQixDQUZFO0FBR2pCdEYsRUFBQUEsY0FBYyxFQUFFdUYsbUNBQXVCRixVQUh0QjtBQUlqQkcsRUFBQUEsY0FBYyxFQUFFTCxtQkFBVU0sSUFKVDtBQU1qQnpELEVBQUFBLFNBQVMsRUFBRW1ELG1CQUFVQyxNQUFWLENBQWlCQyxVQU5YO0FBT2pCSyxFQUFBQSxRQUFRLEVBQUVQLG1CQUFVQyxNQUFWLENBQWlCQyxVQVBWO0FBUWpCTSxFQUFBQSxPQUFPLEVBQUVSLG1CQUFVQyxNQUFWLENBQWlCQyxVQVJUO0FBU2pCTyxFQUFBQSxRQUFRLEVBQUVULG1CQUFVQyxNQUFWLENBQWlCQyxVQVRWO0FBVWpCUSxFQUFBQSxNQUFNLEVBQUVWLG1CQUFVQyxNQUFWLENBQWlCQyxVQVZSO0FBWWpCdEQsRUFBQUEsT0FBTyxFQUFFb0QsbUJBQVVXLElBQVYsQ0FBZVQsVUFaUDtBQWFqQmIsRUFBQUEsWUFBWSxFQUFFVyxtQkFBVVcsSUFiUDtBQWNqQnBGLEVBQUFBLGVBQWUsRUFBRXlFLG1CQUFVVyxJQWRWO0FBZWpCQyxFQUFBQSxPQUFPLEVBQUVaLG1CQUFVVyxJQWZGO0FBaUJqQmhGLEVBQUFBLGdCQUFnQixFQUFFcUUsbUJBQVVXO0FBakJYLEMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNi4xL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBSZWFjdCBmcm9tICdyZWFjdCc7XG5pbXBvcnQgUHJvcFR5cGVzIGZyb20gJ3Byb3AtdHlwZXMnO1xuaW1wb3J0IHBhdGggZnJvbSAncGF0aCc7XG5cbmltcG9ydCB7YXV0b2JpbmQsIGVxdWFsU2V0c30gZnJvbSAnLi4vaGVscGVycyc7XG5pbXBvcnQge2FkZEV2ZW50fSBmcm9tICcuLi9yZXBvcnRlci1wcm94eSc7XG5pbXBvcnQge011bHRpRmlsZVBhdGNoUHJvcFR5cGV9IGZyb20gJy4uL3Byb3AtdHlwZXMnO1xuaW1wb3J0IENoYW5nZWRGaWxlSXRlbSBmcm9tICcuLi9pdGVtcy9jaGFuZ2VkLWZpbGUtaXRlbSc7XG5pbXBvcnQgTXVsdGlGaWxlUGF0Y2hWaWV3IGZyb20gJy4uL3ZpZXdzL211bHRpLWZpbGUtcGF0Y2gtdmlldyc7XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIE11bHRpRmlsZVBhdGNoQ29udHJvbGxlciBleHRlbmRzIFJlYWN0LkNvbXBvbmVudCB7XG4gIHN0YXRpYyBwcm9wVHlwZXMgPSB7XG4gICAgcmVwb3NpdG9yeTogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIHN0YWdpbmdTdGF0dXM6IFByb3BUeXBlcy5vbmVPZihbJ3N0YWdlZCcsICd1bnN0YWdlZCddKSxcbiAgICBtdWx0aUZpbGVQYXRjaDogTXVsdGlGaWxlUGF0Y2hQcm9wVHlwZS5pc1JlcXVpcmVkLFxuICAgIGhhc1VuZG9IaXN0b3J5OiBQcm9wVHlwZXMuYm9vbCxcblxuICAgIHdvcmtzcGFjZTogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIGNvbW1hbmRzOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAga2V5bWFwczogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIHRvb2x0aXBzOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgY29uZmlnOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG5cbiAgICBkZXN0cm95OiBQcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgIGRpc2NhcmRMaW5lczogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdW5kb0xhc3REaXNjYXJkOiBQcm9wVHlwZXMuZnVuYyxcbiAgICBzdXJmYWNlOiBQcm9wVHlwZXMuZnVuYyxcblxuICAgIHN3aXRjaFRvSXNzdWVpc2g6IFByb3BUeXBlcy5mdW5jLFxuICB9XG5cbiAgY29uc3RydWN0b3IocHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgYXV0b2JpbmQoXG4gICAgICB0aGlzLFxuICAgICAgJ3NlbGVjdGVkUm93c0NoYW5nZWQnLFxuICAgICAgJ3VuZG9MYXN0RGlzY2FyZCcsICdkaXZlSW50b01pcnJvclBhdGNoJywgJ29wZW5GaWxlJyxcbiAgICAgICd0b2dnbGVGaWxlJywgJ3RvZ2dsZVJvd3MnLCAndG9nZ2xlTW9kZUNoYW5nZScsICd0b2dnbGVTeW1saW5rQ2hhbmdlJywgJ2Rpc2NhcmRSb3dzJyxcbiAgICApO1xuXG4gICAgdGhpcy5zdGF0ZSA9IHtcbiAgICAgIHNlbGVjdGlvbk1vZGU6ICdodW5rJyxcbiAgICAgIHNlbGVjdGVkUm93czogbmV3IFNldCgpLFxuICAgICAgaGFzTXVsdGlwbGVGaWxlU2VsZWN0aW9uczogZmFsc2UsXG4gICAgfTtcblxuICAgIHRoaXMubW91c2VTZWxlY3Rpb25JblByb2dyZXNzID0gZmFsc2U7XG4gICAgdGhpcy5zdGFnaW5nT3BlcmF0aW9uSW5Qcm9ncmVzcyA9IGZhbHNlO1xuXG4gICAgdGhpcy5sYXN0UGF0Y2hTdHJpbmcgPSBudWxsO1xuICAgIHRoaXMucGF0Y2hDaGFuZ2VQcm9taXNlID0gbmV3IFByb21pc2UocmVzb2x2ZSA9PiB7XG4gICAgICB0aGlzLnJlc29sdmVQYXRjaENoYW5nZVByb21pc2UgPSByZXNvbHZlO1xuICAgIH0pO1xuICB9XG5cbiAgY29tcG9uZW50RGlkVXBkYXRlKHByZXZQcm9wcykge1xuICAgIGlmIChcbiAgICAgIHRoaXMubGFzdFBhdGNoU3RyaW5nICE9PSBudWxsICYmXG4gICAgICB0aGlzLmxhc3RQYXRjaFN0cmluZyAhPT0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC50b1N0cmluZygpXG4gICAgKSB7XG4gICAgICB0aGlzLnJlc29sdmVQYXRjaENoYW5nZVByb21pc2UoKTtcbiAgICAgIHRoaXMucGF0Y2hDaGFuZ2VQcm9taXNlID0gbmV3IFByb21pc2UocmVzb2x2ZSA9PiB7XG4gICAgICAgIHRoaXMucmVzb2x2ZVBhdGNoQ2hhbmdlUHJvbWlzZSA9IHJlc29sdmU7XG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICByZW5kZXIoKSB7XG4gICAgcmV0dXJuIChcbiAgICAgIDxNdWx0aUZpbGVQYXRjaFZpZXdcbiAgICAgICAgey4uLnRoaXMucHJvcHN9XG5cbiAgICAgICAgc2VsZWN0ZWRSb3dzPXt0aGlzLnN0YXRlLnNlbGVjdGVkUm93c31cbiAgICAgICAgc2VsZWN0aW9uTW9kZT17dGhpcy5zdGF0ZS5zZWxlY3Rpb25Nb2RlfVxuICAgICAgICBoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zPXt0aGlzLnN0YXRlLmhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnN9XG4gICAgICAgIHNlbGVjdGVkUm93c0NoYW5nZWQ9e3RoaXMuc2VsZWN0ZWRSb3dzQ2hhbmdlZH1cblxuICAgICAgICBkaXZlSW50b01pcnJvclBhdGNoPXt0aGlzLmRpdmVJbnRvTWlycm9yUGF0Y2h9XG4gICAgICAgIG9wZW5GaWxlPXt0aGlzLm9wZW5GaWxlfVxuICAgICAgICB0b2dnbGVGaWxlPXt0aGlzLnRvZ2dsZUZpbGV9XG4gICAgICAgIHRvZ2dsZVJvd3M9e3RoaXMudG9nZ2xlUm93c31cbiAgICAgICAgdG9nZ2xlTW9kZUNoYW5nZT17dGhpcy50b2dnbGVNb2RlQ2hhbmdlfVxuICAgICAgICB0b2dnbGVTeW1saW5rQ2hhbmdlPXt0aGlzLnRvZ2dsZVN5bWxpbmtDaGFuZ2V9XG4gICAgICAgIHVuZG9MYXN0RGlzY2FyZD17dGhpcy51bmRvTGFzdERpc2NhcmR9XG4gICAgICAgIGRpc2NhcmRSb3dzPXt0aGlzLmRpc2NhcmRSb3dzfVxuICAgICAgICBzZWxlY3ROZXh0SHVuaz17dGhpcy5zZWxlY3ROZXh0SHVua31cbiAgICAgICAgc2VsZWN0UHJldmlvdXNIdW5rPXt0aGlzLnNlbGVjdFByZXZpb3VzSHVua31cbiAgICAgICAgc3dpdGNoVG9Jc3N1ZWlzaD17dGhpcy5wcm9wcy5zd2l0Y2hUb0lzc3VlaXNofVxuICAgICAgLz5cbiAgICApO1xuICB9XG5cbiAgdW5kb0xhc3REaXNjYXJkKGZpbGVQYXRjaCwge2V2ZW50U291cmNlfSA9IHt9KSB7XG4gICAgYWRkRXZlbnQoJ3VuZG8tbGFzdC1kaXNjYXJkJywge1xuICAgICAgcGFja2FnZTogJ2dpdGh1YicsXG4gICAgICBjb21wb25lbnQ6IHRoaXMuY29uc3RydWN0b3IubmFtZSxcbiAgICAgIGV2ZW50U291cmNlLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIHRoaXMucHJvcHMudW5kb0xhc3REaXNjYXJkKGZpbGVQYXRjaC5nZXRQYXRoKCksIHRoaXMucHJvcHMucmVwb3NpdG9yeSk7XG4gIH1cblxuICBkaXZlSW50b01pcnJvclBhdGNoKGZpbGVQYXRjaCkge1xuICAgIGNvbnN0IG1pcnJvclN0YXR1cyA9IHRoaXMud2l0aFN0YWdpbmdTdGF0dXMoe3N0YWdlZDogJ3Vuc3RhZ2VkJywgdW5zdGFnZWQ6ICdzdGFnZWQnfSk7XG4gICAgY29uc3Qgd29ya2luZ0RpcmVjdG9yeSA9IHRoaXMucHJvcHMucmVwb3NpdG9yeS5nZXRXb3JraW5nRGlyZWN0b3J5UGF0aCgpO1xuICAgIGNvbnN0IHVyaSA9IENoYW5nZWRGaWxlSXRlbS5idWlsZFVSSShmaWxlUGF0Y2guZ2V0UGF0aCgpLCB3b3JraW5nRGlyZWN0b3J5LCBtaXJyb3JTdGF0dXMpO1xuXG4gICAgdGhpcy5wcm9wcy5kZXN0cm95KCk7XG4gICAgcmV0dXJuIHRoaXMucHJvcHMud29ya3NwYWNlLm9wZW4odXJpKTtcbiAgfVxuXG4gIGFzeW5jIG9wZW5GaWxlKGZpbGVQYXRjaCwgcG9zaXRpb25zLCBwZW5kaW5nKSB7XG4gICAgY29uc3QgYWJzb2x1dGVQYXRoID0gcGF0aC5qb2luKHRoaXMucHJvcHMucmVwb3NpdG9yeS5nZXRXb3JraW5nRGlyZWN0b3J5UGF0aCgpLCBmaWxlUGF0Y2guZ2V0UGF0aCgpKTtcbiAgICBjb25zdCBlZGl0b3IgPSBhd2FpdCB0aGlzLnByb3BzLndvcmtzcGFjZS5vcGVuKGFic29sdXRlUGF0aCwge3BlbmRpbmd9KTtcbiAgICBpZiAocG9zaXRpb25zLmxlbmd0aCA+IDApIHtcbiAgICAgIGVkaXRvci5zZXRDdXJzb3JCdWZmZXJQb3NpdGlvbihwb3NpdGlvbnNbMF0sIHthdXRvc2Nyb2xsOiBmYWxzZX0pO1xuICAgICAgZm9yIChjb25zdCBwb3NpdGlvbiBvZiBwb3NpdGlvbnMuc2xpY2UoMSkpIHtcbiAgICAgICAgZWRpdG9yLmFkZEN1cnNvckF0QnVmZmVyUG9zaXRpb24ocG9zaXRpb24pO1xuICAgICAgfVxuICAgICAgZWRpdG9yLnNjcm9sbFRvQnVmZmVyUG9zaXRpb24ocG9zaXRpb25zW3Bvc2l0aW9ucy5sZW5ndGggLSAxXSwge2NlbnRlcjogdHJ1ZX0pO1xuICAgIH1cbiAgICByZXR1cm4gZWRpdG9yO1xuICB9XG5cbiAgdG9nZ2xlRmlsZShmaWxlUGF0Y2gpIHtcbiAgICByZXR1cm4gdGhpcy5zdGFnaW5nT3BlcmF0aW9uKCgpID0+IHtcbiAgICAgIGNvbnN0IG1ldGhvZE5hbWUgPSB0aGlzLndpdGhTdGFnaW5nU3RhdHVzKHtzdGFnZWQ6ICd1bnN0YWdlRmlsZXMnLCB1bnN0YWdlZDogJ3N0YWdlRmlsZXMnfSk7XG4gICAgICByZXR1cm4gdGhpcy5wcm9wcy5yZXBvc2l0b3J5W21ldGhvZE5hbWVdKFtmaWxlUGF0Y2guZ2V0UGF0aCgpXSk7XG4gICAgfSk7XG4gIH1cblxuICBhc3luYyB0b2dnbGVSb3dzKHJvd1NldCwgbmV4dFNlbGVjdGlvbk1vZGUpIHtcbiAgICBsZXQgY2hvc2VuUm93cyA9IHJvd1NldDtcbiAgICBpZiAoY2hvc2VuUm93cykge1xuICAgICAgY29uc3QgbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnMgPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLnNwYW5zTXVsdGlwbGVGaWxlcyhjaG9zZW5Sb3dzKTtcbiAgICAgIGF3YWl0IHRoaXMuc2VsZWN0ZWRSb3dzQ2hhbmdlZChjaG9zZW5Sb3dzLCBuZXh0U2VsZWN0aW9uTW9kZSwgbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnMpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjaG9zZW5Sb3dzID0gdGhpcy5zdGF0ZS5zZWxlY3RlZFJvd3M7XG4gICAgfVxuXG4gICAgaWYgKGNob3NlblJvd3Muc2l6ZSA9PT0gMCkge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZSgpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLnN0YWdpbmdPcGVyYXRpb24oKCkgPT4ge1xuICAgICAgY29uc3QgcGF0Y2ggPSB0aGlzLndpdGhTdGFnaW5nU3RhdHVzKHtcbiAgICAgICAgc3RhZ2VkOiAoKSA9PiB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldFVuc3RhZ2VQYXRjaEZvckxpbmVzKGNob3NlblJvd3MpLFxuICAgICAgICB1bnN0YWdlZDogKCkgPT4gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRTdGFnZVBhdGNoRm9yTGluZXMoY2hvc2VuUm93cyksXG4gICAgICB9KTtcbiAgICAgIHJldHVybiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXBwbHlQYXRjaFRvSW5kZXgocGF0Y2gpO1xuICAgIH0pO1xuICB9XG5cbiAgdG9nZ2xlTW9kZUNoYW5nZShmaWxlUGF0Y2gpIHtcbiAgICByZXR1cm4gdGhpcy5zdGFnaW5nT3BlcmF0aW9uKCgpID0+IHtcbiAgICAgIGNvbnN0IHRhcmdldE1vZGUgPSB0aGlzLndpdGhTdGFnaW5nU3RhdHVzKHtcbiAgICAgICAgdW5zdGFnZWQ6IGZpbGVQYXRjaC5nZXROZXdNb2RlKCksXG4gICAgICAgIHN0YWdlZDogZmlsZVBhdGNoLmdldE9sZE1vZGUoKSxcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIHRoaXMucHJvcHMucmVwb3NpdG9yeS5zdGFnZUZpbGVNb2RlQ2hhbmdlKGZpbGVQYXRjaC5nZXRQYXRoKCksIHRhcmdldE1vZGUpO1xuICAgIH0pO1xuICB9XG5cbiAgdG9nZ2xlU3ltbGlua0NoYW5nZShmaWxlUGF0Y2gpIHtcbiAgICByZXR1cm4gdGhpcy5zdGFnaW5nT3BlcmF0aW9uKCgpID0+IHtcbiAgICAgIGNvbnN0IHJlbFBhdGggPSBmaWxlUGF0Y2guZ2V0UGF0aCgpO1xuICAgICAgY29uc3QgcmVwb3NpdG9yeSA9IHRoaXMucHJvcHMucmVwb3NpdG9yeTtcbiAgICAgIHJldHVybiB0aGlzLndpdGhTdGFnaW5nU3RhdHVzKHtcbiAgICAgICAgdW5zdGFnZWQ6ICgpID0+IHtcbiAgICAgICAgICBpZiAoZmlsZVBhdGNoLmhhc1R5cGVjaGFuZ2UoKSAmJiBmaWxlUGF0Y2guZ2V0U3RhdHVzKCkgPT09ICdhZGRlZCcpIHtcbiAgICAgICAgICAgIHJldHVybiByZXBvc2l0b3J5LnN0YWdlRmlsZVN5bWxpbmtDaGFuZ2UocmVsUGF0aCk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgcmV0dXJuIHJlcG9zaXRvcnkuc3RhZ2VGaWxlcyhbcmVsUGF0aF0pO1xuICAgICAgICB9LFxuICAgICAgICBzdGFnZWQ6ICgpID0+IHtcbiAgICAgICAgICBpZiAoZmlsZVBhdGNoLmhhc1R5cGVjaGFuZ2UoKSAmJiBmaWxlUGF0Y2guZ2V0U3RhdHVzKCkgPT09ICdkZWxldGVkJykge1xuICAgICAgICAgICAgcmV0dXJuIHJlcG9zaXRvcnkuc3RhZ2VGaWxlU3ltbGlua0NoYW5nZShyZWxQYXRoKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZXR1cm4gcmVwb3NpdG9yeS51bnN0YWdlRmlsZXMoW3JlbFBhdGhdKTtcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgYXN5bmMgZGlzY2FyZFJvd3Mocm93U2V0LCBuZXh0U2VsZWN0aW9uTW9kZSwge2V2ZW50U291cmNlfSA9IHt9KSB7XG4gICAgLy8gKGt1eWNoYWNvKSBGb3Igbm93IHdlIG9ubHkgc3VwcG9ydCBkaXNjYXJkaW5nIHJvd3MgZm9yIE11bHRpRmlsZVBhdGNoZXMgdGhhdCBjb250YWluIGEgc2luZ2xlIGZpbGUgcGF0Y2hcbiAgICAvLyBUaGUgb25seSB3YXkgdG8gYWNjZXNzIHRoaXMgbWV0aG9kIGZyb20gdGhlIFVJIGlzIHRvIGJlIGluIGEgQ2hhbmdlZEZpbGVJdGVtLCB3aGljaCBvbmx5IGhhcyBhIHNpbmdsZSBmaWxlIHBhdGNoXG4gICAgLy8gVGhpcyBjaGVjayBpcyBkdXBsaWNhdGVkIGluIFJvb3RDb250cm9sbGVyI2Rpc2NhcmRMaW5lcy4gV2UgYWxzbyB3YW50IGl0IGhlcmUgdG8gcHJldmVudCB1cyBmcm9tIHNlbmRpbmcgbWV0cmljc1xuICAgIC8vIHVubmVjZXNzYXJpbHlcbiAgICBpZiAodGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hlcygpLmxlbmd0aCAhPT0gMSkge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZShudWxsKTtcbiAgICB9XG5cbiAgICBsZXQgY2hvc2VuUm93cyA9IHJvd1NldDtcbiAgICBpZiAoY2hvc2VuUm93cykge1xuICAgICAgY29uc3QgbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnMgPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLnNwYW5zTXVsdGlwbGVGaWxlcyhjaG9zZW5Sb3dzKTtcbiAgICAgIGF3YWl0IHRoaXMuc2VsZWN0ZWRSb3dzQ2hhbmdlZChjaG9zZW5Sb3dzLCBuZXh0U2VsZWN0aW9uTW9kZSwgbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnMpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjaG9zZW5Sb3dzID0gdGhpcy5zdGF0ZS5zZWxlY3RlZFJvd3M7XG4gICAgfVxuXG4gICAgYWRkRXZlbnQoJ2Rpc2NhcmQtdW5zdGFnZWQtY2hhbmdlcycsIHtcbiAgICAgIHBhY2thZ2U6ICdnaXRodWInLFxuICAgICAgY29tcG9uZW50OiB0aGlzLmNvbnN0cnVjdG9yLm5hbWUsXG4gICAgICBsaW5lQ291bnQ6IGNob3NlblJvd3Muc2l6ZSxcbiAgICAgIGV2ZW50U291cmNlLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIHRoaXMucHJvcHMuZGlzY2FyZExpbmVzKHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2gsIGNob3NlblJvd3MsIHRoaXMucHJvcHMucmVwb3NpdG9yeSk7XG4gIH1cblxuICBzZWxlY3RlZFJvd3NDaGFuZ2VkKHJvd3MsIG5leHRTZWxlY3Rpb25Nb2RlLCBuZXh0TXVsdGlwbGVGaWxlU2VsZWN0aW9ucykge1xuICAgIGlmIChcbiAgICAgIGVxdWFsU2V0cyh0aGlzLnN0YXRlLnNlbGVjdGVkUm93cywgcm93cykgJiZcbiAgICAgIHRoaXMuc3RhdGUuc2VsZWN0aW9uTW9kZSA9PT0gbmV4dFNlbGVjdGlvbk1vZGUgJiZcbiAgICAgIHRoaXMuc3RhdGUuaGFzTXVsdGlwbGVGaWxlU2VsZWN0aW9ucyA9PT0gbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnNcbiAgICApIHtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IFByb21pc2UocmVzb2x2ZSA9PiB7XG4gICAgICB0aGlzLnNldFN0YXRlKHtcbiAgICAgICAgc2VsZWN0ZWRSb3dzOiByb3dzLFxuICAgICAgICBzZWxlY3Rpb25Nb2RlOiBuZXh0U2VsZWN0aW9uTW9kZSxcbiAgICAgICAgaGFzTXVsdGlwbGVGaWxlU2VsZWN0aW9uczogbmV4dE11bHRpcGxlRmlsZVNlbGVjdGlvbnMsXG4gICAgICB9LCByZXNvbHZlKTtcbiAgICB9KTtcbiAgfVxuXG4gIHdpdGhTdGFnaW5nU3RhdHVzKGNhbGxiYWNrcykge1xuICAgIGNvbnN0IGNhbGxiYWNrID0gY2FsbGJhY2tzW3RoaXMucHJvcHMuc3RhZ2luZ1N0YXR1c107XG4gICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgaWYgKCFjYWxsYmFjaykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIHN0YWdpbmcgc3RhdHVzOiAke3RoaXMucHJvcHMuc3RhZ2luZ1N0YXR1c31gKTtcbiAgICB9XG4gICAgcmV0dXJuIGNhbGxiYWNrIGluc3RhbmNlb2YgRnVuY3Rpb24gPyBjYWxsYmFjaygpIDogY2FsbGJhY2s7XG4gIH1cblxuICBzdGFnaW5nT3BlcmF0aW9uKGZuKSB7XG4gICAgaWYgKHRoaXMuc3RhZ2luZ09wZXJhdGlvbkluUHJvZ3Jlc3MpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cblxuICAgIHRoaXMuc3RhZ2luZ09wZXJhdGlvbkluUHJvZ3Jlc3MgPSB0cnVlO1xuICAgIHRoaXMubGFzdFBhdGNoU3RyaW5nID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC50b1N0cmluZygpO1xuICAgIGNvbnN0IG9wZXJhdGlvblByb21pc2UgPSBmbigpO1xuXG4gICAgb3BlcmF0aW9uUHJvbWlzZVxuICAgICAgLnRoZW4oKCkgPT4gdGhpcy5wYXRjaENoYW5nZVByb21pc2UpXG4gICAgICAudGhlbigoKSA9PiB7XG4gICAgICAgIHRoaXMuc3RhZ2luZ09wZXJhdGlvbkluUHJvZ3Jlc3MgPSBmYWxzZTtcbiAgICAgICAgdGhpcy5sYXN0UGF0Y2hTdHJpbmcgPSBudWxsO1xuICAgICAgfSk7XG5cbiAgICByZXR1cm4gb3BlcmF0aW9uUHJvbWlzZTtcbiAgfVxufVxuIl19