"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _atom = require("atom");

var _bintrees = require("bintrees");

var _patchBuffer = _interopRequireDefault(require("./patch-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultiFilePatch {
  static createNull() {
    return new this({
      patchBuffer: new _patchBuffer.default(),
      filePatches: []
    });
  }

  constructor({
    patchBuffer,
    filePatches
  }) {
    _defineProperty(this, "isPatchVisible", filePatchPath => {
      const patch = this.filePatchesByPath.get(filePatchPath);

      if (!patch) {
        return false;
      }

      return patch.getRenderStatus().isVisible();
    });

    _defineProperty(this, "getBufferRowForDiffPosition", (fileName, diffRow) => {
      const {
        startBufferRow,
        index
      } = this.diffRowOffsetIndices.get(fileName);
      const {
        offset
      } = index.lowerBound({
        diffRow
      }).data();
      return startBufferRow + diffRow - offset;
    });

    this.patchBuffer = patchBuffer;
    this.filePatches = filePatches;
    this.filePatchesByMarker = new Map();
    this.filePatchesByPath = new Map();
    this.hunksByMarker = new Map(); // Store a map of {diffRow, offset} for each FilePatch where offset is the number of Hunk headers within the current
    // FilePatch that occur before this row in the original diff output.

    this.diffRowOffsetIndices = new Map();

    for (const filePatch of this.filePatches) {
      this.filePatchesByPath.set(filePatch.getPath(), filePatch);
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);
      this.populateDiffRowOffsetIndices(filePatch);
    }
  }

  clone(opts = {}) {
    return new this.constructor({
      patchBuffer: opts.patchBuffer !== undefined ? opts.patchBuffer : this.getPatchBuffer(),
      filePatches: opts.filePatches !== undefined ? opts.filePatches : this.getFilePatches()
    });
  }

  getPatchBuffer() {
    return this.patchBuffer;
  }

  getBuffer() {
    return this.getPatchBuffer().getBuffer();
  }

  getPatchLayer() {
    return this.getPatchBuffer().getLayer('patch');
  }

  getHunkLayer() {
    return this.getPatchBuffer().getLayer('hunk');
  }

  getUnchangedLayer() {
    return this.getPatchBuffer().getLayer('unchanged');
  }

  getAdditionLayer() {
    return this.getPatchBuffer().getLayer('addition');
  }

  getDeletionLayer() {
    return this.getPatchBuffer().getLayer('deletion');
  }

  getNoNewlineLayer() {
    return this.getPatchBuffer().getLayer('nonewline');
  }

  getFilePatches() {
    return this.filePatches;
  }

  getPathSet() {
    return this.getFilePatches().reduce((pathSet, filePatch) => {
      for (const file of [filePatch.getOldFile(), filePatch.getNewFile()]) {
        if (file.isPresent()) {
          pathSet.add(file.getPath());
        }
      }

      return pathSet;
    }, new Set());
  }

  getFilePatchAt(bufferRow) {
    if (bufferRow < 0 || bufferRow > this.patchBuffer.getBuffer().getLastRow()) {
      return undefined;
    }

    const [marker] = this.patchBuffer.findMarkers('patch', {
      intersectsRow: bufferRow
    });
    return this.filePatchesByMarker.get(marker);
  }

  getHunkAt(bufferRow) {
    if (bufferRow < 0) {
      return undefined;
    }

    const [marker] = this.patchBuffer.findMarkers('hunk', {
      intersectsRow: bufferRow
    });
    return this.hunksByMarker.get(marker);
  }

  getStagePatchForLines(selectedLineSet) {
    const nextPatchBuffer = new _patchBuffer.default();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildStagePatchForLines(this.getBuffer(), nextPatchBuffer, selectedLineSet);
    });
    return this.clone({
      patchBuffer: nextPatchBuffer,
      filePatches: nextFilePatches
    });
  }

  getStagePatchForHunk(hunk) {
    return this.getStagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getUnstagePatchForLines(selectedLineSet) {
    const nextPatchBuffer = new _patchBuffer.default();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildUnstagePatchForLines(this.getBuffer(), nextPatchBuffer, selectedLineSet);
    });
    return this.clone({
      patchBuffer: nextPatchBuffer,
      filePatches: nextFilePatches
    });
  }

  getUnstagePatchForHunk(hunk) {
    return this.getUnstagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getMaxSelectionIndex(selectedRows) {
    if (selectedRows.size === 0) {
      return 0;
    }

    const lastMax = Math.max(...selectedRows);
    let selectionIndex = 0; // counts unselected lines in changed regions from the old patch
    // until we get to the bottom-most selected line from the old patch (lastMax).

    patchLoop: for (const filePatch of this.getFilePatches()) {
      for (const hunk of filePatch.getHunks()) {
        let includesMax = false;

        for (const change of hunk.getChanges()) {
          for (const _ref of change.intersectRows(selectedRows, true)) {
            const {
              intersection,
              gap
            } = _ref;
            // Only include a partial range if this intersection includes the last selected buffer row.
            includesMax = intersection.intersectsRow(lastMax);
            const delta = includesMax ? lastMax - intersection.start.row + 1 : intersection.getRowCount();

            if (gap) {
              // Range of unselected changes.
              selectionIndex += delta;
            }

            if (includesMax) {
              break patchLoop;
            }
          }
        }
      }
    }

    return selectionIndex;
  }

  getSelectionRangeForIndex(selectionIndex) {
    // Iterate over changed lines in this patch in order to find the
    // new row to be selected based on the last selection index.
    // As we walk through the changed lines, we whittle down the
    // remaining lines until we reach the row that corresponds to the
    // last selected index.
    let selectionRow = 0;
    let remainingChangedLines = selectionIndex;
    let foundRow = false;
    let lastChangedRow = 0;

    patchLoop: for (const filePatch of this.getFilePatches()) {
      for (const hunk of filePatch.getHunks()) {
        for (const change of hunk.getChanges()) {
          if (remainingChangedLines < change.bufferRowCount()) {
            selectionRow = change.getStartBufferRow() + remainingChangedLines;
            foundRow = true;
            break patchLoop;
          } else {
            remainingChangedLines -= change.bufferRowCount();
            lastChangedRow = change.getEndBufferRow();
          }
        }
      }
    } // If we never got to the last selected index, that means it is
    // no longer present in the new patch (ie. we staged the last line of the file).
    // In this case we want the next selected line to be the last changed row in the file


    if (!foundRow) {
      selectionRow = lastChangedRow;
    }

    return _atom.Range.fromObject([[selectionRow, 0], [selectionRow, Infinity]]);
  }

  isDiffRowOffsetIndexEmpty(filePatchPath) {
    const diffRowOffsetIndex = this.diffRowOffsetIndices.get(filePatchPath);
    return diffRowOffsetIndex.index.size === 0;
  }

  populateDiffRowOffsetIndices(filePatch) {
    let diffRow = 1;
    const index = new _bintrees.RBTree((a, b) => a.diffRow - b.diffRow);
    this.diffRowOffsetIndices.set(filePatch.getPath(), {
      startBufferRow: filePatch.getStartRange().start.row,
      index
    });

    for (let hunkIndex = 0; hunkIndex < filePatch.getHunks().length; hunkIndex++) {
      const hunk = filePatch.getHunks()[hunkIndex];
      this.hunksByMarker.set(hunk.getMarker(), hunk); // Advance past the hunk body

      diffRow += hunk.bufferRowCount();
      index.insert({
        diffRow,
        offset: hunkIndex + 1
      }); // Advance past the next hunk header

      diffRow++;
    }
  }

  adoptBuffer(nextPatchBuffer) {
    nextPatchBuffer.clearAllLayers();
    this.filePatchesByMarker.clear();
    this.hunksByMarker.clear();
    const markerMap = nextPatchBuffer.adopt(this.patchBuffer);

    for (const filePatch of this.getFilePatches()) {
      filePatch.updateMarkers(markerMap);
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);

      for (const hunk of filePatch.getHunks()) {
        this.hunksByMarker.set(hunk.getMarker(), hunk);
      }
    }

    this.patchBuffer = nextPatchBuffer;
  }
  /*
   * Efficiently locate the FilePatch instances that contain at least one row from a Set.
   */


  getFilePatchesContaining(rowSet) {
    const sortedRowSet = Array.from(rowSet);
    sortedRowSet.sort((a, b) => a - b);
    const filePatches = [];
    let lastFilePatch = null;

    for (const row of sortedRowSet) {
      // Because the rows are sorted, consecutive rows will almost certainly belong to the same patch, so we can save
      // many avoidable marker index lookups by comparing with the last.
      if (lastFilePatch && lastFilePatch.containsRow(row)) {
        continue;
      }

      lastFilePatch = this.getFilePatchAt(row);
      filePatches.push(lastFilePatch);
    }

    return filePatches;
  }

  anyPresent() {
    return this.patchBuffer !== null && this.filePatches.some(fp => fp.isPresent());
  }

  didAnyChangeExecutableMode() {
    for (const filePatch of this.getFilePatches()) {
      if (filePatch.didChangeExecutableMode()) {
        return true;
      }
    }

    return false;
  }

  anyHaveTypechange() {
    return this.getFilePatches().some(fp => fp.hasTypechange());
  }

  getMaxLineNumberWidth() {
    return this.getFilePatches().reduce((maxWidth, filePatch) => {
      const width = filePatch.getMaxLineNumberWidth();
      return maxWidth >= width ? maxWidth : width;
    }, 0);
  }

  spansMultipleFiles(rows) {
    let lastFilePatch = null;

    for (const row of rows) {
      if (lastFilePatch) {
        if (lastFilePatch.containsRow(row)) {
          continue;
        }

        return true;
      } else {
        lastFilePatch = this.getFilePatchAt(row);
      }
    }

    return false;
  }

  collapseFilePatch(filePatch) {
    const index = this.filePatches.indexOf(filePatch);
    this.filePatchesByMarker.delete(filePatch.getMarker());

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.delete(hunk.getMarker());
    }

    const before = this.getMarkersBefore(index);
    const after = this.getMarkersAfter(index);
    filePatch.triggerCollapseIn(this.patchBuffer, {
      before,
      after
    });
    this.filePatchesByMarker.set(filePatch.getMarker(), filePatch); // This hunk collection should be empty, but let's iterate anyway just in case filePatch was already collapsed

    /* istanbul ignore next */

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.set(hunk.getMarker(), hunk);
    }
  }

  expandFilePatch(filePatch) {
    const index = this.filePatches.indexOf(filePatch);
    this.filePatchesByMarker.delete(filePatch.getMarker());

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.delete(hunk.getMarker());
    }

    const before = this.getMarkersBefore(index);
    const after = this.getMarkersAfter(index);
    filePatch.triggerExpandIn(this.patchBuffer, {
      before,
      after
    });
    this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);

    for (const hunk of filePatch.getHunks()) {
      this.hunksByMarker.set(hunk.getMarker(), hunk);
    } // if the patch was initially collapsed, we need to calculate
    // the diffRowOffsetIndices to calculate comment position.


    if (this.isDiffRowOffsetIndexEmpty(filePatch.getPath())) {
      this.populateDiffRowOffsetIndices(filePatch);
    }
  }

  getMarkersBefore(filePatchIndex) {
    const before = [];
    let beforeIndex = filePatchIndex - 1;

    while (beforeIndex >= 0) {
      const beforeFilePatch = this.filePatches[beforeIndex];
      before.push(...beforeFilePatch.getEndingMarkers());

      if (!beforeFilePatch.getMarker().getRange().isEmpty()) {
        break;
      }

      beforeIndex--;
    }

    return before;
  }

  getMarkersAfter(filePatchIndex) {
    const after = [];
    let afterIndex = filePatchIndex + 1;

    while (afterIndex < this.filePatches.length) {
      const afterFilePatch = this.filePatches[afterIndex];
      after.push(...afterFilePatch.getStartingMarkers());

      if (!afterFilePatch.getMarker().getRange().isEmpty()) {
        break;
      }

      afterIndex++;
    }

    return after;
  }

  /*
   * Construct an apply-able patch String.
   */
  toString() {
    return this.filePatches.map(fp => fp.toStringIn(this.getBuffer())).join('') + '\n';
  }
  /*
   * Construct a string of diagnostic information useful for debugging.
   */

  /* istanbul ignore next */


  inspect() {
    let inspectString = '(MultiFilePatch';
    inspectString += ` filePatchesByMarker=(${Array.from(this.filePatchesByMarker.keys(), m => m.id).join(', ')})`;
    inspectString += ` hunksByMarker=(${Array.from(this.hunksByMarker.keys(), m => m.id).join(', ')})\n`;

    for (const filePatch of this.filePatches) {
      inspectString += filePatch.inspect({
        indent: 2
      });
    }

    inspectString += ')\n';
    return inspectString;
  }
  /* istanbul ignore next */


  isEqual(other) {
    return this.toString() === other.toString();
  }

}

exports.default = MultiFilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm11bHRpLWZpbGUtcGF0Y2guanMiXSwibmFtZXMiOlsiTXVsdGlGaWxlUGF0Y2giLCJjcmVhdGVOdWxsIiwicGF0Y2hCdWZmZXIiLCJQYXRjaEJ1ZmZlciIsImZpbGVQYXRjaGVzIiwiY29uc3RydWN0b3IiLCJmaWxlUGF0Y2hQYXRoIiwicGF0Y2giLCJmaWxlUGF0Y2hlc0J5UGF0aCIsImdldCIsImdldFJlbmRlclN0YXR1cyIsImlzVmlzaWJsZSIsImZpbGVOYW1lIiwiZGlmZlJvdyIsInN0YXJ0QnVmZmVyUm93IiwiaW5kZXgiLCJkaWZmUm93T2Zmc2V0SW5kaWNlcyIsIm9mZnNldCIsImxvd2VyQm91bmQiLCJkYXRhIiwiZmlsZVBhdGNoZXNCeU1hcmtlciIsIk1hcCIsImh1bmtzQnlNYXJrZXIiLCJmaWxlUGF0Y2giLCJzZXQiLCJnZXRQYXRoIiwiZ2V0TWFya2VyIiwicG9wdWxhdGVEaWZmUm93T2Zmc2V0SW5kaWNlcyIsImNsb25lIiwib3B0cyIsInVuZGVmaW5lZCIsImdldFBhdGNoQnVmZmVyIiwiZ2V0RmlsZVBhdGNoZXMiLCJnZXRCdWZmZXIiLCJnZXRQYXRjaExheWVyIiwiZ2V0TGF5ZXIiLCJnZXRIdW5rTGF5ZXIiLCJnZXRVbmNoYW5nZWRMYXllciIsImdldEFkZGl0aW9uTGF5ZXIiLCJnZXREZWxldGlvbkxheWVyIiwiZ2V0Tm9OZXdsaW5lTGF5ZXIiLCJnZXRQYXRoU2V0IiwicmVkdWNlIiwicGF0aFNldCIsImZpbGUiLCJnZXRPbGRGaWxlIiwiZ2V0TmV3RmlsZSIsImlzUHJlc2VudCIsImFkZCIsIlNldCIsImdldEZpbGVQYXRjaEF0IiwiYnVmZmVyUm93IiwiZ2V0TGFzdFJvdyIsIm1hcmtlciIsImZpbmRNYXJrZXJzIiwiaW50ZXJzZWN0c1JvdyIsImdldEh1bmtBdCIsImdldFN0YWdlUGF0Y2hGb3JMaW5lcyIsInNlbGVjdGVkTGluZVNldCIsIm5leHRQYXRjaEJ1ZmZlciIsIm5leHRGaWxlUGF0Y2hlcyIsImdldEZpbGVQYXRjaGVzQ29udGFpbmluZyIsIm1hcCIsImZwIiwiYnVpbGRTdGFnZVBhdGNoRm9yTGluZXMiLCJnZXRTdGFnZVBhdGNoRm9ySHVuayIsImh1bmsiLCJnZXRCdWZmZXJSb3dzIiwiZ2V0VW5zdGFnZVBhdGNoRm9yTGluZXMiLCJidWlsZFVuc3RhZ2VQYXRjaEZvckxpbmVzIiwiZ2V0VW5zdGFnZVBhdGNoRm9ySHVuayIsImdldE1heFNlbGVjdGlvbkluZGV4Iiwic2VsZWN0ZWRSb3dzIiwic2l6ZSIsImxhc3RNYXgiLCJNYXRoIiwibWF4Iiwic2VsZWN0aW9uSW5kZXgiLCJwYXRjaExvb3AiLCJnZXRIdW5rcyIsImluY2x1ZGVzTWF4IiwiY2hhbmdlIiwiZ2V0Q2hhbmdlcyIsImludGVyc2VjdFJvd3MiLCJpbnRlcnNlY3Rpb24iLCJnYXAiLCJkZWx0YSIsInN0YXJ0Iiwicm93IiwiZ2V0Um93Q291bnQiLCJnZXRTZWxlY3Rpb25SYW5nZUZvckluZGV4Iiwic2VsZWN0aW9uUm93IiwicmVtYWluaW5nQ2hhbmdlZExpbmVzIiwiZm91bmRSb3ciLCJsYXN0Q2hhbmdlZFJvdyIsImJ1ZmZlclJvd0NvdW50IiwiZ2V0U3RhcnRCdWZmZXJSb3ciLCJnZXRFbmRCdWZmZXJSb3ciLCJSYW5nZSIsImZyb21PYmplY3QiLCJJbmZpbml0eSIsImlzRGlmZlJvd09mZnNldEluZGV4RW1wdHkiLCJkaWZmUm93T2Zmc2V0SW5kZXgiLCJSQlRyZWUiLCJhIiwiYiIsImdldFN0YXJ0UmFuZ2UiLCJodW5rSW5kZXgiLCJsZW5ndGgiLCJpbnNlcnQiLCJhZG9wdEJ1ZmZlciIsImNsZWFyQWxsTGF5ZXJzIiwiY2xlYXIiLCJtYXJrZXJNYXAiLCJhZG9wdCIsInVwZGF0ZU1hcmtlcnMiLCJyb3dTZXQiLCJzb3J0ZWRSb3dTZXQiLCJBcnJheSIsImZyb20iLCJzb3J0IiwibGFzdEZpbGVQYXRjaCIsImNvbnRhaW5zUm93IiwicHVzaCIsImFueVByZXNlbnQiLCJzb21lIiwiZGlkQW55Q2hhbmdlRXhlY3V0YWJsZU1vZGUiLCJkaWRDaGFuZ2VFeGVjdXRhYmxlTW9kZSIsImFueUhhdmVUeXBlY2hhbmdlIiwiaGFzVHlwZWNoYW5nZSIsImdldE1heExpbmVOdW1iZXJXaWR0aCIsIm1heFdpZHRoIiwid2lkdGgiLCJzcGFuc011bHRpcGxlRmlsZXMiLCJyb3dzIiwiY29sbGFwc2VGaWxlUGF0Y2giLCJpbmRleE9mIiwiZGVsZXRlIiwiYmVmb3JlIiwiZ2V0TWFya2Vyc0JlZm9yZSIsImFmdGVyIiwiZ2V0TWFya2Vyc0FmdGVyIiwidHJpZ2dlckNvbGxhcHNlSW4iLCJleHBhbmRGaWxlUGF0Y2giLCJ0cmlnZ2VyRXhwYW5kSW4iLCJmaWxlUGF0Y2hJbmRleCIsImJlZm9yZUluZGV4IiwiYmVmb3JlRmlsZVBhdGNoIiwiZ2V0RW5kaW5nTWFya2VycyIsImdldFJhbmdlIiwiaXNFbXB0eSIsImFmdGVySW5kZXgiLCJhZnRlckZpbGVQYXRjaCIsImdldFN0YXJ0aW5nTWFya2VycyIsInRvU3RyaW5nIiwidG9TdHJpbmdJbiIsImpvaW4iLCJpbnNwZWN0IiwiaW5zcGVjdFN0cmluZyIsImtleXMiLCJtIiwiaWQiLCJpbmRlbnQiLCJpc0VxdWFsIiwib3RoZXIiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFBQTs7QUFDQTs7QUFFQTs7Ozs7O0FBRWUsTUFBTUEsY0FBTixDQUFxQjtBQUNsQyxTQUFPQyxVQUFQLEdBQW9CO0FBQ2xCLFdBQU8sSUFBSSxJQUFKLENBQVM7QUFBQ0MsTUFBQUEsV0FBVyxFQUFFLElBQUlDLG9CQUFKLEVBQWQ7QUFBaUNDLE1BQUFBLFdBQVcsRUFBRTtBQUE5QyxLQUFULENBQVA7QUFDRDs7QUFFREMsRUFBQUEsV0FBVyxDQUFDO0FBQUNILElBQUFBLFdBQUQ7QUFBY0UsSUFBQUE7QUFBZCxHQUFELEVBQTZCO0FBQUEsNENBa1h2QkUsYUFBYSxJQUFJO0FBQ2hDLFlBQU1DLEtBQUssR0FBRyxLQUFLQyxpQkFBTCxDQUF1QkMsR0FBdkIsQ0FBMkJILGFBQTNCLENBQWQ7O0FBQ0EsVUFBSSxDQUFDQyxLQUFMLEVBQVk7QUFDVixlQUFPLEtBQVA7QUFDRDs7QUFDRCxhQUFPQSxLQUFLLENBQUNHLGVBQU4sR0FBd0JDLFNBQXhCLEVBQVA7QUFDRCxLQXhYdUM7O0FBQUEseURBMFhWLENBQUNDLFFBQUQsRUFBV0MsT0FBWCxLQUF1QjtBQUNuRCxZQUFNO0FBQUNDLFFBQUFBLGNBQUQ7QUFBaUJDLFFBQUFBO0FBQWpCLFVBQTBCLEtBQUtDLG9CQUFMLENBQTBCUCxHQUExQixDQUE4QkcsUUFBOUIsQ0FBaEM7QUFDQSxZQUFNO0FBQUNLLFFBQUFBO0FBQUQsVUFBV0YsS0FBSyxDQUFDRyxVQUFOLENBQWlCO0FBQUNMLFFBQUFBO0FBQUQsT0FBakIsRUFBNEJNLElBQTVCLEVBQWpCO0FBQ0EsYUFBT0wsY0FBYyxHQUFHRCxPQUFqQixHQUEyQkksTUFBbEM7QUFDRCxLQTlYdUM7O0FBQ3RDLFNBQUtmLFdBQUwsR0FBbUJBLFdBQW5CO0FBQ0EsU0FBS0UsV0FBTCxHQUFtQkEsV0FBbkI7QUFFQSxTQUFLZ0IsbUJBQUwsR0FBMkIsSUFBSUMsR0FBSixFQUEzQjtBQUNBLFNBQUtiLGlCQUFMLEdBQXlCLElBQUlhLEdBQUosRUFBekI7QUFDQSxTQUFLQyxhQUFMLEdBQXFCLElBQUlELEdBQUosRUFBckIsQ0FOc0MsQ0FRdEM7QUFDQTs7QUFDQSxTQUFLTCxvQkFBTCxHQUE0QixJQUFJSyxHQUFKLEVBQTVCOztBQUVBLFNBQUssTUFBTUUsU0FBWCxJQUF3QixLQUFLbkIsV0FBN0IsRUFBMEM7QUFDeEMsV0FBS0ksaUJBQUwsQ0FBdUJnQixHQUF2QixDQUEyQkQsU0FBUyxDQUFDRSxPQUFWLEVBQTNCLEVBQWdERixTQUFoRDtBQUNBLFdBQUtILG1CQUFMLENBQXlCSSxHQUF6QixDQUE2QkQsU0FBUyxDQUFDRyxTQUFWLEVBQTdCLEVBQW9ESCxTQUFwRDtBQUVBLFdBQUtJLDRCQUFMLENBQWtDSixTQUFsQztBQUNEO0FBQ0Y7O0FBRURLLEVBQUFBLEtBQUssQ0FBQ0MsSUFBSSxHQUFHLEVBQVIsRUFBWTtBQUNmLFdBQU8sSUFBSSxLQUFLeEIsV0FBVCxDQUFxQjtBQUMxQkgsTUFBQUEsV0FBVyxFQUFFMkIsSUFBSSxDQUFDM0IsV0FBTCxLQUFxQjRCLFNBQXJCLEdBQWlDRCxJQUFJLENBQUMzQixXQUF0QyxHQUFvRCxLQUFLNkIsY0FBTCxFQUR2QztBQUUxQjNCLE1BQUFBLFdBQVcsRUFBRXlCLElBQUksQ0FBQ3pCLFdBQUwsS0FBcUIwQixTQUFyQixHQUFpQ0QsSUFBSSxDQUFDekIsV0FBdEMsR0FBb0QsS0FBSzRCLGNBQUw7QUFGdkMsS0FBckIsQ0FBUDtBQUlEOztBQUVERCxFQUFBQSxjQUFjLEdBQUc7QUFDZixXQUFPLEtBQUs3QixXQUFaO0FBQ0Q7O0FBRUQrQixFQUFBQSxTQUFTLEdBQUc7QUFDVixXQUFPLEtBQUtGLGNBQUwsR0FBc0JFLFNBQXRCLEVBQVA7QUFDRDs7QUFFREMsRUFBQUEsYUFBYSxHQUFHO0FBQ2QsV0FBTyxLQUFLSCxjQUFMLEdBQXNCSSxRQUF0QixDQUErQixPQUEvQixDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLFlBQVksR0FBRztBQUNiLFdBQU8sS0FBS0wsY0FBTCxHQUFzQkksUUFBdEIsQ0FBK0IsTUFBL0IsQ0FBUDtBQUNEOztBQUVERSxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixXQUFPLEtBQUtOLGNBQUwsR0FBc0JJLFFBQXRCLENBQStCLFdBQS9CLENBQVA7QUFDRDs7QUFFREcsRUFBQUEsZ0JBQWdCLEdBQUc7QUFDakIsV0FBTyxLQUFLUCxjQUFMLEdBQXNCSSxRQUF0QixDQUErQixVQUEvQixDQUFQO0FBQ0Q7O0FBRURJLEVBQUFBLGdCQUFnQixHQUFHO0FBQ2pCLFdBQU8sS0FBS1IsY0FBTCxHQUFzQkksUUFBdEIsQ0FBK0IsVUFBL0IsQ0FBUDtBQUNEOztBQUVESyxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixXQUFPLEtBQUtULGNBQUwsR0FBc0JJLFFBQXRCLENBQStCLFdBQS9CLENBQVA7QUFDRDs7QUFFREgsRUFBQUEsY0FBYyxHQUFHO0FBQ2YsV0FBTyxLQUFLNUIsV0FBWjtBQUNEOztBQUVEcUMsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFLVCxjQUFMLEdBQXNCVSxNQUF0QixDQUE2QixDQUFDQyxPQUFELEVBQVVwQixTQUFWLEtBQXdCO0FBQzFELFdBQUssTUFBTXFCLElBQVgsSUFBbUIsQ0FBQ3JCLFNBQVMsQ0FBQ3NCLFVBQVYsRUFBRCxFQUF5QnRCLFNBQVMsQ0FBQ3VCLFVBQVYsRUFBekIsQ0FBbkIsRUFBcUU7QUFDbkUsWUFBSUYsSUFBSSxDQUFDRyxTQUFMLEVBQUosRUFBc0I7QUFDcEJKLFVBQUFBLE9BQU8sQ0FBQ0ssR0FBUixDQUFZSixJQUFJLENBQUNuQixPQUFMLEVBQVo7QUFDRDtBQUNGOztBQUNELGFBQU9rQixPQUFQO0FBQ0QsS0FQTSxFQU9KLElBQUlNLEdBQUosRUFQSSxDQUFQO0FBUUQ7O0FBRURDLEVBQUFBLGNBQWMsQ0FBQ0MsU0FBRCxFQUFZO0FBQ3hCLFFBQUlBLFNBQVMsR0FBRyxDQUFaLElBQWlCQSxTQUFTLEdBQUcsS0FBS2pELFdBQUwsQ0FBaUIrQixTQUFqQixHQUE2Qm1CLFVBQTdCLEVBQWpDLEVBQTRFO0FBQzFFLGFBQU90QixTQUFQO0FBQ0Q7O0FBQ0QsVUFBTSxDQUFDdUIsTUFBRCxJQUFXLEtBQUtuRCxXQUFMLENBQWlCb0QsV0FBakIsQ0FBNkIsT0FBN0IsRUFBc0M7QUFBQ0MsTUFBQUEsYUFBYSxFQUFFSjtBQUFoQixLQUF0QyxDQUFqQjtBQUNBLFdBQU8sS0FBSy9CLG1CQUFMLENBQXlCWCxHQUF6QixDQUE2QjRDLE1BQTdCLENBQVA7QUFDRDs7QUFFREcsRUFBQUEsU0FBUyxDQUFDTCxTQUFELEVBQVk7QUFDbkIsUUFBSUEsU0FBUyxHQUFHLENBQWhCLEVBQW1CO0FBQ2pCLGFBQU9yQixTQUFQO0FBQ0Q7O0FBQ0QsVUFBTSxDQUFDdUIsTUFBRCxJQUFXLEtBQUtuRCxXQUFMLENBQWlCb0QsV0FBakIsQ0FBNkIsTUFBN0IsRUFBcUM7QUFBQ0MsTUFBQUEsYUFBYSxFQUFFSjtBQUFoQixLQUFyQyxDQUFqQjtBQUNBLFdBQU8sS0FBSzdCLGFBQUwsQ0FBbUJiLEdBQW5CLENBQXVCNEMsTUFBdkIsQ0FBUDtBQUNEOztBQUVESSxFQUFBQSxxQkFBcUIsQ0FBQ0MsZUFBRCxFQUFrQjtBQUNyQyxVQUFNQyxlQUFlLEdBQUcsSUFBSXhELG9CQUFKLEVBQXhCO0FBQ0EsVUFBTXlELGVBQWUsR0FBRyxLQUFLQyx3QkFBTCxDQUE4QkgsZUFBOUIsRUFBK0NJLEdBQS9DLENBQW1EQyxFQUFFLElBQUk7QUFDL0UsYUFBT0EsRUFBRSxDQUFDQyx1QkFBSCxDQUEyQixLQUFLL0IsU0FBTCxFQUEzQixFQUE2QzBCLGVBQTdDLEVBQThERCxlQUE5RCxDQUFQO0FBQ0QsS0FGdUIsQ0FBeEI7QUFHQSxXQUFPLEtBQUs5QixLQUFMLENBQVc7QUFBQzFCLE1BQUFBLFdBQVcsRUFBRXlELGVBQWQ7QUFBK0J2RCxNQUFBQSxXQUFXLEVBQUV3RDtBQUE1QyxLQUFYLENBQVA7QUFDRDs7QUFFREssRUFBQUEsb0JBQW9CLENBQUNDLElBQUQsRUFBTztBQUN6QixXQUFPLEtBQUtULHFCQUFMLENBQTJCLElBQUlSLEdBQUosQ0FBUWlCLElBQUksQ0FBQ0MsYUFBTCxFQUFSLENBQTNCLENBQVA7QUFDRDs7QUFFREMsRUFBQUEsdUJBQXVCLENBQUNWLGVBQUQsRUFBa0I7QUFDdkMsVUFBTUMsZUFBZSxHQUFHLElBQUl4RCxvQkFBSixFQUF4QjtBQUNBLFVBQU15RCxlQUFlLEdBQUcsS0FBS0Msd0JBQUwsQ0FBOEJILGVBQTlCLEVBQStDSSxHQUEvQyxDQUFtREMsRUFBRSxJQUFJO0FBQy9FLGFBQU9BLEVBQUUsQ0FBQ00seUJBQUgsQ0FBNkIsS0FBS3BDLFNBQUwsRUFBN0IsRUFBK0MwQixlQUEvQyxFQUFnRUQsZUFBaEUsQ0FBUDtBQUNELEtBRnVCLENBQXhCO0FBR0EsV0FBTyxLQUFLOUIsS0FBTCxDQUFXO0FBQUMxQixNQUFBQSxXQUFXLEVBQUV5RCxlQUFkO0FBQStCdkQsTUFBQUEsV0FBVyxFQUFFd0Q7QUFBNUMsS0FBWCxDQUFQO0FBQ0Q7O0FBRURVLEVBQUFBLHNCQUFzQixDQUFDSixJQUFELEVBQU87QUFDM0IsV0FBTyxLQUFLRSx1QkFBTCxDQUE2QixJQUFJbkIsR0FBSixDQUFRaUIsSUFBSSxDQUFDQyxhQUFMLEVBQVIsQ0FBN0IsQ0FBUDtBQUNEOztBQUVESSxFQUFBQSxvQkFBb0IsQ0FBQ0MsWUFBRCxFQUFlO0FBQ2pDLFFBQUlBLFlBQVksQ0FBQ0MsSUFBYixLQUFzQixDQUExQixFQUE2QjtBQUMzQixhQUFPLENBQVA7QUFDRDs7QUFFRCxVQUFNQyxPQUFPLEdBQUdDLElBQUksQ0FBQ0MsR0FBTCxDQUFTLEdBQUdKLFlBQVosQ0FBaEI7QUFFQSxRQUFJSyxjQUFjLEdBQUcsQ0FBckIsQ0FQaUMsQ0FRakM7QUFDQTs7QUFDQUMsSUFBQUEsU0FBUyxFQUFFLEtBQUssTUFBTXZELFNBQVgsSUFBd0IsS0FBS1MsY0FBTCxFQUF4QixFQUErQztBQUN4RCxXQUFLLE1BQU1rQyxJQUFYLElBQW1CM0MsU0FBUyxDQUFDd0QsUUFBVixFQUFuQixFQUF5QztBQUN2QyxZQUFJQyxXQUFXLEdBQUcsS0FBbEI7O0FBRUEsYUFBSyxNQUFNQyxNQUFYLElBQXFCZixJQUFJLENBQUNnQixVQUFMLEVBQXJCLEVBQXdDO0FBQ3RDLDZCQUFrQ0QsTUFBTSxDQUFDRSxhQUFQLENBQXFCWCxZQUFyQixFQUFtQyxJQUFuQyxDQUFsQyxFQUE0RTtBQUFBLGtCQUFqRTtBQUFDWSxjQUFBQSxZQUFEO0FBQWVDLGNBQUFBO0FBQWYsYUFBaUU7QUFDMUU7QUFDQUwsWUFBQUEsV0FBVyxHQUFHSSxZQUFZLENBQUM3QixhQUFiLENBQTJCbUIsT0FBM0IsQ0FBZDtBQUNBLGtCQUFNWSxLQUFLLEdBQUdOLFdBQVcsR0FBR04sT0FBTyxHQUFHVSxZQUFZLENBQUNHLEtBQWIsQ0FBbUJDLEdBQTdCLEdBQW1DLENBQXRDLEdBQTBDSixZQUFZLENBQUNLLFdBQWIsRUFBbkU7O0FBRUEsZ0JBQUlKLEdBQUosRUFBUztBQUNQO0FBQ0FSLGNBQUFBLGNBQWMsSUFBSVMsS0FBbEI7QUFDRDs7QUFFRCxnQkFBSU4sV0FBSixFQUFpQjtBQUNmLG9CQUFNRixTQUFOO0FBQ0Q7QUFDRjtBQUNGO0FBQ0Y7QUFDRjs7QUFFRCxXQUFPRCxjQUFQO0FBQ0Q7O0FBRURhLEVBQUFBLHlCQUF5QixDQUFDYixjQUFELEVBQWlCO0FBQ3hDO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFFQSxRQUFJYyxZQUFZLEdBQUcsQ0FBbkI7QUFDQSxRQUFJQyxxQkFBcUIsR0FBR2YsY0FBNUI7QUFFQSxRQUFJZ0IsUUFBUSxHQUFHLEtBQWY7QUFDQSxRQUFJQyxjQUFjLEdBQUcsQ0FBckI7O0FBRUFoQixJQUFBQSxTQUFTLEVBQUUsS0FBSyxNQUFNdkQsU0FBWCxJQUF3QixLQUFLUyxjQUFMLEVBQXhCLEVBQStDO0FBQ3hELFdBQUssTUFBTWtDLElBQVgsSUFBbUIzQyxTQUFTLENBQUN3RCxRQUFWLEVBQW5CLEVBQXlDO0FBQ3ZDLGFBQUssTUFBTUUsTUFBWCxJQUFxQmYsSUFBSSxDQUFDZ0IsVUFBTCxFQUFyQixFQUF3QztBQUN0QyxjQUFJVSxxQkFBcUIsR0FBR1gsTUFBTSxDQUFDYyxjQUFQLEVBQTVCLEVBQXFEO0FBQ25ESixZQUFBQSxZQUFZLEdBQUdWLE1BQU0sQ0FBQ2UsaUJBQVAsS0FBNkJKLHFCQUE1QztBQUNBQyxZQUFBQSxRQUFRLEdBQUcsSUFBWDtBQUNBLGtCQUFNZixTQUFOO0FBQ0QsV0FKRCxNQUlPO0FBQ0xjLFlBQUFBLHFCQUFxQixJQUFJWCxNQUFNLENBQUNjLGNBQVAsRUFBekI7QUFDQUQsWUFBQUEsY0FBYyxHQUFHYixNQUFNLENBQUNnQixlQUFQLEVBQWpCO0FBQ0Q7QUFDRjtBQUNGO0FBQ0YsS0ExQnVDLENBNEJ4QztBQUNBO0FBQ0E7OztBQUNBLFFBQUksQ0FBQ0osUUFBTCxFQUFlO0FBQ2JGLE1BQUFBLFlBQVksR0FBR0csY0FBZjtBQUNEOztBQUVELFdBQU9JLFlBQU1DLFVBQU4sQ0FBaUIsQ0FBQyxDQUFDUixZQUFELEVBQWUsQ0FBZixDQUFELEVBQW9CLENBQUNBLFlBQUQsRUFBZVMsUUFBZixDQUFwQixDQUFqQixDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLHlCQUF5QixDQUFDL0YsYUFBRCxFQUFnQjtBQUN2QyxVQUFNZ0csa0JBQWtCLEdBQUcsS0FBS3RGLG9CQUFMLENBQTBCUCxHQUExQixDQUE4QkgsYUFBOUIsQ0FBM0I7QUFDQSxXQUFPZ0csa0JBQWtCLENBQUN2RixLQUFuQixDQUF5QjBELElBQXpCLEtBQWtDLENBQXpDO0FBQ0Q7O0FBQ0Q5QyxFQUFBQSw0QkFBNEIsQ0FBQ0osU0FBRCxFQUFZO0FBQ3RDLFFBQUlWLE9BQU8sR0FBRyxDQUFkO0FBQ0EsVUFBTUUsS0FBSyxHQUFHLElBQUl3RixnQkFBSixDQUFXLENBQUNDLENBQUQsRUFBSUMsQ0FBSixLQUFVRCxDQUFDLENBQUMzRixPQUFGLEdBQVk0RixDQUFDLENBQUM1RixPQUFuQyxDQUFkO0FBQ0EsU0FBS0csb0JBQUwsQ0FBMEJRLEdBQTFCLENBQThCRCxTQUFTLENBQUNFLE9BQVYsRUFBOUIsRUFBbUQ7QUFBQ1gsTUFBQUEsY0FBYyxFQUFFUyxTQUFTLENBQUNtRixhQUFWLEdBQTBCbkIsS0FBMUIsQ0FBZ0NDLEdBQWpEO0FBQXNEekUsTUFBQUE7QUFBdEQsS0FBbkQ7O0FBRUEsU0FBSyxJQUFJNEYsU0FBUyxHQUFHLENBQXJCLEVBQXdCQSxTQUFTLEdBQUdwRixTQUFTLENBQUN3RCxRQUFWLEdBQXFCNkIsTUFBekQsRUFBaUVELFNBQVMsRUFBMUUsRUFBOEU7QUFDNUUsWUFBTXpDLElBQUksR0FBRzNDLFNBQVMsQ0FBQ3dELFFBQVYsR0FBcUI0QixTQUFyQixDQUFiO0FBQ0EsV0FBS3JGLGFBQUwsQ0FBbUJFLEdBQW5CLENBQXVCMEMsSUFBSSxDQUFDeEMsU0FBTCxFQUF2QixFQUF5Q3dDLElBQXpDLEVBRjRFLENBSTVFOztBQUNBckQsTUFBQUEsT0FBTyxJQUFJcUQsSUFBSSxDQUFDNkIsY0FBTCxFQUFYO0FBQ0FoRixNQUFBQSxLQUFLLENBQUM4RixNQUFOLENBQWE7QUFBQ2hHLFFBQUFBLE9BQUQ7QUFBVUksUUFBQUEsTUFBTSxFQUFFMEYsU0FBUyxHQUFHO0FBQTlCLE9BQWIsRUFONEUsQ0FRNUU7O0FBQ0E5RixNQUFBQSxPQUFPO0FBQ1I7QUFDRjs7QUFFRGlHLEVBQUFBLFdBQVcsQ0FBQ25ELGVBQUQsRUFBa0I7QUFDM0JBLElBQUFBLGVBQWUsQ0FBQ29ELGNBQWhCO0FBRUEsU0FBSzNGLG1CQUFMLENBQXlCNEYsS0FBekI7QUFDQSxTQUFLMUYsYUFBTCxDQUFtQjBGLEtBQW5CO0FBRUEsVUFBTUMsU0FBUyxHQUFHdEQsZUFBZSxDQUFDdUQsS0FBaEIsQ0FBc0IsS0FBS2hILFdBQTNCLENBQWxCOztBQUVBLFNBQUssTUFBTXFCLFNBQVgsSUFBd0IsS0FBS1MsY0FBTCxFQUF4QixFQUErQztBQUM3Q1QsTUFBQUEsU0FBUyxDQUFDNEYsYUFBVixDQUF3QkYsU0FBeEI7QUFDQSxXQUFLN0YsbUJBQUwsQ0FBeUJJLEdBQXpCLENBQTZCRCxTQUFTLENBQUNHLFNBQVYsRUFBN0IsRUFBb0RILFNBQXBEOztBQUVBLFdBQUssTUFBTTJDLElBQVgsSUFBbUIzQyxTQUFTLENBQUN3RCxRQUFWLEVBQW5CLEVBQXlDO0FBQ3ZDLGFBQUt6RCxhQUFMLENBQW1CRSxHQUFuQixDQUF1QjBDLElBQUksQ0FBQ3hDLFNBQUwsRUFBdkIsRUFBeUN3QyxJQUF6QztBQUNEO0FBQ0Y7O0FBRUQsU0FBS2hFLFdBQUwsR0FBbUJ5RCxlQUFuQjtBQUNEO0FBRUQ7Ozs7O0FBR0FFLEVBQUFBLHdCQUF3QixDQUFDdUQsTUFBRCxFQUFTO0FBQy9CLFVBQU1DLFlBQVksR0FBR0MsS0FBSyxDQUFDQyxJQUFOLENBQVdILE1BQVgsQ0FBckI7QUFDQUMsSUFBQUEsWUFBWSxDQUFDRyxJQUFiLENBQWtCLENBQUNoQixDQUFELEVBQUlDLENBQUosS0FBVUQsQ0FBQyxHQUFHQyxDQUFoQztBQUVBLFVBQU1yRyxXQUFXLEdBQUcsRUFBcEI7QUFDQSxRQUFJcUgsYUFBYSxHQUFHLElBQXBCOztBQUNBLFNBQUssTUFBTWpDLEdBQVgsSUFBa0I2QixZQUFsQixFQUFnQztBQUM5QjtBQUNBO0FBQ0EsVUFBSUksYUFBYSxJQUFJQSxhQUFhLENBQUNDLFdBQWQsQ0FBMEJsQyxHQUExQixDQUFyQixFQUFxRDtBQUNuRDtBQUNEOztBQUVEaUMsTUFBQUEsYUFBYSxHQUFHLEtBQUt2RSxjQUFMLENBQW9Cc0MsR0FBcEIsQ0FBaEI7QUFDQXBGLE1BQUFBLFdBQVcsQ0FBQ3VILElBQVosQ0FBaUJGLGFBQWpCO0FBQ0Q7O0FBRUQsV0FBT3JILFdBQVA7QUFDRDs7QUFFRHdILEVBQUFBLFVBQVUsR0FBRztBQUNYLFdBQU8sS0FBSzFILFdBQUwsS0FBcUIsSUFBckIsSUFBNkIsS0FBS0UsV0FBTCxDQUFpQnlILElBQWpCLENBQXNCOUQsRUFBRSxJQUFJQSxFQUFFLENBQUNoQixTQUFILEVBQTVCLENBQXBDO0FBQ0Q7O0FBRUQrRSxFQUFBQSwwQkFBMEIsR0FBRztBQUMzQixTQUFLLE1BQU12RyxTQUFYLElBQXdCLEtBQUtTLGNBQUwsRUFBeEIsRUFBK0M7QUFDN0MsVUFBSVQsU0FBUyxDQUFDd0csdUJBQVYsRUFBSixFQUF5QztBQUN2QyxlQUFPLElBQVA7QUFDRDtBQUNGOztBQUNELFdBQU8sS0FBUDtBQUNEOztBQUVEQyxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixXQUFPLEtBQUtoRyxjQUFMLEdBQXNCNkYsSUFBdEIsQ0FBMkI5RCxFQUFFLElBQUlBLEVBQUUsQ0FBQ2tFLGFBQUgsRUFBakMsQ0FBUDtBQUNEOztBQUVEQyxFQUFBQSxxQkFBcUIsR0FBRztBQUN0QixXQUFPLEtBQUtsRyxjQUFMLEdBQXNCVSxNQUF0QixDQUE2QixDQUFDeUYsUUFBRCxFQUFXNUcsU0FBWCxLQUF5QjtBQUMzRCxZQUFNNkcsS0FBSyxHQUFHN0csU0FBUyxDQUFDMkcscUJBQVYsRUFBZDtBQUNBLGFBQU9DLFFBQVEsSUFBSUMsS0FBWixHQUFvQkQsUUFBcEIsR0FBK0JDLEtBQXRDO0FBQ0QsS0FITSxFQUdKLENBSEksQ0FBUDtBQUlEOztBQUVEQyxFQUFBQSxrQkFBa0IsQ0FBQ0MsSUFBRCxFQUFPO0FBQ3ZCLFFBQUliLGFBQWEsR0FBRyxJQUFwQjs7QUFDQSxTQUFLLE1BQU1qQyxHQUFYLElBQWtCOEMsSUFBbEIsRUFBd0I7QUFDdEIsVUFBSWIsYUFBSixFQUFtQjtBQUNqQixZQUFJQSxhQUFhLENBQUNDLFdBQWQsQ0FBMEJsQyxHQUExQixDQUFKLEVBQW9DO0FBQ2xDO0FBQ0Q7O0FBRUQsZUFBTyxJQUFQO0FBQ0QsT0FORCxNQU1PO0FBQ0xpQyxRQUFBQSxhQUFhLEdBQUcsS0FBS3ZFLGNBQUwsQ0FBb0JzQyxHQUFwQixDQUFoQjtBQUNEO0FBQ0Y7O0FBQ0QsV0FBTyxLQUFQO0FBQ0Q7O0FBRUQrQyxFQUFBQSxpQkFBaUIsQ0FBQ2hILFNBQUQsRUFBWTtBQUMzQixVQUFNUixLQUFLLEdBQUcsS0FBS1gsV0FBTCxDQUFpQm9JLE9BQWpCLENBQXlCakgsU0FBekIsQ0FBZDtBQUVBLFNBQUtILG1CQUFMLENBQXlCcUgsTUFBekIsQ0FBZ0NsSCxTQUFTLENBQUNHLFNBQVYsRUFBaEM7O0FBQ0EsU0FBSyxNQUFNd0MsSUFBWCxJQUFtQjNDLFNBQVMsQ0FBQ3dELFFBQVYsRUFBbkIsRUFBeUM7QUFDdkMsV0FBS3pELGFBQUwsQ0FBbUJtSCxNQUFuQixDQUEwQnZFLElBQUksQ0FBQ3hDLFNBQUwsRUFBMUI7QUFDRDs7QUFFRCxVQUFNZ0gsTUFBTSxHQUFHLEtBQUtDLGdCQUFMLENBQXNCNUgsS0FBdEIsQ0FBZjtBQUNBLFVBQU02SCxLQUFLLEdBQUcsS0FBS0MsZUFBTCxDQUFxQjlILEtBQXJCLENBQWQ7QUFFQVEsSUFBQUEsU0FBUyxDQUFDdUgsaUJBQVYsQ0FBNEIsS0FBSzVJLFdBQWpDLEVBQThDO0FBQUN3SSxNQUFBQSxNQUFEO0FBQVNFLE1BQUFBO0FBQVQsS0FBOUM7QUFFQSxTQUFLeEgsbUJBQUwsQ0FBeUJJLEdBQXpCLENBQTZCRCxTQUFTLENBQUNHLFNBQVYsRUFBN0IsRUFBb0RILFNBQXBELEVBYjJCLENBZTNCOztBQUNBOztBQUNBLFNBQUssTUFBTTJDLElBQVgsSUFBbUIzQyxTQUFTLENBQUN3RCxRQUFWLEVBQW5CLEVBQXlDO0FBQ3ZDLFdBQUt6RCxhQUFMLENBQW1CRSxHQUFuQixDQUF1QjBDLElBQUksQ0FBQ3hDLFNBQUwsRUFBdkIsRUFBeUN3QyxJQUF6QztBQUNEO0FBQ0Y7O0FBRUQ2RSxFQUFBQSxlQUFlLENBQUN4SCxTQUFELEVBQVk7QUFDekIsVUFBTVIsS0FBSyxHQUFHLEtBQUtYLFdBQUwsQ0FBaUJvSSxPQUFqQixDQUF5QmpILFNBQXpCLENBQWQ7QUFFQSxTQUFLSCxtQkFBTCxDQUF5QnFILE1BQXpCLENBQWdDbEgsU0FBUyxDQUFDRyxTQUFWLEVBQWhDOztBQUNBLFNBQUssTUFBTXdDLElBQVgsSUFBbUIzQyxTQUFTLENBQUN3RCxRQUFWLEVBQW5CLEVBQXlDO0FBQ3ZDLFdBQUt6RCxhQUFMLENBQW1CbUgsTUFBbkIsQ0FBMEJ2RSxJQUFJLENBQUN4QyxTQUFMLEVBQTFCO0FBQ0Q7O0FBRUQsVUFBTWdILE1BQU0sR0FBRyxLQUFLQyxnQkFBTCxDQUFzQjVILEtBQXRCLENBQWY7QUFDQSxVQUFNNkgsS0FBSyxHQUFHLEtBQUtDLGVBQUwsQ0FBcUI5SCxLQUFyQixDQUFkO0FBRUFRLElBQUFBLFNBQVMsQ0FBQ3lILGVBQVYsQ0FBMEIsS0FBSzlJLFdBQS9CLEVBQTRDO0FBQUN3SSxNQUFBQSxNQUFEO0FBQVNFLE1BQUFBO0FBQVQsS0FBNUM7QUFFQSxTQUFLeEgsbUJBQUwsQ0FBeUJJLEdBQXpCLENBQTZCRCxTQUFTLENBQUNHLFNBQVYsRUFBN0IsRUFBb0RILFNBQXBEOztBQUNBLFNBQUssTUFBTTJDLElBQVgsSUFBbUIzQyxTQUFTLENBQUN3RCxRQUFWLEVBQW5CLEVBQXlDO0FBQ3ZDLFdBQUt6RCxhQUFMLENBQW1CRSxHQUFuQixDQUF1QjBDLElBQUksQ0FBQ3hDLFNBQUwsRUFBdkIsRUFBeUN3QyxJQUF6QztBQUNELEtBaEJ3QixDQWtCekI7QUFDQTs7O0FBQ0EsUUFBSSxLQUFLbUMseUJBQUwsQ0FBK0I5RSxTQUFTLENBQUNFLE9BQVYsRUFBL0IsQ0FBSixFQUF5RDtBQUN2RCxXQUFLRSw0QkFBTCxDQUFrQ0osU0FBbEM7QUFDRDtBQUNGOztBQUVEb0gsRUFBQUEsZ0JBQWdCLENBQUNNLGNBQUQsRUFBaUI7QUFDL0IsVUFBTVAsTUFBTSxHQUFHLEVBQWY7QUFDQSxRQUFJUSxXQUFXLEdBQUdELGNBQWMsR0FBRyxDQUFuQzs7QUFDQSxXQUFPQyxXQUFXLElBQUksQ0FBdEIsRUFBeUI7QUFDdkIsWUFBTUMsZUFBZSxHQUFHLEtBQUsvSSxXQUFMLENBQWlCOEksV0FBakIsQ0FBeEI7QUFDQVIsTUFBQUEsTUFBTSxDQUFDZixJQUFQLENBQVksR0FBR3dCLGVBQWUsQ0FBQ0MsZ0JBQWhCLEVBQWY7O0FBRUEsVUFBSSxDQUFDRCxlQUFlLENBQUN6SCxTQUFoQixHQUE0QjJILFFBQTVCLEdBQXVDQyxPQUF2QyxFQUFMLEVBQXVEO0FBQ3JEO0FBQ0Q7O0FBQ0RKLE1BQUFBLFdBQVc7QUFDWjs7QUFDRCxXQUFPUixNQUFQO0FBQ0Q7O0FBRURHLEVBQUFBLGVBQWUsQ0FBQ0ksY0FBRCxFQUFpQjtBQUM5QixVQUFNTCxLQUFLLEdBQUcsRUFBZDtBQUNBLFFBQUlXLFVBQVUsR0FBR04sY0FBYyxHQUFHLENBQWxDOztBQUNBLFdBQU9NLFVBQVUsR0FBRyxLQUFLbkosV0FBTCxDQUFpQndHLE1BQXJDLEVBQTZDO0FBQzNDLFlBQU00QyxjQUFjLEdBQUcsS0FBS3BKLFdBQUwsQ0FBaUJtSixVQUFqQixDQUF2QjtBQUNBWCxNQUFBQSxLQUFLLENBQUNqQixJQUFOLENBQVcsR0FBRzZCLGNBQWMsQ0FBQ0Msa0JBQWYsRUFBZDs7QUFFQSxVQUFJLENBQUNELGNBQWMsQ0FBQzlILFNBQWYsR0FBMkIySCxRQUEzQixHQUFzQ0MsT0FBdEMsRUFBTCxFQUFzRDtBQUNwRDtBQUNEOztBQUNEQyxNQUFBQSxVQUFVO0FBQ1g7O0FBQ0QsV0FBT1gsS0FBUDtBQUNEOztBQWdCRDs7O0FBR0FjLEVBQUFBLFFBQVEsR0FBRztBQUNULFdBQU8sS0FBS3RKLFdBQUwsQ0FBaUIwRCxHQUFqQixDQUFxQkMsRUFBRSxJQUFJQSxFQUFFLENBQUM0RixVQUFILENBQWMsS0FBSzFILFNBQUwsRUFBZCxDQUEzQixFQUE0RDJILElBQTVELENBQWlFLEVBQWpFLElBQXVFLElBQTlFO0FBQ0Q7QUFFRDs7OztBQUdBOzs7QUFDQUMsRUFBQUEsT0FBTyxHQUFHO0FBQ1IsUUFBSUMsYUFBYSxHQUFHLGlCQUFwQjtBQUNBQSxJQUFBQSxhQUFhLElBQUsseUJBQXdCeEMsS0FBSyxDQUFDQyxJQUFOLENBQVcsS0FBS25HLG1CQUFMLENBQXlCMkksSUFBekIsRUFBWCxFQUE0Q0MsQ0FBQyxJQUFJQSxDQUFDLENBQUNDLEVBQW5ELEVBQXVETCxJQUF2RCxDQUE0RCxJQUE1RCxDQUFrRSxHQUE1RztBQUNBRSxJQUFBQSxhQUFhLElBQUssbUJBQWtCeEMsS0FBSyxDQUFDQyxJQUFOLENBQVcsS0FBS2pHLGFBQUwsQ0FBbUJ5SSxJQUFuQixFQUFYLEVBQXNDQyxDQUFDLElBQUlBLENBQUMsQ0FBQ0MsRUFBN0MsRUFBaURMLElBQWpELENBQXNELElBQXRELENBQTRELEtBQWhHOztBQUNBLFNBQUssTUFBTXJJLFNBQVgsSUFBd0IsS0FBS25CLFdBQTdCLEVBQTBDO0FBQ3hDMEosTUFBQUEsYUFBYSxJQUFJdkksU0FBUyxDQUFDc0ksT0FBVixDQUFrQjtBQUFDSyxRQUFBQSxNQUFNLEVBQUU7QUFBVCxPQUFsQixDQUFqQjtBQUNEOztBQUNESixJQUFBQSxhQUFhLElBQUksS0FBakI7QUFDQSxXQUFPQSxhQUFQO0FBQ0Q7QUFFRDs7O0FBQ0FLLEVBQUFBLE9BQU8sQ0FBQ0MsS0FBRCxFQUFRO0FBQ2IsV0FBTyxLQUFLVixRQUFMLE9BQW9CVSxLQUFLLENBQUNWLFFBQU4sRUFBM0I7QUFDRDs7QUE5WmlDIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tLTEuMzYuMS9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1JhbmdlfSBmcm9tICdhdG9tJztcbmltcG9ydCB7UkJUcmVlfSBmcm9tICdiaW50cmVlcyc7XG5cbmltcG9ydCBQYXRjaEJ1ZmZlciBmcm9tICcuL3BhdGNoLWJ1ZmZlcic7XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIE11bHRpRmlsZVBhdGNoIHtcbiAgc3RhdGljIGNyZWF0ZU51bGwoKSB7XG4gICAgcmV0dXJuIG5ldyB0aGlzKHtwYXRjaEJ1ZmZlcjogbmV3IFBhdGNoQnVmZmVyKCksIGZpbGVQYXRjaGVzOiBbXX0pO1xuICB9XG5cbiAgY29uc3RydWN0b3Ioe3BhdGNoQnVmZmVyLCBmaWxlUGF0Y2hlc30pIHtcbiAgICB0aGlzLnBhdGNoQnVmZmVyID0gcGF0Y2hCdWZmZXI7XG4gICAgdGhpcy5maWxlUGF0Y2hlcyA9IGZpbGVQYXRjaGVzO1xuXG4gICAgdGhpcy5maWxlUGF0Y2hlc0J5TWFya2VyID0gbmV3IE1hcCgpO1xuICAgIHRoaXMuZmlsZVBhdGNoZXNCeVBhdGggPSBuZXcgTWFwKCk7XG4gICAgdGhpcy5odW5rc0J5TWFya2VyID0gbmV3IE1hcCgpO1xuXG4gICAgLy8gU3RvcmUgYSBtYXAgb2Yge2RpZmZSb3csIG9mZnNldH0gZm9yIGVhY2ggRmlsZVBhdGNoIHdoZXJlIG9mZnNldCBpcyB0aGUgbnVtYmVyIG9mIEh1bmsgaGVhZGVycyB3aXRoaW4gdGhlIGN1cnJlbnRcbiAgICAvLyBGaWxlUGF0Y2ggdGhhdCBvY2N1ciBiZWZvcmUgdGhpcyByb3cgaW4gdGhlIG9yaWdpbmFsIGRpZmYgb3V0cHV0LlxuICAgIHRoaXMuZGlmZlJvd09mZnNldEluZGljZXMgPSBuZXcgTWFwKCk7XG5cbiAgICBmb3IgKGNvbnN0IGZpbGVQYXRjaCBvZiB0aGlzLmZpbGVQYXRjaGVzKSB7XG4gICAgICB0aGlzLmZpbGVQYXRjaGVzQnlQYXRoLnNldChmaWxlUGF0Y2guZ2V0UGF0aCgpLCBmaWxlUGF0Y2gpO1xuICAgICAgdGhpcy5maWxlUGF0Y2hlc0J5TWFya2VyLnNldChmaWxlUGF0Y2guZ2V0TWFya2VyKCksIGZpbGVQYXRjaCk7XG5cbiAgICAgIHRoaXMucG9wdWxhdGVEaWZmUm93T2Zmc2V0SW5kaWNlcyhmaWxlUGF0Y2gpO1xuICAgIH1cbiAgfVxuXG4gIGNsb25lKG9wdHMgPSB7fSkge1xuICAgIHJldHVybiBuZXcgdGhpcy5jb25zdHJ1Y3Rvcih7XG4gICAgICBwYXRjaEJ1ZmZlcjogb3B0cy5wYXRjaEJ1ZmZlciAhPT0gdW5kZWZpbmVkID8gb3B0cy5wYXRjaEJ1ZmZlciA6IHRoaXMuZ2V0UGF0Y2hCdWZmZXIoKSxcbiAgICAgIGZpbGVQYXRjaGVzOiBvcHRzLmZpbGVQYXRjaGVzICE9PSB1bmRlZmluZWQgPyBvcHRzLmZpbGVQYXRjaGVzIDogdGhpcy5nZXRGaWxlUGF0Y2hlcygpLFxuICAgIH0pO1xuICB9XG5cbiAgZ2V0UGF0Y2hCdWZmZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMucGF0Y2hCdWZmZXI7XG4gIH1cblxuICBnZXRCdWZmZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UGF0Y2hCdWZmZXIoKS5nZXRCdWZmZXIoKTtcbiAgfVxuXG4gIGdldFBhdGNoTGF5ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UGF0Y2hCdWZmZXIoKS5nZXRMYXllcigncGF0Y2gnKTtcbiAgfVxuXG4gIGdldEh1bmtMYXllcigpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRQYXRjaEJ1ZmZlcigpLmdldExheWVyKCdodW5rJyk7XG4gIH1cblxuICBnZXRVbmNoYW5nZWRMYXllcigpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRQYXRjaEJ1ZmZlcigpLmdldExheWVyKCd1bmNoYW5nZWQnKTtcbiAgfVxuXG4gIGdldEFkZGl0aW9uTGF5ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UGF0Y2hCdWZmZXIoKS5nZXRMYXllcignYWRkaXRpb24nKTtcbiAgfVxuXG4gIGdldERlbGV0aW9uTGF5ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UGF0Y2hCdWZmZXIoKS5nZXRMYXllcignZGVsZXRpb24nKTtcbiAgfVxuXG4gIGdldE5vTmV3bGluZUxheWVyKCkge1xuICAgIHJldHVybiB0aGlzLmdldFBhdGNoQnVmZmVyKCkuZ2V0TGF5ZXIoJ25vbmV3bGluZScpO1xuICB9XG5cbiAgZ2V0RmlsZVBhdGNoZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuZmlsZVBhdGNoZXM7XG4gIH1cblxuICBnZXRQYXRoU2V0KCkge1xuICAgIHJldHVybiB0aGlzLmdldEZpbGVQYXRjaGVzKCkucmVkdWNlKChwYXRoU2V0LCBmaWxlUGF0Y2gpID0+IHtcbiAgICAgIGZvciAoY29uc3QgZmlsZSBvZiBbZmlsZVBhdGNoLmdldE9sZEZpbGUoKSwgZmlsZVBhdGNoLmdldE5ld0ZpbGUoKV0pIHtcbiAgICAgICAgaWYgKGZpbGUuaXNQcmVzZW50KCkpIHtcbiAgICAgICAgICBwYXRoU2V0LmFkZChmaWxlLmdldFBhdGgoKSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiBwYXRoU2V0O1xuICAgIH0sIG5ldyBTZXQoKSk7XG4gIH1cblxuICBnZXRGaWxlUGF0Y2hBdChidWZmZXJSb3cpIHtcbiAgICBpZiAoYnVmZmVyUm93IDwgMCB8fCBidWZmZXJSb3cgPiB0aGlzLnBhdGNoQnVmZmVyLmdldEJ1ZmZlcigpLmdldExhc3RSb3coKSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgY29uc3QgW21hcmtlcl0gPSB0aGlzLnBhdGNoQnVmZmVyLmZpbmRNYXJrZXJzKCdwYXRjaCcsIHtpbnRlcnNlY3RzUm93OiBidWZmZXJSb3d9KTtcbiAgICByZXR1cm4gdGhpcy5maWxlUGF0Y2hlc0J5TWFya2VyLmdldChtYXJrZXIpO1xuICB9XG5cbiAgZ2V0SHVua0F0KGJ1ZmZlclJvdykge1xuICAgIGlmIChidWZmZXJSb3cgPCAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCBbbWFya2VyXSA9IHRoaXMucGF0Y2hCdWZmZXIuZmluZE1hcmtlcnMoJ2h1bmsnLCB7aW50ZXJzZWN0c1JvdzogYnVmZmVyUm93fSk7XG4gICAgcmV0dXJuIHRoaXMuaHVua3NCeU1hcmtlci5nZXQobWFya2VyKTtcbiAgfVxuXG4gIGdldFN0YWdlUGF0Y2hGb3JMaW5lcyhzZWxlY3RlZExpbmVTZXQpIHtcbiAgICBjb25zdCBuZXh0UGF0Y2hCdWZmZXIgPSBuZXcgUGF0Y2hCdWZmZXIoKTtcbiAgICBjb25zdCBuZXh0RmlsZVBhdGNoZXMgPSB0aGlzLmdldEZpbGVQYXRjaGVzQ29udGFpbmluZyhzZWxlY3RlZExpbmVTZXQpLm1hcChmcCA9PiB7XG4gICAgICByZXR1cm4gZnAuYnVpbGRTdGFnZVBhdGNoRm9yTGluZXModGhpcy5nZXRCdWZmZXIoKSwgbmV4dFBhdGNoQnVmZmVyLCBzZWxlY3RlZExpbmVTZXQpO1xuICAgIH0pO1xuICAgIHJldHVybiB0aGlzLmNsb25lKHtwYXRjaEJ1ZmZlcjogbmV4dFBhdGNoQnVmZmVyLCBmaWxlUGF0Y2hlczogbmV4dEZpbGVQYXRjaGVzfSk7XG4gIH1cblxuICBnZXRTdGFnZVBhdGNoRm9ySHVuayhodW5rKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0U3RhZ2VQYXRjaEZvckxpbmVzKG5ldyBTZXQoaHVuay5nZXRCdWZmZXJSb3dzKCkpKTtcbiAgfVxuXG4gIGdldFVuc3RhZ2VQYXRjaEZvckxpbmVzKHNlbGVjdGVkTGluZVNldCkge1xuICAgIGNvbnN0IG5leHRQYXRjaEJ1ZmZlciA9IG5ldyBQYXRjaEJ1ZmZlcigpO1xuICAgIGNvbnN0IG5leHRGaWxlUGF0Y2hlcyA9IHRoaXMuZ2V0RmlsZVBhdGNoZXNDb250YWluaW5nKHNlbGVjdGVkTGluZVNldCkubWFwKGZwID0+IHtcbiAgICAgIHJldHVybiBmcC5idWlsZFVuc3RhZ2VQYXRjaEZvckxpbmVzKHRoaXMuZ2V0QnVmZmVyKCksIG5leHRQYXRjaEJ1ZmZlciwgc2VsZWN0ZWRMaW5lU2V0KTtcbiAgICB9KTtcbiAgICByZXR1cm4gdGhpcy5jbG9uZSh7cGF0Y2hCdWZmZXI6IG5leHRQYXRjaEJ1ZmZlciwgZmlsZVBhdGNoZXM6IG5leHRGaWxlUGF0Y2hlc30pO1xuICB9XG5cbiAgZ2V0VW5zdGFnZVBhdGNoRm9ySHVuayhodW5rKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0VW5zdGFnZVBhdGNoRm9yTGluZXMobmV3IFNldChodW5rLmdldEJ1ZmZlclJvd3MoKSkpO1xuICB9XG5cbiAgZ2V0TWF4U2VsZWN0aW9uSW5kZXgoc2VsZWN0ZWRSb3dzKSB7XG4gICAgaWYgKHNlbGVjdGVkUm93cy5zaXplID09PSAwKSB7XG4gICAgICByZXR1cm4gMDtcbiAgICB9XG5cbiAgICBjb25zdCBsYXN0TWF4ID0gTWF0aC5tYXgoLi4uc2VsZWN0ZWRSb3dzKTtcblxuICAgIGxldCBzZWxlY3Rpb25JbmRleCA9IDA7XG4gICAgLy8gY291bnRzIHVuc2VsZWN0ZWQgbGluZXMgaW4gY2hhbmdlZCByZWdpb25zIGZyb20gdGhlIG9sZCBwYXRjaFxuICAgIC8vIHVudGlsIHdlIGdldCB0byB0aGUgYm90dG9tLW1vc3Qgc2VsZWN0ZWQgbGluZSBmcm9tIHRoZSBvbGQgcGF0Y2ggKGxhc3RNYXgpLlxuICAgIHBhdGNoTG9vcDogZm9yIChjb25zdCBmaWxlUGF0Y2ggb2YgdGhpcy5nZXRGaWxlUGF0Y2hlcygpKSB7XG4gICAgICBmb3IgKGNvbnN0IGh1bmsgb2YgZmlsZVBhdGNoLmdldEh1bmtzKCkpIHtcbiAgICAgICAgbGV0IGluY2x1ZGVzTWF4ID0gZmFsc2U7XG5cbiAgICAgICAgZm9yIChjb25zdCBjaGFuZ2Ugb2YgaHVuay5nZXRDaGFuZ2VzKCkpIHtcbiAgICAgICAgICBmb3IgKGNvbnN0IHtpbnRlcnNlY3Rpb24sIGdhcH0gb2YgY2hhbmdlLmludGVyc2VjdFJvd3Moc2VsZWN0ZWRSb3dzLCB0cnVlKSkge1xuICAgICAgICAgICAgLy8gT25seSBpbmNsdWRlIGEgcGFydGlhbCByYW5nZSBpZiB0aGlzIGludGVyc2VjdGlvbiBpbmNsdWRlcyB0aGUgbGFzdCBzZWxlY3RlZCBidWZmZXIgcm93LlxuICAgICAgICAgICAgaW5jbHVkZXNNYXggPSBpbnRlcnNlY3Rpb24uaW50ZXJzZWN0c1JvdyhsYXN0TWF4KTtcbiAgICAgICAgICAgIGNvbnN0IGRlbHRhID0gaW5jbHVkZXNNYXggPyBsYXN0TWF4IC0gaW50ZXJzZWN0aW9uLnN0YXJ0LnJvdyArIDEgOiBpbnRlcnNlY3Rpb24uZ2V0Um93Q291bnQoKTtcblxuICAgICAgICAgICAgaWYgKGdhcCkge1xuICAgICAgICAgICAgICAvLyBSYW5nZSBvZiB1bnNlbGVjdGVkIGNoYW5nZXMuXG4gICAgICAgICAgICAgIHNlbGVjdGlvbkluZGV4ICs9IGRlbHRhO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICBpZiAoaW5jbHVkZXNNYXgpIHtcbiAgICAgICAgICAgICAgYnJlYWsgcGF0Y2hMb29wO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBzZWxlY3Rpb25JbmRleDtcbiAgfVxuXG4gIGdldFNlbGVjdGlvblJhbmdlRm9ySW5kZXgoc2VsZWN0aW9uSW5kZXgpIHtcbiAgICAvLyBJdGVyYXRlIG92ZXIgY2hhbmdlZCBsaW5lcyBpbiB0aGlzIHBhdGNoIGluIG9yZGVyIHRvIGZpbmQgdGhlXG4gICAgLy8gbmV3IHJvdyB0byBiZSBzZWxlY3RlZCBiYXNlZCBvbiB0aGUgbGFzdCBzZWxlY3Rpb24gaW5kZXguXG4gICAgLy8gQXMgd2Ugd2FsayB0aHJvdWdoIHRoZSBjaGFuZ2VkIGxpbmVzLCB3ZSB3aGl0dGxlIGRvd24gdGhlXG4gICAgLy8gcmVtYWluaW5nIGxpbmVzIHVudGlsIHdlIHJlYWNoIHRoZSByb3cgdGhhdCBjb3JyZXNwb25kcyB0byB0aGVcbiAgICAvLyBsYXN0IHNlbGVjdGVkIGluZGV4LlxuXG4gICAgbGV0IHNlbGVjdGlvblJvdyA9IDA7XG4gICAgbGV0IHJlbWFpbmluZ0NoYW5nZWRMaW5lcyA9IHNlbGVjdGlvbkluZGV4O1xuXG4gICAgbGV0IGZvdW5kUm93ID0gZmFsc2U7XG4gICAgbGV0IGxhc3RDaGFuZ2VkUm93ID0gMDtcblxuICAgIHBhdGNoTG9vcDogZm9yIChjb25zdCBmaWxlUGF0Y2ggb2YgdGhpcy5nZXRGaWxlUGF0Y2hlcygpKSB7XG4gICAgICBmb3IgKGNvbnN0IGh1bmsgb2YgZmlsZVBhdGNoLmdldEh1bmtzKCkpIHtcbiAgICAgICAgZm9yIChjb25zdCBjaGFuZ2Ugb2YgaHVuay5nZXRDaGFuZ2VzKCkpIHtcbiAgICAgICAgICBpZiAocmVtYWluaW5nQ2hhbmdlZExpbmVzIDwgY2hhbmdlLmJ1ZmZlclJvd0NvdW50KCkpIHtcbiAgICAgICAgICAgIHNlbGVjdGlvblJvdyA9IGNoYW5nZS5nZXRTdGFydEJ1ZmZlclJvdygpICsgcmVtYWluaW5nQ2hhbmdlZExpbmVzO1xuICAgICAgICAgICAgZm91bmRSb3cgPSB0cnVlO1xuICAgICAgICAgICAgYnJlYWsgcGF0Y2hMb29wO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICByZW1haW5pbmdDaGFuZ2VkTGluZXMgLT0gY2hhbmdlLmJ1ZmZlclJvd0NvdW50KCk7XG4gICAgICAgICAgICBsYXN0Q2hhbmdlZFJvdyA9IGNoYW5nZS5nZXRFbmRCdWZmZXJSb3coKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBJZiB3ZSBuZXZlciBnb3QgdG8gdGhlIGxhc3Qgc2VsZWN0ZWQgaW5kZXgsIHRoYXQgbWVhbnMgaXQgaXNcbiAgICAvLyBubyBsb25nZXIgcHJlc2VudCBpbiB0aGUgbmV3IHBhdGNoIChpZS4gd2Ugc3RhZ2VkIHRoZSBsYXN0IGxpbmUgb2YgdGhlIGZpbGUpLlxuICAgIC8vIEluIHRoaXMgY2FzZSB3ZSB3YW50IHRoZSBuZXh0IHNlbGVjdGVkIGxpbmUgdG8gYmUgdGhlIGxhc3QgY2hhbmdlZCByb3cgaW4gdGhlIGZpbGVcbiAgICBpZiAoIWZvdW5kUm93KSB7XG4gICAgICBzZWxlY3Rpb25Sb3cgPSBsYXN0Q2hhbmdlZFJvdztcbiAgICB9XG5cbiAgICByZXR1cm4gUmFuZ2UuZnJvbU9iamVjdChbW3NlbGVjdGlvblJvdywgMF0sIFtzZWxlY3Rpb25Sb3csIEluZmluaXR5XV0pO1xuICB9XG5cbiAgaXNEaWZmUm93T2Zmc2V0SW5kZXhFbXB0eShmaWxlUGF0Y2hQYXRoKSB7XG4gICAgY29uc3QgZGlmZlJvd09mZnNldEluZGV4ID0gdGhpcy5kaWZmUm93T2Zmc2V0SW5kaWNlcy5nZXQoZmlsZVBhdGNoUGF0aCk7XG4gICAgcmV0dXJuIGRpZmZSb3dPZmZzZXRJbmRleC5pbmRleC5zaXplID09PSAwO1xuICB9XG4gIHBvcHVsYXRlRGlmZlJvd09mZnNldEluZGljZXMoZmlsZVBhdGNoKSB7XG4gICAgbGV0IGRpZmZSb3cgPSAxO1xuICAgIGNvbnN0IGluZGV4ID0gbmV3IFJCVHJlZSgoYSwgYikgPT4gYS5kaWZmUm93IC0gYi5kaWZmUm93KTtcbiAgICB0aGlzLmRpZmZSb3dPZmZzZXRJbmRpY2VzLnNldChmaWxlUGF0Y2guZ2V0UGF0aCgpLCB7c3RhcnRCdWZmZXJSb3c6IGZpbGVQYXRjaC5nZXRTdGFydFJhbmdlKCkuc3RhcnQucm93LCBpbmRleH0pO1xuXG4gICAgZm9yIChsZXQgaHVua0luZGV4ID0gMDsgaHVua0luZGV4IDwgZmlsZVBhdGNoLmdldEh1bmtzKCkubGVuZ3RoOyBodW5rSW5kZXgrKykge1xuICAgICAgY29uc3QgaHVuayA9IGZpbGVQYXRjaC5nZXRIdW5rcygpW2h1bmtJbmRleF07XG4gICAgICB0aGlzLmh1bmtzQnlNYXJrZXIuc2V0KGh1bmsuZ2V0TWFya2VyKCksIGh1bmspO1xuXG4gICAgICAvLyBBZHZhbmNlIHBhc3QgdGhlIGh1bmsgYm9keVxuICAgICAgZGlmZlJvdyArPSBodW5rLmJ1ZmZlclJvd0NvdW50KCk7XG4gICAgICBpbmRleC5pbnNlcnQoe2RpZmZSb3csIG9mZnNldDogaHVua0luZGV4ICsgMX0pO1xuXG4gICAgICAvLyBBZHZhbmNlIHBhc3QgdGhlIG5leHQgaHVuayBoZWFkZXJcbiAgICAgIGRpZmZSb3crKztcbiAgICB9XG4gIH1cblxuICBhZG9wdEJ1ZmZlcihuZXh0UGF0Y2hCdWZmZXIpIHtcbiAgICBuZXh0UGF0Y2hCdWZmZXIuY2xlYXJBbGxMYXllcnMoKTtcblxuICAgIHRoaXMuZmlsZVBhdGNoZXNCeU1hcmtlci5jbGVhcigpO1xuICAgIHRoaXMuaHVua3NCeU1hcmtlci5jbGVhcigpO1xuXG4gICAgY29uc3QgbWFya2VyTWFwID0gbmV4dFBhdGNoQnVmZmVyLmFkb3B0KHRoaXMucGF0Y2hCdWZmZXIpO1xuXG4gICAgZm9yIChjb25zdCBmaWxlUGF0Y2ggb2YgdGhpcy5nZXRGaWxlUGF0Y2hlcygpKSB7XG4gICAgICBmaWxlUGF0Y2gudXBkYXRlTWFya2VycyhtYXJrZXJNYXApO1xuICAgICAgdGhpcy5maWxlUGF0Y2hlc0J5TWFya2VyLnNldChmaWxlUGF0Y2guZ2V0TWFya2VyKCksIGZpbGVQYXRjaCk7XG5cbiAgICAgIGZvciAoY29uc3QgaHVuayBvZiBmaWxlUGF0Y2guZ2V0SHVua3MoKSkge1xuICAgICAgICB0aGlzLmh1bmtzQnlNYXJrZXIuc2V0KGh1bmsuZ2V0TWFya2VyKCksIGh1bmspO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMucGF0Y2hCdWZmZXIgPSBuZXh0UGF0Y2hCdWZmZXI7XG4gIH1cblxuICAvKlxuICAgKiBFZmZpY2llbnRseSBsb2NhdGUgdGhlIEZpbGVQYXRjaCBpbnN0YW5jZXMgdGhhdCBjb250YWluIGF0IGxlYXN0IG9uZSByb3cgZnJvbSBhIFNldC5cbiAgICovXG4gIGdldEZpbGVQYXRjaGVzQ29udGFpbmluZyhyb3dTZXQpIHtcbiAgICBjb25zdCBzb3J0ZWRSb3dTZXQgPSBBcnJheS5mcm9tKHJvd1NldCk7XG4gICAgc29ydGVkUm93U2V0LnNvcnQoKGEsIGIpID0+IGEgLSBiKTtcblxuICAgIGNvbnN0IGZpbGVQYXRjaGVzID0gW107XG4gICAgbGV0IGxhc3RGaWxlUGF0Y2ggPSBudWxsO1xuICAgIGZvciAoY29uc3Qgcm93IG9mIHNvcnRlZFJvd1NldCkge1xuICAgICAgLy8gQmVjYXVzZSB0aGUgcm93cyBhcmUgc29ydGVkLCBjb25zZWN1dGl2ZSByb3dzIHdpbGwgYWxtb3N0IGNlcnRhaW5seSBiZWxvbmcgdG8gdGhlIHNhbWUgcGF0Y2gsIHNvIHdlIGNhbiBzYXZlXG4gICAgICAvLyBtYW55IGF2b2lkYWJsZSBtYXJrZXIgaW5kZXggbG9va3VwcyBieSBjb21wYXJpbmcgd2l0aCB0aGUgbGFzdC5cbiAgICAgIGlmIChsYXN0RmlsZVBhdGNoICYmIGxhc3RGaWxlUGF0Y2guY29udGFpbnNSb3cocm93KSkge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgbGFzdEZpbGVQYXRjaCA9IHRoaXMuZ2V0RmlsZVBhdGNoQXQocm93KTtcbiAgICAgIGZpbGVQYXRjaGVzLnB1c2gobGFzdEZpbGVQYXRjaCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGZpbGVQYXRjaGVzO1xuICB9XG5cbiAgYW55UHJlc2VudCgpIHtcbiAgICByZXR1cm4gdGhpcy5wYXRjaEJ1ZmZlciAhPT0gbnVsbCAmJiB0aGlzLmZpbGVQYXRjaGVzLnNvbWUoZnAgPT4gZnAuaXNQcmVzZW50KCkpO1xuICB9XG5cbiAgZGlkQW55Q2hhbmdlRXhlY3V0YWJsZU1vZGUoKSB7XG4gICAgZm9yIChjb25zdCBmaWxlUGF0Y2ggb2YgdGhpcy5nZXRGaWxlUGF0Y2hlcygpKSB7XG4gICAgICBpZiAoZmlsZVBhdGNoLmRpZENoYW5nZUV4ZWN1dGFibGVNb2RlKCkpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGFueUhhdmVUeXBlY2hhbmdlKCkge1xuICAgIHJldHVybiB0aGlzLmdldEZpbGVQYXRjaGVzKCkuc29tZShmcCA9PiBmcC5oYXNUeXBlY2hhbmdlKCkpO1xuICB9XG5cbiAgZ2V0TWF4TGluZU51bWJlcldpZHRoKCkge1xuICAgIHJldHVybiB0aGlzLmdldEZpbGVQYXRjaGVzKCkucmVkdWNlKChtYXhXaWR0aCwgZmlsZVBhdGNoKSA9PiB7XG4gICAgICBjb25zdCB3aWR0aCA9IGZpbGVQYXRjaC5nZXRNYXhMaW5lTnVtYmVyV2lkdGgoKTtcbiAgICAgIHJldHVybiBtYXhXaWR0aCA+PSB3aWR0aCA/IG1heFdpZHRoIDogd2lkdGg7XG4gICAgfSwgMCk7XG4gIH1cblxuICBzcGFuc011bHRpcGxlRmlsZXMocm93cykge1xuICAgIGxldCBsYXN0RmlsZVBhdGNoID0gbnVsbDtcbiAgICBmb3IgKGNvbnN0IHJvdyBvZiByb3dzKSB7XG4gICAgICBpZiAobGFzdEZpbGVQYXRjaCkge1xuICAgICAgICBpZiAobGFzdEZpbGVQYXRjaC5jb250YWluc1Jvdyhyb3cpKSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGxhc3RGaWxlUGF0Y2ggPSB0aGlzLmdldEZpbGVQYXRjaEF0KHJvdyk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGNvbGxhcHNlRmlsZVBhdGNoKGZpbGVQYXRjaCkge1xuICAgIGNvbnN0IGluZGV4ID0gdGhpcy5maWxlUGF0Y2hlcy5pbmRleE9mKGZpbGVQYXRjaCk7XG5cbiAgICB0aGlzLmZpbGVQYXRjaGVzQnlNYXJrZXIuZGVsZXRlKGZpbGVQYXRjaC5nZXRNYXJrZXIoKSk7XG4gICAgZm9yIChjb25zdCBodW5rIG9mIGZpbGVQYXRjaC5nZXRIdW5rcygpKSB7XG4gICAgICB0aGlzLmh1bmtzQnlNYXJrZXIuZGVsZXRlKGh1bmsuZ2V0TWFya2VyKCkpO1xuICAgIH1cblxuICAgIGNvbnN0IGJlZm9yZSA9IHRoaXMuZ2V0TWFya2Vyc0JlZm9yZShpbmRleCk7XG4gICAgY29uc3QgYWZ0ZXIgPSB0aGlzLmdldE1hcmtlcnNBZnRlcihpbmRleCk7XG5cbiAgICBmaWxlUGF0Y2gudHJpZ2dlckNvbGxhcHNlSW4odGhpcy5wYXRjaEJ1ZmZlciwge2JlZm9yZSwgYWZ0ZXJ9KTtcblxuICAgIHRoaXMuZmlsZVBhdGNoZXNCeU1hcmtlci5zZXQoZmlsZVBhdGNoLmdldE1hcmtlcigpLCBmaWxlUGF0Y2gpO1xuXG4gICAgLy8gVGhpcyBodW5rIGNvbGxlY3Rpb24gc2hvdWxkIGJlIGVtcHR5LCBidXQgbGV0J3MgaXRlcmF0ZSBhbnl3YXkganVzdCBpbiBjYXNlIGZpbGVQYXRjaCB3YXMgYWxyZWFkeSBjb2xsYXBzZWRcbiAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgIGZvciAoY29uc3QgaHVuayBvZiBmaWxlUGF0Y2guZ2V0SHVua3MoKSkge1xuICAgICAgdGhpcy5odW5rc0J5TWFya2VyLnNldChodW5rLmdldE1hcmtlcigpLCBodW5rKTtcbiAgICB9XG4gIH1cblxuICBleHBhbmRGaWxlUGF0Y2goZmlsZVBhdGNoKSB7XG4gICAgY29uc3QgaW5kZXggPSB0aGlzLmZpbGVQYXRjaGVzLmluZGV4T2YoZmlsZVBhdGNoKTtcblxuICAgIHRoaXMuZmlsZVBhdGNoZXNCeU1hcmtlci5kZWxldGUoZmlsZVBhdGNoLmdldE1hcmtlcigpKTtcbiAgICBmb3IgKGNvbnN0IGh1bmsgb2YgZmlsZVBhdGNoLmdldEh1bmtzKCkpIHtcbiAgICAgIHRoaXMuaHVua3NCeU1hcmtlci5kZWxldGUoaHVuay5nZXRNYXJrZXIoKSk7XG4gICAgfVxuXG4gICAgY29uc3QgYmVmb3JlID0gdGhpcy5nZXRNYXJrZXJzQmVmb3JlKGluZGV4KTtcbiAgICBjb25zdCBhZnRlciA9IHRoaXMuZ2V0TWFya2Vyc0FmdGVyKGluZGV4KTtcblxuICAgIGZpbGVQYXRjaC50cmlnZ2VyRXhwYW5kSW4odGhpcy5wYXRjaEJ1ZmZlciwge2JlZm9yZSwgYWZ0ZXJ9KTtcblxuICAgIHRoaXMuZmlsZVBhdGNoZXNCeU1hcmtlci5zZXQoZmlsZVBhdGNoLmdldE1hcmtlcigpLCBmaWxlUGF0Y2gpO1xuICAgIGZvciAoY29uc3QgaHVuayBvZiBmaWxlUGF0Y2guZ2V0SHVua3MoKSkge1xuICAgICAgdGhpcy5odW5rc0J5TWFya2VyLnNldChodW5rLmdldE1hcmtlcigpLCBodW5rKTtcbiAgICB9XG5cbiAgICAvLyBpZiB0aGUgcGF0Y2ggd2FzIGluaXRpYWxseSBjb2xsYXBzZWQsIHdlIG5lZWQgdG8gY2FsY3VsYXRlXG4gICAgLy8gdGhlIGRpZmZSb3dPZmZzZXRJbmRpY2VzIHRvIGNhbGN1bGF0ZSBjb21tZW50IHBvc2l0aW9uLlxuICAgIGlmICh0aGlzLmlzRGlmZlJvd09mZnNldEluZGV4RW1wdHkoZmlsZVBhdGNoLmdldFBhdGgoKSkpIHtcbiAgICAgIHRoaXMucG9wdWxhdGVEaWZmUm93T2Zmc2V0SW5kaWNlcyhmaWxlUGF0Y2gpO1xuICAgIH1cbiAgfVxuXG4gIGdldE1hcmtlcnNCZWZvcmUoZmlsZVBhdGNoSW5kZXgpIHtcbiAgICBjb25zdCBiZWZvcmUgPSBbXTtcbiAgICBsZXQgYmVmb3JlSW5kZXggPSBmaWxlUGF0Y2hJbmRleCAtIDE7XG4gICAgd2hpbGUgKGJlZm9yZUluZGV4ID49IDApIHtcbiAgICAgIGNvbnN0IGJlZm9yZUZpbGVQYXRjaCA9IHRoaXMuZmlsZVBhdGNoZXNbYmVmb3JlSW5kZXhdO1xuICAgICAgYmVmb3JlLnB1c2goLi4uYmVmb3JlRmlsZVBhdGNoLmdldEVuZGluZ01hcmtlcnMoKSk7XG5cbiAgICAgIGlmICghYmVmb3JlRmlsZVBhdGNoLmdldE1hcmtlcigpLmdldFJhbmdlKCkuaXNFbXB0eSgpKSB7XG4gICAgICAgIGJyZWFrO1xuICAgICAgfVxuICAgICAgYmVmb3JlSW5kZXgtLTtcbiAgICB9XG4gICAgcmV0dXJuIGJlZm9yZTtcbiAgfVxuXG4gIGdldE1hcmtlcnNBZnRlcihmaWxlUGF0Y2hJbmRleCkge1xuICAgIGNvbnN0IGFmdGVyID0gW107XG4gICAgbGV0IGFmdGVySW5kZXggPSBmaWxlUGF0Y2hJbmRleCArIDE7XG4gICAgd2hpbGUgKGFmdGVySW5kZXggPCB0aGlzLmZpbGVQYXRjaGVzLmxlbmd0aCkge1xuICAgICAgY29uc3QgYWZ0ZXJGaWxlUGF0Y2ggPSB0aGlzLmZpbGVQYXRjaGVzW2FmdGVySW5kZXhdO1xuICAgICAgYWZ0ZXIucHVzaCguLi5hZnRlckZpbGVQYXRjaC5nZXRTdGFydGluZ01hcmtlcnMoKSk7XG5cbiAgICAgIGlmICghYWZ0ZXJGaWxlUGF0Y2guZ2V0TWFya2VyKCkuZ2V0UmFuZ2UoKS5pc0VtcHR5KCkpIHtcbiAgICAgICAgYnJlYWs7XG4gICAgICB9XG4gICAgICBhZnRlckluZGV4Kys7XG4gICAgfVxuICAgIHJldHVybiBhZnRlcjtcbiAgfVxuXG4gIGlzUGF0Y2hWaXNpYmxlID0gZmlsZVBhdGNoUGF0aCA9PiB7XG4gICAgY29uc3QgcGF0Y2ggPSB0aGlzLmZpbGVQYXRjaGVzQnlQYXRoLmdldChmaWxlUGF0Y2hQYXRoKTtcbiAgICBpZiAoIXBhdGNoKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICAgIHJldHVybiBwYXRjaC5nZXRSZW5kZXJTdGF0dXMoKS5pc1Zpc2libGUoKTtcbiAgfVxuXG4gIGdldEJ1ZmZlclJvd0ZvckRpZmZQb3NpdGlvbiA9IChmaWxlTmFtZSwgZGlmZlJvdykgPT4ge1xuICAgIGNvbnN0IHtzdGFydEJ1ZmZlclJvdywgaW5kZXh9ID0gdGhpcy5kaWZmUm93T2Zmc2V0SW5kaWNlcy5nZXQoZmlsZU5hbWUpO1xuICAgIGNvbnN0IHtvZmZzZXR9ID0gaW5kZXgubG93ZXJCb3VuZCh7ZGlmZlJvd30pLmRhdGEoKTtcbiAgICByZXR1cm4gc3RhcnRCdWZmZXJSb3cgKyBkaWZmUm93IC0gb2Zmc2V0O1xuICB9XG5cbiAgLypcbiAgICogQ29uc3RydWN0IGFuIGFwcGx5LWFibGUgcGF0Y2ggU3RyaW5nLlxuICAgKi9cbiAgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMuZmlsZVBhdGNoZXMubWFwKGZwID0+IGZwLnRvU3RyaW5nSW4odGhpcy5nZXRCdWZmZXIoKSkpLmpvaW4oJycpICsgJ1xcbic7XG4gIH1cblxuICAvKlxuICAgKiBDb25zdHJ1Y3QgYSBzdHJpbmcgb2YgZGlhZ25vc3RpYyBpbmZvcm1hdGlvbiB1c2VmdWwgZm9yIGRlYnVnZ2luZy5cbiAgICovXG4gIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gIGluc3BlY3QoKSB7XG4gICAgbGV0IGluc3BlY3RTdHJpbmcgPSAnKE11bHRpRmlsZVBhdGNoJztcbiAgICBpbnNwZWN0U3RyaW5nICs9IGAgZmlsZVBhdGNoZXNCeU1hcmtlcj0oJHtBcnJheS5mcm9tKHRoaXMuZmlsZVBhdGNoZXNCeU1hcmtlci5rZXlzKCksIG0gPT4gbS5pZCkuam9pbignLCAnKX0pYDtcbiAgICBpbnNwZWN0U3RyaW5nICs9IGAgaHVua3NCeU1hcmtlcj0oJHtBcnJheS5mcm9tKHRoaXMuaHVua3NCeU1hcmtlci5rZXlzKCksIG0gPT4gbS5pZCkuam9pbignLCAnKX0pXFxuYDtcbiAgICBmb3IgKGNvbnN0IGZpbGVQYXRjaCBvZiB0aGlzLmZpbGVQYXRjaGVzKSB7XG4gICAgICBpbnNwZWN0U3RyaW5nICs9IGZpbGVQYXRjaC5pbnNwZWN0KHtpbmRlbnQ6IDJ9KTtcbiAgICB9XG4gICAgaW5zcGVjdFN0cmluZyArPSAnKVxcbic7XG4gICAgcmV0dXJuIGluc3BlY3RTdHJpbmc7XG4gIH1cblxuICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICBpc0VxdWFsKG90aGVyKSB7XG4gICAgcmV0dXJuIHRoaXMudG9TdHJpbmcoKSA9PT0gb3RoZXIudG9TdHJpbmcoKTtcbiAgfVxufVxuIl19