(function() {
  var CompositeDisposable, Range, SelectNext, _, ref;

  _ = require('underscore-plus');

  ref = require('atom'), CompositeDisposable = ref.CompositeDisposable, Range = ref.Range;

  module.exports = SelectNext = (function() {
    SelectNext.prototype.selectionRanges = null;

    function SelectNext(editor) {
      this.editor = editor;
      this.selectionRanges = [];
    }

    SelectNext.prototype.findAndSelectNext = function() {
      if (this.editor.getLastSelection().isEmpty()) {
        return this.selectWord();
      } else {
        return this.selectNextOccurrence();
      }
    };

    SelectNext.prototype.findAndSelectAll = function() {
      if (this.editor.getLastSelection().isEmpty()) {
        this.selectWord();
      }
      return this.selectAllOccurrences();
    };

    SelectNext.prototype.undoLastSelection = function() {
      this.updateSavedSelections();
      if (this.selectionRanges.length < 1) {
        return;
      }
      if (this.selectionRanges.length > 1) {
        this.selectionRanges.pop();
        this.editor.setSelectedBufferRanges(this.selectionRanges);
      } else {
        this.editor.clearSelections();
      }
      return this.editor.scrollToCursorPosition();
    };

    SelectNext.prototype.skipCurrentSelection = function() {
      var lastSelection;
      this.updateSavedSelections();
      if (this.selectionRanges.length < 1) {
        return;
      }
      if (this.selectionRanges.length > 1) {
        lastSelection = this.selectionRanges.pop();
        this.editor.setSelectedBufferRanges(this.selectionRanges);
        return this.selectNextOccurrence({
          start: lastSelection.end
        });
      } else {
        this.selectNextOccurrence();
        this.selectionRanges.shift();
        if (this.selectionRanges.length < 1) {
          return;
        }
        return this.editor.setSelectedBufferRanges(this.selectionRanges);
      }
    };

    SelectNext.prototype.selectWord = function() {
      var clearWordSelected, disposables, lastSelection;
      this.editor.selectWordsContainingCursors();
      lastSelection = this.editor.getLastSelection();
      if (this.wordSelected = this.isWordSelected(lastSelection)) {
        disposables = new CompositeDisposable;
        clearWordSelected = (function(_this) {
          return function() {
            _this.wordSelected = null;
            return disposables.dispose();
          };
        })(this);
        disposables.add(lastSelection.onDidChangeRange(clearWordSelected));
        return disposables.add(lastSelection.onDidDestroy(clearWordSelected));
      }
    };

    SelectNext.prototype.selectAllOccurrences = function() {
      var range;
      range = [[0, 0], this.editor.getEofBufferPosition()];
      return this.scanForNextOccurrence(range, (function(_this) {
        return function(arg) {
          var range, stop;
          range = arg.range, stop = arg.stop;
          return _this.addSelection(range);
        };
      })(this));
    };

    SelectNext.prototype.selectNextOccurrence = function(options) {
      var range, ref1, startingRange;
      if (options == null) {
        options = {};
      }
      startingRange = (ref1 = options.start) != null ? ref1 : this.editor.getSelectedBufferRange().end;
      range = this.findNextOccurrence([startingRange, this.editor.getEofBufferPosition()]);
      if (range == null) {
        range = this.findNextOccurrence([[0, 0], this.editor.getSelections()[0].getBufferRange().start]);
      }
      if (range != null) {
        return this.addSelection(range);
      }
    };

    SelectNext.prototype.findNextOccurrence = function(scanRange) {
      var foundRange;
      foundRange = null;
      this.scanForNextOccurrence(scanRange, function(arg) {
        var range, stop;
        range = arg.range, stop = arg.stop;
        foundRange = range;
        return stop();
      });
      return foundRange;
    };

    SelectNext.prototype.addSelection = function(range) {
      var reversed, selection;
      reversed = this.editor.getLastSelection().isReversed();
      selection = this.editor.addSelectionForBufferRange(range, {
        reversed: reversed
      });
      return this.updateSavedSelections(selection);
    };

    SelectNext.prototype.scanForNextOccurrence = function(range, callback) {
      var nonWordCharacters, selection, text;
      selection = this.editor.getLastSelection();
      text = _.escapeRegExp(selection.getText());
      if (this.wordSelected) {
        nonWordCharacters = atom.config.get('editor.nonWordCharacters');
        text = "(^|[ \t" + (_.escapeRegExp(nonWordCharacters)) + "]+)" + text + "(?=$|[\\s" + (_.escapeRegExp(nonWordCharacters)) + "]+)";
      }
      return this.editor.scanInBufferRange(new RegExp(text, 'g'), range, function(result) {
        var prefix;
        if (prefix = result.match[1]) {
          result.range = result.range.translate([0, prefix.length], [0, 0]);
        }
        return callback(result);
      });
    };

    SelectNext.prototype.updateSavedSelections = function(selection) {
      var i, len, results, s, selectionRange, selections;
      if (selection == null) {
        selection = null;
      }
      selections = this.editor.getSelections();
      if (selections.length < 3) {
        this.selectionRanges = [];
      }
      if (this.selectionRanges.length === 0) {
        results = [];
        for (i = 0, len = selections.length; i < len; i++) {
          s = selections[i];
          results.push(this.selectionRanges.push(s.getBufferRange()));
        }
        return results;
      } else if (selection) {
        selectionRange = selection.getBufferRange();
        if (this.selectionRanges.some(function(existingRange) {
          return existingRange.isEqual(selectionRange);
        })) {
          return;
        }
        return this.selectionRanges.push(selectionRange);
      }
    };

    SelectNext.prototype.isNonWordCharacter = function(character) {
      var nonWordCharacters;
      nonWordCharacters = atom.config.get('editor.nonWordCharacters');
      return new RegExp("[ \t" + (_.escapeRegExp(nonWordCharacters)) + "]").test(character);
    };

    SelectNext.prototype.isNonWordCharacterToTheLeft = function(selection) {
      var range, selectionStart;
      selectionStart = selection.getBufferRange().start;
      range = Range.fromPointWithDelta(selectionStart, 0, -1);
      return this.isNonWordCharacter(this.editor.getTextInBufferRange(range));
    };

    SelectNext.prototype.isNonWordCharacterToTheRight = function(selection) {
      var range, selectionEnd;
      selectionEnd = selection.getBufferRange().end;
      range = Range.fromPointWithDelta(selectionEnd, 0, 1);
      return this.isNonWordCharacter(this.editor.getTextInBufferRange(range));
    };

    SelectNext.prototype.isWordSelected = function(selection) {
      var containsOnlyWordCharacters, lineRange, nonWordCharacterToTheLeft, nonWordCharacterToTheRight, selectionRange;
      if (selection.getBufferRange().isSingleLine()) {
        selectionRange = selection.getBufferRange();
        lineRange = this.editor.bufferRangeForBufferRow(selectionRange.start.row);
        nonWordCharacterToTheLeft = _.isEqual(selectionRange.start, lineRange.start) || this.isNonWordCharacterToTheLeft(selection);
        nonWordCharacterToTheRight = _.isEqual(selectionRange.end, lineRange.end) || this.isNonWordCharacterToTheRight(selection);
        containsOnlyWordCharacters = !this.isNonWordCharacter(selection.getText());
        return nonWordCharacterToTheLeft && nonWordCharacterToTheRight && containsOnlyWordCharacters;
      } else {
        return false;
      }
    };

    return SelectNext;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
