(function() {
  var CompositeDisposable, GitRepositoryAsync, GitView, _, ref;

  _ = require("underscore-plus");

  ref = require("atom"), CompositeDisposable = ref.CompositeDisposable, GitRepositoryAsync = ref.GitRepositoryAsync;

  module.exports = GitView = (function() {
    function GitView() {
      this.element = document.createElement('status-bar-git');
      this.element.classList.add('git-view');
      this.createBranchArea();
      this.createCommitsArea();
      this.createStatusArea();
      this.activeItemSubscription = atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          return _this.subscribeToActiveItem();
        };
      })(this));
      this.projectPathSubscription = atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.subscribeToRepositories();
        };
      })(this));
      this.subscribeToRepositories();
      this.subscribeToActiveItem();
    }

    GitView.prototype.createBranchArea = function() {
      var branchIcon;
      this.branchArea = document.createElement('div');
      this.branchArea.classList.add('git-branch', 'inline-block');
      this.element.appendChild(this.branchArea);
      this.element.branchArea = this.branchArea;
      branchIcon = document.createElement('span');
      branchIcon.classList.add('icon', 'icon-git-branch');
      this.branchArea.appendChild(branchIcon);
      this.branchLabel = document.createElement('span');
      this.branchLabel.classList.add('branch-label');
      this.branchArea.appendChild(this.branchLabel);
      return this.element.branchLabel = this.branchLabel;
    };

    GitView.prototype.createCommitsArea = function() {
      this.commitsArea = document.createElement('div');
      this.commitsArea.classList.add('git-commits', 'inline-block');
      this.element.appendChild(this.commitsArea);
      this.commitsAhead = document.createElement('span');
      this.commitsAhead.classList.add('icon', 'icon-arrow-up', 'commits-ahead-label');
      this.commitsArea.appendChild(this.commitsAhead);
      this.commitsBehind = document.createElement('span');
      this.commitsBehind.classList.add('icon', 'icon-arrow-down', 'commits-behind-label');
      return this.commitsArea.appendChild(this.commitsBehind);
    };

    GitView.prototype.createStatusArea = function() {
      this.gitStatus = document.createElement('div');
      this.gitStatus.classList.add('git-status', 'inline-block');
      this.element.appendChild(this.gitStatus);
      this.gitStatusIcon = document.createElement('span');
      this.gitStatusIcon.classList.add('icon');
      this.gitStatus.appendChild(this.gitStatusIcon);
      return this.element.gitStatusIcon = this.gitStatusIcon;
    };

    GitView.prototype.subscribeToActiveItem = function() {
      var activeItem, ref1;
      activeItem = this.getActiveItem();
      if ((ref1 = this.savedSubscription) != null) {
        ref1.dispose();
      }
      this.savedSubscription = activeItem != null ? typeof activeItem.onDidSave === "function" ? activeItem.onDidSave((function(_this) {
        return function() {
          return _this.update();
        };
      })(this)) : void 0 : void 0;
      return this.update();
    };

    GitView.prototype.subscribeToRepositories = function() {
      var i, len, ref1, ref2, repo, results;
      if ((ref1 = this.repositorySubscriptions) != null) {
        ref1.dispose();
      }
      this.repositorySubscriptions = new CompositeDisposable;
      ref2 = atom.project.getRepositories();
      results = [];
      for (i = 0, len = ref2.length; i < len; i++) {
        repo = ref2[i];
        if (!(repo != null)) {
          continue;
        }
        this.repositorySubscriptions.add(repo.onDidChangeStatus((function(_this) {
          return function(arg) {
            var path, status;
            path = arg.path, status = arg.status;
            if (path === _this.getActiveItemPath()) {
              return _this.update();
            }
          };
        })(this)));
        results.push(this.repositorySubscriptions.add(repo.onDidChangeStatuses((function(_this) {
          return function() {
            return _this.update();
          };
        })(this))));
      }
      return results;
    };

    GitView.prototype.destroy = function() {
      var ref1, ref2, ref3, ref4, ref5, ref6, ref7, ref8;
      if ((ref1 = this.activeItemSubscription) != null) {
        ref1.dispose();
      }
      if ((ref2 = this.projectPathSubscription) != null) {
        ref2.dispose();
      }
      if ((ref3 = this.savedSubscription) != null) {
        ref3.dispose();
      }
      if ((ref4 = this.repositorySubscriptions) != null) {
        ref4.dispose();
      }
      if ((ref5 = this.branchTooltipDisposable) != null) {
        ref5.dispose();
      }
      if ((ref6 = this.commitsAheadTooltipDisposable) != null) {
        ref6.dispose();
      }
      if ((ref7 = this.commitsBehindTooltipDisposable) != null) {
        ref7.dispose();
      }
      return (ref8 = this.statusTooltipDisposable) != null ? ref8.dispose() : void 0;
    };

    GitView.prototype.getActiveItemPath = function() {
      var ref1;
      return (ref1 = this.getActiveItem()) != null ? typeof ref1.getPath === "function" ? ref1.getPath() : void 0 : void 0;
    };

    GitView.prototype.getRepositoryForActiveItem = function() {
      var i, len, ref1, repo, rootDir, rootDirIndex;
      rootDir = atom.project.relativizePath(this.getActiveItemPath())[0];
      rootDirIndex = atom.project.getPaths().indexOf(rootDir);
      if (rootDirIndex >= 0) {
        return atom.project.getRepositories()[rootDirIndex];
      } else {
        ref1 = atom.project.getRepositories();
        for (i = 0, len = ref1.length; i < len; i++) {
          repo = ref1[i];
          if (repo) {
            return repo;
          }
        }
      }
    };

    GitView.prototype.getActiveItem = function() {
      return atom.workspace.getCenter().getActivePaneItem();
    };

    GitView.prototype.update = function() {
      var repo;
      repo = this.getRepositoryForActiveItem();
      this.updateBranchText(repo);
      this.updateAheadBehindCount(repo);
      return this.updateStatusText(repo);
    };

    GitView.prototype.updateBranchText = function(repo) {
      var head, ref1;
      if (this.showGitInformation(repo)) {
        head = repo.getShortHead(this.getActiveItemPath());
        this.branchLabel.textContent = head;
        if (head) {
          this.branchArea.style.display = '';
        }
        if ((ref1 = this.branchTooltipDisposable) != null) {
          ref1.dispose();
        }
        return this.branchTooltipDisposable = atom.tooltips.add(this.branchArea, {
          title: "On branch " + head
        });
      } else {
        return this.branchArea.style.display = 'none';
      }
    };

    GitView.prototype.showGitInformation = function(repo) {
      var itemPath;
      if (repo == null) {
        return false;
      }
      if (itemPath = this.getActiveItemPath()) {
        return atom.project.contains(itemPath);
      } else {
        return this.getActiveItem() == null;
      }
    };

    GitView.prototype.updateAheadBehindCount = function(repo) {
      var ahead, behind, itemPath, ref1, ref2, ref3;
      if (!this.showGitInformation(repo)) {
        this.commitsArea.style.display = 'none';
        return;
      }
      itemPath = this.getActiveItemPath();
      ref1 = repo.getCachedUpstreamAheadBehindCount(itemPath), ahead = ref1.ahead, behind = ref1.behind;
      if (ahead > 0) {
        this.commitsAhead.textContent = ahead;
        this.commitsAhead.style.display = '';
        if ((ref2 = this.commitsAheadTooltipDisposable) != null) {
          ref2.dispose();
        }
        this.commitsAheadTooltipDisposable = atom.tooltips.add(this.commitsAhead, {
          title: (_.pluralize(ahead, 'commit')) + " ahead of upstream"
        });
      } else {
        this.commitsAhead.style.display = 'none';
      }
      if (behind > 0) {
        this.commitsBehind.textContent = behind;
        this.commitsBehind.style.display = '';
        if ((ref3 = this.commitsBehindTooltipDisposable) != null) {
          ref3.dispose();
        }
        this.commitsBehindTooltipDisposable = atom.tooltips.add(this.commitsBehind, {
          title: (_.pluralize(behind, 'commit')) + " behind upstream"
        });
      } else {
        this.commitsBehind.style.display = 'none';
      }
      if (ahead > 0 || behind > 0) {
        return this.commitsArea.style.display = '';
      } else {
        return this.commitsArea.style.display = 'none';
      }
    };

    GitView.prototype.clearStatus = function() {
      return this.gitStatusIcon.classList.remove('icon-diff-modified', 'status-modified', 'icon-diff-added', 'status-added', 'icon-diff-ignored', 'status-ignored');
    };

    GitView.prototype.updateAsNewFile = function() {
      var textEditor;
      this.clearStatus();
      this.gitStatusIcon.classList.add('icon-diff-added', 'status-added');
      if (textEditor = atom.workspace.getActiveTextEditor()) {
        this.gitStatusIcon.textContent = "+" + (textEditor.getLineCount());
        this.updateTooltipText((_.pluralize(textEditor.getLineCount(), 'line')) + " in this new file not yet committed");
      } else {
        this.gitStatusIcon.textContent = '';
        this.updateTooltipText();
      }
      return this.gitStatus.style.display = '';
    };

    GitView.prototype.updateAsModifiedFile = function(repo, path) {
      var stats;
      stats = repo.getDiffStats(path);
      this.clearStatus();
      this.gitStatusIcon.classList.add('icon-diff-modified', 'status-modified');
      if (stats.added && stats.deleted) {
        this.gitStatusIcon.textContent = "+" + stats.added + ", -" + stats.deleted;
        this.updateTooltipText((_.pluralize(stats.added, 'line')) + " added and " + (_.pluralize(stats.deleted, 'line')) + " deleted in this file not yet committed");
      } else if (stats.added) {
        this.gitStatusIcon.textContent = "+" + stats.added;
        this.updateTooltipText((_.pluralize(stats.added, 'line')) + " added to this file not yet committed");
      } else if (stats.deleted) {
        this.gitStatusIcon.textContent = "-" + stats.deleted;
        this.updateTooltipText((_.pluralize(stats.deleted, 'line')) + " deleted from this file not yet committed");
      } else {
        this.gitStatusIcon.textContent = '';
        this.updateTooltipText();
      }
      return this.gitStatus.style.display = '';
    };

    GitView.prototype.updateAsIgnoredFile = function() {
      this.clearStatus();
      this.gitStatusIcon.classList.add('icon-diff-ignored', 'status-ignored');
      this.gitStatusIcon.textContent = '';
      this.gitStatus.style.display = '';
      return this.updateTooltipText("File is ignored by git");
    };

    GitView.prototype.updateTooltipText = function(text) {
      var ref1;
      if ((ref1 = this.statusTooltipDisposable) != null) {
        ref1.dispose();
      }
      if (text) {
        return this.statusTooltipDisposable = atom.tooltips.add(this.gitStatusIcon, {
          title: text
        });
      }
    };

    GitView.prototype.updateStatusText = function(repo) {
      var hideStatus, itemPath, ref1, status;
      hideStatus = (function(_this) {
        return function() {
          _this.clearStatus();
          return _this.gitStatus.style.display = 'none';
        };
      })(this);
      itemPath = this.getActiveItemPath();
      if (this.showGitInformation(repo) && (itemPath != null)) {
        status = (ref1 = repo.getCachedPathStatus(itemPath)) != null ? ref1 : 0;
        if (repo.isStatusNew(status)) {
          return this.updateAsNewFile();
        }
        if (repo.isStatusModified(status)) {
          return this.updateAsModifiedFile(repo, itemPath);
        }
        if (repo.isPathIgnored(itemPath)) {
          return this.updateAsIgnoredFile();
        } else {
          return hideStatus();
        }
      } else {
        return hideStatus();
      }
    };

    return GitView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
