"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _util = require("util");

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const sortOrders = {
  'by key': (a, b) => a.key.localeCompare(b.key),
  'oldest first': (a, b) => b.age - a.age,
  'newest first': (a, b) => a.age - b.age,
  'most hits': (a, b) => b.hits - a.hits,
  'fewest hits': (a, b) => a.hits - b.hits
};

class GitCacheView extends _react.default.Component {
  static buildURI() {
    return this.uriPattern;
  }

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'fetchRepositoryData', 'fetchCacheData', 'renderCache', 'didSelectItem', 'clearCache');
    this.state = {
      order: 'by key'
    };
  }

  getURI() {
    return 'atom-github://debug/cache';
  }

  getTitle() {
    return 'GitHub Package Cache View';
  }

  serialize() {
    return null;
  }

  fetchRepositoryData(repository) {
    return repository.getCache();
  }

  fetchCacheData(cache) {
    const cached = {};
    const promises = [];
    const now = performance.now();

    for (const [key, value] of cache) {
      cached[key] = {
        hits: value.hits,
        age: now - value.createdAt
      };
      promises.push(value.promise.then(payload => (0, _util.inspect)(payload, {
        depth: 3,
        breakLength: 30
      }), err => `${err.message}\n${err.stack}`).then(resolved => {
        cached[key].value = resolved;
      }));
    }

    return Promise.all(promises).then(() => cached);
  }

  render() {
    return _react.default.createElement(_observeModel.default, {
      model: this.props.repository,
      fetchData: this.fetchRepositoryData
    }, cache => _react.default.createElement(_observeModel.default, {
      model: cache,
      fetchData: this.fetchCacheData
    }, this.renderCache));
  }

  renderCache(contents) {
    const rows = Object.keys(contents || {}).map(key => {
      return {
        key,
        age: contents[key].age,
        hits: contents[key].hits,
        content: contents[key].value
      };
    });
    rows.sort(sortOrders[this.state.order]);
    const orders = Object.keys(sortOrders);
    return _react.default.createElement("div", {
      className: "github-CacheView"
    }, _react.default.createElement("header", null, _react.default.createElement("h1", null, "Cache contents"), _react.default.createElement("p", null, _react.default.createElement("span", {
      className: "badge"
    }, rows.length), " cached items")), _react.default.createElement("main", null, _react.default.createElement("p", {
      className: "github-CacheView-Controls"
    }, _react.default.createElement("span", {
      className: "github-CacheView-Order"
    }, "order", _react.default.createElement("select", {
      className: "input-select",
      onChange: this.didSelectItem,
      value: this.state.order
    }, orders.map(order => {
      return _react.default.createElement("option", {
        key: order,
        value: order
      }, order);
    }))), _react.default.createElement("span", {
      className: "github-CacheView-Clear"
    }, _react.default.createElement("button", {
      className: "btn icon icon-trashcan",
      onClick: this.clearCache
    }, "Clear"))), _react.default.createElement("table", null, _react.default.createElement("thead", null, _react.default.createElement("tr", null, _react.default.createElement("td", {
      className: "github-CacheView-Key"
    }, "key"), _react.default.createElement("td", {
      className: "github-CacheView-Age"
    }, "age"), _react.default.createElement("td", {
      className: "github-CacheView-Hits"
    }, "hits"), _react.default.createElement("td", {
      className: "github-CacheView-Content"
    }, "content"))), _react.default.createElement("tbody", null, rows.map(row => _react.default.createElement("tr", {
      key: row.key,
      className: "github-CacheView-Row"
    }, _react.default.createElement("td", {
      className: "github-CacheView-Key"
    }, _react.default.createElement("button", {
      className: "btn",
      onClick: () => this.didClickKey(row.key)
    }, row.key)), _react.default.createElement("td", {
      className: "github-CacheView-Age"
    }, this.formatAge(row.age)), _react.default.createElement("td", {
      className: "github-CacheView-Hits"
    }, row.hits), _react.default.createElement("td", {
      className: "github-CacheView-Content"
    }, _react.default.createElement("code", null, row.content))))))));
  }

  formatAge(ageMs) {
    let remaining = ageMs;
    const parts = [];

    if (remaining > 3600000) {
      const hours = Math.floor(remaining / 3600000);
      parts.push(`${hours}h`);
      remaining -= 3600000 * hours;
    }

    if (remaining > 60000) {
      const minutes = Math.floor(remaining / 60000);
      parts.push(`${minutes}m`);
      remaining -= 60000 * minutes;
    }

    if (remaining > 1000) {
      const seconds = Math.floor(remaining / 1000);
      parts.push(`${seconds}s`);
      remaining -= 1000 * seconds;
    }

    parts.push(`${Math.floor(remaining)}ms`);
    return parts.slice(parts.length - 2).join(' ');
  }

  didSelectItem(event) {
    this.setState({
      order: event.target.value
    });
  }

  didClickKey(key) {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.removePrimary(key);
  }

  clearCache() {
    const cache = this.props.repository.getCache();

    if (!cache) {
      return;
    }

    cache.clear();
  }

}

exports.default = GitCacheView;

_defineProperty(GitCacheView, "uriPattern", 'atom-github://debug/cache');

_defineProperty(GitCacheView, "propTypes", {
  repository: _propTypes.default.object.isRequired
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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