"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _state = _interopRequireDefault(require("./state"));

var _gitShellOutStrategy = require("../../git-shell-out-strategy");

var _workspaceChangeObserver = require("../workspace-change-observer");

var _patch = require("../patch");

var _discardHistory = _interopRequireDefault(require("../discard-history"));

var _branch = _interopRequireWildcard(require("../branch"));

var _author = _interopRequireDefault(require("../author"));

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remote = _interopRequireDefault(require("../remote"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _commit = _interopRequireDefault(require("../commit"));

var _operationStates = _interopRequireDefault(require("../operation-states"));

var _reporterProxy = require("../../reporter-proxy");

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state.default {
  constructor(repository, history) {
    super(repository);
    this.cache = new Cache();
    this.discardHistory = new _discardHistory.default(this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), {
      maxHistoryLength: 60
    });
    this.operationStates = new _operationStates.default({
      didUpdate: this.didUpdate.bind(this)
    });
    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();
    /* istanbul ignore else */

    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, {
    suppressUpdate
  } = {
    suppressUpdate: false
  }) {
    this.commitMessage = message;

    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      this.commitMessageTemplate = template;
    }

    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  acceptInvalidation(spec) {
    this.cache.invalidate(spec());
    this.didUpdate();
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();

    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(Keys.statusBundle);

        for (const k of Keys.filePatch.eachWithOpts({
          staged: false
        })) {
          keys.add(k);
        }

        continue;
      }

      const includes = (...segments) => fullPath.includes(_path.default.join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(Keys.stagedChanges);
        keys.add(Keys.filePatch.all);
        keys.add(Keys.index.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(Keys.remotes);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(Keys.remotes);
        keys.add(Keys.config.all);
        keys.add(Keys.statusBundle);
        continue;
      } // File change within the working directory


      const relativePath = _path.default.relative(this.workdir(), fullPath);

      keys.add(Keys.filePatch.oneWith(relativePath, {
        staged: false
      }));
      keys.add(Keys.statusBundle);
    }
    /* istanbul ignore else */


    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }

    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();

        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }

        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Staging and unstaging


  stageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, {
        index: true
      });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  } // Committing


  commit(message, options) {
    return this.invalidate(() => [...Keys.headOperationKeys(), ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.headDescription, Keys.branches, Keys.stagedChanges], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _objectSpread({}, options, {
        coAuthors: coAuthors.map(author => {
          return {
            email: author.getEmail(),
            name: author.getFullName()
          };
        })
      });
      await this.git().commit(message, opts); // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here

      const {
        unstagedFiles,
        mergeConflictFiles
      } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_objectSpread({}, unstagedFiles, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        package: 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  } // Merging


  merge(branchName) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, Keys.filePatch.all, Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...Keys.filePatch.eachWithFileOpts([filePath], [{
      staged: false
    }, {
      staged: true
    }]), Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  } // Checkout


  checkout(revision, options = {}) {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...paths.map(fileName => Keys.index.oneWith(fileName)), ...Keys.filePatch.eachWithFileOpts(paths, [{
      staged: true
    }])], () => this.git().checkoutFiles(paths, revision));
  } // Reset


  undoLastCommit() {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', {
          package: 'github'
        });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  } // Remote interactions


  fetch(branchName, options = {}) {
    return this.invalidate(() => [Keys.statusBundle, Keys.headDescription], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [Keys.statusBundle, Keys.headDescription];

      if (options.setUpstream) {
        keys.push(Keys.branches);
        keys.push(...Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    }, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  } // Configuration


  setConfig(setting, value, options) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options));
  }

  unsetConfig(setting) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  } // Direct blob interactions


  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  } // Discard history


  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    /* istanbul ignore else */

    if (snapshots) {
      await this.saveDiscardHistory();
    }

    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);

    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [Keys.statusBundle, ...paths.map(filePath => Keys.filePatch.oneWith(filePath, {
      staged: false
    }))], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path.default.join(this.workdir(), filePath);

        return _fsExtra.default.remove(absPath);
      }));
    });
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Index queries


  getStatusBundle() {
    return this.cache.getOrSet(Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({
    changedEntries,
    untrackedEntries,
    renamedEntries,
    unmergedEntries
  }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };
    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};
    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }

      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });
    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });
    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }

      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });
    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const {
        stagedStatus,
        unstagedStatus,
        filePath
      } = unmergedEntries[i];

      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({
            target: 'HEAD'
          });
        }

        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  async getStatusesForChangedFiles() {
    const {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    } = await this.getStatusBundle();
    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  getFilePatchForPath(filePath, options) {
    const opts = _objectSpread({
      staged: false,
      patchBuffer: null,
      builder: {},
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, {
      staged: opts.staged
    }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, {
        staged: opts.staged
      });
      const payload = opts.before();
      const patch = (0, _patch.buildFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  getStagedChangesPatch(options) {
    const opts = _objectSpread({
      builder: {},
      patchBuffer: null,
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.stagedChanges, async () => {
      const diffs = await this.git().getStagedChangesPatch();
      const payload = opts.before();
      const patch = (0, _patch.buildMultiFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  } // Commit access


  getLastCommit() {
    return this.cache.getOrSet(Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit.default.createUnborn() : new _commit.default(headCommit);
    });
  }

  getCommit(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), async () => {
      const [rawCommit] = await this.git().getCommits({
        max: 1,
        ref: sha,
        includePatch: true
      });
      const commit = new _commit.default(rawCommit);
      return commit;
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_objectSpread({
        ref: 'HEAD'
      }, options));
      return commits.map(commit => new _commit.default(commit));
    });
  }

  async isCommitPushed(sha) {
    const remoteBranchesWithCommit = await this.git().getBranchesWithCommit(sha, {
      showLocal: false,
      showRemote: true
    });
    const currentRemote = (await this.repository.getCurrentBranch()).getUpstream();
    return remoteBranchesWithCommit.includes(currentRemote.getFullRef());
  } // Author information


  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author.default(email, authorMap[email]));
    });
  } // Branches


  getBranches() {
    return this.cache.getOrSet(Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet.default();

      for (const payload of payloads) {
        let upstream = _branch.nullBranch;

        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch.default.createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch.default(payload.upstream.trackingRef);
        }

        let push = upstream;

        if (payload.push) {
          push = payload.push.remoteName ? _branch.default.createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch.default(payload.push.trackingRef);
        }

        branches.add(new _branch.default(payload.name, upstream, push, payload.head, {
          sha: payload.sha
        }));
      }

      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(Keys.headDescription, () => {
      return this.git().describeHead();
    });
  } // Merging and rebasing status


  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  } // Remotes


  getRemotes() {
    return this.cache.getOrSet(Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet.default(remotesInfo.map(({
        name,
        url
      }) => new _remote.default(name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [...Keys.config.eachWithSetting(`remote.${name}.url`), ...Keys.config.eachWithSetting(`remote.${name}.fetch`), Keys.remotes], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote.default(name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, {
    local
  } = {
    local: false
  }) {
    return this.cache.getOrSet(Keys.config.oneWith(option, {
      local
    }), () => {
      return this.git().getConfig(option, {
        local
      });
    });
  }

  directGetConfig(key, options) {
    return this.getConfig(key, options);
  } // Direct blob access


  getBlobContents(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  directGetBlobContents(sha) {
    return this.getBlobContents(sha);
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  } // Cache

  /* istanbul ignore next */


  getCache() {
    return this.cache;
  }

  invalidate(spec, body) {
    return body().then(result => {
      this.acceptInvalidation(spec);
      return result;
    }, err => {
      this.acceptInvalidation(spec);
      return Promise.reject(err);
    });
  }

}

exports.default = Present;

_state.default.register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);

    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();
    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });
    const groups = key.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);

      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }

      groupSet.add(key);
    }

    this.didUpdate();
    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage.delete(primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet.delete(key);
    this.didUpdate();
  }
  /* istanbul ignore next */


  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }
  /* istanbul ignore next */


  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }

}

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());
    const groups = this.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];

      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `CacheKey(${this.primary})`;
  }

}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `GroupKey(${this.group})`;
  }

}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),
  stagedChanges: new CacheKey('staged-changes'),
  filePatch: {
    _optKey: ({
      staged
    }) => {
      return staged ? 's' : 'u';
    },
    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);

      return new CacheKey(`file-patch:${optKey}:${fileName}`, ['file-patch', `file-patch:${optKey}`]);
    },
    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];

      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(Keys.filePatch.oneWith(fileNames[i], opts[j]));
        }
      }

      return keys;
    },
    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:${Keys.filePatch._optKey(opt)}`)),
    all: new GroupKey('file-patch')
  },
  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),
    all: new GroupKey('index')
  },
  lastCommit: new CacheKey('last-commit'),
  recentCommits: new CacheKey('recent-commits'),
  authors: new CacheKey('authors'),
  branches: new CacheKey('branches'),
  headDescription: new CacheKey('head-description'),
  remotes: new CacheKey('remotes'),
  config: {
    _optKey: options => options.local ? 'l' : '',
    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);

      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },
    eachWithSetting: setting => [Keys.config.oneWith(setting, {
      local: true
    }), Keys.config.oneWith(setting, {
      local: false
    })],
    all: new GroupKey('config')
  },
  blob: {
    oneWith: sha => new CacheKey(`blob:${sha}`, ['blob'])
  },
  // Common collections of keys and patterns for use with invalidate().
  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: false
  }])],
  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: true
  }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],
  headOperationKeys: () => [...Keys.filePatch.eachWithOpts({
    staged: true
  }), Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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