(function() {
  var CompositeDisposable, Disposable, Emitter, NotificationsLog, NotificationsLogItem, ref, typeIcons;

  ref = require('atom'), Emitter = ref.Emitter, CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable;

  NotificationsLogItem = require('./notifications-log-item');

  typeIcons = {
    fatal: 'bug',
    error: 'flame',
    warning: 'alert',
    info: 'info',
    success: 'check'
  };

  module.exports = NotificationsLog = (function() {
    NotificationsLog.prototype.subscriptions = null;

    NotificationsLog.prototype.logItems = [];

    NotificationsLog.prototype.typesHidden = {
      fatal: false,
      error: false,
      warning: false,
      info: false,
      success: false
    };

    function NotificationsLog(duplicateTimeDelay, typesHidden) {
      this.duplicateTimeDelay = duplicateTimeDelay;
      if (typesHidden == null) {
        typesHidden = null;
      }
      if (typesHidden != null) {
        this.typesHidden = typesHidden;
      }
      this.emitter = new Emitter;
      this.subscriptions = new CompositeDisposable;
      this.render();
    }

    NotificationsLog.prototype.render = function() {
      var button, header, i, icon, lastNotification, len, notification, ref1, timeSpan, type;
      this.element = document.createElement('div');
      this.element.classList.add('notifications-log');
      header = document.createElement('header');
      this.element.appendChild(header);
      this.list = document.createElement('ul');
      this.list.classList.add('notifications-log-items');
      this.element.appendChild(this.list);
      for (type in typeIcons) {
        icon = typeIcons[type];
        button = document.createElement('button');
        button.classList.add('notification-type', 'btn', 'icon', "icon-" + icon, type);
        button.classList.toggle('show-type', !this.typesHidden[type]);
        this.list.classList.toggle("hide-" + type, this.typesHidden[type]);
        button.dataset.type = type;
        button.addEventListener('click', (function(_this) {
          return function(e) {
            return _this.toggleType(e.target.dataset.type);
          };
        })(this));
        this.subscriptions.add(atom.tooltips.add(button, {
          title: "Toggle " + type + " notifications"
        }));
        header.appendChild(button);
      }
      lastNotification = null;
      ref1 = atom.notifications.getNotifications();
      for (i = 0, len = ref1.length; i < len; i++) {
        notification = ref1[i];
        if (lastNotification != null) {
          timeSpan = notification.getTimestamp() - lastNotification.getTimestamp();
          if (!(timeSpan < this.duplicateTimeDelay && notification.isEqual(lastNotification))) {
            this.addNotification(notification);
          }
        } else {
          this.addNotification(notification);
        }
        lastNotification = notification;
      }
      return this.subscriptions.add(new Disposable((function(_this) {
        return function() {
          return _this.element.remove();
        };
      })(this)));
    };

    NotificationsLog.prototype.destroy = function() {
      this.subscriptions.dispose();
      return this.emitter.emit('did-destroy');
    };

    NotificationsLog.prototype.getElement = function() {
      return this.element;
    };

    NotificationsLog.prototype.getURI = function() {
      return 'atom://notifications/log';
    };

    NotificationsLog.prototype.getTitle = function() {
      return 'Log';
    };

    NotificationsLog.prototype.getLongTitle = function() {
      return 'Notifications Log';
    };

    NotificationsLog.prototype.getIconName = function() {
      return 'alert';
    };

    NotificationsLog.prototype.getDefaultLocation = function() {
      return 'bottom';
    };

    NotificationsLog.prototype.getAllowedLocations = function() {
      return ['left', 'right', 'bottom'];
    };

    NotificationsLog.prototype.serialize = function() {
      return {
        typesHidden: this.typesHidden,
        deserializer: 'notifications/NotificationsLog'
      };
    };

    NotificationsLog.prototype.toggleType = function(type, force) {
      var button, hide;
      button = this.element.querySelector(".notification-type." + type);
      hide = !button.classList.toggle('show-type', force);
      this.list.classList.toggle("hide-" + type, hide);
      return this.typesHidden[type] = hide;
    };

    NotificationsLog.prototype.addNotification = function(notification) {
      var logItem;
      logItem = new NotificationsLogItem(notification);
      logItem.onClick((function(_this) {
        return function() {
          return _this.emitter.emit('item-clicked', notification);
        };
      })(this));
      this.logItems.push(logItem);
      this.list.insertBefore(logItem.getElement(), this.list.firstChild);
      return this.subscriptions.add(new Disposable(function() {
        return logItem.destroy();
      }));
    };

    NotificationsLog.prototype.onItemClick = function(callback) {
      return this.emitter.on('item-clicked', callback);
    };

    NotificationsLog.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    return NotificationsLog;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
