"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _classnames = _interopRequireDefault(require("classnames"));

var _atom = require("atom");

var _eventKit = require("event-kit");

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

var _propTypes2 = require("../prop-types");

var _atomTextEditor = _interopRequireDefault(require("../atom/atom-text-editor"));

var _marker = _interopRequireDefault(require("../atom/marker"));

var _markerLayer = _interopRequireDefault(require("../atom/marker-layer"));

var _decoration = _interopRequireDefault(require("../atom/decoration"));

var _gutter = _interopRequireDefault(require("../atom/gutter"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _filePatchHeaderView = _interopRequireDefault(require("./file-patch-header-view"));

var _filePatchMetaView = _interopRequireDefault(require("./file-patch-meta-view"));

var _hunkHeaderView = _interopRequireDefault(require("./hunk-header-view"));

var _prReviewsContainer = _interopRequireDefault(require("../containers/pr-reviews-container"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _file = _interopRequireDefault(require("../models/patch/file"));

var _patch = require("../models/patch/patch");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const executableText = {
  [_file.default.modes.NORMAL]: 'non executable',
  [_file.default.modes.EXECUTABLE]: 'executable'
};
const NBSP_CHARACTER = '\u00a0';

const BLANK_LABEL = () => NBSP_CHARACTER;

class MultiFilePatchView extends _react.default.Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "renderFilePatchDecorations", (filePatch, index) => {
      const isCollapsed = !filePatch.getRenderStatus().isVisible();
      const isEmpty = filePatch.getMarker().getRange().isEmpty();
      const atEnd = filePatch.getStartRange().start.isEqual(this.props.multiFilePatch.getBuffer().getEndPosition());
      const position = isEmpty && atEnd ? 'after' : 'before';
      return _react.default.createElement(_react.Fragment, {
        key: filePatch.getPath()
      }, _react.default.createElement(_marker.default, {
        invalidate: "never",
        bufferRange: filePatch.getStartRange()
      }, _react.default.createElement(_decoration.default, {
        type: "block",
        position: position,
        order: index,
        className: "github-FilePatchView-controlBlock"
      }, _react.default.createElement(_filePatchHeaderView.default, {
        itemType: this.props.itemType,
        relPath: filePatch.getPath(),
        newPath: filePatch.getStatus() === 'renamed' ? filePatch.getNewPath() : null,
        stagingStatus: this.props.stagingStatus,
        isPartiallyStaged: this.props.isPartiallyStaged,
        hasUndoHistory: this.props.hasUndoHistory,
        hasMultipleFileSelections: this.props.hasMultipleFileSelections,
        tooltips: this.props.tooltips,
        undoLastDiscard: () => this.undoLastDiscardFromButton(filePatch),
        diveIntoMirrorPatch: () => this.props.diveIntoMirrorPatch(filePatch),
        openFile: () => this.didOpenFile({
          selectedFilePatch: filePatch
        }),
        toggleFile: () => this.props.toggleFile(filePatch),
        isCollapsed: isCollapsed,
        triggerCollapse: () => this.props.multiFilePatch.collapseFilePatch(filePatch),
        triggerExpand: () => this.props.multiFilePatch.expandFilePatch(filePatch)
      }), !isCollapsed && this.renderSymlinkChangeMeta(filePatch), !isCollapsed && this.renderExecutableModeChangeMeta(filePatch))), filePatch.getPatch().getRenderStatus() === _patch.TOO_LARGE && this.renderDiffGate(filePatch, position, index), this.renderHunkHeaders(filePatch, index));
    });

    _defineProperty(this, "undoLastDiscardFromCoreUndo", () => {
      if (this.props.hasUndoHistory) {
        const selectedFilePatches = Array.from(this.getSelectedFilePatches());
        /* istanbul ignore else */

        if (this.props.itemType === _changedFileItem.default) {
          this.props.undoLastDiscard(selectedFilePatches[0], {
            eventSource: {
              command: 'core:undo'
            }
          });
        }
      }
    });

    _defineProperty(this, "undoLastDiscardFromButton", filePatch => {
      this.props.undoLastDiscard(filePatch, {
        eventSource: 'button'
      });
    });

    _defineProperty(this, "discardSelectionFromCommand", () => {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: {
          command: 'github:discard-selected-lines'
        }
      });
    });

    _defineProperty(this, "didToggleModeChange", () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.didChangeExecutableMode()).map(this.props.toggleModeChange));
    });

    _defineProperty(this, "didToggleSymlinkChange", () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.hasTypechange()).map(this.props.toggleSymlinkChange));
    });

    (0, _helpers.autobind)(this, 'didMouseDownOnHeader', 'didMouseDownOnLineNumber', 'didMouseMoveOnLineNumber', 'didMouseUp', 'didConfirm', 'didToggleSelectionMode', 'selectNextHunk', 'selectPreviousHunk', 'didOpenFile', 'didAddSelection', 'didChangeSelectionRange', 'didDestroySelection', 'oldLineNumberLabel', 'newLineNumberLabel');
    this.mouseSelectionInProgress = false;
    this.lastMouseMoveLine = null;
    this.nextSelectionMode = null;
    this.refRoot = new _refHolder.default();
    this.refEditor = new _refHolder.default();
    this.refEditorElement = new _refHolder.default();
    this.mounted = false;
    this.subs = new _eventKit.CompositeDisposable();
    this.subs.add(this.refEditor.observe(editor => {
      this.refEditorElement.setter(editor.getElement());

      if (this.props.refEditor) {
        this.props.refEditor.setter(editor);
      }
    }), this.refEditorElement.observe(element => {
      this.props.refInitialFocus && this.props.refInitialFocus.setter(element);
    })); // Synchronously maintain the editor's scroll position and logical selection across buffer updates.

    this.suppressChanges = false;
    let lastScrollTop = null;
    let lastScrollLeft = null;
    let lastSelectionIndex = null;
    this.subs.add(this.props.onWillUpdatePatch(() => {
      this.suppressChanges = true;
      this.refEditor.map(editor => {
        lastSelectionIndex = this.props.multiFilePatch.getMaxSelectionIndex(this.props.selectedRows);
        lastScrollTop = editor.getElement().getScrollTop();
        lastScrollLeft = editor.getElement().getScrollLeft();
        return null;
      });
    }), this.props.onDidUpdatePatch(nextPatch => {
      this.refEditor.map(editor => {
        /* istanbul ignore else */
        if (lastSelectionIndex !== null) {
          const nextSelectionRange = nextPatch.getSelectionRangeForIndex(lastSelectionIndex);

          if (this.props.selectionMode === 'line') {
            this.nextSelectionMode = 'line';
            editor.setSelectedBufferRange(nextSelectionRange);
          } else {
            const nextHunks = new Set(_atom.Range.fromObject(nextSelectionRange).getRows().map(row => nextPatch.getHunkAt(row)).filter(Boolean));
            /* istanbul ignore next */

            const nextRanges = nextHunks.size > 0 ? Array.from(nextHunks, hunk => hunk.getRange()) : [[[0, 0], [0, 0]]];
            this.nextSelectionMode = 'hunk';
            editor.setSelectedBufferRanges(nextRanges);
          }
        }
        /* istanbul ignore else */


        if (lastScrollTop !== null) {
          editor.getElement().setScrollTop(lastScrollTop);
        }
        /* istanbul ignore else */


        if (lastScrollLeft !== null) {
          editor.getElement().setScrollLeft(lastScrollLeft);
        }

        return null;
      });
      this.suppressChanges = false;
      this.didChangeSelectedRows();
    }));
  }

  componentDidMount() {
    this.mounted = true;
    this.measurePerformance('mount');
    window.addEventListener('mouseup', this.didMouseUp);
    this.refEditor.map(editor => {
      // this.props.multiFilePatch is guaranteed to contain at least one FilePatch if <AtomTextEditor> is rendered.
      const [firstPatch] = this.props.multiFilePatch.getFilePatches();
      const [firstHunk] = firstPatch.getHunks();

      if (!firstHunk) {
        return null;
      }

      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRange(firstHunk.getRange());
      return null;
    });
    this.subs.add(this.props.config.onDidChange('github.showDiffIconGutter', () => this.forceUpdate()));
  }

  componentDidUpdate(prevProps) {
    this.measurePerformance('update');

    if (prevProps.refInitialFocus !== this.props.refInitialFocus) {
      prevProps.refInitialFocus && prevProps.refInitialFocus.setter(null);
      this.props.refInitialFocus && this.refEditorElement.map(this.props.refInitialFocus.setter);
    }

    if (this.props.multiFilePatch === prevProps.multiFilePatch) {
      this.nextSelectionMode = null;
    }
  }

  componentWillUnmount() {
    window.removeEventListener('mouseup', this.didMouseUp);
    this.subs.dispose();
    this.mounted = false;
    performance.clearMarks();
    performance.clearMeasures();
  }

  render() {
    const rootClass = (0, _classnames.default)('github-FilePatchView', {
      [`github-FilePatchView--${this.props.stagingStatus}`]: this.props.stagingStatus
    }, {
      'github-FilePatchView--blank': !this.props.multiFilePatch.anyPresent()
    }, {
      'github-FilePatchView--hunkMode': this.props.selectionMode === 'hunk'
    });

    if (this.mounted) {
      performance.mark('MultiFilePatchView-update-start');
    } else {
      performance.mark('MultiFilePatchView-mount-start');
    }

    return _react.default.createElement("div", {
      className: rootClass,
      ref: this.refRoot.setter
    }, this.renderCommands(), _react.default.createElement("main", {
      className: "github-FilePatchView-container"
    }, this.props.multiFilePatch.anyPresent() ? this.renderNonEmptyPatch() : this.renderEmptyPatch()));
  }

  renderCommands() {
    if (this.props.itemType === _commitDetailItem.default || this.props.itemType === _issueishDetailItem.default) {
      return _react.default.createElement(_commands.default, {
        registry: this.props.commands,
        target: this.refRoot
      }, _react.default.createElement(_commands.Command, {
        command: "github:select-next-hunk",
        callback: this.selectNextHunk
      }), _react.default.createElement(_commands.Command, {
        command: "github:select-previous-hunk",
        callback: this.selectPreviousHunk
      }), _react.default.createElement(_commands.Command, {
        command: "github:toggle-patch-selection-mode",
        callback: this.didToggleSelectionMode
      }));
    }

    let stageModeCommand = null;
    let stageSymlinkCommand = null;

    if (this.props.multiFilePatch.didAnyChangeExecutableMode()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-file-mode-change' : 'github:unstage-file-mode-change';
      stageModeCommand = _react.default.createElement(_commands.Command, {
        command: command,
        callback: this.didToggleModeChange
      });
    }

    if (this.props.multiFilePatch.anyHaveTypechange()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-symlink-change' : 'github:unstage-symlink-change';
      stageSymlinkCommand = _react.default.createElement(_commands.Command, {
        command: command,
        callback: this.didToggleSymlinkChange
      });
    }

    return _react.default.createElement(_commands.default, {
      registry: this.props.commands,
      target: this.refRoot
    }, _react.default.createElement(_commands.Command, {
      command: "github:select-next-hunk",
      callback: this.selectNextHunk
    }), _react.default.createElement(_commands.Command, {
      command: "github:select-previous-hunk",
      callback: this.selectPreviousHunk
    }), _react.default.createElement(_commands.Command, {
      command: "core:confirm",
      callback: this.didConfirm
    }), _react.default.createElement(_commands.Command, {
      command: "core:undo",
      callback: this.undoLastDiscardFromCoreUndo
    }), _react.default.createElement(_commands.Command, {
      command: "github:discard-selected-lines",
      callback: this.discardSelectionFromCommand
    }), _react.default.createElement(_commands.Command, {
      command: "github:jump-to-file",
      callback: this.didOpenFile
    }), _react.default.createElement(_commands.Command, {
      command: "github:surface",
      callback: this.props.surface
    }), _react.default.createElement(_commands.Command, {
      command: "github:toggle-patch-selection-mode",
      callback: this.didToggleSelectionMode
    }), stageModeCommand, stageSymlinkCommand,
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-patch",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.getPatchBuffer().inspect({
          layerNames: ['patch', 'hunk']
        }));
      }
    }),
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-regions",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.getPatchBuffer().inspect({
          layerNames: ['unchanged', 'deletion', 'addition', 'nonewline']
        }));
      }
    }),
    /* istanbul ignore next */
    atom.inDevMode() && _react.default.createElement(_commands.Command, {
      command: "github:inspect-mfp",
      callback: () => {
        // eslint-disable-next-line no-console
        console.log(this.props.multiFilePatch.inspect());
      }
    }));
  }

  renderEmptyPatch() {
    return _react.default.createElement("p", {
      className: "github-FilePatchView-message icon icon-info"
    }, "No changes to display");
  }

  renderNonEmptyPatch() {
    return _react.default.createElement(_atomTextEditor.default, {
      workspace: this.props.workspace,
      buffer: this.props.multiFilePatch.getBuffer(),
      lineNumberGutterVisible: false,
      autoWidth: false,
      autoHeight: false,
      readOnly: true,
      softWrapped: true,
      didAddSelection: this.didAddSelection,
      didChangeSelectionRange: this.didChangeSelectionRange,
      didDestroySelection: this.didDestroySelection,
      refModel: this.refEditor,
      hideEmptiness: true
    }, _react.default.createElement(_gutter.default, {
      name: "old-line-numbers",
      priority: 1,
      className: "old",
      type: "line-number",
      labelFn: this.oldLineNumberLabel,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), _react.default.createElement(_gutter.default, {
      name: "new-line-numbers",
      priority: 2,
      className: "new",
      type: "line-number",
      labelFn: this.newLineNumberLabel,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), this.props.config.get('github.showDiffIconGutter') && _react.default.createElement(_gutter.default, {
      name: "diff-icons",
      priority: 3,
      type: "line-number",
      className: "icons",
      labelFn: BLANK_LABEL,
      onMouseDown: this.didMouseDownOnLineNumber,
      onMouseMove: this.didMouseMoveOnLineNumber
    }), this.renderPullRequestReviews(), this.props.multiFilePatch.getFilePatches().map(this.renderFilePatchDecorations), this.renderLineDecorations(Array.from(this.props.selectedRows, row => _atom.Range.fromObject([[row, 0], [row, Infinity]])), 'github-FilePatchView-line--selected', {
      gutter: true,
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getAdditionLayer(), 'github-FilePatchView-line--added', {
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getDeletionLayer(), 'github-FilePatchView-line--deleted', {
      icon: true,
      line: true
    }), this.renderDecorationsOnLayer(this.props.multiFilePatch.getNoNewlineLayer(), 'github-FilePatchView-line--nonewline', {
      icon: true,
      line: true
    }));
  }

  renderPullRequestReviews() {
    if (this.props.itemType === _issueishDetailItem.default) {
      // "forceRerender" ensures that the PullRequestCommentsView re-renders each time that the MultiFilePatchView does.
      // It doesn't re-query for reviews, but it does re-check patch visibility.
      return _react.default.createElement(_prReviewsContainer.default, {
        pullRequest: this.props.pullRequest,
        getBufferRowForDiffPosition: this.props.multiFilePatch.getBufferRowForDiffPosition,
        isPatchVisible: this.props.multiFilePatch.isPatchVisible,
        switchToIssueish: this.props.switchToIssueish,
        forceRerender: {}
      });
    } else {
      return null;
    }
  }

  renderDiffGate(filePatch, position, orderOffset) {
    const showDiff = () => {
      (0, _reporterProxy.addEvent)('expand-file-patch', {
        component: this.constructor.name,
        package: 'github'
      });
      this.props.multiFilePatch.expandFilePatch(filePatch);
    };

    return _react.default.createElement(_marker.default, {
      invalidate: "never",
      bufferRange: filePatch.getStartRange()
    }, _react.default.createElement(_decoration.default, {
      type: "block",
      order: orderOffset + 0.1,
      position: position,
      className: "github-FilePatchView-controlBlock"
    }, _react.default.createElement("p", {
      className: "github-FilePatchView-message icon icon-info"
    }, "Large diffs are collapsed by default for performance reasons.", _react.default.createElement("br", null), _react.default.createElement("button", {
      className: "github-FilePatchView-showDiffButton",
      onClick: showDiff
    }, " Load Diff"))));
  }

  renderExecutableModeChangeMeta(filePatch) {
    if (!filePatch.didChangeExecutableMode()) {
      return null;
    }

    const oldMode = filePatch.getOldMode();
    const newMode = filePatch.getNewMode();
    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Mode Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Mode Change'
    };
    return _react.default.createElement(_filePatchMetaView.default, {
      title: "Mode change",
      actionIcon: attrs.actionIcon,
      actionText: attrs.actionText,
      itemType: this.props.itemType,
      action: () => this.props.toggleModeChange(filePatch)
    }, _react.default.createElement(_react.Fragment, null, "File changed mode", _react.default.createElement("span", {
      className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed"
    }, "from ", executableText[oldMode], " ", _react.default.createElement("code", null, oldMode)), _react.default.createElement("span", {
      className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added"
    }, "to ", executableText[newMode], " ", _react.default.createElement("code", null, newMode))));
  }

  renderSymlinkChangeMeta(filePatch) {
    if (!filePatch.hasSymlink()) {
      return null;
    }

    let detail = _react.default.createElement("div", null);

    let title = '';
    const oldSymlink = filePatch.getOldSymlink();
    const newSymlink = filePatch.getNewSymlink();

    if (oldSymlink && newSymlink) {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink changed", _react.default.createElement("span", {
        className: (0, _classnames.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--removed')
      }, "from ", _react.default.createElement("code", null, oldSymlink)), _react.default.createElement("span", {
        className: (0, _classnames.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--added')
      }, "to ", _react.default.createElement("code", null, newSymlink)), ".");
      title = 'Symlink changed';
    } else if (oldSymlink && !newSymlink) {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink", _react.default.createElement("span", {
        className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed"
      }, "to ", _react.default.createElement("code", null, oldSymlink)), "deleted.");
      title = 'Symlink deleted';
    } else {
      detail = _react.default.createElement(_react.Fragment, null, "Symlink", _react.default.createElement("span", {
        className: "github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added"
      }, "to ", _react.default.createElement("code", null, newSymlink)), "created.");
      title = 'Symlink created';
    }

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Symlink Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Symlink Change'
    };
    return _react.default.createElement(_filePatchMetaView.default, {
      title: title,
      actionIcon: attrs.actionIcon,
      actionText: attrs.actionText,
      itemType: this.props.itemType,
      action: () => this.props.toggleSymlinkChange(filePatch)
    }, _react.default.createElement(_react.Fragment, null, detail));
  }

  renderHunkHeaders(filePatch, orderOffset) {
    const toggleVerb = this.props.stagingStatus === 'unstaged' ? 'Stage' : 'Unstage';
    const selectedHunks = new Set(Array.from(this.props.selectedRows, row => this.props.multiFilePatch.getHunkAt(row)));
    return _react.default.createElement(_react.Fragment, null, _react.default.createElement(_markerLayer.default, null, filePatch.getHunks().map((hunk, index) => {
      const containsSelection = this.props.selectionMode === 'line' && selectedHunks.has(hunk);
      const isSelected = this.props.selectionMode === 'hunk' && selectedHunks.has(hunk);
      let buttonSuffix = '';

      if (containsSelection) {
        buttonSuffix += 'Selected Line';

        if (this.props.selectedRows.size > 1) {
          buttonSuffix += 's';
        }
      } else {
        buttonSuffix += 'Hunk';

        if (selectedHunks.size > 1) {
          buttonSuffix += 's';
        }
      }

      const toggleSelectionLabel = `${toggleVerb} ${buttonSuffix}`;
      const discardSelectionLabel = `Discard ${buttonSuffix}`;
      const startPoint = hunk.getRange().start;
      const startRange = new _atom.Range(startPoint, startPoint);
      return _react.default.createElement(_marker.default, {
        key: `hunkHeader-${index}`,
        bufferRange: startRange,
        invalidate: "never"
      }, _react.default.createElement(_decoration.default, {
        type: "block",
        order: orderOffset + 0.2,
        className: "github-FilePatchView-controlBlock"
      }, _react.default.createElement(_hunkHeaderView.default, {
        refTarget: this.refEditorElement,
        hunk: hunk,
        isSelected: isSelected,
        stagingStatus: this.props.stagingStatus,
        selectionMode: "line",
        toggleSelectionLabel: toggleSelectionLabel,
        discardSelectionLabel: discardSelectionLabel,
        tooltips: this.props.tooltips,
        keymaps: this.props.keymaps,
        toggleSelection: () => this.toggleHunkSelection(hunk, containsSelection),
        discardSelection: () => this.discardHunkSelection(hunk, containsSelection),
        mouseDown: this.didMouseDownOnHeader,
        itemType: this.props.itemType
      })));
    })));
  }

  renderLineDecorations(ranges, lineClass, {
    line,
    gutter,
    icon,
    refHolder
  }) {
    if (ranges.length === 0) {
      return null;
    }

    const holder = refHolder || new _refHolder.default();
    return _react.default.createElement(_markerLayer.default, {
      handleLayer: holder.setter
    }, ranges.map((range, index) => {
      return _react.default.createElement(_marker.default, {
        key: `line-${lineClass}-${index}`,
        bufferRange: range,
        invalidate: "never"
      });
    }), this.renderDecorations(lineClass, {
      line,
      gutter,
      icon
    }));
  }

  renderDecorationsOnLayer(layer, lineClass, {
    line,
    gutter,
    icon
  }) {
    if (layer.getMarkerCount() === 0) {
      return null;
    }

    return _react.default.createElement(_markerLayer.default, {
      external: layer
    }, this.renderDecorations(lineClass, {
      line,
      gutter,
      icon
    }));
  }

  renderDecorations(lineClass, {
    line,
    gutter,
    icon
  }) {
    return _react.default.createElement(_react.Fragment, null, line && _react.default.createElement(_decoration.default, {
      type: "line",
      className: lineClass,
      omitEmptyLastRow: false
    }), gutter && _react.default.createElement(_react.Fragment, null, _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "old-line-numbers",
      className: lineClass,
      omitEmptyLastRow: false
    }), _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "new-line-numbers",
      className: lineClass,
      omitEmptyLastRow: false
    })), icon && _react.default.createElement(_decoration.default, {
      type: "line-number",
      gutterName: "diff-icons",
      className: lineClass,
      omitEmptyLastRow: false
    }));
  }

  toggleHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: 'button'
      });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.toggleRows(changeRows, 'hunk', {
        eventSource: 'button'
      });
    }
  }

  discardHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, {
        eventSource: 'button'
      });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.discardRows(changeRows, 'hunk', {
        eventSource: 'button'
      });
    }
  }

  didMouseDownOnHeader(event, hunk) {
    this.nextSelectionMode = 'hunk';
    this.handleSelectionEvent(event, hunk.getRange());
  }

  didMouseDownOnLineNumber(event) {
    const line = event.bufferRow;

    if (line === undefined || isNaN(line)) {
      return;
    }

    this.nextSelectionMode = 'line';

    if (this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]])) {
      this.mouseSelectionInProgress = true;
    }
  }

  didMouseMoveOnLineNumber(event) {
    if (!this.mouseSelectionInProgress) {
      return;
    }

    const line = event.bufferRow;

    if (this.lastMouseMoveLine === line || line === undefined || isNaN(line)) {
      return;
    }

    this.lastMouseMoveLine = line;
    this.nextSelectionMode = 'line';
    this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]], {
      add: true
    });
  }

  didMouseUp() {
    this.mouseSelectionInProgress = false;
  }

  handleSelectionEvent(event, rangeLike, opts) {
    if (event.button !== 0) {
      return false;
    }

    const isWindows = process.platform === 'win32';

    if (event.ctrlKey && !isWindows) {
      // Allow the context menu to open.
      return false;
    }

    const options = _objectSpread({
      add: false
    }, opts); // Normalize the target selection range


    const converted = _atom.Range.fromObject(rangeLike);

    const range = this.refEditor.map(editor => editor.clipBufferRange(converted)).getOr(converted);

    if (event.metaKey ||
    /* istanbul ignore next */
    event.ctrlKey && isWindows) {
      this.refEditor.map(editor => {
        let intersects = false;
        let without = null;

        for (const selection of editor.getSelections()) {
          if (selection.intersectsBufferRange(range)) {
            // Remove range from this selection by truncating it to the "near edge" of the range and creating a
            // new selection from the "far edge" to the previous end. Omit either side if it is empty.
            intersects = true;
            const selectionRange = selection.getBufferRange();
            const newRanges = [];

            if (!range.start.isEqual(selectionRange.start)) {
              // Include the bit from the selection's previous start to the range's start.
              let nudged = range.start;

              if (range.start.column === 0) {
                const lastColumn = editor.getBuffer().lineLengthForRow(range.start.row - 1);
                nudged = [range.start.row - 1, lastColumn];
              }

              newRanges.push([selectionRange.start, nudged]);
            }

            if (!range.end.isEqual(selectionRange.end)) {
              // Include the bit from the range's end to the selection's end.
              let nudged = range.end;
              const lastColumn = editor.getBuffer().lineLengthForRow(range.end.row);

              if (range.end.column === lastColumn) {
                nudged = [range.end.row + 1, 0];
              }

              newRanges.push([nudged, selectionRange.end]);
            }

            if (newRanges.length > 0) {
              selection.setBufferRange(newRanges[0]);

              for (const newRange of newRanges.slice(1)) {
                editor.addSelectionForBufferRange(newRange, {
                  reversed: selection.isReversed()
                });
              }
            } else {
              without = selection;
            }
          }
        }

        if (without !== null) {
          const replacementRanges = editor.getSelections().filter(each => each !== without).map(each => each.getBufferRange());

          if (replacementRanges.length > 0) {
            editor.setSelectedBufferRanges(replacementRanges);
          }
        }

        if (!intersects) {
          // Add this range as a new, distinct selection.
          editor.addSelectionForBufferRange(range);
        }

        return null;
      });
    } else if (options.add || event.shiftKey) {
      // Extend the existing selection to encompass this range.
      this.refEditor.map(editor => {
        const lastSelection = editor.getLastSelection();
        const lastSelectionRange = lastSelection.getBufferRange(); // You are now entering the wall of ternery operators. This is your last exit before the tollbooth

        const isBefore = range.start.isLessThan(lastSelectionRange.start);
        const farEdge = isBefore ? range.start : range.end;
        const newRange = isBefore ? [farEdge, lastSelectionRange.end] : [lastSelectionRange.start, farEdge];
        lastSelection.setBufferRange(newRange, {
          reversed: isBefore
        });
        return null;
      });
    } else {
      this.refEditor.map(editor => editor.setSelectedBufferRange(range));
    }

    return true;
  }

  didConfirm() {
    return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode);
  }

  didToggleSelectionMode() {
    const selectedHunks = this.getSelectedHunks();
    this.withSelectionMode({
      line: () => {
        const hunkRanges = selectedHunks.map(hunk => hunk.getRange());
        this.nextSelectionMode = 'hunk';
        this.refEditor.map(editor => editor.setSelectedBufferRanges(hunkRanges));
      },
      hunk: () => {
        let firstChangeRow = Infinity;

        for (const hunk of selectedHunks) {
          const [firstChange] = hunk.getChanges();
          /* istanbul ignore else */

          if (firstChange && (!firstChangeRow || firstChange.getStartBufferRow() < firstChangeRow)) {
            firstChangeRow = firstChange.getStartBufferRow();
          }
        }

        this.nextSelectionMode = 'line';
        this.refEditor.map(editor => {
          editor.setSelectedBufferRanges([[[firstChangeRow, 0], [firstChangeRow, Infinity]]]);
          return null;
        });
      }
    });
  }

  selectNextHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkAfter(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  selectPreviousHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkBefore(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  didOpenFile({
    selectedFilePatch
  }) {
    const cursorsByFilePatch = new Map();
    this.refEditor.map(editor => {
      const placedRows = new Set();

      for (const cursor of editor.getCursors()) {
        const cursorRow = cursor.getBufferPosition().row;
        const hunk = this.props.multiFilePatch.getHunkAt(cursorRow);
        const filePatch = this.props.multiFilePatch.getFilePatchAt(cursorRow);
        /* istanbul ignore next */

        if (!hunk) {
          continue;
        }

        let newRow = hunk.getNewRowAt(cursorRow);
        let newColumn = cursor.getBufferPosition().column;

        if (newRow === null) {
          let nearestRow = hunk.getNewStartRow();

          for (const region of hunk.getRegions()) {
            if (!region.includesBufferRow(cursorRow)) {
              region.when({
                unchanged: () => {
                  nearestRow += region.bufferRowCount();
                },
                addition: () => {
                  nearestRow += region.bufferRowCount();
                }
              });
            } else {
              break;
            }
          }

          if (!placedRows.has(nearestRow)) {
            newRow = nearestRow;
            newColumn = 0;
            placedRows.add(nearestRow);
          }
        }

        if (newRow !== null) {
          // Why is this needed? I _think_ everything is in terms of buffer position
          // so there shouldn't be an off-by-one issue
          newRow -= 1;
          const cursors = cursorsByFilePatch.get(filePatch);

          if (!cursors) {
            cursorsByFilePatch.set(filePatch, [[newRow, newColumn]]);
          } else {
            cursors.push([newRow, newColumn]);
          }
        }
      }

      return null;
    });
    const filePatchesWithCursors = new Set(cursorsByFilePatch.keys());

    if (selectedFilePatch && !filePatchesWithCursors.has(selectedFilePatch)) {
      const [firstHunk] = selectedFilePatch.getHunks();
      const cursorRow = firstHunk ? firstHunk.getNewStartRow() - 1 : 0;
      return this.props.openFile(selectedFilePatch, [[cursorRow, 0]], true);
    } else {
      const pending = cursorsByFilePatch.size === 1;
      return Promise.all(Array.from(cursorsByFilePatch, value => {
        const [filePatch, cursors] = value;
        return this.props.openFile(filePatch, cursors, pending);
      }));
    }
  }

  getSelectedRows() {
    return this.refEditor.map(editor => {
      return new Set(editor.getSelections().map(selection => selection.getBufferRange()).reduce((acc, range) => {
        for (const row of range.getRows()) {
          if (this.isChangeRow(row)) {
            acc.push(row);
          }
        }

        return acc;
      }, []));
    }).getOr(new Set());
  }

  didAddSelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectionRange(event) {
    if (!event || event.oldBufferRange.start.row !== event.newBufferRange.start.row || event.oldBufferRange.end.row !== event.newBufferRange.end.row) {
      this.didChangeSelectedRows();
    }
  }

  didDestroySelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectedRows() {
    if (this.suppressChanges) {
      return;
    }

    const nextCursorRows = this.refEditor.map(editor => {
      return editor.getCursorBufferPositions().map(position => position.row);
    }).getOr([]);
    const hasMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(nextCursorRows);
    this.props.selectedRowsChanged(this.getSelectedRows(), this.nextSelectionMode || 'line', hasMultipleFileSelections);
  }

  oldLineNumberLabel({
    bufferRow,
    softWrapped
  }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);

    if (hunk === undefined) {
      return this.pad('');
    }

    const oldRow = hunk.getOldRowAt(bufferRow);

    if (softWrapped) {
      return this.pad(oldRow === null ? '' : '•');
    }

    return this.pad(oldRow);
  }

  newLineNumberLabel({
    bufferRow,
    softWrapped
  }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);

    if (hunk === undefined) {
      return this.pad('');
    }

    const newRow = hunk.getNewRowAt(bufferRow);

    if (softWrapped) {
      return this.pad(newRow === null ? '' : '•');
    }

    return this.pad(newRow);
  }
  /*
   * Return a Set of the Hunks that include at least one editor selection. The selection need not contain an actual
   * change row.
   */


  getSelectedHunks() {
    return this.withSelectedHunks(each => each);
  }

  withSelectedHunks(callback) {
    return this.refEditor.map(editor => {
      const seen = new Set();
      return editor.getSelectedBufferRanges().reduce((acc, range) => {
        for (const row of range.getRows()) {
          const hunk = this.props.multiFilePatch.getHunkAt(row);

          if (!hunk || seen.has(hunk)) {
            continue;
          }

          seen.add(hunk);
          acc.push(callback(hunk));
        }

        return acc;
      }, []);
    }).getOr([]);
  }
  /*
   * Return a Set of FilePatches that include at least one editor selection. The selection need not contain an actual
   * change row.
   */


  getSelectedFilePatches() {
    return this.refEditor.map(editor => {
      const patches = new Set();

      for (const range of editor.getSelectedBufferRanges()) {
        for (const row of range.getRows()) {
          const patch = this.props.multiFilePatch.getFilePatchAt(row);
          patches.add(patch);
        }
      }

      return patches;
    }).getOr(new Set());
  }

  getHunkBefore(hunk) {
    const prevRow = hunk.getRange().start.row - 1;
    return this.props.multiFilePatch.getHunkAt(prevRow);
  }

  getHunkAfter(hunk) {
    const nextRow = hunk.getRange().end.row + 1;
    return this.props.multiFilePatch.getHunkAt(nextRow);
  }

  isChangeRow(bufferRow) {
    const changeLayers = [this.props.multiFilePatch.getAdditionLayer(), this.props.multiFilePatch.getDeletionLayer()];
    return changeLayers.some(layer => layer.findMarkers({
      intersectsRow: bufferRow
    }).length > 0);
  }

  withSelectionMode(callbacks) {
    const callback = callbacks[this.props.selectionMode];
    /* istanbul ignore if */

    if (!callback) {
      throw new Error(`Unknown selection mode: ${this.props.selectionMode}`);
    }

    return callback();
  }

  pad(num) {
    const maxDigits = this.props.multiFilePatch.getMaxLineNumberWidth();

    if (num === null) {
      return NBSP_CHARACTER.repeat(maxDigits);
    } else {
      return NBSP_CHARACTER.repeat(maxDigits - num.toString().length) + num.toString();
    }
  }

  measurePerformance(action) {
    /* istanbul ignore else */
    if ((action === 'update' || action === 'mount') && performance.getEntriesByName(`MultiFilePatchView-${action}-start`).length > 0) {
      performance.mark(`MultiFilePatchView-${action}-end`);
      performance.measure(`MultiFilePatchView-${action}`, `MultiFilePatchView-${action}-start`, `MultiFilePatchView-${action}-end`);
      const perf = performance.getEntriesByName(`MultiFilePatchView-${action}`)[0];
      performance.clearMarks(`MultiFilePatchView-${action}-start`);
      performance.clearMarks(`MultiFilePatchView-${action}-end`);
      performance.clearMeasures(`MultiFilePatchView-${action}`);
      (0, _reporterProxy.addEvent)(`MultiFilePatchView-${action}`, {
        package: 'github',
        filePatchesLineCounts: this.props.multiFilePatch.getFilePatches().map(fp => fp.getPatch().getChangedLineCount()),
        duration: perf.duration
      });
    }
  }

}

exports.default = MultiFilePatchView;

_defineProperty(MultiFilePatchView, "propTypes", {
  stagingStatus: _propTypes.default.oneOf(['staged', 'unstaged']),
  isPartiallyStaged: _propTypes.default.bool,
  multiFilePatch: _propTypes2.MultiFilePatchPropType.isRequired,
  selectionMode: _propTypes.default.oneOf(['hunk', 'line']).isRequired,
  selectedRows: _propTypes.default.object.isRequired,
  hasMultipleFileSelections: _propTypes.default.bool.isRequired,
  repository: _propTypes.default.object.isRequired,
  hasUndoHistory: _propTypes.default.bool,
  workspace: _propTypes.default.object.isRequired,
  commands: _propTypes.default.object.isRequired,
  keymaps: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  pullRequest: _propTypes.default.object,
  selectedRowsChanged: _propTypes.default.func,
  switchToIssueish: _propTypes.default.func,
  diveIntoMirrorPatch: _propTypes.default.func,
  surface: _propTypes.default.func,
  openFile: _propTypes.default.func,
  toggleFile: _propTypes.default.func,
  toggleRows: _propTypes.default.func,
  toggleModeChange: _propTypes.default.func,
  toggleSymlinkChange: _propTypes.default.func,
  undoLastDiscard: _propTypes.default.func,
  discardRows: _propTypes.default.func,
  onWillUpdatePatch: _propTypes.default.func,
  onDidUpdatePatch: _propTypes.default.func,
  refEditor: _propTypes2.RefHolderPropType,
  refInitialFocus: _propTypes2.RefHolderPropType,
  itemType: _propTypes2.ItemTypePropType.isRequired
});

_defineProperty(MultiFilePatchView, "defaultProps", {
  onWillUpdatePatch: () => new _eventKit.Disposable(),
  onDidUpdatePatch: () => new _eventKit.Disposable()
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm11bHRpLWZpbGUtcGF0Y2gtdmlldy5qcyJdLCJuYW1lcyI6WyJleGVjdXRhYmxlVGV4dCIsIkZpbGUiLCJtb2RlcyIsIk5PUk1BTCIsIkVYRUNVVEFCTEUiLCJOQlNQX0NIQVJBQ1RFUiIsIkJMQU5LX0xBQkVMIiwiTXVsdGlGaWxlUGF0Y2hWaWV3IiwiUmVhY3QiLCJDb21wb25lbnQiLCJjb25zdHJ1Y3RvciIsInByb3BzIiwiZmlsZVBhdGNoIiwiaW5kZXgiLCJpc0NvbGxhcHNlZCIsImdldFJlbmRlclN0YXR1cyIsImlzVmlzaWJsZSIsImlzRW1wdHkiLCJnZXRNYXJrZXIiLCJnZXRSYW5nZSIsImF0RW5kIiwiZ2V0U3RhcnRSYW5nZSIsInN0YXJ0IiwiaXNFcXVhbCIsIm11bHRpRmlsZVBhdGNoIiwiZ2V0QnVmZmVyIiwiZ2V0RW5kUG9zaXRpb24iLCJwb3NpdGlvbiIsImdldFBhdGgiLCJpdGVtVHlwZSIsImdldFN0YXR1cyIsImdldE5ld1BhdGgiLCJzdGFnaW5nU3RhdHVzIiwiaXNQYXJ0aWFsbHlTdGFnZWQiLCJoYXNVbmRvSGlzdG9yeSIsImhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnMiLCJ0b29sdGlwcyIsInVuZG9MYXN0RGlzY2FyZEZyb21CdXR0b24iLCJkaXZlSW50b01pcnJvclBhdGNoIiwiZGlkT3BlbkZpbGUiLCJzZWxlY3RlZEZpbGVQYXRjaCIsInRvZ2dsZUZpbGUiLCJjb2xsYXBzZUZpbGVQYXRjaCIsImV4cGFuZEZpbGVQYXRjaCIsInJlbmRlclN5bWxpbmtDaGFuZ2VNZXRhIiwicmVuZGVyRXhlY3V0YWJsZU1vZGVDaGFuZ2VNZXRhIiwiZ2V0UGF0Y2giLCJUT09fTEFSR0UiLCJyZW5kZXJEaWZmR2F0ZSIsInJlbmRlckh1bmtIZWFkZXJzIiwic2VsZWN0ZWRGaWxlUGF0Y2hlcyIsIkFycmF5IiwiZnJvbSIsImdldFNlbGVjdGVkRmlsZVBhdGNoZXMiLCJDaGFuZ2VkRmlsZUl0ZW0iLCJ1bmRvTGFzdERpc2NhcmQiLCJldmVudFNvdXJjZSIsImNvbW1hbmQiLCJkaXNjYXJkUm93cyIsInNlbGVjdGVkUm93cyIsInNlbGVjdGlvbk1vZGUiLCJQcm9taXNlIiwiYWxsIiwiZmlsdGVyIiwiZnAiLCJkaWRDaGFuZ2VFeGVjdXRhYmxlTW9kZSIsIm1hcCIsInRvZ2dsZU1vZGVDaGFuZ2UiLCJoYXNUeXBlY2hhbmdlIiwidG9nZ2xlU3ltbGlua0NoYW5nZSIsIm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcyIsImxhc3RNb3VzZU1vdmVMaW5lIiwibmV4dFNlbGVjdGlvbk1vZGUiLCJyZWZSb290IiwiUmVmSG9sZGVyIiwicmVmRWRpdG9yIiwicmVmRWRpdG9yRWxlbWVudCIsIm1vdW50ZWQiLCJzdWJzIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsImFkZCIsIm9ic2VydmUiLCJlZGl0b3IiLCJzZXR0ZXIiLCJnZXRFbGVtZW50IiwiZWxlbWVudCIsInJlZkluaXRpYWxGb2N1cyIsInN1cHByZXNzQ2hhbmdlcyIsImxhc3RTY3JvbGxUb3AiLCJsYXN0U2Nyb2xsTGVmdCIsImxhc3RTZWxlY3Rpb25JbmRleCIsIm9uV2lsbFVwZGF0ZVBhdGNoIiwiZ2V0TWF4U2VsZWN0aW9uSW5kZXgiLCJnZXRTY3JvbGxUb3AiLCJnZXRTY3JvbGxMZWZ0Iiwib25EaWRVcGRhdGVQYXRjaCIsIm5leHRQYXRjaCIsIm5leHRTZWxlY3Rpb25SYW5nZSIsImdldFNlbGVjdGlvblJhbmdlRm9ySW5kZXgiLCJzZXRTZWxlY3RlZEJ1ZmZlclJhbmdlIiwibmV4dEh1bmtzIiwiU2V0IiwiUmFuZ2UiLCJmcm9tT2JqZWN0IiwiZ2V0Um93cyIsInJvdyIsImdldEh1bmtBdCIsIkJvb2xlYW4iLCJuZXh0UmFuZ2VzIiwic2l6ZSIsImh1bmsiLCJzZXRTZWxlY3RlZEJ1ZmZlclJhbmdlcyIsInNldFNjcm9sbFRvcCIsInNldFNjcm9sbExlZnQiLCJkaWRDaGFuZ2VTZWxlY3RlZFJvd3MiLCJjb21wb25lbnREaWRNb3VudCIsIm1lYXN1cmVQZXJmb3JtYW5jZSIsIndpbmRvdyIsImFkZEV2ZW50TGlzdGVuZXIiLCJkaWRNb3VzZVVwIiwiZmlyc3RQYXRjaCIsImdldEZpbGVQYXRjaGVzIiwiZmlyc3RIdW5rIiwiZ2V0SHVua3MiLCJjb25maWciLCJvbkRpZENoYW5nZSIsImZvcmNlVXBkYXRlIiwiY29tcG9uZW50RGlkVXBkYXRlIiwicHJldlByb3BzIiwiY29tcG9uZW50V2lsbFVubW91bnQiLCJyZW1vdmVFdmVudExpc3RlbmVyIiwiZGlzcG9zZSIsInBlcmZvcm1hbmNlIiwiY2xlYXJNYXJrcyIsImNsZWFyTWVhc3VyZXMiLCJyZW5kZXIiLCJyb290Q2xhc3MiLCJhbnlQcmVzZW50IiwibWFyayIsInJlbmRlckNvbW1hbmRzIiwicmVuZGVyTm9uRW1wdHlQYXRjaCIsInJlbmRlckVtcHR5UGF0Y2giLCJDb21taXREZXRhaWxJdGVtIiwiSXNzdWVpc2hEZXRhaWxJdGVtIiwiY29tbWFuZHMiLCJzZWxlY3ROZXh0SHVuayIsInNlbGVjdFByZXZpb3VzSHVuayIsImRpZFRvZ2dsZVNlbGVjdGlvbk1vZGUiLCJzdGFnZU1vZGVDb21tYW5kIiwic3RhZ2VTeW1saW5rQ29tbWFuZCIsImRpZEFueUNoYW5nZUV4ZWN1dGFibGVNb2RlIiwiZGlkVG9nZ2xlTW9kZUNoYW5nZSIsImFueUhhdmVUeXBlY2hhbmdlIiwiZGlkVG9nZ2xlU3ltbGlua0NoYW5nZSIsImRpZENvbmZpcm0iLCJ1bmRvTGFzdERpc2NhcmRGcm9tQ29yZVVuZG8iLCJkaXNjYXJkU2VsZWN0aW9uRnJvbUNvbW1hbmQiLCJzdXJmYWNlIiwiYXRvbSIsImluRGV2TW9kZSIsImNvbnNvbGUiLCJsb2ciLCJnZXRQYXRjaEJ1ZmZlciIsImluc3BlY3QiLCJsYXllck5hbWVzIiwid29ya3NwYWNlIiwiZGlkQWRkU2VsZWN0aW9uIiwiZGlkQ2hhbmdlU2VsZWN0aW9uUmFuZ2UiLCJkaWREZXN0cm95U2VsZWN0aW9uIiwib2xkTGluZU51bWJlckxhYmVsIiwiZGlkTW91c2VEb3duT25MaW5lTnVtYmVyIiwiZGlkTW91c2VNb3ZlT25MaW5lTnVtYmVyIiwibmV3TGluZU51bWJlckxhYmVsIiwiZ2V0IiwicmVuZGVyUHVsbFJlcXVlc3RSZXZpZXdzIiwicmVuZGVyRmlsZVBhdGNoRGVjb3JhdGlvbnMiLCJyZW5kZXJMaW5lRGVjb3JhdGlvbnMiLCJJbmZpbml0eSIsImd1dHRlciIsImljb24iLCJsaW5lIiwicmVuZGVyRGVjb3JhdGlvbnNPbkxheWVyIiwiZ2V0QWRkaXRpb25MYXllciIsImdldERlbGV0aW9uTGF5ZXIiLCJnZXROb05ld2xpbmVMYXllciIsInB1bGxSZXF1ZXN0IiwiZ2V0QnVmZmVyUm93Rm9yRGlmZlBvc2l0aW9uIiwiaXNQYXRjaFZpc2libGUiLCJzd2l0Y2hUb0lzc3VlaXNoIiwib3JkZXJPZmZzZXQiLCJzaG93RGlmZiIsImNvbXBvbmVudCIsIm5hbWUiLCJwYWNrYWdlIiwib2xkTW9kZSIsImdldE9sZE1vZGUiLCJuZXdNb2RlIiwiZ2V0TmV3TW9kZSIsImF0dHJzIiwiYWN0aW9uSWNvbiIsImFjdGlvblRleHQiLCJoYXNTeW1saW5rIiwiZGV0YWlsIiwidGl0bGUiLCJvbGRTeW1saW5rIiwiZ2V0T2xkU3ltbGluayIsIm5ld1N5bWxpbmsiLCJnZXROZXdTeW1saW5rIiwidG9nZ2xlVmVyYiIsInNlbGVjdGVkSHVua3MiLCJjb250YWluc1NlbGVjdGlvbiIsImhhcyIsImlzU2VsZWN0ZWQiLCJidXR0b25TdWZmaXgiLCJ0b2dnbGVTZWxlY3Rpb25MYWJlbCIsImRpc2NhcmRTZWxlY3Rpb25MYWJlbCIsInN0YXJ0UG9pbnQiLCJzdGFydFJhbmdlIiwia2V5bWFwcyIsInRvZ2dsZUh1bmtTZWxlY3Rpb24iLCJkaXNjYXJkSHVua1NlbGVjdGlvbiIsImRpZE1vdXNlRG93bk9uSGVhZGVyIiwicmFuZ2VzIiwibGluZUNsYXNzIiwicmVmSG9sZGVyIiwibGVuZ3RoIiwiaG9sZGVyIiwicmFuZ2UiLCJyZW5kZXJEZWNvcmF0aW9ucyIsImxheWVyIiwiZ2V0TWFya2VyQ291bnQiLCJ0b2dnbGVSb3dzIiwiY2hhbmdlUm93cyIsImdldENoYW5nZXMiLCJyZWR1Y2UiLCJyb3dzIiwiY2hhbmdlIiwicHVzaCIsImdldEJ1ZmZlclJvd3MiLCJldmVudCIsImhhbmRsZVNlbGVjdGlvbkV2ZW50IiwiYnVmZmVyUm93IiwidW5kZWZpbmVkIiwiaXNOYU4iLCJkb21FdmVudCIsInJhbmdlTGlrZSIsIm9wdHMiLCJidXR0b24iLCJpc1dpbmRvd3MiLCJwcm9jZXNzIiwicGxhdGZvcm0iLCJjdHJsS2V5Iiwib3B0aW9ucyIsImNvbnZlcnRlZCIsImNsaXBCdWZmZXJSYW5nZSIsImdldE9yIiwibWV0YUtleSIsImludGVyc2VjdHMiLCJ3aXRob3V0Iiwic2VsZWN0aW9uIiwiZ2V0U2VsZWN0aW9ucyIsImludGVyc2VjdHNCdWZmZXJSYW5nZSIsInNlbGVjdGlvblJhbmdlIiwiZ2V0QnVmZmVyUmFuZ2UiLCJuZXdSYW5nZXMiLCJudWRnZWQiLCJjb2x1bW4iLCJsYXN0Q29sdW1uIiwibGluZUxlbmd0aEZvclJvdyIsImVuZCIsInNldEJ1ZmZlclJhbmdlIiwibmV3UmFuZ2UiLCJzbGljZSIsImFkZFNlbGVjdGlvbkZvckJ1ZmZlclJhbmdlIiwicmV2ZXJzZWQiLCJpc1JldmVyc2VkIiwicmVwbGFjZW1lbnRSYW5nZXMiLCJlYWNoIiwic2hpZnRLZXkiLCJsYXN0U2VsZWN0aW9uIiwiZ2V0TGFzdFNlbGVjdGlvbiIsImxhc3RTZWxlY3Rpb25SYW5nZSIsImlzQmVmb3JlIiwiaXNMZXNzVGhhbiIsImZhckVkZ2UiLCJnZXRTZWxlY3RlZEh1bmtzIiwid2l0aFNlbGVjdGlvbk1vZGUiLCJodW5rUmFuZ2VzIiwiZmlyc3RDaGFuZ2VSb3ciLCJmaXJzdENoYW5nZSIsImdldFN0YXJ0QnVmZmVyUm93Iiwid2l0aFNlbGVjdGVkSHVua3MiLCJnZXRIdW5rQWZ0ZXIiLCJnZXRIdW5rQmVmb3JlIiwiY3Vyc29yc0J5RmlsZVBhdGNoIiwiTWFwIiwicGxhY2VkUm93cyIsImN1cnNvciIsImdldEN1cnNvcnMiLCJjdXJzb3JSb3ciLCJnZXRCdWZmZXJQb3NpdGlvbiIsImdldEZpbGVQYXRjaEF0IiwibmV3Um93IiwiZ2V0TmV3Um93QXQiLCJuZXdDb2x1bW4iLCJuZWFyZXN0Um93IiwiZ2V0TmV3U3RhcnRSb3ciLCJyZWdpb24iLCJnZXRSZWdpb25zIiwiaW5jbHVkZXNCdWZmZXJSb3ciLCJ3aGVuIiwidW5jaGFuZ2VkIiwiYnVmZmVyUm93Q291bnQiLCJhZGRpdGlvbiIsImN1cnNvcnMiLCJzZXQiLCJmaWxlUGF0Y2hlc1dpdGhDdXJzb3JzIiwia2V5cyIsIm9wZW5GaWxlIiwicGVuZGluZyIsInZhbHVlIiwiZ2V0U2VsZWN0ZWRSb3dzIiwiYWNjIiwiaXNDaGFuZ2VSb3ciLCJvbGRCdWZmZXJSYW5nZSIsIm5ld0J1ZmZlclJhbmdlIiwibmV4dEN1cnNvclJvd3MiLCJnZXRDdXJzb3JCdWZmZXJQb3NpdGlvbnMiLCJzcGFuc011bHRpcGxlRmlsZXMiLCJzZWxlY3RlZFJvd3NDaGFuZ2VkIiwic29mdFdyYXBwZWQiLCJwYWQiLCJvbGRSb3ciLCJnZXRPbGRSb3dBdCIsImNhbGxiYWNrIiwic2VlbiIsImdldFNlbGVjdGVkQnVmZmVyUmFuZ2VzIiwicGF0Y2hlcyIsInBhdGNoIiwicHJldlJvdyIsIm5leHRSb3ciLCJjaGFuZ2VMYXllcnMiLCJzb21lIiwiZmluZE1hcmtlcnMiLCJpbnRlcnNlY3RzUm93IiwiY2FsbGJhY2tzIiwiRXJyb3IiLCJudW0iLCJtYXhEaWdpdHMiLCJnZXRNYXhMaW5lTnVtYmVyV2lkdGgiLCJyZXBlYXQiLCJ0b1N0cmluZyIsImFjdGlvbiIsImdldEVudHJpZXNCeU5hbWUiLCJtZWFzdXJlIiwicGVyZiIsImZpbGVQYXRjaGVzTGluZUNvdW50cyIsImdldENoYW5nZWRMaW5lQ291bnQiLCJkdXJhdGlvbiIsIlByb3BUeXBlcyIsIm9uZU9mIiwiYm9vbCIsIk11bHRpRmlsZVBhdGNoUHJvcFR5cGUiLCJpc1JlcXVpcmVkIiwib2JqZWN0IiwicmVwb3NpdG9yeSIsImZ1bmMiLCJSZWZIb2xkZXJQcm9wVHlwZSIsIkl0ZW1UeXBlUHJvcFR5cGUiLCJEaXNwb3NhYmxlIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7Ozs7Ozs7Ozs7QUFFQSxNQUFNQSxjQUFjLEdBQUc7QUFDckIsR0FBQ0MsY0FBS0MsS0FBTCxDQUFXQyxNQUFaLEdBQXFCLGdCQURBO0FBRXJCLEdBQUNGLGNBQUtDLEtBQUwsQ0FBV0UsVUFBWixHQUF5QjtBQUZKLENBQXZCO0FBS0EsTUFBTUMsY0FBYyxHQUFHLFFBQXZCOztBQUVBLE1BQU1DLFdBQVcsR0FBRyxNQUFNRCxjQUExQjs7QUFFZSxNQUFNRSxrQkFBTixTQUFpQ0MsZUFBTUMsU0FBdkMsQ0FBaUQ7QUEwQzlEQyxFQUFBQSxXQUFXLENBQUNDLEtBQUQsRUFBUTtBQUNqQixVQUFNQSxLQUFOOztBQURpQix3REFrVVUsQ0FBQ0MsU0FBRCxFQUFZQyxLQUFaLEtBQXNCO0FBQ2pELFlBQU1DLFdBQVcsR0FBRyxDQUFDRixTQUFTLENBQUNHLGVBQVYsR0FBNEJDLFNBQTVCLEVBQXJCO0FBQ0EsWUFBTUMsT0FBTyxHQUFHTCxTQUFTLENBQUNNLFNBQVYsR0FBc0JDLFFBQXRCLEdBQWlDRixPQUFqQyxFQUFoQjtBQUNBLFlBQU1HLEtBQUssR0FBR1IsU0FBUyxDQUFDUyxhQUFWLEdBQTBCQyxLQUExQixDQUFnQ0MsT0FBaEMsQ0FBd0MsS0FBS1osS0FBTCxDQUFXYSxjQUFYLENBQTBCQyxTQUExQixHQUFzQ0MsY0FBdEMsRUFBeEMsQ0FBZDtBQUNBLFlBQU1DLFFBQVEsR0FBR1YsT0FBTyxJQUFJRyxLQUFYLEdBQW1CLE9BQW5CLEdBQTZCLFFBQTlDO0FBRUEsYUFDRSw2QkFBQyxlQUFEO0FBQVUsUUFBQSxHQUFHLEVBQUVSLFNBQVMsQ0FBQ2dCLE9BQVY7QUFBZixTQUNFLDZCQUFDLGVBQUQ7QUFBUSxRQUFBLFVBQVUsRUFBQyxPQUFuQjtBQUEyQixRQUFBLFdBQVcsRUFBRWhCLFNBQVMsQ0FBQ1MsYUFBVjtBQUF4QyxTQUNFLDZCQUFDLG1CQUFEO0FBQVksUUFBQSxJQUFJLEVBQUMsT0FBakI7QUFBeUIsUUFBQSxRQUFRLEVBQUVNLFFBQW5DO0FBQTZDLFFBQUEsS0FBSyxFQUFFZCxLQUFwRDtBQUEyRCxRQUFBLFNBQVMsRUFBQztBQUFyRSxTQUNFLDZCQUFDLDRCQUFEO0FBQ0UsUUFBQSxRQUFRLEVBQUUsS0FBS0YsS0FBTCxDQUFXa0IsUUFEdkI7QUFFRSxRQUFBLE9BQU8sRUFBRWpCLFNBQVMsQ0FBQ2dCLE9BQVYsRUFGWDtBQUdFLFFBQUEsT0FBTyxFQUFFaEIsU0FBUyxDQUFDa0IsU0FBVixPQUEwQixTQUExQixHQUFzQ2xCLFNBQVMsQ0FBQ21CLFVBQVYsRUFBdEMsR0FBK0QsSUFIMUU7QUFJRSxRQUFBLGFBQWEsRUFBRSxLQUFLcEIsS0FBTCxDQUFXcUIsYUFKNUI7QUFLRSxRQUFBLGlCQUFpQixFQUFFLEtBQUtyQixLQUFMLENBQVdzQixpQkFMaEM7QUFNRSxRQUFBLGNBQWMsRUFBRSxLQUFLdEIsS0FBTCxDQUFXdUIsY0FON0I7QUFPRSxRQUFBLHlCQUF5QixFQUFFLEtBQUt2QixLQUFMLENBQVd3Qix5QkFQeEM7QUFTRSxRQUFBLFFBQVEsRUFBRSxLQUFLeEIsS0FBTCxDQUFXeUIsUUFUdkI7QUFXRSxRQUFBLGVBQWUsRUFBRSxNQUFNLEtBQUtDLHlCQUFMLENBQStCekIsU0FBL0IsQ0FYekI7QUFZRSxRQUFBLG1CQUFtQixFQUFFLE1BQU0sS0FBS0QsS0FBTCxDQUFXMkIsbUJBQVgsQ0FBK0IxQixTQUEvQixDQVo3QjtBQWFFLFFBQUEsUUFBUSxFQUFFLE1BQU0sS0FBSzJCLFdBQUwsQ0FBaUI7QUFBQ0MsVUFBQUEsaUJBQWlCLEVBQUU1QjtBQUFwQixTQUFqQixDQWJsQjtBQWNFLFFBQUEsVUFBVSxFQUFFLE1BQU0sS0FBS0QsS0FBTCxDQUFXOEIsVUFBWCxDQUFzQjdCLFNBQXRCLENBZHBCO0FBZ0JFLFFBQUEsV0FBVyxFQUFFRSxXQWhCZjtBQWlCRSxRQUFBLGVBQWUsRUFBRSxNQUFNLEtBQUtILEtBQUwsQ0FBV2EsY0FBWCxDQUEwQmtCLGlCQUExQixDQUE0QzlCLFNBQTVDLENBakJ6QjtBQWtCRSxRQUFBLGFBQWEsRUFBRSxNQUFNLEtBQUtELEtBQUwsQ0FBV2EsY0FBWCxDQUEwQm1CLGVBQTFCLENBQTBDL0IsU0FBMUM7QUFsQnZCLFFBREYsRUFxQkcsQ0FBQ0UsV0FBRCxJQUFnQixLQUFLOEIsdUJBQUwsQ0FBNkJoQyxTQUE3QixDQXJCbkIsRUFzQkcsQ0FBQ0UsV0FBRCxJQUFnQixLQUFLK0IsOEJBQUwsQ0FBb0NqQyxTQUFwQyxDQXRCbkIsQ0FERixDQURGLEVBNEJHQSxTQUFTLENBQUNrQyxRQUFWLEdBQXFCL0IsZUFBckIsT0FBMkNnQyxnQkFBM0MsSUFBd0QsS0FBS0MsY0FBTCxDQUFvQnBDLFNBQXBCLEVBQStCZSxRQUEvQixFQUF5Q2QsS0FBekMsQ0E1QjNELEVBOEJHLEtBQUtvQyxpQkFBTCxDQUF1QnJDLFNBQXZCLEVBQWtDQyxLQUFsQyxDQTlCSCxDQURGO0FBa0NELEtBMVdrQjs7QUFBQSx5REE2bkJXLE1BQU07QUFDbEMsVUFBSSxLQUFLRixLQUFMLENBQVd1QixjQUFmLEVBQStCO0FBQzdCLGNBQU1nQixtQkFBbUIsR0FBR0MsS0FBSyxDQUFDQyxJQUFOLENBQVcsS0FBS0Msc0JBQUwsRUFBWCxDQUE1QjtBQUNBOztBQUNBLFlBQUksS0FBSzFDLEtBQUwsQ0FBV2tCLFFBQVgsS0FBd0J5Qix3QkFBNUIsRUFBNkM7QUFDM0MsZUFBSzNDLEtBQUwsQ0FBVzRDLGVBQVgsQ0FBMkJMLG1CQUFtQixDQUFDLENBQUQsQ0FBOUMsRUFBbUQ7QUFBQ00sWUFBQUEsV0FBVyxFQUFFO0FBQUNDLGNBQUFBLE9BQU8sRUFBRTtBQUFWO0FBQWQsV0FBbkQ7QUFDRDtBQUNGO0FBQ0YsS0Fyb0JrQjs7QUFBQSx1REF1b0JTN0MsU0FBUyxJQUFJO0FBQ3ZDLFdBQUtELEtBQUwsQ0FBVzRDLGVBQVgsQ0FBMkIzQyxTQUEzQixFQUFzQztBQUFDNEMsUUFBQUEsV0FBVyxFQUFFO0FBQWQsT0FBdEM7QUFDRCxLQXpvQmtCOztBQUFBLHlEQTJvQlcsTUFBTTtBQUNsQyxhQUFPLEtBQUs3QyxLQUFMLENBQVcrQyxXQUFYLENBQ0wsS0FBSy9DLEtBQUwsQ0FBV2dELFlBRE4sRUFFTCxLQUFLaEQsS0FBTCxDQUFXaUQsYUFGTixFQUdMO0FBQUNKLFFBQUFBLFdBQVcsRUFBRTtBQUFDQyxVQUFBQSxPQUFPLEVBQUU7QUFBVjtBQUFkLE9BSEssQ0FBUDtBQUtELEtBanBCa0I7O0FBQUEsaURBdzJCRyxNQUFNO0FBQzFCLGFBQU9JLE9BQU8sQ0FBQ0MsR0FBUixDQUNMWCxLQUFLLENBQUNDLElBQU4sQ0FBVyxLQUFLQyxzQkFBTCxFQUFYLEVBQ0dVLE1BREgsQ0FDVUMsRUFBRSxJQUFJQSxFQUFFLENBQUNDLHVCQUFILEVBRGhCLEVBRUdDLEdBRkgsQ0FFTyxLQUFLdkQsS0FBTCxDQUFXd0QsZ0JBRmxCLENBREssQ0FBUDtBQUtELEtBOTJCa0I7O0FBQUEsb0RBZzNCTSxNQUFNO0FBQzdCLGFBQU9OLE9BQU8sQ0FBQ0MsR0FBUixDQUNMWCxLQUFLLENBQUNDLElBQU4sQ0FBVyxLQUFLQyxzQkFBTCxFQUFYLEVBQ0dVLE1BREgsQ0FDVUMsRUFBRSxJQUFJQSxFQUFFLENBQUNJLGFBQUgsRUFEaEIsRUFFR0YsR0FGSCxDQUVPLEtBQUt2RCxLQUFMLENBQVcwRCxtQkFGbEIsQ0FESyxDQUFQO0FBS0QsS0F0M0JrQjs7QUFFakIsMkJBQ0UsSUFERixFQUVFLHNCQUZGLEVBRTBCLDBCQUYxQixFQUVzRCwwQkFGdEQsRUFFa0YsWUFGbEYsRUFHRSxZQUhGLEVBR2dCLHdCQUhoQixFQUcwQyxnQkFIMUMsRUFHNEQsb0JBSDVELEVBSUUsYUFKRixFQUlpQixpQkFKakIsRUFJb0MseUJBSnBDLEVBSStELHFCQUovRCxFQUtFLG9CQUxGLEVBS3dCLG9CQUx4QjtBQVFBLFNBQUtDLHdCQUFMLEdBQWdDLEtBQWhDO0FBQ0EsU0FBS0MsaUJBQUwsR0FBeUIsSUFBekI7QUFDQSxTQUFLQyxpQkFBTCxHQUF5QixJQUF6QjtBQUNBLFNBQUtDLE9BQUwsR0FBZSxJQUFJQyxrQkFBSixFQUFmO0FBQ0EsU0FBS0MsU0FBTCxHQUFpQixJQUFJRCxrQkFBSixFQUFqQjtBQUNBLFNBQUtFLGdCQUFMLEdBQXdCLElBQUlGLGtCQUFKLEVBQXhCO0FBQ0EsU0FBS0csT0FBTCxHQUFlLEtBQWY7QUFFQSxTQUFLQyxJQUFMLEdBQVksSUFBSUMsNkJBQUosRUFBWjtBQUVBLFNBQUtELElBQUwsQ0FBVUUsR0FBVixDQUNFLEtBQUtMLFNBQUwsQ0FBZU0sT0FBZixDQUF1QkMsTUFBTSxJQUFJO0FBQy9CLFdBQUtOLGdCQUFMLENBQXNCTyxNQUF0QixDQUE2QkQsTUFBTSxDQUFDRSxVQUFQLEVBQTdCOztBQUNBLFVBQUksS0FBS3pFLEtBQUwsQ0FBV2dFLFNBQWYsRUFBMEI7QUFDeEIsYUFBS2hFLEtBQUwsQ0FBV2dFLFNBQVgsQ0FBcUJRLE1BQXJCLENBQTRCRCxNQUE1QjtBQUNEO0FBQ0YsS0FMRCxDQURGLEVBT0UsS0FBS04sZ0JBQUwsQ0FBc0JLLE9BQXRCLENBQThCSSxPQUFPLElBQUk7QUFDdkMsV0FBSzFFLEtBQUwsQ0FBVzJFLGVBQVgsSUFBOEIsS0FBSzNFLEtBQUwsQ0FBVzJFLGVBQVgsQ0FBMkJILE1BQTNCLENBQWtDRSxPQUFsQyxDQUE5QjtBQUNELEtBRkQsQ0FQRixFQXBCaUIsQ0FnQ2pCOztBQUNBLFNBQUtFLGVBQUwsR0FBdUIsS0FBdkI7QUFDQSxRQUFJQyxhQUFhLEdBQUcsSUFBcEI7QUFDQSxRQUFJQyxjQUFjLEdBQUcsSUFBckI7QUFDQSxRQUFJQyxrQkFBa0IsR0FBRyxJQUF6QjtBQUNBLFNBQUtaLElBQUwsQ0FBVUUsR0FBVixDQUNFLEtBQUtyRSxLQUFMLENBQVdnRixpQkFBWCxDQUE2QixNQUFNO0FBQ2pDLFdBQUtKLGVBQUwsR0FBdUIsSUFBdkI7QUFDQSxXQUFLWixTQUFMLENBQWVULEdBQWYsQ0FBbUJnQixNQUFNLElBQUk7QUFDM0JRLFFBQUFBLGtCQUFrQixHQUFHLEtBQUsvRSxLQUFMLENBQVdhLGNBQVgsQ0FBMEJvRSxvQkFBMUIsQ0FBK0MsS0FBS2pGLEtBQUwsQ0FBV2dELFlBQTFELENBQXJCO0FBQ0E2QixRQUFBQSxhQUFhLEdBQUdOLE1BQU0sQ0FBQ0UsVUFBUCxHQUFvQlMsWUFBcEIsRUFBaEI7QUFDQUosUUFBQUEsY0FBYyxHQUFHUCxNQUFNLENBQUNFLFVBQVAsR0FBb0JVLGFBQXBCLEVBQWpCO0FBQ0EsZUFBTyxJQUFQO0FBQ0QsT0FMRDtBQU1ELEtBUkQsQ0FERixFQVVFLEtBQUtuRixLQUFMLENBQVdvRixnQkFBWCxDQUE0QkMsU0FBUyxJQUFJO0FBQ3ZDLFdBQUtyQixTQUFMLENBQWVULEdBQWYsQ0FBbUJnQixNQUFNLElBQUk7QUFDM0I7QUFDQSxZQUFJUSxrQkFBa0IsS0FBSyxJQUEzQixFQUFpQztBQUMvQixnQkFBTU8sa0JBQWtCLEdBQUdELFNBQVMsQ0FBQ0UseUJBQVYsQ0FBb0NSLGtCQUFwQyxDQUEzQjs7QUFDQSxjQUFJLEtBQUsvRSxLQUFMLENBQVdpRCxhQUFYLEtBQTZCLE1BQWpDLEVBQXlDO0FBQ3ZDLGlCQUFLWSxpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxZQUFBQSxNQUFNLENBQUNpQixzQkFBUCxDQUE4QkYsa0JBQTlCO0FBQ0QsV0FIRCxNQUdPO0FBQ0wsa0JBQU1HLFNBQVMsR0FBRyxJQUFJQyxHQUFKLENBQ2hCQyxZQUFNQyxVQUFOLENBQWlCTixrQkFBakIsRUFBcUNPLE9BQXJDLEdBQ0d0QyxHQURILENBQ091QyxHQUFHLElBQUlULFNBQVMsQ0FBQ1UsU0FBVixDQUFvQkQsR0FBcEIsQ0FEZCxFQUVHMUMsTUFGSCxDQUVVNEMsT0FGVixDQURnQixDQUFsQjtBQUtFOztBQUNGLGtCQUFNQyxVQUFVLEdBQUdSLFNBQVMsQ0FBQ1MsSUFBVixHQUFpQixDQUFqQixHQUNmMUQsS0FBSyxDQUFDQyxJQUFOLENBQVdnRCxTQUFYLEVBQXNCVSxJQUFJLElBQUlBLElBQUksQ0FBQzNGLFFBQUwsRUFBOUIsQ0FEZSxHQUVmLENBQUMsQ0FBQyxDQUFDLENBQUQsRUFBSSxDQUFKLENBQUQsRUFBUyxDQUFDLENBQUQsRUFBSSxDQUFKLENBQVQsQ0FBRCxDQUZKO0FBSUEsaUJBQUtxRCxpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxZQUFBQSxNQUFNLENBQUM2Qix1QkFBUCxDQUErQkgsVUFBL0I7QUFDRDtBQUNGO0FBRUQ7OztBQUNBLFlBQUlwQixhQUFhLEtBQUssSUFBdEIsRUFBNEI7QUFBRU4sVUFBQUEsTUFBTSxDQUFDRSxVQUFQLEdBQW9CNEIsWUFBcEIsQ0FBaUN4QixhQUFqQztBQUFrRDtBQUVoRjs7O0FBQ0EsWUFBSUMsY0FBYyxLQUFLLElBQXZCLEVBQTZCO0FBQUVQLFVBQUFBLE1BQU0sQ0FBQ0UsVUFBUCxHQUFvQjZCLGFBQXBCLENBQWtDeEIsY0FBbEM7QUFBb0Q7O0FBQ25GLGVBQU8sSUFBUDtBQUNELE9BN0JEO0FBOEJBLFdBQUtGLGVBQUwsR0FBdUIsS0FBdkI7QUFDQSxXQUFLMkIscUJBQUw7QUFDRCxLQWpDRCxDQVZGO0FBNkNEOztBQUVEQyxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixTQUFLdEMsT0FBTCxHQUFlLElBQWY7QUFDQSxTQUFLdUMsa0JBQUwsQ0FBd0IsT0FBeEI7QUFFQUMsSUFBQUEsTUFBTSxDQUFDQyxnQkFBUCxDQUF3QixTQUF4QixFQUFtQyxLQUFLQyxVQUF4QztBQUNBLFNBQUs1QyxTQUFMLENBQWVULEdBQWYsQ0FBbUJnQixNQUFNLElBQUk7QUFDM0I7QUFDQSxZQUFNLENBQUNzQyxVQUFELElBQWUsS0FBSzdHLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQmlHLGNBQTFCLEVBQXJCO0FBQ0EsWUFBTSxDQUFDQyxTQUFELElBQWNGLFVBQVUsQ0FBQ0csUUFBWCxFQUFwQjs7QUFDQSxVQUFJLENBQUNELFNBQUwsRUFBZ0I7QUFDZCxlQUFPLElBQVA7QUFDRDs7QUFFRCxXQUFLbEQsaUJBQUwsR0FBeUIsTUFBekI7QUFDQVUsTUFBQUEsTUFBTSxDQUFDaUIsc0JBQVAsQ0FBOEJ1QixTQUFTLENBQUN2RyxRQUFWLEVBQTlCO0FBQ0EsYUFBTyxJQUFQO0FBQ0QsS0FYRDtBQWFBLFNBQUsyRCxJQUFMLENBQVVFLEdBQVYsQ0FDRSxLQUFLckUsS0FBTCxDQUFXaUgsTUFBWCxDQUFrQkMsV0FBbEIsQ0FBOEIsMkJBQTlCLEVBQTJELE1BQU0sS0FBS0MsV0FBTCxFQUFqRSxDQURGO0FBR0Q7O0FBRURDLEVBQUFBLGtCQUFrQixDQUFDQyxTQUFELEVBQVk7QUFDNUIsU0FBS1osa0JBQUwsQ0FBd0IsUUFBeEI7O0FBRUEsUUFBSVksU0FBUyxDQUFDMUMsZUFBVixLQUE4QixLQUFLM0UsS0FBTCxDQUFXMkUsZUFBN0MsRUFBOEQ7QUFDNUQwQyxNQUFBQSxTQUFTLENBQUMxQyxlQUFWLElBQTZCMEMsU0FBUyxDQUFDMUMsZUFBVixDQUEwQkgsTUFBMUIsQ0FBaUMsSUFBakMsQ0FBN0I7QUFDQSxXQUFLeEUsS0FBTCxDQUFXMkUsZUFBWCxJQUE4QixLQUFLVixnQkFBTCxDQUFzQlYsR0FBdEIsQ0FBMEIsS0FBS3ZELEtBQUwsQ0FBVzJFLGVBQVgsQ0FBMkJILE1BQXJELENBQTlCO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLeEUsS0FBTCxDQUFXYSxjQUFYLEtBQThCd0csU0FBUyxDQUFDeEcsY0FBNUMsRUFBNEQ7QUFDMUQsV0FBS2dELGlCQUFMLEdBQXlCLElBQXpCO0FBQ0Q7QUFDRjs7QUFFRHlELEVBQUFBLG9CQUFvQixHQUFHO0FBQ3JCWixJQUFBQSxNQUFNLENBQUNhLG1CQUFQLENBQTJCLFNBQTNCLEVBQXNDLEtBQUtYLFVBQTNDO0FBQ0EsU0FBS3pDLElBQUwsQ0FBVXFELE9BQVY7QUFDQSxTQUFLdEQsT0FBTCxHQUFlLEtBQWY7QUFDQXVELElBQUFBLFdBQVcsQ0FBQ0MsVUFBWjtBQUNBRCxJQUFBQSxXQUFXLENBQUNFLGFBQVo7QUFDRDs7QUFFREMsRUFBQUEsTUFBTSxHQUFHO0FBQ1AsVUFBTUMsU0FBUyxHQUFHLHlCQUNoQixzQkFEZ0IsRUFFaEI7QUFBQyxPQUFFLHlCQUF3QixLQUFLN0gsS0FBTCxDQUFXcUIsYUFBYyxFQUFuRCxHQUF1RCxLQUFLckIsS0FBTCxDQUFXcUI7QUFBbkUsS0FGZ0IsRUFHaEI7QUFBQyxxQ0FBK0IsQ0FBQyxLQUFLckIsS0FBTCxDQUFXYSxjQUFYLENBQTBCaUgsVUFBMUI7QUFBakMsS0FIZ0IsRUFJaEI7QUFBQyx3Q0FBa0MsS0FBSzlILEtBQUwsQ0FBV2lELGFBQVgsS0FBNkI7QUFBaEUsS0FKZ0IsQ0FBbEI7O0FBT0EsUUFBSSxLQUFLaUIsT0FBVCxFQUFrQjtBQUNoQnVELE1BQUFBLFdBQVcsQ0FBQ00sSUFBWixDQUFpQixpQ0FBakI7QUFDRCxLQUZELE1BRU87QUFDTE4sTUFBQUEsV0FBVyxDQUFDTSxJQUFaLENBQWlCLGdDQUFqQjtBQUNEOztBQUVELFdBQ0U7QUFBSyxNQUFBLFNBQVMsRUFBRUYsU0FBaEI7QUFBMkIsTUFBQSxHQUFHLEVBQUUsS0FBSy9ELE9BQUwsQ0FBYVU7QUFBN0MsT0FDRyxLQUFLd0QsY0FBTCxFQURILEVBR0U7QUFBTSxNQUFBLFNBQVMsRUFBQztBQUFoQixPQUNHLEtBQUtoSSxLQUFMLENBQVdhLGNBQVgsQ0FBMEJpSCxVQUExQixLQUF5QyxLQUFLRyxtQkFBTCxFQUF6QyxHQUFzRSxLQUFLQyxnQkFBTCxFQUR6RSxDQUhGLENBREY7QUFTRDs7QUFFREYsRUFBQUEsY0FBYyxHQUFHO0FBQ2YsUUFBSSxLQUFLaEksS0FBTCxDQUFXa0IsUUFBWCxLQUF3QmlILHlCQUF4QixJQUE0QyxLQUFLbkksS0FBTCxDQUFXa0IsUUFBWCxLQUF3QmtILDJCQUF4RSxFQUE0RjtBQUMxRixhQUNFLDZCQUFDLGlCQUFEO0FBQVUsUUFBQSxRQUFRLEVBQUUsS0FBS3BJLEtBQUwsQ0FBV3FJLFFBQS9CO0FBQXlDLFFBQUEsTUFBTSxFQUFFLEtBQUt2RTtBQUF0RCxTQUNFLDZCQUFDLGlCQUFEO0FBQVMsUUFBQSxPQUFPLEVBQUMseUJBQWpCO0FBQTJDLFFBQUEsUUFBUSxFQUFFLEtBQUt3RTtBQUExRCxRQURGLEVBRUUsNkJBQUMsaUJBQUQ7QUFBUyxRQUFBLE9BQU8sRUFBQyw2QkFBakI7QUFBK0MsUUFBQSxRQUFRLEVBQUUsS0FBS0M7QUFBOUQsUUFGRixFQUdFLDZCQUFDLGlCQUFEO0FBQVMsUUFBQSxPQUFPLEVBQUMsb0NBQWpCO0FBQXNELFFBQUEsUUFBUSxFQUFFLEtBQUtDO0FBQXJFLFFBSEYsQ0FERjtBQU9EOztBQUVELFFBQUlDLGdCQUFnQixHQUFHLElBQXZCO0FBQ0EsUUFBSUMsbUJBQW1CLEdBQUcsSUFBMUI7O0FBRUEsUUFBSSxLQUFLMUksS0FBTCxDQUFXYSxjQUFYLENBQTBCOEgsMEJBQTFCLEVBQUosRUFBNEQ7QUFDMUQsWUFBTTdGLE9BQU8sR0FBRyxLQUFLOUMsS0FBTCxDQUFXcUIsYUFBWCxLQUE2QixVQUE3QixHQUNaLCtCQURZLEdBRVosaUNBRko7QUFHQW9ILE1BQUFBLGdCQUFnQixHQUFHLDZCQUFDLGlCQUFEO0FBQVMsUUFBQSxPQUFPLEVBQUUzRixPQUFsQjtBQUEyQixRQUFBLFFBQVEsRUFBRSxLQUFLOEY7QUFBMUMsUUFBbkI7QUFDRDs7QUFFRCxRQUFJLEtBQUs1SSxLQUFMLENBQVdhLGNBQVgsQ0FBMEJnSSxpQkFBMUIsRUFBSixFQUFtRDtBQUNqRCxZQUFNL0YsT0FBTyxHQUFHLEtBQUs5QyxLQUFMLENBQVdxQixhQUFYLEtBQTZCLFVBQTdCLEdBQ1osNkJBRFksR0FFWiwrQkFGSjtBQUdBcUgsTUFBQUEsbUJBQW1CLEdBQUcsNkJBQUMsaUJBQUQ7QUFBUyxRQUFBLE9BQU8sRUFBRTVGLE9BQWxCO0FBQTJCLFFBQUEsUUFBUSxFQUFFLEtBQUtnRztBQUExQyxRQUF0QjtBQUNEOztBQUVELFdBQ0UsNkJBQUMsaUJBQUQ7QUFBVSxNQUFBLFFBQVEsRUFBRSxLQUFLOUksS0FBTCxDQUFXcUksUUFBL0I7QUFBeUMsTUFBQSxNQUFNLEVBQUUsS0FBS3ZFO0FBQXRELE9BQ0UsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQyx5QkFBakI7QUFBMkMsTUFBQSxRQUFRLEVBQUUsS0FBS3dFO0FBQTFELE1BREYsRUFFRSw2QkFBQyxpQkFBRDtBQUFTLE1BQUEsT0FBTyxFQUFDLDZCQUFqQjtBQUErQyxNQUFBLFFBQVEsRUFBRSxLQUFLQztBQUE5RCxNQUZGLEVBR0UsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQyxjQUFqQjtBQUFnQyxNQUFBLFFBQVEsRUFBRSxLQUFLUTtBQUEvQyxNQUhGLEVBSUUsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQyxXQUFqQjtBQUE2QixNQUFBLFFBQVEsRUFBRSxLQUFLQztBQUE1QyxNQUpGLEVBS0UsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQywrQkFBakI7QUFBaUQsTUFBQSxRQUFRLEVBQUUsS0FBS0M7QUFBaEUsTUFMRixFQU1FLDZCQUFDLGlCQUFEO0FBQVMsTUFBQSxPQUFPLEVBQUMscUJBQWpCO0FBQXVDLE1BQUEsUUFBUSxFQUFFLEtBQUtySDtBQUF0RCxNQU5GLEVBT0UsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQyxnQkFBakI7QUFBa0MsTUFBQSxRQUFRLEVBQUUsS0FBSzVCLEtBQUwsQ0FBV2tKO0FBQXZELE1BUEYsRUFRRSw2QkFBQyxpQkFBRDtBQUFTLE1BQUEsT0FBTyxFQUFDLG9DQUFqQjtBQUFzRCxNQUFBLFFBQVEsRUFBRSxLQUFLVjtBQUFyRSxNQVJGLEVBU0dDLGdCQVRILEVBVUdDLG1CQVZIO0FBV0c7QUFBMkJTLElBQUFBLElBQUksQ0FBQ0MsU0FBTCxNQUMxQiw2QkFBQyxpQkFBRDtBQUFTLE1BQUEsT0FBTyxFQUFDLHNCQUFqQjtBQUF3QyxNQUFBLFFBQVEsRUFBRSxNQUFNO0FBQ3REO0FBQ0FDLFFBQUFBLE9BQU8sQ0FBQ0MsR0FBUixDQUFZLEtBQUt0SixLQUFMLENBQVdhLGNBQVgsQ0FBMEIwSSxjQUExQixHQUEyQ0MsT0FBM0MsQ0FBbUQ7QUFDN0RDLFVBQUFBLFVBQVUsRUFBRSxDQUFDLE9BQUQsRUFBVSxNQUFWO0FBRGlELFNBQW5ELENBQVo7QUFHRDtBQUxELE1BWko7QUFvQkc7QUFBMkJOLElBQUFBLElBQUksQ0FBQ0MsU0FBTCxNQUMxQiw2QkFBQyxpQkFBRDtBQUFTLE1BQUEsT0FBTyxFQUFDLHdCQUFqQjtBQUEwQyxNQUFBLFFBQVEsRUFBRSxNQUFNO0FBQ3hEO0FBQ0FDLFFBQUFBLE9BQU8sQ0FBQ0MsR0FBUixDQUFZLEtBQUt0SixLQUFMLENBQVdhLGNBQVgsQ0FBMEIwSSxjQUExQixHQUEyQ0MsT0FBM0MsQ0FBbUQ7QUFDN0RDLFVBQUFBLFVBQVUsRUFBRSxDQUFDLFdBQUQsRUFBYyxVQUFkLEVBQTBCLFVBQTFCLEVBQXNDLFdBQXRDO0FBRGlELFNBQW5ELENBQVo7QUFHRDtBQUxELE1BckJKO0FBNkJHO0FBQTJCTixJQUFBQSxJQUFJLENBQUNDLFNBQUwsTUFDMUIsNkJBQUMsaUJBQUQ7QUFBUyxNQUFBLE9BQU8sRUFBQyxvQkFBakI7QUFBc0MsTUFBQSxRQUFRLEVBQUUsTUFBTTtBQUNwRDtBQUNBQyxRQUFBQSxPQUFPLENBQUNDLEdBQVIsQ0FBWSxLQUFLdEosS0FBTCxDQUFXYSxjQUFYLENBQTBCMkksT0FBMUIsRUFBWjtBQUNEO0FBSEQsTUE5QkosQ0FERjtBQXVDRDs7QUFFRHRCLEVBQUFBLGdCQUFnQixHQUFHO0FBQ2pCLFdBQU87QUFBRyxNQUFBLFNBQVMsRUFBQztBQUFiLCtCQUFQO0FBQ0Q7O0FBRURELEVBQUFBLG1CQUFtQixHQUFHO0FBQ3BCLFdBQ0UsNkJBQUMsdUJBQUQ7QUFDRSxNQUFBLFNBQVMsRUFBRSxLQUFLakksS0FBTCxDQUFXMEosU0FEeEI7QUFHRSxNQUFBLE1BQU0sRUFBRSxLQUFLMUosS0FBTCxDQUFXYSxjQUFYLENBQTBCQyxTQUExQixFQUhWO0FBSUUsTUFBQSx1QkFBdUIsRUFBRSxLQUozQjtBQUtFLE1BQUEsU0FBUyxFQUFFLEtBTGI7QUFNRSxNQUFBLFVBQVUsRUFBRSxLQU5kO0FBT0UsTUFBQSxRQUFRLEVBQUUsSUFQWjtBQVFFLE1BQUEsV0FBVyxFQUFFLElBUmY7QUFVRSxNQUFBLGVBQWUsRUFBRSxLQUFLNkksZUFWeEI7QUFXRSxNQUFBLHVCQUF1QixFQUFFLEtBQUtDLHVCQVhoQztBQVlFLE1BQUEsbUJBQW1CLEVBQUUsS0FBS0MsbUJBWjVCO0FBYUUsTUFBQSxRQUFRLEVBQUUsS0FBSzdGLFNBYmpCO0FBY0UsTUFBQSxhQUFhLEVBQUU7QUFkakIsT0FnQkUsNkJBQUMsZUFBRDtBQUNFLE1BQUEsSUFBSSxFQUFDLGtCQURQO0FBRUUsTUFBQSxRQUFRLEVBQUUsQ0FGWjtBQUdFLE1BQUEsU0FBUyxFQUFDLEtBSFo7QUFJRSxNQUFBLElBQUksRUFBQyxhQUpQO0FBS0UsTUFBQSxPQUFPLEVBQUUsS0FBSzhGLGtCQUxoQjtBQU1FLE1BQUEsV0FBVyxFQUFFLEtBQUtDLHdCQU5wQjtBQU9FLE1BQUEsV0FBVyxFQUFFLEtBQUtDO0FBUHBCLE1BaEJGLEVBeUJFLDZCQUFDLGVBQUQ7QUFDRSxNQUFBLElBQUksRUFBQyxrQkFEUDtBQUVFLE1BQUEsUUFBUSxFQUFFLENBRlo7QUFHRSxNQUFBLFNBQVMsRUFBQyxLQUhaO0FBSUUsTUFBQSxJQUFJLEVBQUMsYUFKUDtBQUtFLE1BQUEsT0FBTyxFQUFFLEtBQUtDLGtCQUxoQjtBQU1FLE1BQUEsV0FBVyxFQUFFLEtBQUtGLHdCQU5wQjtBQU9FLE1BQUEsV0FBVyxFQUFFLEtBQUtDO0FBUHBCLE1BekJGLEVBa0NHLEtBQUtoSyxLQUFMLENBQVdpSCxNQUFYLENBQWtCaUQsR0FBbEIsQ0FBc0IsMkJBQXRCLEtBQ0MsNkJBQUMsZUFBRDtBQUNFLE1BQUEsSUFBSSxFQUFDLFlBRFA7QUFFRSxNQUFBLFFBQVEsRUFBRSxDQUZaO0FBR0UsTUFBQSxJQUFJLEVBQUMsYUFIUDtBQUlFLE1BQUEsU0FBUyxFQUFDLE9BSlo7QUFLRSxNQUFBLE9BQU8sRUFBRXZLLFdBTFg7QUFNRSxNQUFBLFdBQVcsRUFBRSxLQUFLb0ssd0JBTnBCO0FBT0UsTUFBQSxXQUFXLEVBQUUsS0FBS0M7QUFQcEIsTUFuQ0osRUE4Q0csS0FBS0csd0JBQUwsRUE5Q0gsRUFnREcsS0FBS25LLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQmlHLGNBQTFCLEdBQTJDdkQsR0FBM0MsQ0FBK0MsS0FBSzZHLDBCQUFwRCxDQWhESCxFQWtERyxLQUFLQyxxQkFBTCxDQUNDN0gsS0FBSyxDQUFDQyxJQUFOLENBQVcsS0FBS3pDLEtBQUwsQ0FBV2dELFlBQXRCLEVBQW9DOEMsR0FBRyxJQUFJSCxZQUFNQyxVQUFOLENBQWlCLENBQUMsQ0FBQ0UsR0FBRCxFQUFNLENBQU4sQ0FBRCxFQUFXLENBQUNBLEdBQUQsRUFBTXdFLFFBQU4sQ0FBWCxDQUFqQixDQUEzQyxDQURELEVBRUMscUNBRkQsRUFHQztBQUFDQyxNQUFBQSxNQUFNLEVBQUUsSUFBVDtBQUFlQyxNQUFBQSxJQUFJLEVBQUUsSUFBckI7QUFBMkJDLE1BQUFBLElBQUksRUFBRTtBQUFqQyxLQUhELENBbERILEVBd0RHLEtBQUtDLHdCQUFMLENBQ0MsS0FBSzFLLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQjhKLGdCQUExQixFQURELEVBRUMsa0NBRkQsRUFHQztBQUFDSCxNQUFBQSxJQUFJLEVBQUUsSUFBUDtBQUFhQyxNQUFBQSxJQUFJLEVBQUU7QUFBbkIsS0FIRCxDQXhESCxFQTZERyxLQUFLQyx3QkFBTCxDQUNDLEtBQUsxSyxLQUFMLENBQVdhLGNBQVgsQ0FBMEIrSixnQkFBMUIsRUFERCxFQUVDLG9DQUZELEVBR0M7QUFBQ0osTUFBQUEsSUFBSSxFQUFFLElBQVA7QUFBYUMsTUFBQUEsSUFBSSxFQUFFO0FBQW5CLEtBSEQsQ0E3REgsRUFrRUcsS0FBS0Msd0JBQUwsQ0FDQyxLQUFLMUssS0FBTCxDQUFXYSxjQUFYLENBQTBCZ0ssaUJBQTFCLEVBREQsRUFFQyxzQ0FGRCxFQUdDO0FBQUNMLE1BQUFBLElBQUksRUFBRSxJQUFQO0FBQWFDLE1BQUFBLElBQUksRUFBRTtBQUFuQixLQUhELENBbEVILENBREY7QUEyRUQ7O0FBRUROLEVBQUFBLHdCQUF3QixHQUFHO0FBQ3pCLFFBQUksS0FBS25LLEtBQUwsQ0FBV2tCLFFBQVgsS0FBd0JrSCwyQkFBNUIsRUFBZ0Q7QUFDOUM7QUFDQTtBQUNBLGFBQ0UsNkJBQUMsMkJBQUQ7QUFDRSxRQUFBLFdBQVcsRUFBRSxLQUFLcEksS0FBTCxDQUFXOEssV0FEMUI7QUFFRSxRQUFBLDJCQUEyQixFQUFFLEtBQUs5SyxLQUFMLENBQVdhLGNBQVgsQ0FBMEJrSywyQkFGekQ7QUFHRSxRQUFBLGNBQWMsRUFBRSxLQUFLL0ssS0FBTCxDQUFXYSxjQUFYLENBQTBCbUssY0FINUM7QUFJRSxRQUFBLGdCQUFnQixFQUFFLEtBQUtoTCxLQUFMLENBQVdpTCxnQkFKL0I7QUFLRSxRQUFBLGFBQWEsRUFBRTtBQUxqQixRQURGO0FBU0QsS0FaRCxNQVlPO0FBQ0wsYUFBTyxJQUFQO0FBQ0Q7QUFDRjs7QUE0Q0Q1SSxFQUFBQSxjQUFjLENBQUNwQyxTQUFELEVBQVllLFFBQVosRUFBc0JrSyxXQUF0QixFQUFtQztBQUMvQyxVQUFNQyxRQUFRLEdBQUcsTUFBTTtBQUNyQixtQ0FBUyxtQkFBVCxFQUE4QjtBQUFDQyxRQUFBQSxTQUFTLEVBQUUsS0FBS3JMLFdBQUwsQ0FBaUJzTCxJQUE3QjtBQUFtQ0MsUUFBQUEsT0FBTyxFQUFFO0FBQTVDLE9BQTlCO0FBQ0EsV0FBS3RMLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQm1CLGVBQTFCLENBQTBDL0IsU0FBMUM7QUFDRCxLQUhEOztBQUlBLFdBQ0UsNkJBQUMsZUFBRDtBQUFRLE1BQUEsVUFBVSxFQUFDLE9BQW5CO0FBQTJCLE1BQUEsV0FBVyxFQUFFQSxTQUFTLENBQUNTLGFBQVY7QUFBeEMsT0FDRSw2QkFBQyxtQkFBRDtBQUNFLE1BQUEsSUFBSSxFQUFDLE9BRFA7QUFFRSxNQUFBLEtBQUssRUFBRXdLLFdBQVcsR0FBRyxHQUZ2QjtBQUdFLE1BQUEsUUFBUSxFQUFFbEssUUFIWjtBQUlFLE1BQUEsU0FBUyxFQUFDO0FBSlosT0FNRTtBQUFHLE1BQUEsU0FBUyxFQUFDO0FBQWIsd0VBRUUsd0NBRkYsRUFHRTtBQUFRLE1BQUEsU0FBUyxFQUFDLHFDQUFsQjtBQUF3RCxNQUFBLE9BQU8sRUFBRW1LO0FBQWpFLG9CQUhGLENBTkYsQ0FERixDQURGO0FBaUJEOztBQUVEakosRUFBQUEsOEJBQThCLENBQUNqQyxTQUFELEVBQVk7QUFDeEMsUUFBSSxDQUFDQSxTQUFTLENBQUNxRCx1QkFBVixFQUFMLEVBQTBDO0FBQ3hDLGFBQU8sSUFBUDtBQUNEOztBQUVELFVBQU1pSSxPQUFPLEdBQUd0TCxTQUFTLENBQUN1TCxVQUFWLEVBQWhCO0FBQ0EsVUFBTUMsT0FBTyxHQUFHeEwsU0FBUyxDQUFDeUwsVUFBVixFQUFoQjtBQUVBLFVBQU1DLEtBQUssR0FBRyxLQUFLM0wsS0FBTCxDQUFXcUIsYUFBWCxLQUE2QixVQUE3QixHQUNWO0FBQ0F1SyxNQUFBQSxVQUFVLEVBQUUsZ0JBRFo7QUFFQUMsTUFBQUEsVUFBVSxFQUFFO0FBRlosS0FEVSxHQUtWO0FBQ0FELE1BQUFBLFVBQVUsRUFBRSxjQURaO0FBRUFDLE1BQUFBLFVBQVUsRUFBRTtBQUZaLEtBTEo7QUFVQSxXQUNFLDZCQUFDLDBCQUFEO0FBQ0UsTUFBQSxLQUFLLEVBQUMsYUFEUjtBQUVFLE1BQUEsVUFBVSxFQUFFRixLQUFLLENBQUNDLFVBRnBCO0FBR0UsTUFBQSxVQUFVLEVBQUVELEtBQUssQ0FBQ0UsVUFIcEI7QUFJRSxNQUFBLFFBQVEsRUFBRSxLQUFLN0wsS0FBTCxDQUFXa0IsUUFKdkI7QUFLRSxNQUFBLE1BQU0sRUFBRSxNQUFNLEtBQUtsQixLQUFMLENBQVd3RCxnQkFBWCxDQUE0QnZELFNBQTVCO0FBTGhCLE9BTUUsNkJBQUMsZUFBRCw2QkFFRTtBQUFNLE1BQUEsU0FBUyxFQUFDO0FBQWhCLGdCQUNRWixjQUFjLENBQUNrTSxPQUFELENBRHRCLE9BQ2lDLDJDQUFPQSxPQUFQLENBRGpDLENBRkYsRUFLRTtBQUFNLE1BQUEsU0FBUyxFQUFDO0FBQWhCLGNBQ01sTSxjQUFjLENBQUNvTSxPQUFELENBRHBCLE9BQytCLDJDQUFPQSxPQUFQLENBRC9CLENBTEYsQ0FORixDQURGO0FBa0JEOztBQUVEeEosRUFBQUEsdUJBQXVCLENBQUNoQyxTQUFELEVBQVk7QUFDakMsUUFBSSxDQUFDQSxTQUFTLENBQUM2TCxVQUFWLEVBQUwsRUFBNkI7QUFDM0IsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQsUUFBSUMsTUFBTSxHQUFHLHlDQUFiOztBQUNBLFFBQUlDLEtBQUssR0FBRyxFQUFaO0FBQ0EsVUFBTUMsVUFBVSxHQUFHaE0sU0FBUyxDQUFDaU0sYUFBVixFQUFuQjtBQUNBLFVBQU1DLFVBQVUsR0FBR2xNLFNBQVMsQ0FBQ21NLGFBQVYsRUFBbkI7O0FBQ0EsUUFBSUgsVUFBVSxJQUFJRSxVQUFsQixFQUE4QjtBQUM1QkosTUFBQUEsTUFBTSxHQUNKLDZCQUFDLGVBQUQsMkJBRUU7QUFBTSxRQUFBLFNBQVMsRUFBRSx5QkFDZiwrQkFEZSxFQUVmLDBDQUZlLEVBR2Ysd0NBSGU7QUFBakIsa0JBS08sMkNBQU9FLFVBQVAsQ0FMUCxDQUZGLEVBU0U7QUFBTSxRQUFBLFNBQVMsRUFBRSx5QkFDZiwrQkFEZSxFQUVmLDBDQUZlLEVBR2Ysc0NBSGU7QUFBakIsZ0JBS0ssMkNBQU9FLFVBQVAsQ0FMTCxDQVRGLE1BREY7QUFtQkFILE1BQUFBLEtBQUssR0FBRyxpQkFBUjtBQUNELEtBckJELE1BcUJPLElBQUlDLFVBQVUsSUFBSSxDQUFDRSxVQUFuQixFQUErQjtBQUNwQ0osTUFBQUEsTUFBTSxHQUNKLDZCQUFDLGVBQUQsbUJBRUU7QUFBTSxRQUFBLFNBQVMsRUFBQztBQUFoQixnQkFDSywyQ0FBT0UsVUFBUCxDQURMLENBRkYsYUFERjtBQVNBRCxNQUFBQSxLQUFLLEdBQUcsaUJBQVI7QUFDRCxLQVhNLE1BV0E7QUFDTEQsTUFBQUEsTUFBTSxHQUNKLDZCQUFDLGVBQUQsbUJBRUU7QUFBTSxRQUFBLFNBQVMsRUFBQztBQUFoQixnQkFDSywyQ0FBT0ksVUFBUCxDQURMLENBRkYsYUFERjtBQVNBSCxNQUFBQSxLQUFLLEdBQUcsaUJBQVI7QUFDRDs7QUFFRCxVQUFNTCxLQUFLLEdBQUcsS0FBSzNMLEtBQUwsQ0FBV3FCLGFBQVgsS0FBNkIsVUFBN0IsR0FDVjtBQUNBdUssTUFBQUEsVUFBVSxFQUFFLGdCQURaO0FBRUFDLE1BQUFBLFVBQVUsRUFBRTtBQUZaLEtBRFUsR0FLVjtBQUNBRCxNQUFBQSxVQUFVLEVBQUUsY0FEWjtBQUVBQyxNQUFBQSxVQUFVLEVBQUU7QUFGWixLQUxKO0FBVUEsV0FDRSw2QkFBQywwQkFBRDtBQUNFLE1BQUEsS0FBSyxFQUFFRyxLQURUO0FBRUUsTUFBQSxVQUFVLEVBQUVMLEtBQUssQ0FBQ0MsVUFGcEI7QUFHRSxNQUFBLFVBQVUsRUFBRUQsS0FBSyxDQUFDRSxVQUhwQjtBQUlFLE1BQUEsUUFBUSxFQUFFLEtBQUs3TCxLQUFMLENBQVdrQixRQUp2QjtBQUtFLE1BQUEsTUFBTSxFQUFFLE1BQU0sS0FBS2xCLEtBQUwsQ0FBVzBELG1CQUFYLENBQStCekQsU0FBL0I7QUFMaEIsT0FNRSw2QkFBQyxlQUFELFFBQ0c4TCxNQURILENBTkYsQ0FERjtBQVlEOztBQUVEekosRUFBQUEsaUJBQWlCLENBQUNyQyxTQUFELEVBQVlpTCxXQUFaLEVBQXlCO0FBQ3hDLFVBQU1tQixVQUFVLEdBQUcsS0FBS3JNLEtBQUwsQ0FBV3FCLGFBQVgsS0FBNkIsVUFBN0IsR0FBMEMsT0FBMUMsR0FBb0QsU0FBdkU7QUFDQSxVQUFNaUwsYUFBYSxHQUFHLElBQUk1RyxHQUFKLENBQ3BCbEQsS0FBSyxDQUFDQyxJQUFOLENBQVcsS0FBS3pDLEtBQUwsQ0FBV2dELFlBQXRCLEVBQW9DOEMsR0FBRyxJQUFJLEtBQUs5RixLQUFMLENBQVdhLGNBQVgsQ0FBMEJrRixTQUExQixDQUFvQ0QsR0FBcEMsQ0FBM0MsQ0FEb0IsQ0FBdEI7QUFJQSxXQUNFLDZCQUFDLGVBQUQsUUFDRSw2QkFBQyxvQkFBRCxRQUNHN0YsU0FBUyxDQUFDK0csUUFBVixHQUFxQnpELEdBQXJCLENBQXlCLENBQUM0QyxJQUFELEVBQU9qRyxLQUFQLEtBQWlCO0FBQ3pDLFlBQU1xTSxpQkFBaUIsR0FBRyxLQUFLdk0sS0FBTCxDQUFXaUQsYUFBWCxLQUE2QixNQUE3QixJQUF1Q3FKLGFBQWEsQ0FBQ0UsR0FBZCxDQUFrQnJHLElBQWxCLENBQWpFO0FBQ0EsWUFBTXNHLFVBQVUsR0FBSSxLQUFLek0sS0FBTCxDQUFXaUQsYUFBWCxLQUE2QixNQUE5QixJQUF5Q3FKLGFBQWEsQ0FBQ0UsR0FBZCxDQUFrQnJHLElBQWxCLENBQTVEO0FBRUEsVUFBSXVHLFlBQVksR0FBRyxFQUFuQjs7QUFDQSxVQUFJSCxpQkFBSixFQUF1QjtBQUNyQkcsUUFBQUEsWUFBWSxJQUFJLGVBQWhCOztBQUNBLFlBQUksS0FBSzFNLEtBQUwsQ0FBV2dELFlBQVgsQ0FBd0JrRCxJQUF4QixHQUErQixDQUFuQyxFQUFzQztBQUNwQ3dHLFVBQUFBLFlBQVksSUFBSSxHQUFoQjtBQUNEO0FBQ0YsT0FMRCxNQUtPO0FBQ0xBLFFBQUFBLFlBQVksSUFBSSxNQUFoQjs7QUFDQSxZQUFJSixhQUFhLENBQUNwRyxJQUFkLEdBQXFCLENBQXpCLEVBQTRCO0FBQzFCd0csVUFBQUEsWUFBWSxJQUFJLEdBQWhCO0FBQ0Q7QUFDRjs7QUFFRCxZQUFNQyxvQkFBb0IsR0FBSSxHQUFFTixVQUFXLElBQUdLLFlBQWEsRUFBM0Q7QUFDQSxZQUFNRSxxQkFBcUIsR0FBSSxXQUFVRixZQUFhLEVBQXREO0FBRUEsWUFBTUcsVUFBVSxHQUFHMUcsSUFBSSxDQUFDM0YsUUFBTCxHQUFnQkcsS0FBbkM7QUFDQSxZQUFNbU0sVUFBVSxHQUFHLElBQUluSCxXQUFKLENBQVVrSCxVQUFWLEVBQXNCQSxVQUF0QixDQUFuQjtBQUVBLGFBQ0UsNkJBQUMsZUFBRDtBQUFRLFFBQUEsR0FBRyxFQUFHLGNBQWEzTSxLQUFNLEVBQWpDO0FBQW9DLFFBQUEsV0FBVyxFQUFFNE0sVUFBakQ7QUFBNkQsUUFBQSxVQUFVLEVBQUM7QUFBeEUsU0FDRSw2QkFBQyxtQkFBRDtBQUFZLFFBQUEsSUFBSSxFQUFDLE9BQWpCO0FBQXlCLFFBQUEsS0FBSyxFQUFFNUIsV0FBVyxHQUFHLEdBQTlDO0FBQW1ELFFBQUEsU0FBUyxFQUFDO0FBQTdELFNBQ0UsNkJBQUMsdUJBQUQ7QUFDRSxRQUFBLFNBQVMsRUFBRSxLQUFLakgsZ0JBRGxCO0FBRUUsUUFBQSxJQUFJLEVBQUVrQyxJQUZSO0FBR0UsUUFBQSxVQUFVLEVBQUVzRyxVQUhkO0FBSUUsUUFBQSxhQUFhLEVBQUUsS0FBS3pNLEtBQUwsQ0FBV3FCLGFBSjVCO0FBS0UsUUFBQSxhQUFhLEVBQUMsTUFMaEI7QUFNRSxRQUFBLG9CQUFvQixFQUFFc0wsb0JBTnhCO0FBT0UsUUFBQSxxQkFBcUIsRUFBRUMscUJBUHpCO0FBU0UsUUFBQSxRQUFRLEVBQUUsS0FBSzVNLEtBQUwsQ0FBV3lCLFFBVHZCO0FBVUUsUUFBQSxPQUFPLEVBQUUsS0FBS3pCLEtBQUwsQ0FBVytNLE9BVnRCO0FBWUUsUUFBQSxlQUFlLEVBQUUsTUFBTSxLQUFLQyxtQkFBTCxDQUF5QjdHLElBQXpCLEVBQStCb0csaUJBQS9CLENBWnpCO0FBYUUsUUFBQSxnQkFBZ0IsRUFBRSxNQUFNLEtBQUtVLG9CQUFMLENBQTBCOUcsSUFBMUIsRUFBZ0NvRyxpQkFBaEMsQ0FiMUI7QUFjRSxRQUFBLFNBQVMsRUFBRSxLQUFLVyxvQkFkbEI7QUFlRSxRQUFBLFFBQVEsRUFBRSxLQUFLbE4sS0FBTCxDQUFXa0I7QUFmdkIsUUFERixDQURGLENBREY7QUF1QkQsS0E5Q0EsQ0FESCxDQURGLENBREY7QUFxREQ7O0FBRURtSixFQUFBQSxxQkFBcUIsQ0FBQzhDLE1BQUQsRUFBU0MsU0FBVCxFQUFvQjtBQUFDM0MsSUFBQUEsSUFBRDtBQUFPRixJQUFBQSxNQUFQO0FBQWVDLElBQUFBLElBQWY7QUFBcUI2QyxJQUFBQTtBQUFyQixHQUFwQixFQUFxRDtBQUN4RSxRQUFJRixNQUFNLENBQUNHLE1BQVAsS0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkIsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQsVUFBTUMsTUFBTSxHQUFHRixTQUFTLElBQUksSUFBSXRKLGtCQUFKLEVBQTVCO0FBQ0EsV0FDRSw2QkFBQyxvQkFBRDtBQUFhLE1BQUEsV0FBVyxFQUFFd0osTUFBTSxDQUFDL0k7QUFBakMsT0FDRzJJLE1BQU0sQ0FBQzVKLEdBQVAsQ0FBVyxDQUFDaUssS0FBRCxFQUFRdE4sS0FBUixLQUFrQjtBQUM1QixhQUNFLDZCQUFDLGVBQUQ7QUFDRSxRQUFBLEdBQUcsRUFBRyxRQUFPa04sU0FBVSxJQUFHbE4sS0FBTSxFQURsQztBQUVFLFFBQUEsV0FBVyxFQUFFc04sS0FGZjtBQUdFLFFBQUEsVUFBVSxFQUFDO0FBSGIsUUFERjtBQU9ELEtBUkEsQ0FESCxFQVVHLEtBQUtDLGlCQUFMLENBQXVCTCxTQUF2QixFQUFrQztBQUFDM0MsTUFBQUEsSUFBRDtBQUFPRixNQUFBQSxNQUFQO0FBQWVDLE1BQUFBO0FBQWYsS0FBbEMsQ0FWSCxDQURGO0FBY0Q7O0FBRURFLEVBQUFBLHdCQUF3QixDQUFDZ0QsS0FBRCxFQUFRTixTQUFSLEVBQW1CO0FBQUMzQyxJQUFBQSxJQUFEO0FBQU9GLElBQUFBLE1BQVA7QUFBZUMsSUFBQUE7QUFBZixHQUFuQixFQUF5QztBQUMvRCxRQUFJa0QsS0FBSyxDQUFDQyxjQUFOLE9BQTJCLENBQS9CLEVBQWtDO0FBQ2hDLGFBQU8sSUFBUDtBQUNEOztBQUVELFdBQ0UsNkJBQUMsb0JBQUQ7QUFBYSxNQUFBLFFBQVEsRUFBRUQ7QUFBdkIsT0FDRyxLQUFLRCxpQkFBTCxDQUF1QkwsU0FBdkIsRUFBa0M7QUFBQzNDLE1BQUFBLElBQUQ7QUFBT0YsTUFBQUEsTUFBUDtBQUFlQyxNQUFBQTtBQUFmLEtBQWxDLENBREgsQ0FERjtBQUtEOztBQUVEaUQsRUFBQUEsaUJBQWlCLENBQUNMLFNBQUQsRUFBWTtBQUFDM0MsSUFBQUEsSUFBRDtBQUFPRixJQUFBQSxNQUFQO0FBQWVDLElBQUFBO0FBQWYsR0FBWixFQUFrQztBQUNqRCxXQUNFLDZCQUFDLGVBQUQsUUFDR0MsSUFBSSxJQUNILDZCQUFDLG1CQUFEO0FBQ0UsTUFBQSxJQUFJLEVBQUMsTUFEUDtBQUVFLE1BQUEsU0FBUyxFQUFFMkMsU0FGYjtBQUdFLE1BQUEsZ0JBQWdCLEVBQUU7QUFIcEIsTUFGSixFQVFHN0MsTUFBTSxJQUNMLDZCQUFDLGVBQUQsUUFDRSw2QkFBQyxtQkFBRDtBQUNFLE1BQUEsSUFBSSxFQUFDLGFBRFA7QUFFRSxNQUFBLFVBQVUsRUFBQyxrQkFGYjtBQUdFLE1BQUEsU0FBUyxFQUFFNkMsU0FIYjtBQUlFLE1BQUEsZ0JBQWdCLEVBQUU7QUFKcEIsTUFERixFQU9FLDZCQUFDLG1CQUFEO0FBQ0UsTUFBQSxJQUFJLEVBQUMsYUFEUDtBQUVFLE1BQUEsVUFBVSxFQUFDLGtCQUZiO0FBR0UsTUFBQSxTQUFTLEVBQUVBLFNBSGI7QUFJRSxNQUFBLGdCQUFnQixFQUFFO0FBSnBCLE1BUEYsQ0FUSixFQXdCRzVDLElBQUksSUFDSCw2QkFBQyxtQkFBRDtBQUNFLE1BQUEsSUFBSSxFQUFDLGFBRFA7QUFFRSxNQUFBLFVBQVUsRUFBQyxZQUZiO0FBR0UsTUFBQSxTQUFTLEVBQUU0QyxTQUhiO0FBSUUsTUFBQSxnQkFBZ0IsRUFBRTtBQUpwQixNQXpCSixDQURGO0FBbUNEOztBQXdCREosRUFBQUEsbUJBQW1CLENBQUM3RyxJQUFELEVBQU9vRyxpQkFBUCxFQUEwQjtBQUMzQyxRQUFJQSxpQkFBSixFQUF1QjtBQUNyQixhQUFPLEtBQUt2TSxLQUFMLENBQVc0TixVQUFYLENBQ0wsS0FBSzVOLEtBQUwsQ0FBV2dELFlBRE4sRUFFTCxLQUFLaEQsS0FBTCxDQUFXaUQsYUFGTixFQUdMO0FBQUNKLFFBQUFBLFdBQVcsRUFBRTtBQUFkLE9BSEssQ0FBUDtBQUtELEtBTkQsTUFNTztBQUNMLFlBQU1nTCxVQUFVLEdBQUcsSUFBSW5JLEdBQUosQ0FDakJTLElBQUksQ0FBQzJILFVBQUwsR0FDR0MsTUFESCxDQUNVLENBQUNDLElBQUQsRUFBT0MsTUFBUCxLQUFrQjtBQUN4QkQsUUFBQUEsSUFBSSxDQUFDRSxJQUFMLENBQVUsR0FBR0QsTUFBTSxDQUFDRSxhQUFQLEVBQWI7QUFDQSxlQUFPSCxJQUFQO0FBQ0QsT0FKSCxFQUlLLEVBSkwsQ0FEaUIsQ0FBbkI7QUFPQSxhQUFPLEtBQUtoTyxLQUFMLENBQVc0TixVQUFYLENBQ0xDLFVBREssRUFFTCxNQUZLLEVBR0w7QUFBQ2hMLFFBQUFBLFdBQVcsRUFBRTtBQUFkLE9BSEssQ0FBUDtBQUtEO0FBQ0Y7O0FBRURvSyxFQUFBQSxvQkFBb0IsQ0FBQzlHLElBQUQsRUFBT29HLGlCQUFQLEVBQTBCO0FBQzVDLFFBQUlBLGlCQUFKLEVBQXVCO0FBQ3JCLGFBQU8sS0FBS3ZNLEtBQUwsQ0FBVytDLFdBQVgsQ0FDTCxLQUFLL0MsS0FBTCxDQUFXZ0QsWUFETixFQUVMLEtBQUtoRCxLQUFMLENBQVdpRCxhQUZOLEVBR0w7QUFBQ0osUUFBQUEsV0FBVyxFQUFFO0FBQWQsT0FISyxDQUFQO0FBS0QsS0FORCxNQU1PO0FBQ0wsWUFBTWdMLFVBQVUsR0FBRyxJQUFJbkksR0FBSixDQUNqQlMsSUFBSSxDQUFDMkgsVUFBTCxHQUNHQyxNQURILENBQ1UsQ0FBQ0MsSUFBRCxFQUFPQyxNQUFQLEtBQWtCO0FBQ3hCRCxRQUFBQSxJQUFJLENBQUNFLElBQUwsQ0FBVSxHQUFHRCxNQUFNLENBQUNFLGFBQVAsRUFBYjtBQUNBLGVBQU9ILElBQVA7QUFDRCxPQUpILEVBSUssRUFKTCxDQURpQixDQUFuQjtBQU9BLGFBQU8sS0FBS2hPLEtBQUwsQ0FBVytDLFdBQVgsQ0FBdUI4SyxVQUF2QixFQUFtQyxNQUFuQyxFQUEyQztBQUFDaEwsUUFBQUEsV0FBVyxFQUFFO0FBQWQsT0FBM0MsQ0FBUDtBQUNEO0FBQ0Y7O0FBRURxSyxFQUFBQSxvQkFBb0IsQ0FBQ2tCLEtBQUQsRUFBUWpJLElBQVIsRUFBYztBQUNoQyxTQUFLdEMsaUJBQUwsR0FBeUIsTUFBekI7QUFDQSxTQUFLd0ssb0JBQUwsQ0FBMEJELEtBQTFCLEVBQWlDakksSUFBSSxDQUFDM0YsUUFBTCxFQUFqQztBQUNEOztBQUVEdUosRUFBQUEsd0JBQXdCLENBQUNxRSxLQUFELEVBQVE7QUFDOUIsVUFBTTNELElBQUksR0FBRzJELEtBQUssQ0FBQ0UsU0FBbkI7O0FBQ0EsUUFBSTdELElBQUksS0FBSzhELFNBQVQsSUFBc0JDLEtBQUssQ0FBQy9ELElBQUQsQ0FBL0IsRUFBdUM7QUFDckM7QUFDRDs7QUFFRCxTQUFLNUcsaUJBQUwsR0FBeUIsTUFBekI7O0FBQ0EsUUFBSSxLQUFLd0ssb0JBQUwsQ0FBMEJELEtBQUssQ0FBQ0ssUUFBaEMsRUFBMEMsQ0FBQyxDQUFDaEUsSUFBRCxFQUFPLENBQVAsQ0FBRCxFQUFZLENBQUNBLElBQUQsRUFBT0gsUUFBUCxDQUFaLENBQTFDLENBQUosRUFBOEU7QUFDNUUsV0FBSzNHLHdCQUFMLEdBQWdDLElBQWhDO0FBQ0Q7QUFDRjs7QUFFRHFHLEVBQUFBLHdCQUF3QixDQUFDb0UsS0FBRCxFQUFRO0FBQzlCLFFBQUksQ0FBQyxLQUFLekssd0JBQVYsRUFBb0M7QUFDbEM7QUFDRDs7QUFFRCxVQUFNOEcsSUFBSSxHQUFHMkQsS0FBSyxDQUFDRSxTQUFuQjs7QUFDQSxRQUFJLEtBQUsxSyxpQkFBTCxLQUEyQjZHLElBQTNCLElBQW1DQSxJQUFJLEtBQUs4RCxTQUE1QyxJQUF5REMsS0FBSyxDQUFDL0QsSUFBRCxDQUFsRSxFQUEwRTtBQUN4RTtBQUNEOztBQUNELFNBQUs3RyxpQkFBTCxHQUF5QjZHLElBQXpCO0FBRUEsU0FBSzVHLGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0EsU0FBS3dLLG9CQUFMLENBQTBCRCxLQUFLLENBQUNLLFFBQWhDLEVBQTBDLENBQUMsQ0FBQ2hFLElBQUQsRUFBTyxDQUFQLENBQUQsRUFBWSxDQUFDQSxJQUFELEVBQU9ILFFBQVAsQ0FBWixDQUExQyxFQUF5RTtBQUFDakcsTUFBQUEsR0FBRyxFQUFFO0FBQU4sS0FBekU7QUFDRDs7QUFFRHVDLEVBQUFBLFVBQVUsR0FBRztBQUNYLFNBQUtqRCx3QkFBTCxHQUFnQyxLQUFoQztBQUNEOztBQUVEMEssRUFBQUEsb0JBQW9CLENBQUNELEtBQUQsRUFBUU0sU0FBUixFQUFtQkMsSUFBbkIsRUFBeUI7QUFDM0MsUUFBSVAsS0FBSyxDQUFDUSxNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQ3RCLGFBQU8sS0FBUDtBQUNEOztBQUVELFVBQU1DLFNBQVMsR0FBR0MsT0FBTyxDQUFDQyxRQUFSLEtBQXFCLE9BQXZDOztBQUNBLFFBQUlYLEtBQUssQ0FBQ1ksT0FBTixJQUFpQixDQUFDSCxTQUF0QixFQUFpQztBQUMvQjtBQUNBLGFBQU8sS0FBUDtBQUNEOztBQUVELFVBQU1JLE9BQU87QUFDWDVLLE1BQUFBLEdBQUcsRUFBRTtBQURNLE9BRVJzSyxJQUZRLENBQWIsQ0FYMkMsQ0FnQjNDOzs7QUFDQSxVQUFNTyxTQUFTLEdBQUd2SixZQUFNQyxVQUFOLENBQWlCOEksU0FBakIsQ0FBbEI7O0FBQ0EsVUFBTWxCLEtBQUssR0FBRyxLQUFLeEosU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJQSxNQUFNLENBQUM0SyxlQUFQLENBQXVCRCxTQUF2QixDQUE3QixFQUFnRUUsS0FBaEUsQ0FBc0VGLFNBQXRFLENBQWQ7O0FBRUEsUUFBSWQsS0FBSyxDQUFDaUIsT0FBTjtBQUFpQjtBQUE0QmpCLElBQUFBLEtBQUssQ0FBQ1ksT0FBTixJQUFpQkgsU0FBbEUsRUFBOEU7QUFDNUUsV0FBSzdLLFNBQUwsQ0FBZVQsR0FBZixDQUFtQmdCLE1BQU0sSUFBSTtBQUMzQixZQUFJK0ssVUFBVSxHQUFHLEtBQWpCO0FBQ0EsWUFBSUMsT0FBTyxHQUFHLElBQWQ7O0FBRUEsYUFBSyxNQUFNQyxTQUFYLElBQXdCakwsTUFBTSxDQUFDa0wsYUFBUCxFQUF4QixFQUFnRDtBQUM5QyxjQUFJRCxTQUFTLENBQUNFLHFCQUFWLENBQWdDbEMsS0FBaEMsQ0FBSixFQUE0QztBQUMxQztBQUNBO0FBQ0E4QixZQUFBQSxVQUFVLEdBQUcsSUFBYjtBQUNBLGtCQUFNSyxjQUFjLEdBQUdILFNBQVMsQ0FBQ0ksY0FBVixFQUF2QjtBQUVBLGtCQUFNQyxTQUFTLEdBQUcsRUFBbEI7O0FBRUEsZ0JBQUksQ0FBQ3JDLEtBQUssQ0FBQzdNLEtBQU4sQ0FBWUMsT0FBWixDQUFvQitPLGNBQWMsQ0FBQ2hQLEtBQW5DLENBQUwsRUFBZ0Q7QUFDOUM7QUFDQSxrQkFBSW1QLE1BQU0sR0FBR3RDLEtBQUssQ0FBQzdNLEtBQW5COztBQUNBLGtCQUFJNk0sS0FBSyxDQUFDN00sS0FBTixDQUFZb1AsTUFBWixLQUF1QixDQUEzQixFQUE4QjtBQUM1QixzQkFBTUMsVUFBVSxHQUFHekwsTUFBTSxDQUFDekQsU0FBUCxHQUFtQm1QLGdCQUFuQixDQUFvQ3pDLEtBQUssQ0FBQzdNLEtBQU4sQ0FBWW1GLEdBQVosR0FBa0IsQ0FBdEQsQ0FBbkI7QUFDQWdLLGdCQUFBQSxNQUFNLEdBQUcsQ0FBQ3RDLEtBQUssQ0FBQzdNLEtBQU4sQ0FBWW1GLEdBQVosR0FBa0IsQ0FBbkIsRUFBc0JrSyxVQUF0QixDQUFUO0FBQ0Q7O0FBRURILGNBQUFBLFNBQVMsQ0FBQzNCLElBQVYsQ0FBZSxDQUFDeUIsY0FBYyxDQUFDaFAsS0FBaEIsRUFBdUJtUCxNQUF2QixDQUFmO0FBQ0Q7O0FBRUQsZ0JBQUksQ0FBQ3RDLEtBQUssQ0FBQzBDLEdBQU4sQ0FBVXRQLE9BQVYsQ0FBa0IrTyxjQUFjLENBQUNPLEdBQWpDLENBQUwsRUFBNEM7QUFDMUM7QUFDQSxrQkFBSUosTUFBTSxHQUFHdEMsS0FBSyxDQUFDMEMsR0FBbkI7QUFDQSxvQkFBTUYsVUFBVSxHQUFHekwsTUFBTSxDQUFDekQsU0FBUCxHQUFtQm1QLGdCQUFuQixDQUFvQ3pDLEtBQUssQ0FBQzBDLEdBQU4sQ0FBVXBLLEdBQTlDLENBQW5COztBQUNBLGtCQUFJMEgsS0FBSyxDQUFDMEMsR0FBTixDQUFVSCxNQUFWLEtBQXFCQyxVQUF6QixFQUFxQztBQUNuQ0YsZ0JBQUFBLE1BQU0sR0FBRyxDQUFDdEMsS0FBSyxDQUFDMEMsR0FBTixDQUFVcEssR0FBVixHQUFnQixDQUFqQixFQUFvQixDQUFwQixDQUFUO0FBQ0Q7O0FBRUQrSixjQUFBQSxTQUFTLENBQUMzQixJQUFWLENBQWUsQ0FBQzRCLE1BQUQsRUFBU0gsY0FBYyxDQUFDTyxHQUF4QixDQUFmO0FBQ0Q7O0FBRUQsZ0JBQUlMLFNBQVMsQ0FBQ3ZDLE1BQVYsR0FBbUIsQ0FBdkIsRUFBMEI7QUFDeEJrQyxjQUFBQSxTQUFTLENBQUNXLGNBQVYsQ0FBeUJOLFNBQVMsQ0FBQyxDQUFELENBQWxDOztBQUNBLG1CQUFLLE1BQU1PLFFBQVgsSUFBdUJQLFNBQVMsQ0FBQ1EsS0FBVixDQUFnQixDQUFoQixDQUF2QixFQUEyQztBQUN6QzlMLGdCQUFBQSxNQUFNLENBQUMrTCwwQkFBUCxDQUFrQ0YsUUFBbEMsRUFBNEM7QUFBQ0csa0JBQUFBLFFBQVEsRUFBRWYsU0FBUyxDQUFDZ0IsVUFBVjtBQUFYLGlCQUE1QztBQUNEO0FBQ0YsYUFMRCxNQUtPO0FBQ0xqQixjQUFBQSxPQUFPLEdBQUdDLFNBQVY7QUFDRDtBQUNGO0FBQ0Y7O0FBRUQsWUFBSUQsT0FBTyxLQUFLLElBQWhCLEVBQXNCO0FBQ3BCLGdCQUFNa0IsaUJBQWlCLEdBQUdsTSxNQUFNLENBQUNrTCxhQUFQLEdBQ3ZCck0sTUFEdUIsQ0FDaEJzTixJQUFJLElBQUlBLElBQUksS0FBS25CLE9BREQsRUFFdkJoTSxHQUZ1QixDQUVuQm1OLElBQUksSUFBSUEsSUFBSSxDQUFDZCxjQUFMLEVBRlcsQ0FBMUI7O0FBR0EsY0FBSWEsaUJBQWlCLENBQUNuRCxNQUFsQixHQUEyQixDQUEvQixFQUFrQztBQUNoQy9JLFlBQUFBLE1BQU0sQ0FBQzZCLHVCQUFQLENBQStCcUssaUJBQS9CO0FBQ0Q7QUFDRjs7QUFFRCxZQUFJLENBQUNuQixVQUFMLEVBQWlCO0FBQ2Y7QUFDQS9LLFVBQUFBLE1BQU0sQ0FBQytMLDBCQUFQLENBQWtDOUMsS0FBbEM7QUFDRDs7QUFFRCxlQUFPLElBQVA7QUFDRCxPQTdERDtBQThERCxLQS9ERCxNQStETyxJQUFJeUIsT0FBTyxDQUFDNUssR0FBUixJQUFlK0osS0FBSyxDQUFDdUMsUUFBekIsRUFBbUM7QUFDeEM7QUFDQSxXQUFLM00sU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJO0FBQzNCLGNBQU1xTSxhQUFhLEdBQUdyTSxNQUFNLENBQUNzTSxnQkFBUCxFQUF0QjtBQUNBLGNBQU1DLGtCQUFrQixHQUFHRixhQUFhLENBQUNoQixjQUFkLEVBQTNCLENBRjJCLENBSTNCOztBQUNBLGNBQU1tQixRQUFRLEdBQUd2RCxLQUFLLENBQUM3TSxLQUFOLENBQVlxUSxVQUFaLENBQXVCRixrQkFBa0IsQ0FBQ25RLEtBQTFDLENBQWpCO0FBQ0EsY0FBTXNRLE9BQU8sR0FBR0YsUUFBUSxHQUFHdkQsS0FBSyxDQUFDN00sS0FBVCxHQUFpQjZNLEtBQUssQ0FBQzBDLEdBQS9DO0FBQ0EsY0FBTUUsUUFBUSxHQUFHVyxRQUFRLEdBQUcsQ0FBQ0UsT0FBRCxFQUFVSCxrQkFBa0IsQ0FBQ1osR0FBN0IsQ0FBSCxHQUF1QyxDQUFDWSxrQkFBa0IsQ0FBQ25RLEtBQXBCLEVBQTJCc1EsT0FBM0IsQ0FBaEU7QUFFQUwsUUFBQUEsYUFBYSxDQUFDVCxjQUFkLENBQTZCQyxRQUE3QixFQUF1QztBQUFDRyxVQUFBQSxRQUFRLEVBQUVRO0FBQVgsU0FBdkM7QUFDQSxlQUFPLElBQVA7QUFDRCxPQVhEO0FBWUQsS0FkTSxNQWNBO0FBQ0wsV0FBSy9NLFNBQUwsQ0FBZVQsR0FBZixDQUFtQmdCLE1BQU0sSUFBSUEsTUFBTSxDQUFDaUIsc0JBQVAsQ0FBOEJnSSxLQUE5QixDQUE3QjtBQUNEOztBQUVELFdBQU8sSUFBUDtBQUNEOztBQUVEekUsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFLL0ksS0FBTCxDQUFXNE4sVUFBWCxDQUFzQixLQUFLNU4sS0FBTCxDQUFXZ0QsWUFBakMsRUFBK0MsS0FBS2hELEtBQUwsQ0FBV2lELGFBQTFELENBQVA7QUFDRDs7QUFFRHVGLEVBQUFBLHNCQUFzQixHQUFHO0FBQ3ZCLFVBQU04RCxhQUFhLEdBQUcsS0FBSzRFLGdCQUFMLEVBQXRCO0FBQ0EsU0FBS0MsaUJBQUwsQ0FBdUI7QUFDckIxRyxNQUFBQSxJQUFJLEVBQUUsTUFBTTtBQUNWLGNBQU0yRyxVQUFVLEdBQUc5RSxhQUFhLENBQUMvSSxHQUFkLENBQWtCNEMsSUFBSSxJQUFJQSxJQUFJLENBQUMzRixRQUFMLEVBQTFCLENBQW5CO0FBQ0EsYUFBS3FELGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0EsYUFBS0csU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJQSxNQUFNLENBQUM2Qix1QkFBUCxDQUErQmdMLFVBQS9CLENBQTdCO0FBQ0QsT0FMb0I7QUFNckJqTCxNQUFBQSxJQUFJLEVBQUUsTUFBTTtBQUNWLFlBQUlrTCxjQUFjLEdBQUcvRyxRQUFyQjs7QUFDQSxhQUFLLE1BQU1uRSxJQUFYLElBQW1CbUcsYUFBbkIsRUFBa0M7QUFDaEMsZ0JBQU0sQ0FBQ2dGLFdBQUQsSUFBZ0JuTCxJQUFJLENBQUMySCxVQUFMLEVBQXRCO0FBQ0E7O0FBQ0EsY0FBSXdELFdBQVcsS0FBSyxDQUFDRCxjQUFELElBQW1CQyxXQUFXLENBQUNDLGlCQUFaLEtBQWtDRixjQUExRCxDQUFmLEVBQTBGO0FBQ3hGQSxZQUFBQSxjQUFjLEdBQUdDLFdBQVcsQ0FBQ0MsaUJBQVosRUFBakI7QUFDRDtBQUNGOztBQUVELGFBQUsxTixpQkFBTCxHQUF5QixNQUF6QjtBQUNBLGFBQUtHLFNBQUwsQ0FBZVQsR0FBZixDQUFtQmdCLE1BQU0sSUFBSTtBQUMzQkEsVUFBQUEsTUFBTSxDQUFDNkIsdUJBQVAsQ0FBK0IsQ0FBQyxDQUFDLENBQUNpTCxjQUFELEVBQWlCLENBQWpCLENBQUQsRUFBc0IsQ0FBQ0EsY0FBRCxFQUFpQi9HLFFBQWpCLENBQXRCLENBQUQsQ0FBL0I7QUFDQSxpQkFBTyxJQUFQO0FBQ0QsU0FIRDtBQUlEO0FBckJvQixLQUF2QjtBQXVCRDs7QUFrQkRoQyxFQUFBQSxjQUFjLEdBQUc7QUFDZixTQUFLdEUsU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJO0FBQzNCLFlBQU1rQixTQUFTLEdBQUcsSUFBSUMsR0FBSixDQUNoQixLQUFLOEwsaUJBQUwsQ0FBdUJyTCxJQUFJLElBQUksS0FBS3NMLFlBQUwsQ0FBa0J0TCxJQUFsQixLQUEyQkEsSUFBMUQsQ0FEZ0IsQ0FBbEI7QUFHQSxZQUFNRixVQUFVLEdBQUd6RCxLQUFLLENBQUNDLElBQU4sQ0FBV2dELFNBQVgsRUFBc0JVLElBQUksSUFBSUEsSUFBSSxDQUFDM0YsUUFBTCxFQUE5QixDQUFuQjtBQUNBLFdBQUtxRCxpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxNQUFBQSxNQUFNLENBQUM2Qix1QkFBUCxDQUErQkgsVUFBL0I7QUFDQSxhQUFPLElBQVA7QUFDRCxLQVJEO0FBU0Q7O0FBRURzQyxFQUFBQSxrQkFBa0IsR0FBRztBQUNuQixTQUFLdkUsU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJO0FBQzNCLFlBQU1rQixTQUFTLEdBQUcsSUFBSUMsR0FBSixDQUNoQixLQUFLOEwsaUJBQUwsQ0FBdUJyTCxJQUFJLElBQUksS0FBS3VMLGFBQUwsQ0FBbUJ2TCxJQUFuQixLQUE0QkEsSUFBM0QsQ0FEZ0IsQ0FBbEI7QUFHQSxZQUFNRixVQUFVLEdBQUd6RCxLQUFLLENBQUNDLElBQU4sQ0FBV2dELFNBQVgsRUFBc0JVLElBQUksSUFBSUEsSUFBSSxDQUFDM0YsUUFBTCxFQUE5QixDQUFuQjtBQUNBLFdBQUtxRCxpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxNQUFBQSxNQUFNLENBQUM2Qix1QkFBUCxDQUErQkgsVUFBL0I7QUFDQSxhQUFPLElBQVA7QUFDRCxLQVJEO0FBU0Q7O0FBRURyRSxFQUFBQSxXQUFXLENBQUM7QUFBQ0MsSUFBQUE7QUFBRCxHQUFELEVBQXNCO0FBQy9CLFVBQU04UCxrQkFBa0IsR0FBRyxJQUFJQyxHQUFKLEVBQTNCO0FBRUEsU0FBSzVOLFNBQUwsQ0FBZVQsR0FBZixDQUFtQmdCLE1BQU0sSUFBSTtBQUMzQixZQUFNc04sVUFBVSxHQUFHLElBQUluTSxHQUFKLEVBQW5COztBQUVBLFdBQUssTUFBTW9NLE1BQVgsSUFBcUJ2TixNQUFNLENBQUN3TixVQUFQLEVBQXJCLEVBQTBDO0FBQ3hDLGNBQU1DLFNBQVMsR0FBR0YsTUFBTSxDQUFDRyxpQkFBUCxHQUEyQm5NLEdBQTdDO0FBQ0EsY0FBTUssSUFBSSxHQUFHLEtBQUtuRyxLQUFMLENBQVdhLGNBQVgsQ0FBMEJrRixTQUExQixDQUFvQ2lNLFNBQXBDLENBQWI7QUFDQSxjQUFNL1IsU0FBUyxHQUFHLEtBQUtELEtBQUwsQ0FBV2EsY0FBWCxDQUEwQnFSLGNBQTFCLENBQXlDRixTQUF6QyxDQUFsQjtBQUNBOztBQUNBLFlBQUksQ0FBQzdMLElBQUwsRUFBVztBQUNUO0FBQ0Q7O0FBRUQsWUFBSWdNLE1BQU0sR0FBR2hNLElBQUksQ0FBQ2lNLFdBQUwsQ0FBaUJKLFNBQWpCLENBQWI7QUFDQSxZQUFJSyxTQUFTLEdBQUdQLE1BQU0sQ0FBQ0csaUJBQVAsR0FBMkJsQyxNQUEzQzs7QUFDQSxZQUFJb0MsTUFBTSxLQUFLLElBQWYsRUFBcUI7QUFDbkIsY0FBSUcsVUFBVSxHQUFHbk0sSUFBSSxDQUFDb00sY0FBTCxFQUFqQjs7QUFDQSxlQUFLLE1BQU1DLE1BQVgsSUFBcUJyTSxJQUFJLENBQUNzTSxVQUFMLEVBQXJCLEVBQXdDO0FBQ3RDLGdCQUFJLENBQUNELE1BQU0sQ0FBQ0UsaUJBQVAsQ0FBeUJWLFNBQXpCLENBQUwsRUFBMEM7QUFDeENRLGNBQUFBLE1BQU0sQ0FBQ0csSUFBUCxDQUFZO0FBQ1ZDLGdCQUFBQSxTQUFTLEVBQUUsTUFBTTtBQUNmTixrQkFBQUEsVUFBVSxJQUFJRSxNQUFNLENBQUNLLGNBQVAsRUFBZDtBQUNELGlCQUhTO0FBSVZDLGdCQUFBQSxRQUFRLEVBQUUsTUFBTTtBQUNkUixrQkFBQUEsVUFBVSxJQUFJRSxNQUFNLENBQUNLLGNBQVAsRUFBZDtBQUNEO0FBTlMsZUFBWjtBQVFELGFBVEQsTUFTTztBQUNMO0FBQ0Q7QUFDRjs7QUFFRCxjQUFJLENBQUNoQixVQUFVLENBQUNyRixHQUFYLENBQWU4RixVQUFmLENBQUwsRUFBaUM7QUFDL0JILFlBQUFBLE1BQU0sR0FBR0csVUFBVDtBQUNBRCxZQUFBQSxTQUFTLEdBQUcsQ0FBWjtBQUNBUixZQUFBQSxVQUFVLENBQUN4TixHQUFYLENBQWVpTyxVQUFmO0FBQ0Q7QUFDRjs7QUFFRCxZQUFJSCxNQUFNLEtBQUssSUFBZixFQUFxQjtBQUNuQjtBQUNBO0FBQ0FBLFVBQUFBLE1BQU0sSUFBSSxDQUFWO0FBQ0EsZ0JBQU1ZLE9BQU8sR0FBR3BCLGtCQUFrQixDQUFDekgsR0FBbkIsQ0FBdUJqSyxTQUF2QixDQUFoQjs7QUFDQSxjQUFJLENBQUM4UyxPQUFMLEVBQWM7QUFDWnBCLFlBQUFBLGtCQUFrQixDQUFDcUIsR0FBbkIsQ0FBdUIvUyxTQUF2QixFQUFrQyxDQUFDLENBQUNrUyxNQUFELEVBQVNFLFNBQVQsQ0FBRCxDQUFsQztBQUNELFdBRkQsTUFFTztBQUNMVSxZQUFBQSxPQUFPLENBQUM3RSxJQUFSLENBQWEsQ0FBQ2lFLE1BQUQsRUFBU0UsU0FBVCxDQUFiO0FBQ0Q7QUFDRjtBQUNGOztBQUVELGFBQU8sSUFBUDtBQUNELEtBcEREO0FBc0RBLFVBQU1ZLHNCQUFzQixHQUFHLElBQUl2TixHQUFKLENBQVFpTSxrQkFBa0IsQ0FBQ3VCLElBQW5CLEVBQVIsQ0FBL0I7O0FBQ0EsUUFBSXJSLGlCQUFpQixJQUFJLENBQUNvUixzQkFBc0IsQ0FBQ3pHLEdBQXZCLENBQTJCM0ssaUJBQTNCLENBQTFCLEVBQXlFO0FBQ3ZFLFlBQU0sQ0FBQ2tGLFNBQUQsSUFBY2xGLGlCQUFpQixDQUFDbUYsUUFBbEIsRUFBcEI7QUFDQSxZQUFNZ0wsU0FBUyxHQUFHakwsU0FBUyxHQUFHQSxTQUFTLENBQUN3TCxjQUFWLEtBQTZCLENBQWhDLEdBQW9DLENBQS9EO0FBQ0EsYUFBTyxLQUFLdlMsS0FBTCxDQUFXbVQsUUFBWCxDQUFvQnRSLGlCQUFwQixFQUF1QyxDQUFDLENBQUNtUSxTQUFELEVBQVksQ0FBWixDQUFELENBQXZDLEVBQXlELElBQXpELENBQVA7QUFDRCxLQUpELE1BSU87QUFDTCxZQUFNb0IsT0FBTyxHQUFHekIsa0JBQWtCLENBQUN6TCxJQUFuQixLQUE0QixDQUE1QztBQUNBLGFBQU9oRCxPQUFPLENBQUNDLEdBQVIsQ0FBWVgsS0FBSyxDQUFDQyxJQUFOLENBQVdrUCxrQkFBWCxFQUErQjBCLEtBQUssSUFBSTtBQUN6RCxjQUFNLENBQUNwVCxTQUFELEVBQVk4UyxPQUFaLElBQXVCTSxLQUE3QjtBQUNBLGVBQU8sS0FBS3JULEtBQUwsQ0FBV21ULFFBQVgsQ0FBb0JsVCxTQUFwQixFQUErQjhTLE9BQS9CLEVBQXdDSyxPQUF4QyxDQUFQO0FBQ0QsT0FIa0IsQ0FBWixDQUFQO0FBSUQ7QUFFRjs7QUFFREUsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFdBQU8sS0FBS3RQLFNBQUwsQ0FBZVQsR0FBZixDQUFtQmdCLE1BQU0sSUFBSTtBQUNsQyxhQUFPLElBQUltQixHQUFKLENBQ0xuQixNQUFNLENBQUNrTCxhQUFQLEdBQ0dsTSxHQURILENBQ09pTSxTQUFTLElBQUlBLFNBQVMsQ0FBQ0ksY0FBVixFQURwQixFQUVHN0IsTUFGSCxDQUVVLENBQUN3RixHQUFELEVBQU0vRixLQUFOLEtBQWdCO0FBQ3RCLGFBQUssTUFBTTFILEdBQVgsSUFBa0IwSCxLQUFLLENBQUMzSCxPQUFOLEVBQWxCLEVBQW1DO0FBQ2pDLGNBQUksS0FBSzJOLFdBQUwsQ0FBaUIxTixHQUFqQixDQUFKLEVBQTJCO0FBQ3pCeU4sWUFBQUEsR0FBRyxDQUFDckYsSUFBSixDQUFTcEksR0FBVDtBQUNEO0FBQ0Y7O0FBQ0QsZUFBT3lOLEdBQVA7QUFDRCxPQVRILEVBU0ssRUFUTCxDQURLLENBQVA7QUFZRCxLQWJNLEVBYUpuRSxLQWJJLENBYUUsSUFBSTFKLEdBQUosRUFiRixDQUFQO0FBY0Q7O0FBRURpRSxFQUFBQSxlQUFlLEdBQUc7QUFDaEIsU0FBS3BELHFCQUFMO0FBQ0Q7O0FBRURxRCxFQUFBQSx1QkFBdUIsQ0FBQ3dFLEtBQUQsRUFBUTtBQUM3QixRQUNFLENBQUNBLEtBQUQsSUFDQUEsS0FBSyxDQUFDcUYsY0FBTixDQUFxQjlTLEtBQXJCLENBQTJCbUYsR0FBM0IsS0FBbUNzSSxLQUFLLENBQUNzRixjQUFOLENBQXFCL1MsS0FBckIsQ0FBMkJtRixHQUQ5RCxJQUVBc0ksS0FBSyxDQUFDcUYsY0FBTixDQUFxQnZELEdBQXJCLENBQXlCcEssR0FBekIsS0FBaUNzSSxLQUFLLENBQUNzRixjQUFOLENBQXFCeEQsR0FBckIsQ0FBeUJwSyxHQUg1RCxFQUlFO0FBQ0EsV0FBS1MscUJBQUw7QUFDRDtBQUNGOztBQUVEc0QsRUFBQUEsbUJBQW1CLEdBQUc7QUFDcEIsU0FBS3RELHFCQUFMO0FBQ0Q7O0FBRURBLEVBQUFBLHFCQUFxQixHQUFHO0FBQ3RCLFFBQUksS0FBSzNCLGVBQVQsRUFBMEI7QUFDeEI7QUFDRDs7QUFFRCxVQUFNK08sY0FBYyxHQUFHLEtBQUszUCxTQUFMLENBQWVULEdBQWYsQ0FBbUJnQixNQUFNLElBQUk7QUFDbEQsYUFBT0EsTUFBTSxDQUFDcVAsd0JBQVAsR0FBa0NyUSxHQUFsQyxDQUFzQ3ZDLFFBQVEsSUFBSUEsUUFBUSxDQUFDOEUsR0FBM0QsQ0FBUDtBQUNELEtBRnNCLEVBRXBCc0osS0FGb0IsQ0FFZCxFQUZjLENBQXZCO0FBR0EsVUFBTTVOLHlCQUF5QixHQUFHLEtBQUt4QixLQUFMLENBQVdhLGNBQVgsQ0FBMEJnVCxrQkFBMUIsQ0FBNkNGLGNBQTdDLENBQWxDO0FBRUEsU0FBSzNULEtBQUwsQ0FBVzhULG1CQUFYLENBQ0UsS0FBS1IsZUFBTCxFQURGLEVBRUUsS0FBS3pQLGlCQUFMLElBQTBCLE1BRjVCLEVBR0VyQyx5QkFIRjtBQUtEOztBQUVEc0ksRUFBQUEsa0JBQWtCLENBQUM7QUFBQ3dFLElBQUFBLFNBQUQ7QUFBWXlGLElBQUFBO0FBQVosR0FBRCxFQUEyQjtBQUMzQyxVQUFNNU4sSUFBSSxHQUFHLEtBQUtuRyxLQUFMLENBQVdhLGNBQVgsQ0FBMEJrRixTQUExQixDQUFvQ3VJLFNBQXBDLENBQWI7O0FBQ0EsUUFBSW5JLElBQUksS0FBS29JLFNBQWIsRUFBd0I7QUFDdEIsYUFBTyxLQUFLeUYsR0FBTCxDQUFTLEVBQVQsQ0FBUDtBQUNEOztBQUVELFVBQU1DLE1BQU0sR0FBRzlOLElBQUksQ0FBQytOLFdBQUwsQ0FBaUI1RixTQUFqQixDQUFmOztBQUNBLFFBQUl5RixXQUFKLEVBQWlCO0FBQ2YsYUFBTyxLQUFLQyxHQUFMLENBQVNDLE1BQU0sS0FBSyxJQUFYLEdBQWtCLEVBQWxCLEdBQXVCLEdBQWhDLENBQVA7QUFDRDs7QUFFRCxXQUFPLEtBQUtELEdBQUwsQ0FBU0MsTUFBVCxDQUFQO0FBQ0Q7O0FBRURoSyxFQUFBQSxrQkFBa0IsQ0FBQztBQUFDcUUsSUFBQUEsU0FBRDtBQUFZeUYsSUFBQUE7QUFBWixHQUFELEVBQTJCO0FBQzNDLFVBQU01TixJQUFJLEdBQUcsS0FBS25HLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQmtGLFNBQTFCLENBQW9DdUksU0FBcEMsQ0FBYjs7QUFDQSxRQUFJbkksSUFBSSxLQUFLb0ksU0FBYixFQUF3QjtBQUN0QixhQUFPLEtBQUt5RixHQUFMLENBQVMsRUFBVCxDQUFQO0FBQ0Q7O0FBRUQsVUFBTTdCLE1BQU0sR0FBR2hNLElBQUksQ0FBQ2lNLFdBQUwsQ0FBaUI5RCxTQUFqQixDQUFmOztBQUNBLFFBQUl5RixXQUFKLEVBQWlCO0FBQ2YsYUFBTyxLQUFLQyxHQUFMLENBQVM3QixNQUFNLEtBQUssSUFBWCxHQUFrQixFQUFsQixHQUF1QixHQUFoQyxDQUFQO0FBQ0Q7O0FBQ0QsV0FBTyxLQUFLNkIsR0FBTCxDQUFTN0IsTUFBVCxDQUFQO0FBQ0Q7QUFFRDs7Ozs7O0FBSUFqQixFQUFBQSxnQkFBZ0IsR0FBRztBQUNqQixXQUFPLEtBQUtNLGlCQUFMLENBQXVCZCxJQUFJLElBQUlBLElBQS9CLENBQVA7QUFDRDs7QUFFRGMsRUFBQUEsaUJBQWlCLENBQUMyQyxRQUFELEVBQVc7QUFDMUIsV0FBTyxLQUFLblEsU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJO0FBQ2xDLFlBQU02UCxJQUFJLEdBQUcsSUFBSTFPLEdBQUosRUFBYjtBQUNBLGFBQU9uQixNQUFNLENBQUM4UCx1QkFBUCxHQUFpQ3RHLE1BQWpDLENBQXdDLENBQUN3RixHQUFELEVBQU0vRixLQUFOLEtBQWdCO0FBQzdELGFBQUssTUFBTTFILEdBQVgsSUFBa0IwSCxLQUFLLENBQUMzSCxPQUFOLEVBQWxCLEVBQW1DO0FBQ2pDLGdCQUFNTSxJQUFJLEdBQUcsS0FBS25HLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQmtGLFNBQTFCLENBQW9DRCxHQUFwQyxDQUFiOztBQUNBLGNBQUksQ0FBQ0ssSUFBRCxJQUFTaU8sSUFBSSxDQUFDNUgsR0FBTCxDQUFTckcsSUFBVCxDQUFiLEVBQTZCO0FBQzNCO0FBQ0Q7O0FBRURpTyxVQUFBQSxJQUFJLENBQUMvUCxHQUFMLENBQVM4QixJQUFUO0FBQ0FvTixVQUFBQSxHQUFHLENBQUNyRixJQUFKLENBQVNpRyxRQUFRLENBQUNoTyxJQUFELENBQWpCO0FBQ0Q7O0FBQ0QsZUFBT29OLEdBQVA7QUFDRCxPQVhNLEVBV0osRUFYSSxDQUFQO0FBWUQsS0FkTSxFQWNKbkUsS0FkSSxDQWNFLEVBZEYsQ0FBUDtBQWVEO0FBRUQ7Ozs7OztBQUlBMU0sRUFBQUEsc0JBQXNCLEdBQUc7QUFDdkIsV0FBTyxLQUFLc0IsU0FBTCxDQUFlVCxHQUFmLENBQW1CZ0IsTUFBTSxJQUFJO0FBQ2xDLFlBQU0rUCxPQUFPLEdBQUcsSUFBSTVPLEdBQUosRUFBaEI7O0FBQ0EsV0FBSyxNQUFNOEgsS0FBWCxJQUFvQmpKLE1BQU0sQ0FBQzhQLHVCQUFQLEVBQXBCLEVBQXNEO0FBQ3BELGFBQUssTUFBTXZPLEdBQVgsSUFBa0IwSCxLQUFLLENBQUMzSCxPQUFOLEVBQWxCLEVBQW1DO0FBQ2pDLGdCQUFNME8sS0FBSyxHQUFHLEtBQUt2VSxLQUFMLENBQVdhLGNBQVgsQ0FBMEJxUixjQUExQixDQUF5Q3BNLEdBQXpDLENBQWQ7QUFDQXdPLFVBQUFBLE9BQU8sQ0FBQ2pRLEdBQVIsQ0FBWWtRLEtBQVo7QUFDRDtBQUNGOztBQUNELGFBQU9ELE9BQVA7QUFDRCxLQVRNLEVBU0psRixLQVRJLENBU0UsSUFBSTFKLEdBQUosRUFURixDQUFQO0FBVUQ7O0FBRURnTSxFQUFBQSxhQUFhLENBQUN2TCxJQUFELEVBQU87QUFDbEIsVUFBTXFPLE9BQU8sR0FBR3JPLElBQUksQ0FBQzNGLFFBQUwsR0FBZ0JHLEtBQWhCLENBQXNCbUYsR0FBdEIsR0FBNEIsQ0FBNUM7QUFDQSxXQUFPLEtBQUs5RixLQUFMLENBQVdhLGNBQVgsQ0FBMEJrRixTQUExQixDQUFvQ3lPLE9BQXBDLENBQVA7QUFDRDs7QUFFRC9DLEVBQUFBLFlBQVksQ0FBQ3RMLElBQUQsRUFBTztBQUNqQixVQUFNc08sT0FBTyxHQUFHdE8sSUFBSSxDQUFDM0YsUUFBTCxHQUFnQjBQLEdBQWhCLENBQW9CcEssR0FBcEIsR0FBMEIsQ0FBMUM7QUFDQSxXQUFPLEtBQUs5RixLQUFMLENBQVdhLGNBQVgsQ0FBMEJrRixTQUExQixDQUFvQzBPLE9BQXBDLENBQVA7QUFDRDs7QUFFRGpCLEVBQUFBLFdBQVcsQ0FBQ2xGLFNBQUQsRUFBWTtBQUNyQixVQUFNb0csWUFBWSxHQUFHLENBQUMsS0FBSzFVLEtBQUwsQ0FBV2EsY0FBWCxDQUEwQjhKLGdCQUExQixFQUFELEVBQStDLEtBQUszSyxLQUFMLENBQVdhLGNBQVgsQ0FBMEIrSixnQkFBMUIsRUFBL0MsQ0FBckI7QUFDQSxXQUFPOEosWUFBWSxDQUFDQyxJQUFiLENBQWtCakgsS0FBSyxJQUFJQSxLQUFLLENBQUNrSCxXQUFOLENBQWtCO0FBQUNDLE1BQUFBLGFBQWEsRUFBRXZHO0FBQWhCLEtBQWxCLEVBQThDaEIsTUFBOUMsR0FBdUQsQ0FBbEYsQ0FBUDtBQUNEOztBQUVENkQsRUFBQUEsaUJBQWlCLENBQUMyRCxTQUFELEVBQVk7QUFDM0IsVUFBTVgsUUFBUSxHQUFHVyxTQUFTLENBQUMsS0FBSzlVLEtBQUwsQ0FBV2lELGFBQVosQ0FBMUI7QUFDQTs7QUFDQSxRQUFJLENBQUNrUixRQUFMLEVBQWU7QUFDYixZQUFNLElBQUlZLEtBQUosQ0FBVywyQkFBMEIsS0FBSy9VLEtBQUwsQ0FBV2lELGFBQWMsRUFBOUQsQ0FBTjtBQUNEOztBQUNELFdBQU9rUixRQUFRLEVBQWY7QUFDRDs7QUFFREgsRUFBQUEsR0FBRyxDQUFDZ0IsR0FBRCxFQUFNO0FBQ1AsVUFBTUMsU0FBUyxHQUFHLEtBQUtqVixLQUFMLENBQVdhLGNBQVgsQ0FBMEJxVSxxQkFBMUIsRUFBbEI7O0FBQ0EsUUFBSUYsR0FBRyxLQUFLLElBQVosRUFBa0I7QUFDaEIsYUFBT3RWLGNBQWMsQ0FBQ3lWLE1BQWYsQ0FBc0JGLFNBQXRCLENBQVA7QUFDRCxLQUZELE1BRU87QUFDTCxhQUFPdlYsY0FBYyxDQUFDeVYsTUFBZixDQUFzQkYsU0FBUyxHQUFHRCxHQUFHLENBQUNJLFFBQUosR0FBZTlILE1BQWpELElBQTJEMEgsR0FBRyxDQUFDSSxRQUFKLEVBQWxFO0FBQ0Q7QUFDRjs7QUFFRDNPLEVBQUFBLGtCQUFrQixDQUFDNE8sTUFBRCxFQUFTO0FBQ3pCO0FBQ0EsUUFBSSxDQUFDQSxNQUFNLEtBQUssUUFBWCxJQUF1QkEsTUFBTSxLQUFLLE9BQW5DLEtBQ0M1TixXQUFXLENBQUM2TixnQkFBWixDQUE4QixzQkFBcUJELE1BQU8sUUFBMUQsRUFBbUUvSCxNQUFuRSxHQUE0RSxDQURqRixFQUNvRjtBQUNsRjdGLE1BQUFBLFdBQVcsQ0FBQ00sSUFBWixDQUFrQixzQkFBcUJzTixNQUFPLE1BQTlDO0FBQ0E1TixNQUFBQSxXQUFXLENBQUM4TixPQUFaLENBQ0csc0JBQXFCRixNQUFPLEVBRC9CLEVBRUcsc0JBQXFCQSxNQUFPLFFBRi9CLEVBR0csc0JBQXFCQSxNQUFPLE1BSC9CO0FBSUEsWUFBTUcsSUFBSSxHQUFHL04sV0FBVyxDQUFDNk4sZ0JBQVosQ0FBOEIsc0JBQXFCRCxNQUFPLEVBQTFELEVBQTZELENBQTdELENBQWI7QUFDQTVOLE1BQUFBLFdBQVcsQ0FBQ0MsVUFBWixDQUF3QixzQkFBcUIyTixNQUFPLFFBQXBEO0FBQ0E1TixNQUFBQSxXQUFXLENBQUNDLFVBQVosQ0FBd0Isc0JBQXFCMk4sTUFBTyxNQUFwRDtBQUNBNU4sTUFBQUEsV0FBVyxDQUFDRSxhQUFaLENBQTJCLHNCQUFxQjBOLE1BQU8sRUFBdkQ7QUFDQSxtQ0FBVSxzQkFBcUJBLE1BQU8sRUFBdEMsRUFBeUM7QUFDdkMvSixRQUFBQSxPQUFPLEVBQUUsUUFEOEI7QUFFdkNtSyxRQUFBQSxxQkFBcUIsRUFBRSxLQUFLelYsS0FBTCxDQUFXYSxjQUFYLENBQTBCaUcsY0FBMUIsR0FBMkN2RCxHQUEzQyxDQUNyQkYsRUFBRSxJQUFJQSxFQUFFLENBQUNsQixRQUFILEdBQWN1VCxtQkFBZCxFQURlLENBRmdCO0FBS3ZDQyxRQUFBQSxRQUFRLEVBQUVILElBQUksQ0FBQ0c7QUFMd0IsT0FBekM7QUFPRDtBQUNGOztBQWxyQzZEOzs7O2dCQUEzQy9WLGtCLGVBQ0E7QUFDakJ5QixFQUFBQSxhQUFhLEVBQUV1VSxtQkFBVUMsS0FBVixDQUFnQixDQUFDLFFBQUQsRUFBVyxVQUFYLENBQWhCLENBREU7QUFFakJ2VSxFQUFBQSxpQkFBaUIsRUFBRXNVLG1CQUFVRSxJQUZaO0FBR2pCalYsRUFBQUEsY0FBYyxFQUFFa1YsbUNBQXVCQyxVQUh0QjtBQUlqQi9TLEVBQUFBLGFBQWEsRUFBRTJTLG1CQUFVQyxLQUFWLENBQWdCLENBQUMsTUFBRCxFQUFTLE1BQVQsQ0FBaEIsRUFBa0NHLFVBSmhDO0FBS2pCaFQsRUFBQUEsWUFBWSxFQUFFNFMsbUJBQVVLLE1BQVYsQ0FBaUJELFVBTGQ7QUFNakJ4VSxFQUFBQSx5QkFBeUIsRUFBRW9VLG1CQUFVRSxJQUFWLENBQWVFLFVBTnpCO0FBT2pCRSxFQUFBQSxVQUFVLEVBQUVOLG1CQUFVSyxNQUFWLENBQWlCRCxVQVBaO0FBUWpCelUsRUFBQUEsY0FBYyxFQUFFcVUsbUJBQVVFLElBUlQ7QUFVakJwTSxFQUFBQSxTQUFTLEVBQUVrTSxtQkFBVUssTUFBVixDQUFpQkQsVUFWWDtBQVdqQjNOLEVBQUFBLFFBQVEsRUFBRXVOLG1CQUFVSyxNQUFWLENBQWlCRCxVQVhWO0FBWWpCakosRUFBQUEsT0FBTyxFQUFFNkksbUJBQVVLLE1BQVYsQ0FBaUJELFVBWlQ7QUFhakJ2VSxFQUFBQSxRQUFRLEVBQUVtVSxtQkFBVUssTUFBVixDQUFpQkQsVUFiVjtBQWNqQi9PLEVBQUFBLE1BQU0sRUFBRTJPLG1CQUFVSyxNQUFWLENBQWlCRCxVQWRSO0FBZWpCbEwsRUFBQUEsV0FBVyxFQUFFOEssbUJBQVVLLE1BZk47QUFpQmpCbkMsRUFBQUEsbUJBQW1CLEVBQUU4QixtQkFBVU8sSUFqQmQ7QUFrQmpCbEwsRUFBQUEsZ0JBQWdCLEVBQUUySyxtQkFBVU8sSUFsQlg7QUFvQmpCeFUsRUFBQUEsbUJBQW1CLEVBQUVpVSxtQkFBVU8sSUFwQmQ7QUFxQmpCak4sRUFBQUEsT0FBTyxFQUFFME0sbUJBQVVPLElBckJGO0FBc0JqQmhELEVBQUFBLFFBQVEsRUFBRXlDLG1CQUFVTyxJQXRCSDtBQXVCakJyVSxFQUFBQSxVQUFVLEVBQUU4VCxtQkFBVU8sSUF2Qkw7QUF3QmpCdkksRUFBQUEsVUFBVSxFQUFFZ0ksbUJBQVVPLElBeEJMO0FBeUJqQjNTLEVBQUFBLGdCQUFnQixFQUFFb1MsbUJBQVVPLElBekJYO0FBMEJqQnpTLEVBQUFBLG1CQUFtQixFQUFFa1MsbUJBQVVPLElBMUJkO0FBMkJqQnZULEVBQUFBLGVBQWUsRUFBRWdULG1CQUFVTyxJQTNCVjtBQTRCakJwVCxFQUFBQSxXQUFXLEVBQUU2UyxtQkFBVU8sSUE1Qk47QUE2QmpCblIsRUFBQUEsaUJBQWlCLEVBQUU0USxtQkFBVU8sSUE3Qlo7QUE4QmpCL1EsRUFBQUEsZ0JBQWdCLEVBQUV3USxtQkFBVU8sSUE5Qlg7QUErQmpCblMsRUFBQUEsU0FBUyxFQUFFb1MsNkJBL0JNO0FBZ0NqQnpSLEVBQUFBLGVBQWUsRUFBRXlSLDZCQWhDQTtBQWlDakJsVixFQUFBQSxRQUFRLEVBQUVtViw2QkFBaUJMO0FBakNWLEM7O2dCQURBcFcsa0Isa0JBcUNHO0FBQ3BCb0YsRUFBQUEsaUJBQWlCLEVBQUUsTUFBTSxJQUFJc1Isb0JBQUosRUFETDtBQUVwQmxSLEVBQUFBLGdCQUFnQixFQUFFLE1BQU0sSUFBSWtSLG9CQUFKO0FBRkosQyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM2LjAvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IFJlYWN0LCB7RnJhZ21lbnR9IGZyb20gJ3JlYWN0JztcbmltcG9ydCBQcm9wVHlwZXMgZnJvbSAncHJvcC10eXBlcyc7XG5pbXBvcnQgY3ggZnJvbSAnY2xhc3NuYW1lcyc7XG5pbXBvcnQge1JhbmdlfSBmcm9tICdhdG9tJztcbmltcG9ydCB7Q29tcG9zaXRlRGlzcG9zYWJsZSwgRGlzcG9zYWJsZX0gZnJvbSAnZXZlbnQta2l0JztcblxuaW1wb3J0IHthdXRvYmluZH0gZnJvbSAnLi4vaGVscGVycyc7XG5pbXBvcnQge2FkZEV2ZW50fSBmcm9tICcuLi9yZXBvcnRlci1wcm94eSc7XG5pbXBvcnQge1JlZkhvbGRlclByb3BUeXBlLCBNdWx0aUZpbGVQYXRjaFByb3BUeXBlLCBJdGVtVHlwZVByb3BUeXBlfSBmcm9tICcuLi9wcm9wLXR5cGVzJztcbmltcG9ydCBBdG9tVGV4dEVkaXRvciBmcm9tICcuLi9hdG9tL2F0b20tdGV4dC1lZGl0b3InO1xuaW1wb3J0IE1hcmtlciBmcm9tICcuLi9hdG9tL21hcmtlcic7XG5pbXBvcnQgTWFya2VyTGF5ZXIgZnJvbSAnLi4vYXRvbS9tYXJrZXItbGF5ZXInO1xuaW1wb3J0IERlY29yYXRpb24gZnJvbSAnLi4vYXRvbS9kZWNvcmF0aW9uJztcbmltcG9ydCBHdXR0ZXIgZnJvbSAnLi4vYXRvbS9ndXR0ZXInO1xuaW1wb3J0IENvbW1hbmRzLCB7Q29tbWFuZH0gZnJvbSAnLi4vYXRvbS9jb21tYW5kcyc7XG5pbXBvcnQgRmlsZVBhdGNoSGVhZGVyVmlldyBmcm9tICcuL2ZpbGUtcGF0Y2gtaGVhZGVyLXZpZXcnO1xuaW1wb3J0IEZpbGVQYXRjaE1ldGFWaWV3IGZyb20gJy4vZmlsZS1wYXRjaC1tZXRhLXZpZXcnO1xuaW1wb3J0IEh1bmtIZWFkZXJWaWV3IGZyb20gJy4vaHVuay1oZWFkZXItdmlldyc7XG5pbXBvcnQgUHVsbFJlcXVlc3RzUmV2aWV3c0NvbnRhaW5lciBmcm9tICcuLi9jb250YWluZXJzL3ByLXJldmlld3MtY29udGFpbmVyJztcbmltcG9ydCBSZWZIb2xkZXIgZnJvbSAnLi4vbW9kZWxzL3JlZi1ob2xkZXInO1xuaW1wb3J0IENoYW5nZWRGaWxlSXRlbSBmcm9tICcuLi9pdGVtcy9jaGFuZ2VkLWZpbGUtaXRlbSc7XG5pbXBvcnQgQ29tbWl0RGV0YWlsSXRlbSBmcm9tICcuLi9pdGVtcy9jb21taXQtZGV0YWlsLWl0ZW0nO1xuaW1wb3J0IElzc3VlaXNoRGV0YWlsSXRlbSBmcm9tICcuLi9pdGVtcy9pc3N1ZWlzaC1kZXRhaWwtaXRlbSc7XG5pbXBvcnQgRmlsZSBmcm9tICcuLi9tb2RlbHMvcGF0Y2gvZmlsZSc7XG5cbmltcG9ydCB7VE9PX0xBUkdFfSBmcm9tICcuLi9tb2RlbHMvcGF0Y2gvcGF0Y2gnO1xuXG5jb25zdCBleGVjdXRhYmxlVGV4dCA9IHtcbiAgW0ZpbGUubW9kZXMuTk9STUFMXTogJ25vbiBleGVjdXRhYmxlJyxcbiAgW0ZpbGUubW9kZXMuRVhFQ1VUQUJMRV06ICdleGVjdXRhYmxlJyxcbn07XG5cbmNvbnN0IE5CU1BfQ0hBUkFDVEVSID0gJ1xcdTAwYTAnO1xuXG5jb25zdCBCTEFOS19MQUJFTCA9ICgpID0+IE5CU1BfQ0hBUkFDVEVSO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBNdWx0aUZpbGVQYXRjaFZpZXcgZXh0ZW5kcyBSZWFjdC5Db21wb25lbnQge1xuICBzdGF0aWMgcHJvcFR5cGVzID0ge1xuICAgIHN0YWdpbmdTdGF0dXM6IFByb3BUeXBlcy5vbmVPZihbJ3N0YWdlZCcsICd1bnN0YWdlZCddKSxcbiAgICBpc1BhcnRpYWxseVN0YWdlZDogUHJvcFR5cGVzLmJvb2wsXG4gICAgbXVsdGlGaWxlUGF0Y2g6IE11bHRpRmlsZVBhdGNoUHJvcFR5cGUuaXNSZXF1aXJlZCxcbiAgICBzZWxlY3Rpb25Nb2RlOiBQcm9wVHlwZXMub25lT2YoWydodW5rJywgJ2xpbmUnXSkuaXNSZXF1aXJlZCxcbiAgICBzZWxlY3RlZFJvd3M6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICBoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zOiBQcm9wVHlwZXMuYm9vbC5pc1JlcXVpcmVkLFxuICAgIHJlcG9zaXRvcnk6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICBoYXNVbmRvSGlzdG9yeTogUHJvcFR5cGVzLmJvb2wsXG5cbiAgICB3b3Jrc3BhY2U6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICBjb21tYW5kczogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIGtleW1hcHM6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICB0b29sdGlwczogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIGNvbmZpZzogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIHB1bGxSZXF1ZXN0OiBQcm9wVHlwZXMub2JqZWN0LFxuXG4gICAgc2VsZWN0ZWRSb3dzQ2hhbmdlZDogUHJvcFR5cGVzLmZ1bmMsXG4gICAgc3dpdGNoVG9Jc3N1ZWlzaDogUHJvcFR5cGVzLmZ1bmMsXG5cbiAgICBkaXZlSW50b01pcnJvclBhdGNoOiBQcm9wVHlwZXMuZnVuYyxcbiAgICBzdXJmYWNlOiBQcm9wVHlwZXMuZnVuYyxcbiAgICBvcGVuRmlsZTogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdG9nZ2xlRmlsZTogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdG9nZ2xlUm93czogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdG9nZ2xlTW9kZUNoYW5nZTogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdG9nZ2xlU3ltbGlua0NoYW5nZTogUHJvcFR5cGVzLmZ1bmMsXG4gICAgdW5kb0xhc3REaXNjYXJkOiBQcm9wVHlwZXMuZnVuYyxcbiAgICBkaXNjYXJkUm93czogUHJvcFR5cGVzLmZ1bmMsXG4gICAgb25XaWxsVXBkYXRlUGF0Y2g6IFByb3BUeXBlcy5mdW5jLFxuICAgIG9uRGlkVXBkYXRlUGF0Y2g6IFByb3BUeXBlcy5mdW5jLFxuICAgIHJlZkVkaXRvcjogUmVmSG9sZGVyUHJvcFR5cGUsXG4gICAgcmVmSW5pdGlhbEZvY3VzOiBSZWZIb2xkZXJQcm9wVHlwZSxcbiAgICBpdGVtVHlwZTogSXRlbVR5cGVQcm9wVHlwZS5pc1JlcXVpcmVkLFxuICB9XG5cbiAgc3RhdGljIGRlZmF1bHRQcm9wcyA9IHtcbiAgICBvbldpbGxVcGRhdGVQYXRjaDogKCkgPT4gbmV3IERpc3Bvc2FibGUoKSxcbiAgICBvbkRpZFVwZGF0ZVBhdGNoOiAoKSA9PiBuZXcgRGlzcG9zYWJsZSgpLFxuICB9XG5cbiAgY29uc3RydWN0b3IocHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgYXV0b2JpbmQoXG4gICAgICB0aGlzLFxuICAgICAgJ2RpZE1vdXNlRG93bk9uSGVhZGVyJywgJ2RpZE1vdXNlRG93bk9uTGluZU51bWJlcicsICdkaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXInLCAnZGlkTW91c2VVcCcsXG4gICAgICAnZGlkQ29uZmlybScsICdkaWRUb2dnbGVTZWxlY3Rpb25Nb2RlJywgJ3NlbGVjdE5leHRIdW5rJywgJ3NlbGVjdFByZXZpb3VzSHVuaycsXG4gICAgICAnZGlkT3BlbkZpbGUnLCAnZGlkQWRkU2VsZWN0aW9uJywgJ2RpZENoYW5nZVNlbGVjdGlvblJhbmdlJywgJ2RpZERlc3Ryb3lTZWxlY3Rpb24nLFxuICAgICAgJ29sZExpbmVOdW1iZXJMYWJlbCcsICduZXdMaW5lTnVtYmVyTGFiZWwnLFxuICAgICk7XG5cbiAgICB0aGlzLm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcyA9IGZhbHNlO1xuICAgIHRoaXMubGFzdE1vdXNlTW92ZUxpbmUgPSBudWxsO1xuICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSBudWxsO1xuICAgIHRoaXMucmVmUm9vdCA9IG5ldyBSZWZIb2xkZXIoKTtcbiAgICB0aGlzLnJlZkVkaXRvciA9IG5ldyBSZWZIb2xkZXIoKTtcbiAgICB0aGlzLnJlZkVkaXRvckVsZW1lbnQgPSBuZXcgUmVmSG9sZGVyKCk7XG4gICAgdGhpcy5tb3VudGVkID0gZmFsc2U7XG5cbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuXG4gICAgdGhpcy5zdWJzLmFkZChcbiAgICAgIHRoaXMucmVmRWRpdG9yLm9ic2VydmUoZWRpdG9yID0+IHtcbiAgICAgICAgdGhpcy5yZWZFZGl0b3JFbGVtZW50LnNldHRlcihlZGl0b3IuZ2V0RWxlbWVudCgpKTtcbiAgICAgICAgaWYgKHRoaXMucHJvcHMucmVmRWRpdG9yKSB7XG4gICAgICAgICAgdGhpcy5wcm9wcy5yZWZFZGl0b3Iuc2V0dGVyKGVkaXRvcik7XG4gICAgICAgIH1cbiAgICAgIH0pLFxuICAgICAgdGhpcy5yZWZFZGl0b3JFbGVtZW50Lm9ic2VydmUoZWxlbWVudCA9PiB7XG4gICAgICAgIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzICYmIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzLnNldHRlcihlbGVtZW50KTtcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICAvLyBTeW5jaHJvbm91c2x5IG1haW50YWluIHRoZSBlZGl0b3IncyBzY3JvbGwgcG9zaXRpb24gYW5kIGxvZ2ljYWwgc2VsZWN0aW9uIGFjcm9zcyBidWZmZXIgdXBkYXRlcy5cbiAgICB0aGlzLnN1cHByZXNzQ2hhbmdlcyA9IGZhbHNlO1xuICAgIGxldCBsYXN0U2Nyb2xsVG9wID0gbnVsbDtcbiAgICBsZXQgbGFzdFNjcm9sbExlZnQgPSBudWxsO1xuICAgIGxldCBsYXN0U2VsZWN0aW9uSW5kZXggPSBudWxsO1xuICAgIHRoaXMuc3Vicy5hZGQoXG4gICAgICB0aGlzLnByb3BzLm9uV2lsbFVwZGF0ZVBhdGNoKCgpID0+IHtcbiAgICAgICAgdGhpcy5zdXBwcmVzc0NoYW5nZXMgPSB0cnVlO1xuICAgICAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgICAgICBsYXN0U2VsZWN0aW9uSW5kZXggPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldE1heFNlbGVjdGlvbkluZGV4KHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzKTtcbiAgICAgICAgICBsYXN0U2Nyb2xsVG9wID0gZWRpdG9yLmdldEVsZW1lbnQoKS5nZXRTY3JvbGxUb3AoKTtcbiAgICAgICAgICBsYXN0U2Nyb2xsTGVmdCA9IGVkaXRvci5nZXRFbGVtZW50KCkuZ2V0U2Nyb2xsTGVmdCgpO1xuICAgICAgICAgIHJldHVybiBudWxsO1xuICAgICAgICB9KTtcbiAgICAgIH0pLFxuICAgICAgdGhpcy5wcm9wcy5vbkRpZFVwZGF0ZVBhdGNoKG5leHRQYXRjaCA9PiB7XG4gICAgICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBlbHNlICovXG4gICAgICAgICAgaWYgKGxhc3RTZWxlY3Rpb25JbmRleCAhPT0gbnVsbCkge1xuICAgICAgICAgICAgY29uc3QgbmV4dFNlbGVjdGlvblJhbmdlID0gbmV4dFBhdGNoLmdldFNlbGVjdGlvblJhbmdlRm9ySW5kZXgobGFzdFNlbGVjdGlvbkluZGV4KTtcbiAgICAgICAgICAgIGlmICh0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUgPT09ICdsaW5lJykge1xuICAgICAgICAgICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2xpbmUnO1xuICAgICAgICAgICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZShuZXh0U2VsZWN0aW9uUmFuZ2UpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgY29uc3QgbmV4dEh1bmtzID0gbmV3IFNldChcbiAgICAgICAgICAgICAgICBSYW5nZS5mcm9tT2JqZWN0KG5leHRTZWxlY3Rpb25SYW5nZSkuZ2V0Um93cygpXG4gICAgICAgICAgICAgICAgICAubWFwKHJvdyA9PiBuZXh0UGF0Y2guZ2V0SHVua0F0KHJvdykpXG4gICAgICAgICAgICAgICAgICAuZmlsdGVyKEJvb2xlYW4pLFxuICAgICAgICAgICAgICApO1xuICAgICAgICAgICAgICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgICAgICAgICAgIGNvbnN0IG5leHRSYW5nZXMgPSBuZXh0SHVua3Muc2l6ZSA+IDBcbiAgICAgICAgICAgICAgICA/IEFycmF5LmZyb20obmV4dEh1bmtzLCBodW5rID0+IGh1bmsuZ2V0UmFuZ2UoKSlcbiAgICAgICAgICAgICAgICA6IFtbWzAsIDBdLCBbMCwgMF1dXTtcblxuICAgICAgICAgICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2h1bmsnO1xuICAgICAgICAgICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMobmV4dFJhbmdlcyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGVsc2UgKi9cbiAgICAgICAgICBpZiAobGFzdFNjcm9sbFRvcCAhPT0gbnVsbCkgeyBlZGl0b3IuZ2V0RWxlbWVudCgpLnNldFNjcm9sbFRvcChsYXN0U2Nyb2xsVG9wKTsgfVxuXG4gICAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIGVsc2UgKi9cbiAgICAgICAgICBpZiAobGFzdFNjcm9sbExlZnQgIT09IG51bGwpIHsgZWRpdG9yLmdldEVsZW1lbnQoKS5zZXRTY3JvbGxMZWZ0KGxhc3RTY3JvbGxMZWZ0KTsgfVxuICAgICAgICAgIHJldHVybiBudWxsO1xuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5zdXBwcmVzc0NoYW5nZXMgPSBmYWxzZTtcbiAgICAgICAgdGhpcy5kaWRDaGFuZ2VTZWxlY3RlZFJvd3MoKTtcbiAgICAgIH0pLFxuICAgICk7XG4gIH1cblxuICBjb21wb25lbnREaWRNb3VudCgpIHtcbiAgICB0aGlzLm1vdW50ZWQgPSB0cnVlO1xuICAgIHRoaXMubWVhc3VyZVBlcmZvcm1hbmNlKCdtb3VudCcpO1xuXG4gICAgd2luZG93LmFkZEV2ZW50TGlzdGVuZXIoJ21vdXNldXAnLCB0aGlzLmRpZE1vdXNlVXApO1xuICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgLy8gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaCBpcyBndWFyYW50ZWVkIHRvIGNvbnRhaW4gYXQgbGVhc3Qgb25lIEZpbGVQYXRjaCBpZiA8QXRvbVRleHRFZGl0b3I+IGlzIHJlbmRlcmVkLlxuICAgICAgY29uc3QgW2ZpcnN0UGF0Y2hdID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hlcygpO1xuICAgICAgY29uc3QgW2ZpcnN0SHVua10gPSBmaXJzdFBhdGNoLmdldEh1bmtzKCk7XG4gICAgICBpZiAoIWZpcnN0SHVuaykge1xuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgIH1cblxuICAgICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdodW5rJztcbiAgICAgIGVkaXRvci5zZXRTZWxlY3RlZEJ1ZmZlclJhbmdlKGZpcnN0SHVuay5nZXRSYW5nZSgpKTtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH0pO1xuXG4gICAgdGhpcy5zdWJzLmFkZChcbiAgICAgIHRoaXMucHJvcHMuY29uZmlnLm9uRGlkQ2hhbmdlKCdnaXRodWIuc2hvd0RpZmZJY29uR3V0dGVyJywgKCkgPT4gdGhpcy5mb3JjZVVwZGF0ZSgpKSxcbiAgICApO1xuICB9XG5cbiAgY29tcG9uZW50RGlkVXBkYXRlKHByZXZQcm9wcykge1xuICAgIHRoaXMubWVhc3VyZVBlcmZvcm1hbmNlKCd1cGRhdGUnKTtcblxuICAgIGlmIChwcmV2UHJvcHMucmVmSW5pdGlhbEZvY3VzICE9PSB0aGlzLnByb3BzLnJlZkluaXRpYWxGb2N1cykge1xuICAgICAgcHJldlByb3BzLnJlZkluaXRpYWxGb2N1cyAmJiBwcmV2UHJvcHMucmVmSW5pdGlhbEZvY3VzLnNldHRlcihudWxsKTtcbiAgICAgIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzICYmIHRoaXMucmVmRWRpdG9yRWxlbWVudC5tYXAodGhpcy5wcm9wcy5yZWZJbml0aWFsRm9jdXMuc2V0dGVyKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaCA9PT0gcHJldlByb3BzLm11bHRpRmlsZVBhdGNoKSB7XG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gbnVsbDtcbiAgICB9XG4gIH1cblxuICBjb21wb25lbnRXaWxsVW5tb3VudCgpIHtcbiAgICB3aW5kb3cucmVtb3ZlRXZlbnRMaXN0ZW5lcignbW91c2V1cCcsIHRoaXMuZGlkTW91c2VVcCk7XG4gICAgdGhpcy5zdWJzLmRpc3Bvc2UoKTtcbiAgICB0aGlzLm1vdW50ZWQgPSBmYWxzZTtcbiAgICBwZXJmb3JtYW5jZS5jbGVhck1hcmtzKCk7XG4gICAgcGVyZm9ybWFuY2UuY2xlYXJNZWFzdXJlcygpO1xuICB9XG5cbiAgcmVuZGVyKCkge1xuICAgIGNvbnN0IHJvb3RDbGFzcyA9IGN4KFxuICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3JyxcbiAgICAgIHtbYGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LS0ke3RoaXMucHJvcHMuc3RhZ2luZ1N0YXR1c31gXTogdGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzfSxcbiAgICAgIHsnZ2l0aHViLUZpbGVQYXRjaFZpZXctLWJsYW5rJzogIXRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guYW55UHJlc2VudCgpfSxcbiAgICAgIHsnZ2l0aHViLUZpbGVQYXRjaFZpZXctLWh1bmtNb2RlJzogdGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlID09PSAnaHVuayd9LFxuICAgICk7XG5cbiAgICBpZiAodGhpcy5tb3VudGVkKSB7XG4gICAgICBwZXJmb3JtYW5jZS5tYXJrKCdNdWx0aUZpbGVQYXRjaFZpZXctdXBkYXRlLXN0YXJ0Jyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHBlcmZvcm1hbmNlLm1hcmsoJ011bHRpRmlsZVBhdGNoVmlldy1tb3VudC1zdGFydCcpO1xuICAgIH1cblxuICAgIHJldHVybiAoXG4gICAgICA8ZGl2IGNsYXNzTmFtZT17cm9vdENsYXNzfSByZWY9e3RoaXMucmVmUm9vdC5zZXR0ZXJ9PlxuICAgICAgICB7dGhpcy5yZW5kZXJDb21tYW5kcygpfVxuXG4gICAgICAgIDxtYWluIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LWNvbnRhaW5lclwiPlxuICAgICAgICAgIHt0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmFueVByZXNlbnQoKSA/IHRoaXMucmVuZGVyTm9uRW1wdHlQYXRjaCgpIDogdGhpcy5yZW5kZXJFbXB0eVBhdGNoKCl9XG4gICAgICAgIDwvbWFpbj5cbiAgICAgIDwvZGl2PlxuICAgICk7XG4gIH1cblxuICByZW5kZXJDb21tYW5kcygpIHtcbiAgICBpZiAodGhpcy5wcm9wcy5pdGVtVHlwZSA9PT0gQ29tbWl0RGV0YWlsSXRlbSB8fCB0aGlzLnByb3BzLml0ZW1UeXBlID09PSBJc3N1ZWlzaERldGFpbEl0ZW0pIHtcbiAgICAgIHJldHVybiAoXG4gICAgICAgIDxDb21tYW5kcyByZWdpc3RyeT17dGhpcy5wcm9wcy5jb21tYW5kc30gdGFyZ2V0PXt0aGlzLnJlZlJvb3R9PlxuICAgICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6c2VsZWN0LW5leHQtaHVua1wiIGNhbGxiYWNrPXt0aGlzLnNlbGVjdE5leHRIdW5rfSAvPlxuICAgICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6c2VsZWN0LXByZXZpb3VzLWh1bmtcIiBjYWxsYmFjaz17dGhpcy5zZWxlY3RQcmV2aW91c0h1bmt9IC8+XG4gICAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1Yjp0b2dnbGUtcGF0Y2gtc2VsZWN0aW9uLW1vZGVcIiBjYWxsYmFjaz17dGhpcy5kaWRUb2dnbGVTZWxlY3Rpb25Nb2RlfSAvPlxuICAgICAgICA8L0NvbW1hbmRzPlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBsZXQgc3RhZ2VNb2RlQ29tbWFuZCA9IG51bGw7XG4gICAgbGV0IHN0YWdlU3ltbGlua0NvbW1hbmQgPSBudWxsO1xuXG4gICAgaWYgKHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZGlkQW55Q2hhbmdlRXhlY3V0YWJsZU1vZGUoKSkge1xuICAgICAgY29uc3QgY29tbWFuZCA9IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1cyA9PT0gJ3Vuc3RhZ2VkJ1xuICAgICAgICA/ICdnaXRodWI6c3RhZ2UtZmlsZS1tb2RlLWNoYW5nZSdcbiAgICAgICAgOiAnZ2l0aHViOnVuc3RhZ2UtZmlsZS1tb2RlLWNoYW5nZSc7XG4gICAgICBzdGFnZU1vZGVDb21tYW5kID0gPENvbW1hbmQgY29tbWFuZD17Y29tbWFuZH0gY2FsbGJhY2s9e3RoaXMuZGlkVG9nZ2xlTW9kZUNoYW5nZX0gLz47XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guYW55SGF2ZVR5cGVjaGFuZ2UoKSkge1xuICAgICAgY29uc3QgY29tbWFuZCA9IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1cyA9PT0gJ3Vuc3RhZ2VkJ1xuICAgICAgICA/ICdnaXRodWI6c3RhZ2Utc3ltbGluay1jaGFuZ2UnXG4gICAgICAgIDogJ2dpdGh1Yjp1bnN0YWdlLXN5bWxpbmstY2hhbmdlJztcbiAgICAgIHN0YWdlU3ltbGlua0NvbW1hbmQgPSA8Q29tbWFuZCBjb21tYW5kPXtjb21tYW5kfSBjYWxsYmFjaz17dGhpcy5kaWRUb2dnbGVTeW1saW5rQ2hhbmdlfSAvPjtcbiAgICB9XG5cbiAgICByZXR1cm4gKFxuICAgICAgPENvbW1hbmRzIHJlZ2lzdHJ5PXt0aGlzLnByb3BzLmNvbW1hbmRzfSB0YXJnZXQ9e3RoaXMucmVmUm9vdH0+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6c2VsZWN0LW5leHQtaHVua1wiIGNhbGxiYWNrPXt0aGlzLnNlbGVjdE5leHRIdW5rfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOnNlbGVjdC1wcmV2aW91cy1odW5rXCIgY2FsbGJhY2s9e3RoaXMuc2VsZWN0UHJldmlvdXNIdW5rfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiY29yZTpjb25maXJtXCIgY2FsbGJhY2s9e3RoaXMuZGlkQ29uZmlybX0gLz5cbiAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImNvcmU6dW5kb1wiIGNhbGxiYWNrPXt0aGlzLnVuZG9MYXN0RGlzY2FyZEZyb21Db3JlVW5kb30gLz5cbiAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjpkaXNjYXJkLXNlbGVjdGVkLWxpbmVzXCIgY2FsbGJhY2s9e3RoaXMuZGlzY2FyZFNlbGVjdGlvbkZyb21Db21tYW5kfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOmp1bXAtdG8tZmlsZVwiIGNhbGxiYWNrPXt0aGlzLmRpZE9wZW5GaWxlfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOnN1cmZhY2VcIiBjYWxsYmFjaz17dGhpcy5wcm9wcy5zdXJmYWNlfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOnRvZ2dsZS1wYXRjaC1zZWxlY3Rpb24tbW9kZVwiIGNhbGxiYWNrPXt0aGlzLmRpZFRvZ2dsZVNlbGVjdGlvbk1vZGV9IC8+XG4gICAgICAgIHtzdGFnZU1vZGVDb21tYW5kfVxuICAgICAgICB7c3RhZ2VTeW1saW5rQ29tbWFuZH1cbiAgICAgICAgey8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovIGF0b20uaW5EZXZNb2RlKCkgJiZcbiAgICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOmluc3BlY3QtcGF0Y2hcIiBjYWxsYmFjaz17KCkgPT4ge1xuICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0UGF0Y2hCdWZmZXIoKS5pbnNwZWN0KHtcbiAgICAgICAgICAgICAgbGF5ZXJOYW1lczogWydwYXRjaCcsICdodW5rJ10sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgICAgfX1cbiAgICAgICAgICAvPlxuICAgICAgICB9XG4gICAgICAgIHsvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqLyBhdG9tLmluRGV2TW9kZSgpICYmXG4gICAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjppbnNwZWN0LXJlZ2lvbnNcIiBjYWxsYmFjaz17KCkgPT4ge1xuICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWNvbnNvbGVcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0UGF0Y2hCdWZmZXIoKS5pbnNwZWN0KHtcbiAgICAgICAgICAgICAgbGF5ZXJOYW1lczogWyd1bmNoYW5nZWQnLCAnZGVsZXRpb24nLCAnYWRkaXRpb24nLCAnbm9uZXdsaW5lJ10sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgICAgfX1cbiAgICAgICAgICAvPlxuICAgICAgICB9XG4gICAgICAgIHsvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqLyBhdG9tLmluRGV2TW9kZSgpICYmXG4gICAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjppbnNwZWN0LW1mcFwiIGNhbGxiYWNrPXsoKSA9PiB7XG4gICAgICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tY29uc29sZVxuICAgICAgICAgICAgY29uc29sZS5sb2codGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5pbnNwZWN0KCkpO1xuICAgICAgICAgIH19XG4gICAgICAgICAgLz5cbiAgICAgICAgfVxuICAgICAgPC9Db21tYW5kcz5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyRW1wdHlQYXRjaCgpIHtcbiAgICByZXR1cm4gPHAgY2xhc3NOYW1lPVwiZ2l0aHViLUZpbGVQYXRjaFZpZXctbWVzc2FnZSBpY29uIGljb24taW5mb1wiPk5vIGNoYW5nZXMgdG8gZGlzcGxheTwvcD47XG4gIH1cblxuICByZW5kZXJOb25FbXB0eVBhdGNoKCkge1xuICAgIHJldHVybiAoXG4gICAgICA8QXRvbVRleHRFZGl0b3JcbiAgICAgICAgd29ya3NwYWNlPXt0aGlzLnByb3BzLndvcmtzcGFjZX1cblxuICAgICAgICBidWZmZXI9e3RoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0QnVmZmVyKCl9XG4gICAgICAgIGxpbmVOdW1iZXJHdXR0ZXJWaXNpYmxlPXtmYWxzZX1cbiAgICAgICAgYXV0b1dpZHRoPXtmYWxzZX1cbiAgICAgICAgYXV0b0hlaWdodD17ZmFsc2V9XG4gICAgICAgIHJlYWRPbmx5PXt0cnVlfVxuICAgICAgICBzb2Z0V3JhcHBlZD17dHJ1ZX1cblxuICAgICAgICBkaWRBZGRTZWxlY3Rpb249e3RoaXMuZGlkQWRkU2VsZWN0aW9ufVxuICAgICAgICBkaWRDaGFuZ2VTZWxlY3Rpb25SYW5nZT17dGhpcy5kaWRDaGFuZ2VTZWxlY3Rpb25SYW5nZX1cbiAgICAgICAgZGlkRGVzdHJveVNlbGVjdGlvbj17dGhpcy5kaWREZXN0cm95U2VsZWN0aW9ufVxuICAgICAgICByZWZNb2RlbD17dGhpcy5yZWZFZGl0b3J9XG4gICAgICAgIGhpZGVFbXB0aW5lc3M9e3RydWV9PlxuXG4gICAgICAgIDxHdXR0ZXJcbiAgICAgICAgICBuYW1lPVwib2xkLWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgcHJpb3JpdHk9ezF9XG4gICAgICAgICAgY2xhc3NOYW1lPVwib2xkXCJcbiAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgIGxhYmVsRm49e3RoaXMub2xkTGluZU51bWJlckxhYmVsfVxuICAgICAgICAgIG9uTW91c2VEb3duPXt0aGlzLmRpZE1vdXNlRG93bk9uTGluZU51bWJlcn1cbiAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgIC8+XG4gICAgICAgIDxHdXR0ZXJcbiAgICAgICAgICBuYW1lPVwibmV3LWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgcHJpb3JpdHk9ezJ9XG4gICAgICAgICAgY2xhc3NOYW1lPVwibmV3XCJcbiAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgIGxhYmVsRm49e3RoaXMubmV3TGluZU51bWJlckxhYmVsfVxuICAgICAgICAgIG9uTW91c2VEb3duPXt0aGlzLmRpZE1vdXNlRG93bk9uTGluZU51bWJlcn1cbiAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgIC8+XG4gICAgICAgIHt0aGlzLnByb3BzLmNvbmZpZy5nZXQoJ2dpdGh1Yi5zaG93RGlmZkljb25HdXR0ZXInKSAmJiAoXG4gICAgICAgICAgPEd1dHRlclxuICAgICAgICAgICAgbmFtZT1cImRpZmYtaWNvbnNcIlxuICAgICAgICAgICAgcHJpb3JpdHk9ezN9XG4gICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPVwiaWNvbnNcIlxuICAgICAgICAgICAgbGFiZWxGbj17QkxBTktfTEFCRUx9XG4gICAgICAgICAgICBvbk1vdXNlRG93bj17dGhpcy5kaWRNb3VzZURvd25PbkxpbmVOdW1iZXJ9XG4gICAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgICAgLz5cbiAgICAgICAgKX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJQdWxsUmVxdWVzdFJldmlld3MoKX1cblxuICAgICAgICB7dGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hlcygpLm1hcCh0aGlzLnJlbmRlckZpbGVQYXRjaERlY29yYXRpb25zKX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJMaW5lRGVjb3JhdGlvbnMoXG4gICAgICAgICAgQXJyYXkuZnJvbSh0aGlzLnByb3BzLnNlbGVjdGVkUm93cywgcm93ID0+IFJhbmdlLmZyb21PYmplY3QoW1tyb3csIDBdLCBbcm93LCBJbmZpbml0eV1dKSksXG4gICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LWxpbmUtLXNlbGVjdGVkJyxcbiAgICAgICAgICB7Z3V0dGVyOiB0cnVlLCBpY29uOiB0cnVlLCBsaW5lOiB0cnVlfSxcbiAgICAgICAgKX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9uc09uTGF5ZXIoXG4gICAgICAgICAgdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRBZGRpdGlvbkxheWVyKCksXG4gICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LWxpbmUtLWFkZGVkJyxcbiAgICAgICAgICB7aWNvbjogdHJ1ZSwgbGluZTogdHJ1ZX0sXG4gICAgICAgICl9XG4gICAgICAgIHt0aGlzLnJlbmRlckRlY29yYXRpb25zT25MYXllcihcbiAgICAgICAgICB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldERlbGV0aW9uTGF5ZXIoKSxcbiAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbGluZS0tZGVsZXRlZCcsXG4gICAgICAgICAge2ljb246IHRydWUsIGxpbmU6IHRydWV9LFxuICAgICAgICApfVxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9uc09uTGF5ZXIoXG4gICAgICAgICAgdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXROb05ld2xpbmVMYXllcigpLFxuICAgICAgICAgICdnaXRodWItRmlsZVBhdGNoVmlldy1saW5lLS1ub25ld2xpbmUnLFxuICAgICAgICAgIHtpY29uOiB0cnVlLCBsaW5lOiB0cnVlfSxcbiAgICAgICAgKX1cblxuICAgICAgPC9BdG9tVGV4dEVkaXRvcj5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyUHVsbFJlcXVlc3RSZXZpZXdzKCkge1xuICAgIGlmICh0aGlzLnByb3BzLml0ZW1UeXBlID09PSBJc3N1ZWlzaERldGFpbEl0ZW0pIHtcbiAgICAgIC8vIFwiZm9yY2VSZXJlbmRlclwiIGVuc3VyZXMgdGhhdCB0aGUgUHVsbFJlcXVlc3RDb21tZW50c1ZpZXcgcmUtcmVuZGVycyBlYWNoIHRpbWUgdGhhdCB0aGUgTXVsdGlGaWxlUGF0Y2hWaWV3IGRvZXMuXG4gICAgICAvLyBJdCBkb2Vzbid0IHJlLXF1ZXJ5IGZvciByZXZpZXdzLCBidXQgaXQgZG9lcyByZS1jaGVjayBwYXRjaCB2aXNpYmlsaXR5LlxuICAgICAgcmV0dXJuIChcbiAgICAgICAgPFB1bGxSZXF1ZXN0c1Jldmlld3NDb250YWluZXJcbiAgICAgICAgICBwdWxsUmVxdWVzdD17dGhpcy5wcm9wcy5wdWxsUmVxdWVzdH1cbiAgICAgICAgICBnZXRCdWZmZXJSb3dGb3JEaWZmUG9zaXRpb249e3RoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0QnVmZmVyUm93Rm9yRGlmZlBvc2l0aW9ufVxuICAgICAgICAgIGlzUGF0Y2hWaXNpYmxlPXt0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmlzUGF0Y2hWaXNpYmxlfVxuICAgICAgICAgIHN3aXRjaFRvSXNzdWVpc2g9e3RoaXMucHJvcHMuc3dpdGNoVG9Jc3N1ZWlzaH1cbiAgICAgICAgICBmb3JjZVJlcmVuZGVyPXt7fX1cbiAgICAgICAgLz5cbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgfVxuXG4gIHJlbmRlckZpbGVQYXRjaERlY29yYXRpb25zID0gKGZpbGVQYXRjaCwgaW5kZXgpID0+IHtcbiAgICBjb25zdCBpc0NvbGxhcHNlZCA9ICFmaWxlUGF0Y2guZ2V0UmVuZGVyU3RhdHVzKCkuaXNWaXNpYmxlKCk7XG4gICAgY29uc3QgaXNFbXB0eSA9IGZpbGVQYXRjaC5nZXRNYXJrZXIoKS5nZXRSYW5nZSgpLmlzRW1wdHkoKTtcbiAgICBjb25zdCBhdEVuZCA9IGZpbGVQYXRjaC5nZXRTdGFydFJhbmdlKCkuc3RhcnQuaXNFcXVhbCh0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldEJ1ZmZlcigpLmdldEVuZFBvc2l0aW9uKCkpO1xuICAgIGNvbnN0IHBvc2l0aW9uID0gaXNFbXB0eSAmJiBhdEVuZCA/ICdhZnRlcicgOiAnYmVmb3JlJztcblxuICAgIHJldHVybiAoXG4gICAgICA8RnJhZ21lbnQga2V5PXtmaWxlUGF0Y2guZ2V0UGF0aCgpfT5cbiAgICAgICAgPE1hcmtlciBpbnZhbGlkYXRlPVwibmV2ZXJcIiBidWZmZXJSYW5nZT17ZmlsZVBhdGNoLmdldFN0YXJ0UmFuZ2UoKX0+XG4gICAgICAgICAgPERlY29yYXRpb24gdHlwZT1cImJsb2NrXCIgcG9zaXRpb249e3Bvc2l0aW9ufSBvcmRlcj17aW5kZXh9IGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LWNvbnRyb2xCbG9ja1wiPlxuICAgICAgICAgICAgPEZpbGVQYXRjaEhlYWRlclZpZXdcbiAgICAgICAgICAgICAgaXRlbVR5cGU9e3RoaXMucHJvcHMuaXRlbVR5cGV9XG4gICAgICAgICAgICAgIHJlbFBhdGg9e2ZpbGVQYXRjaC5nZXRQYXRoKCl9XG4gICAgICAgICAgICAgIG5ld1BhdGg9e2ZpbGVQYXRjaC5nZXRTdGF0dXMoKSA9PT0gJ3JlbmFtZWQnID8gZmlsZVBhdGNoLmdldE5ld1BhdGgoKSA6IG51bGx9XG4gICAgICAgICAgICAgIHN0YWdpbmdTdGF0dXM9e3RoaXMucHJvcHMuc3RhZ2luZ1N0YXR1c31cbiAgICAgICAgICAgICAgaXNQYXJ0aWFsbHlTdGFnZWQ9e3RoaXMucHJvcHMuaXNQYXJ0aWFsbHlTdGFnZWR9XG4gICAgICAgICAgICAgIGhhc1VuZG9IaXN0b3J5PXt0aGlzLnByb3BzLmhhc1VuZG9IaXN0b3J5fVxuICAgICAgICAgICAgICBoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zPXt0aGlzLnByb3BzLmhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnN9XG5cbiAgICAgICAgICAgICAgdG9vbHRpcHM9e3RoaXMucHJvcHMudG9vbHRpcHN9XG5cbiAgICAgICAgICAgICAgdW5kb0xhc3REaXNjYXJkPXsoKSA9PiB0aGlzLnVuZG9MYXN0RGlzY2FyZEZyb21CdXR0b24oZmlsZVBhdGNoKX1cbiAgICAgICAgICAgICAgZGl2ZUludG9NaXJyb3JQYXRjaD17KCkgPT4gdGhpcy5wcm9wcy5kaXZlSW50b01pcnJvclBhdGNoKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICAgIG9wZW5GaWxlPXsoKSA9PiB0aGlzLmRpZE9wZW5GaWxlKHtzZWxlY3RlZEZpbGVQYXRjaDogZmlsZVBhdGNofSl9XG4gICAgICAgICAgICAgIHRvZ2dsZUZpbGU9eygpID0+IHRoaXMucHJvcHMudG9nZ2xlRmlsZShmaWxlUGF0Y2gpfVxuXG4gICAgICAgICAgICAgIGlzQ29sbGFwc2VkPXtpc0NvbGxhcHNlZH1cbiAgICAgICAgICAgICAgdHJpZ2dlckNvbGxhcHNlPXsoKSA9PiB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmNvbGxhcHNlRmlsZVBhdGNoKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICAgIHRyaWdnZXJFeHBhbmQ9eygpID0+IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZXhwYW5kRmlsZVBhdGNoKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICAvPlxuICAgICAgICAgICAgeyFpc0NvbGxhcHNlZCAmJiB0aGlzLnJlbmRlclN5bWxpbmtDaGFuZ2VNZXRhKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICB7IWlzQ29sbGFwc2VkICYmIHRoaXMucmVuZGVyRXhlY3V0YWJsZU1vZGVDaGFuZ2VNZXRhKGZpbGVQYXRjaCl9XG4gICAgICAgICAgPC9EZWNvcmF0aW9uPlxuICAgICAgICA8L01hcmtlcj5cblxuICAgICAgICB7ZmlsZVBhdGNoLmdldFBhdGNoKCkuZ2V0UmVuZGVyU3RhdHVzKCkgPT09IFRPT19MQVJHRSAmJiB0aGlzLnJlbmRlckRpZmZHYXRlKGZpbGVQYXRjaCwgcG9zaXRpb24sIGluZGV4KX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJIdW5rSGVhZGVycyhmaWxlUGF0Y2gsIGluZGV4KX1cbiAgICAgIDwvRnJhZ21lbnQ+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckRpZmZHYXRlKGZpbGVQYXRjaCwgcG9zaXRpb24sIG9yZGVyT2Zmc2V0KSB7XG4gICAgY29uc3Qgc2hvd0RpZmYgPSAoKSA9PiB7XG4gICAgICBhZGRFdmVudCgnZXhwYW5kLWZpbGUtcGF0Y2gnLCB7Y29tcG9uZW50OiB0aGlzLmNvbnN0cnVjdG9yLm5hbWUsIHBhY2thZ2U6ICdnaXRodWInfSk7XG4gICAgICB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmV4cGFuZEZpbGVQYXRjaChmaWxlUGF0Y2gpO1xuICAgIH07XG4gICAgcmV0dXJuIChcbiAgICAgIDxNYXJrZXIgaW52YWxpZGF0ZT1cIm5ldmVyXCIgYnVmZmVyUmFuZ2U9e2ZpbGVQYXRjaC5nZXRTdGFydFJhbmdlKCl9PlxuICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgIHR5cGU9XCJibG9ja1wiXG4gICAgICAgICAgb3JkZXI9e29yZGVyT2Zmc2V0ICsgMC4xfVxuICAgICAgICAgIHBvc2l0aW9uPXtwb3NpdGlvbn1cbiAgICAgICAgICBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1jb250cm9sQmxvY2tcIj5cblxuICAgICAgICAgIDxwIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1lc3NhZ2UgaWNvbiBpY29uLWluZm9cIj5cbiAgICAgICAgICAgIExhcmdlIGRpZmZzIGFyZSBjb2xsYXBzZWQgYnkgZGVmYXVsdCBmb3IgcGVyZm9ybWFuY2UgcmVhc29ucy5cbiAgICAgICAgICAgIDxiciAvPlxuICAgICAgICAgICAgPGJ1dHRvbiBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1zaG93RGlmZkJ1dHRvblwiIG9uQ2xpY2s9e3Nob3dEaWZmfT4gTG9hZCBEaWZmPC9idXR0b24+XG4gICAgICAgICAgPC9wPlxuXG4gICAgICAgIDwvRGVjb3JhdGlvbj5cbiAgICAgIDwvTWFya2VyPlxuICAgICk7XG4gIH1cblxuICByZW5kZXJFeGVjdXRhYmxlTW9kZUNoYW5nZU1ldGEoZmlsZVBhdGNoKSB7XG4gICAgaWYgKCFmaWxlUGF0Y2guZGlkQ2hhbmdlRXhlY3V0YWJsZU1vZGUoKSkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuXG4gICAgY29uc3Qgb2xkTW9kZSA9IGZpbGVQYXRjaC5nZXRPbGRNb2RlKCk7XG4gICAgY29uc3QgbmV3TW9kZSA9IGZpbGVQYXRjaC5nZXROZXdNb2RlKCk7XG5cbiAgICBjb25zdCBhdHRycyA9IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1cyA9PT0gJ3Vuc3RhZ2VkJ1xuICAgICAgPyB7XG4gICAgICAgIGFjdGlvbkljb246ICdpY29uLW1vdmUtZG93bicsXG4gICAgICAgIGFjdGlvblRleHQ6ICdTdGFnZSBNb2RlIENoYW5nZScsXG4gICAgICB9XG4gICAgICA6IHtcbiAgICAgICAgYWN0aW9uSWNvbjogJ2ljb24tbW92ZS11cCcsXG4gICAgICAgIGFjdGlvblRleHQ6ICdVbnN0YWdlIE1vZGUgQ2hhbmdlJyxcbiAgICAgIH07XG5cbiAgICByZXR1cm4gKFxuICAgICAgPEZpbGVQYXRjaE1ldGFWaWV3XG4gICAgICAgIHRpdGxlPVwiTW9kZSBjaGFuZ2VcIlxuICAgICAgICBhY3Rpb25JY29uPXthdHRycy5hY3Rpb25JY29ufVxuICAgICAgICBhY3Rpb25UZXh0PXthdHRycy5hY3Rpb25UZXh0fVxuICAgICAgICBpdGVtVHlwZT17dGhpcy5wcm9wcy5pdGVtVHlwZX1cbiAgICAgICAgYWN0aW9uPXsoKSA9PiB0aGlzLnByb3BzLnRvZ2dsZU1vZGVDaGFuZ2UoZmlsZVBhdGNoKX0+XG4gICAgICAgIDxGcmFnbWVudD5cbiAgICAgICAgICBGaWxlIGNoYW5nZWQgbW9kZVxuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1yZW1vdmVkXCI+XG4gICAgICAgICAgICBmcm9tIHtleGVjdXRhYmxlVGV4dFtvbGRNb2RlXX0gPGNvZGU+e29sZE1vZGV9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgICA8c3BhbiBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZiBnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZi0tYWRkZWRcIj5cbiAgICAgICAgICAgIHRvIHtleGVjdXRhYmxlVGV4dFtuZXdNb2RlXX0gPGNvZGU+e25ld01vZGV9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgIDwvRmlsZVBhdGNoTWV0YVZpZXc+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlclN5bWxpbmtDaGFuZ2VNZXRhKGZpbGVQYXRjaCkge1xuICAgIGlmICghZmlsZVBhdGNoLmhhc1N5bWxpbmsoKSkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuXG4gICAgbGV0IGRldGFpbCA9IDxkaXYgLz47XG4gICAgbGV0IHRpdGxlID0gJyc7XG4gICAgY29uc3Qgb2xkU3ltbGluayA9IGZpbGVQYXRjaC5nZXRPbGRTeW1saW5rKCk7XG4gICAgY29uc3QgbmV3U3ltbGluayA9IGZpbGVQYXRjaC5nZXROZXdTeW1saW5rKCk7XG4gICAgaWYgKG9sZFN5bWxpbmsgJiYgbmV3U3ltbGluaykge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGluayBjaGFuZ2VkXG4gICAgICAgICAgPHNwYW4gY2xhc3NOYW1lPXtjeChcbiAgICAgICAgICAgICdnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZicsXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYtLWZ1bGxXaWR0aCcsXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYtLXJlbW92ZWQnLFxuICAgICAgICAgICl9PlxuICAgICAgICAgICAgZnJvbSA8Y29kZT57b2xkU3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPlxuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT17Y3goXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYnLFxuICAgICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1mdWxsV2lkdGgnLFxuICAgICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1hZGRlZCcsXG4gICAgICAgICAgKX0+XG4gICAgICAgICAgICB0byA8Y29kZT57bmV3U3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPi5cbiAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgICk7XG4gICAgICB0aXRsZSA9ICdTeW1saW5rIGNoYW5nZWQnO1xuICAgIH0gZWxzZSBpZiAob2xkU3ltbGluayAmJiAhbmV3U3ltbGluaykge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGlua1xuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1yZW1vdmVkXCI+XG4gICAgICAgICAgICB0byA8Y29kZT57b2xkU3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPlxuICAgICAgICAgIGRlbGV0ZWQuXG4gICAgICAgIDwvRnJhZ21lbnQ+XG4gICAgICApO1xuICAgICAgdGl0bGUgPSAnU3ltbGluayBkZWxldGVkJztcbiAgICB9IGVsc2Uge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGlua1xuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1hZGRlZFwiPlxuICAgICAgICAgICAgdG8gPGNvZGU+e25ld1N5bWxpbmt9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgICBjcmVhdGVkLlxuICAgICAgICA8L0ZyYWdtZW50PlxuICAgICAgKTtcbiAgICAgIHRpdGxlID0gJ1N5bWxpbmsgY3JlYXRlZCc7XG4gICAgfVxuXG4gICAgY29uc3QgYXR0cnMgPSB0aGlzLnByb3BzLnN0YWdpbmdTdGF0dXMgPT09ICd1bnN0YWdlZCdcbiAgICAgID8ge1xuICAgICAgICBhY3Rpb25JY29uOiAnaWNvbi1tb3ZlLWRvd24nLFxuICAgICAgICBhY3Rpb25UZXh0OiAnU3RhZ2UgU3ltbGluayBDaGFuZ2UnLFxuICAgICAgfVxuICAgICAgOiB7XG4gICAgICAgIGFjdGlvbkljb246ICdpY29uLW1vdmUtdXAnLFxuICAgICAgICBhY3Rpb25UZXh0OiAnVW5zdGFnZSBTeW1saW5rIENoYW5nZScsXG4gICAgICB9O1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxGaWxlUGF0Y2hNZXRhVmlld1xuICAgICAgICB0aXRsZT17dGl0bGV9XG4gICAgICAgIGFjdGlvbkljb249e2F0dHJzLmFjdGlvbkljb259XG4gICAgICAgIGFjdGlvblRleHQ9e2F0dHJzLmFjdGlvblRleHR9XG4gICAgICAgIGl0ZW1UeXBlPXt0aGlzLnByb3BzLml0ZW1UeXBlfVxuICAgICAgICBhY3Rpb249eygpID0+IHRoaXMucHJvcHMudG9nZ2xlU3ltbGlua0NoYW5nZShmaWxlUGF0Y2gpfT5cbiAgICAgICAgPEZyYWdtZW50PlxuICAgICAgICAgIHtkZXRhaWx9XG4gICAgICAgIDwvRnJhZ21lbnQ+XG4gICAgICA8L0ZpbGVQYXRjaE1ldGFWaWV3PlxuICAgICk7XG4gIH1cblxuICByZW5kZXJIdW5rSGVhZGVycyhmaWxlUGF0Y2gsIG9yZGVyT2Zmc2V0KSB7XG4gICAgY29uc3QgdG9nZ2xlVmVyYiA9IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1cyA9PT0gJ3Vuc3RhZ2VkJyA/ICdTdGFnZScgOiAnVW5zdGFnZSc7XG4gICAgY29uc3Qgc2VsZWN0ZWRIdW5rcyA9IG5ldyBTZXQoXG4gICAgICBBcnJheS5mcm9tKHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzLCByb3cgPT4gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQocm93KSksXG4gICAgKTtcblxuICAgIHJldHVybiAoXG4gICAgICA8RnJhZ21lbnQ+XG4gICAgICAgIDxNYXJrZXJMYXllcj5cbiAgICAgICAgICB7ZmlsZVBhdGNoLmdldEh1bmtzKCkubWFwKChodW5rLCBpbmRleCkgPT4ge1xuICAgICAgICAgICAgY29uc3QgY29udGFpbnNTZWxlY3Rpb24gPSB0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUgPT09ICdsaW5lJyAmJiBzZWxlY3RlZEh1bmtzLmhhcyhodW5rKTtcbiAgICAgICAgICAgIGNvbnN0IGlzU2VsZWN0ZWQgPSAodGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlID09PSAnaHVuaycpICYmIHNlbGVjdGVkSHVua3MuaGFzKGh1bmspO1xuXG4gICAgICAgICAgICBsZXQgYnV0dG9uU3VmZml4ID0gJyc7XG4gICAgICAgICAgICBpZiAoY29udGFpbnNTZWxlY3Rpb24pIHtcbiAgICAgICAgICAgICAgYnV0dG9uU3VmZml4ICs9ICdTZWxlY3RlZCBMaW5lJztcbiAgICAgICAgICAgICAgaWYgKHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzLnNpemUgPiAxKSB7XG4gICAgICAgICAgICAgICAgYnV0dG9uU3VmZml4ICs9ICdzJztcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgYnV0dG9uU3VmZml4ICs9ICdIdW5rJztcbiAgICAgICAgICAgICAgaWYgKHNlbGVjdGVkSHVua3Muc2l6ZSA+IDEpIHtcbiAgICAgICAgICAgICAgICBidXR0b25TdWZmaXggKz0gJ3MnO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIGNvbnN0IHRvZ2dsZVNlbGVjdGlvbkxhYmVsID0gYCR7dG9nZ2xlVmVyYn0gJHtidXR0b25TdWZmaXh9YDtcbiAgICAgICAgICAgIGNvbnN0IGRpc2NhcmRTZWxlY3Rpb25MYWJlbCA9IGBEaXNjYXJkICR7YnV0dG9uU3VmZml4fWA7XG5cbiAgICAgICAgICAgIGNvbnN0IHN0YXJ0UG9pbnQgPSBodW5rLmdldFJhbmdlKCkuc3RhcnQ7XG4gICAgICAgICAgICBjb25zdCBzdGFydFJhbmdlID0gbmV3IFJhbmdlKHN0YXJ0UG9pbnQsIHN0YXJ0UG9pbnQpO1xuXG4gICAgICAgICAgICByZXR1cm4gKFxuICAgICAgICAgICAgICA8TWFya2VyIGtleT17YGh1bmtIZWFkZXItJHtpbmRleH1gfSBidWZmZXJSYW5nZT17c3RhcnRSYW5nZX0gaW52YWxpZGF0ZT1cIm5ldmVyXCI+XG4gICAgICAgICAgICAgICAgPERlY29yYXRpb24gdHlwZT1cImJsb2NrXCIgb3JkZXI9e29yZGVyT2Zmc2V0ICsgMC4yfSBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1jb250cm9sQmxvY2tcIj5cbiAgICAgICAgICAgICAgICAgIDxIdW5rSGVhZGVyVmlld1xuICAgICAgICAgICAgICAgICAgICByZWZUYXJnZXQ9e3RoaXMucmVmRWRpdG9yRWxlbWVudH1cbiAgICAgICAgICAgICAgICAgICAgaHVuaz17aHVua31cbiAgICAgICAgICAgICAgICAgICAgaXNTZWxlY3RlZD17aXNTZWxlY3RlZH1cbiAgICAgICAgICAgICAgICAgICAgc3RhZ2luZ1N0YXR1cz17dGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzfVxuICAgICAgICAgICAgICAgICAgICBzZWxlY3Rpb25Nb2RlPVwibGluZVwiXG4gICAgICAgICAgICAgICAgICAgIHRvZ2dsZVNlbGVjdGlvbkxhYmVsPXt0b2dnbGVTZWxlY3Rpb25MYWJlbH1cbiAgICAgICAgICAgICAgICAgICAgZGlzY2FyZFNlbGVjdGlvbkxhYmVsPXtkaXNjYXJkU2VsZWN0aW9uTGFiZWx9XG5cbiAgICAgICAgICAgICAgICAgICAgdG9vbHRpcHM9e3RoaXMucHJvcHMudG9vbHRpcHN9XG4gICAgICAgICAgICAgICAgICAgIGtleW1hcHM9e3RoaXMucHJvcHMua2V5bWFwc31cblxuICAgICAgICAgICAgICAgICAgICB0b2dnbGVTZWxlY3Rpb249eygpID0+IHRoaXMudG9nZ2xlSHVua1NlbGVjdGlvbihodW5rLCBjb250YWluc1NlbGVjdGlvbil9XG4gICAgICAgICAgICAgICAgICAgIGRpc2NhcmRTZWxlY3Rpb249eygpID0+IHRoaXMuZGlzY2FyZEh1bmtTZWxlY3Rpb24oaHVuaywgY29udGFpbnNTZWxlY3Rpb24pfVxuICAgICAgICAgICAgICAgICAgICBtb3VzZURvd249e3RoaXMuZGlkTW91c2VEb3duT25IZWFkZXJ9XG4gICAgICAgICAgICAgICAgICAgIGl0ZW1UeXBlPXt0aGlzLnByb3BzLml0ZW1UeXBlfVxuICAgICAgICAgICAgICAgICAgLz5cbiAgICAgICAgICAgICAgICA8L0RlY29yYXRpb24+XG4gICAgICAgICAgICAgIDwvTWFya2VyPlxuICAgICAgICAgICAgKTtcbiAgICAgICAgICB9KX1cbiAgICAgICAgPC9NYXJrZXJMYXllcj5cbiAgICAgIDwvRnJhZ21lbnQ+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckxpbmVEZWNvcmF0aW9ucyhyYW5nZXMsIGxpbmVDbGFzcywge2xpbmUsIGd1dHRlciwgaWNvbiwgcmVmSG9sZGVyfSkge1xuICAgIGlmIChyYW5nZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICBjb25zdCBob2xkZXIgPSByZWZIb2xkZXIgfHwgbmV3IFJlZkhvbGRlcigpO1xuICAgIHJldHVybiAoXG4gICAgICA8TWFya2VyTGF5ZXIgaGFuZGxlTGF5ZXI9e2hvbGRlci5zZXR0ZXJ9PlxuICAgICAgICB7cmFuZ2VzLm1hcCgocmFuZ2UsIGluZGV4KSA9PiB7XG4gICAgICAgICAgcmV0dXJuIChcbiAgICAgICAgICAgIDxNYXJrZXJcbiAgICAgICAgICAgICAga2V5PXtgbGluZS0ke2xpbmVDbGFzc30tJHtpbmRleH1gfVxuICAgICAgICAgICAgICBidWZmZXJSYW5nZT17cmFuZ2V9XG4gICAgICAgICAgICAgIGludmFsaWRhdGU9XCJuZXZlclwiXG4gICAgICAgICAgICAvPlxuICAgICAgICAgICk7XG4gICAgICAgIH0pfVxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9ucyhsaW5lQ2xhc3MsIHtsaW5lLCBndXR0ZXIsIGljb259KX1cbiAgICAgIDwvTWFya2VyTGF5ZXI+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckRlY29yYXRpb25zT25MYXllcihsYXllciwgbGluZUNsYXNzLCB7bGluZSwgZ3V0dGVyLCBpY29ufSkge1xuICAgIGlmIChsYXllci5nZXRNYXJrZXJDb3VudCgpID09PSAwKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICByZXR1cm4gKFxuICAgICAgPE1hcmtlckxheWVyIGV4dGVybmFsPXtsYXllcn0+XG4gICAgICAgIHt0aGlzLnJlbmRlckRlY29yYXRpb25zKGxpbmVDbGFzcywge2xpbmUsIGd1dHRlciwgaWNvbn0pfVxuICAgICAgPC9NYXJrZXJMYXllcj5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyRGVjb3JhdGlvbnMobGluZUNsYXNzLCB7bGluZSwgZ3V0dGVyLCBpY29ufSkge1xuICAgIHJldHVybiAoXG4gICAgICA8RnJhZ21lbnQ+XG4gICAgICAgIHtsaW5lICYmIChcbiAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgdHlwZT1cImxpbmVcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPXtsaW5lQ2xhc3N9XG4gICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAvPlxuICAgICAgICApfVxuICAgICAgICB7Z3V0dGVyICYmIChcbiAgICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgICBndXR0ZXJOYW1lPVwib2xkLWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgICAgIGNsYXNzTmFtZT17bGluZUNsYXNzfVxuICAgICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAgIC8+XG4gICAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgICBndXR0ZXJOYW1lPVwibmV3LWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgICAgIGNsYXNzTmFtZT17bGluZUNsYXNzfVxuICAgICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAgIC8+XG4gICAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgICAgKX1cbiAgICAgICAge2ljb24gJiYgKFxuICAgICAgICAgIDxEZWNvcmF0aW9uXG4gICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgZ3V0dGVyTmFtZT1cImRpZmYtaWNvbnNcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPXtsaW5lQ2xhc3N9XG4gICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAvPlxuICAgICAgICApfVxuICAgICAgPC9GcmFnbWVudD5cbiAgICApO1xuICB9XG5cbiAgdW5kb0xhc3REaXNjYXJkRnJvbUNvcmVVbmRvID0gKCkgPT4ge1xuICAgIGlmICh0aGlzLnByb3BzLmhhc1VuZG9IaXN0b3J5KSB7XG4gICAgICBjb25zdCBzZWxlY3RlZEZpbGVQYXRjaGVzID0gQXJyYXkuZnJvbSh0aGlzLmdldFNlbGVjdGVkRmlsZVBhdGNoZXMoKSk7XG4gICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgZWxzZSAqL1xuICAgICAgaWYgKHRoaXMucHJvcHMuaXRlbVR5cGUgPT09IENoYW5nZWRGaWxlSXRlbSkge1xuICAgICAgICB0aGlzLnByb3BzLnVuZG9MYXN0RGlzY2FyZChzZWxlY3RlZEZpbGVQYXRjaGVzWzBdLCB7ZXZlbnRTb3VyY2U6IHtjb21tYW5kOiAnY29yZTp1bmRvJ319KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICB1bmRvTGFzdERpc2NhcmRGcm9tQnV0dG9uID0gZmlsZVBhdGNoID0+IHtcbiAgICB0aGlzLnByb3BzLnVuZG9MYXN0RGlzY2FyZChmaWxlUGF0Y2gsIHtldmVudFNvdXJjZTogJ2J1dHRvbid9KTtcbiAgfVxuXG4gIGRpc2NhcmRTZWxlY3Rpb25Gcm9tQ29tbWFuZCA9ICgpID0+IHtcbiAgICByZXR1cm4gdGhpcy5wcm9wcy5kaXNjYXJkUm93cyhcbiAgICAgIHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzLFxuICAgICAgdGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlLFxuICAgICAge2V2ZW50U291cmNlOiB7Y29tbWFuZDogJ2dpdGh1YjpkaXNjYXJkLXNlbGVjdGVkLWxpbmVzJ319LFxuICAgICk7XG4gIH1cblxuICB0b2dnbGVIdW5rU2VsZWN0aW9uKGh1bmssIGNvbnRhaW5zU2VsZWN0aW9uKSB7XG4gICAgaWYgKGNvbnRhaW5zU2VsZWN0aW9uKSB7XG4gICAgICByZXR1cm4gdGhpcy5wcm9wcy50b2dnbGVSb3dzKFxuICAgICAgICB0aGlzLnByb3BzLnNlbGVjdGVkUm93cyxcbiAgICAgICAgdGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlLFxuICAgICAgICB7ZXZlbnRTb3VyY2U6ICdidXR0b24nfSxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGNoYW5nZVJvd3MgPSBuZXcgU2V0KFxuICAgICAgICBodW5rLmdldENoYW5nZXMoKVxuICAgICAgICAgIC5yZWR1Y2UoKHJvd3MsIGNoYW5nZSkgPT4ge1xuICAgICAgICAgICAgcm93cy5wdXNoKC4uLmNoYW5nZS5nZXRCdWZmZXJSb3dzKCkpO1xuICAgICAgICAgICAgcmV0dXJuIHJvd3M7XG4gICAgICAgICAgfSwgW10pLFxuICAgICAgKTtcbiAgICAgIHJldHVybiB0aGlzLnByb3BzLnRvZ2dsZVJvd3MoXG4gICAgICAgIGNoYW5nZVJvd3MsXG4gICAgICAgICdodW5rJyxcbiAgICAgICAge2V2ZW50U291cmNlOiAnYnV0dG9uJ30sXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIGRpc2NhcmRIdW5rU2VsZWN0aW9uKGh1bmssIGNvbnRhaW5zU2VsZWN0aW9uKSB7XG4gICAgaWYgKGNvbnRhaW5zU2VsZWN0aW9uKSB7XG4gICAgICByZXR1cm4gdGhpcy5wcm9wcy5kaXNjYXJkUm93cyhcbiAgICAgICAgdGhpcy5wcm9wcy5zZWxlY3RlZFJvd3MsXG4gICAgICAgIHRoaXMucHJvcHMuc2VsZWN0aW9uTW9kZSxcbiAgICAgICAge2V2ZW50U291cmNlOiAnYnV0dG9uJ30sXG4gICAgICApO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBjaGFuZ2VSb3dzID0gbmV3IFNldChcbiAgICAgICAgaHVuay5nZXRDaGFuZ2VzKClcbiAgICAgICAgICAucmVkdWNlKChyb3dzLCBjaGFuZ2UpID0+IHtcbiAgICAgICAgICAgIHJvd3MucHVzaCguLi5jaGFuZ2UuZ2V0QnVmZmVyUm93cygpKTtcbiAgICAgICAgICAgIHJldHVybiByb3dzO1xuICAgICAgICAgIH0sIFtdKSxcbiAgICAgICk7XG4gICAgICByZXR1cm4gdGhpcy5wcm9wcy5kaXNjYXJkUm93cyhjaGFuZ2VSb3dzLCAnaHVuaycsIHtldmVudFNvdXJjZTogJ2J1dHRvbid9KTtcbiAgICB9XG4gIH1cblxuICBkaWRNb3VzZURvd25PbkhlYWRlcihldmVudCwgaHVuaykge1xuICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnaHVuayc7XG4gICAgdGhpcy5oYW5kbGVTZWxlY3Rpb25FdmVudChldmVudCwgaHVuay5nZXRSYW5nZSgpKTtcbiAgfVxuXG4gIGRpZE1vdXNlRG93bk9uTGluZU51bWJlcihldmVudCkge1xuICAgIGNvbnN0IGxpbmUgPSBldmVudC5idWZmZXJSb3c7XG4gICAgaWYgKGxpbmUgPT09IHVuZGVmaW5lZCB8fCBpc05hTihsaW5lKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnbGluZSc7XG4gICAgaWYgKHRoaXMuaGFuZGxlU2VsZWN0aW9uRXZlbnQoZXZlbnQuZG9tRXZlbnQsIFtbbGluZSwgMF0sIFtsaW5lLCBJbmZpbml0eV1dKSkge1xuICAgICAgdGhpcy5tb3VzZVNlbGVjdGlvbkluUHJvZ3Jlc3MgPSB0cnVlO1xuICAgIH1cbiAgfVxuXG4gIGRpZE1vdXNlTW92ZU9uTGluZU51bWJlcihldmVudCkge1xuICAgIGlmICghdGhpcy5tb3VzZVNlbGVjdGlvbkluUHJvZ3Jlc3MpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBsaW5lID0gZXZlbnQuYnVmZmVyUm93O1xuICAgIGlmICh0aGlzLmxhc3RNb3VzZU1vdmVMaW5lID09PSBsaW5lIHx8IGxpbmUgPT09IHVuZGVmaW5lZCB8fCBpc05hTihsaW5lKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICB0aGlzLmxhc3RNb3VzZU1vdmVMaW5lID0gbGluZTtcblxuICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnbGluZSc7XG4gICAgdGhpcy5oYW5kbGVTZWxlY3Rpb25FdmVudChldmVudC5kb21FdmVudCwgW1tsaW5lLCAwXSwgW2xpbmUsIEluZmluaXR5XV0sIHthZGQ6IHRydWV9KTtcbiAgfVxuXG4gIGRpZE1vdXNlVXAoKSB7XG4gICAgdGhpcy5tb3VzZVNlbGVjdGlvbkluUHJvZ3Jlc3MgPSBmYWxzZTtcbiAgfVxuXG4gIGhhbmRsZVNlbGVjdGlvbkV2ZW50KGV2ZW50LCByYW5nZUxpa2UsIG9wdHMpIHtcbiAgICBpZiAoZXZlbnQuYnV0dG9uICE9PSAwKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgY29uc3QgaXNXaW5kb3dzID0gcHJvY2Vzcy5wbGF0Zm9ybSA9PT0gJ3dpbjMyJztcbiAgICBpZiAoZXZlbnQuY3RybEtleSAmJiAhaXNXaW5kb3dzKSB7XG4gICAgICAvLyBBbGxvdyB0aGUgY29udGV4dCBtZW51IHRvIG9wZW4uXG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgY29uc3Qgb3B0aW9ucyA9IHtcbiAgICAgIGFkZDogZmFsc2UsXG4gICAgICAuLi5vcHRzLFxuICAgIH07XG5cbiAgICAvLyBOb3JtYWxpemUgdGhlIHRhcmdldCBzZWxlY3Rpb24gcmFuZ2VcbiAgICBjb25zdCBjb252ZXJ0ZWQgPSBSYW5nZS5mcm9tT2JqZWN0KHJhbmdlTGlrZSk7XG4gICAgY29uc3QgcmFuZ2UgPSB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IGVkaXRvci5jbGlwQnVmZmVyUmFuZ2UoY29udmVydGVkKSkuZ2V0T3IoY29udmVydGVkKTtcblxuICAgIGlmIChldmVudC5tZXRhS2V5IHx8IC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovIChldmVudC5jdHJsS2V5ICYmIGlzV2luZG93cykpIHtcbiAgICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgICBsZXQgaW50ZXJzZWN0cyA9IGZhbHNlO1xuICAgICAgICBsZXQgd2l0aG91dCA9IG51bGw7XG5cbiAgICAgICAgZm9yIChjb25zdCBzZWxlY3Rpb24gb2YgZWRpdG9yLmdldFNlbGVjdGlvbnMoKSkge1xuICAgICAgICAgIGlmIChzZWxlY3Rpb24uaW50ZXJzZWN0c0J1ZmZlclJhbmdlKHJhbmdlKSkge1xuICAgICAgICAgICAgLy8gUmVtb3ZlIHJhbmdlIGZyb20gdGhpcyBzZWxlY3Rpb24gYnkgdHJ1bmNhdGluZyBpdCB0byB0aGUgXCJuZWFyIGVkZ2VcIiBvZiB0aGUgcmFuZ2UgYW5kIGNyZWF0aW5nIGFcbiAgICAgICAgICAgIC8vIG5ldyBzZWxlY3Rpb24gZnJvbSB0aGUgXCJmYXIgZWRnZVwiIHRvIHRoZSBwcmV2aW91cyBlbmQuIE9taXQgZWl0aGVyIHNpZGUgaWYgaXQgaXMgZW1wdHkuXG4gICAgICAgICAgICBpbnRlcnNlY3RzID0gdHJ1ZTtcbiAgICAgICAgICAgIGNvbnN0IHNlbGVjdGlvblJhbmdlID0gc2VsZWN0aW9uLmdldEJ1ZmZlclJhbmdlKCk7XG5cbiAgICAgICAgICAgIGNvbnN0IG5ld1JhbmdlcyA9IFtdO1xuXG4gICAgICAgICAgICBpZiAoIXJhbmdlLnN0YXJ0LmlzRXF1YWwoc2VsZWN0aW9uUmFuZ2Uuc3RhcnQpKSB7XG4gICAgICAgICAgICAgIC8vIEluY2x1ZGUgdGhlIGJpdCBmcm9tIHRoZSBzZWxlY3Rpb24ncyBwcmV2aW91cyBzdGFydCB0byB0aGUgcmFuZ2UncyBzdGFydC5cbiAgICAgICAgICAgICAgbGV0IG51ZGdlZCA9IHJhbmdlLnN0YXJ0O1xuICAgICAgICAgICAgICBpZiAocmFuZ2Uuc3RhcnQuY29sdW1uID09PSAwKSB7XG4gICAgICAgICAgICAgICAgY29uc3QgbGFzdENvbHVtbiA9IGVkaXRvci5nZXRCdWZmZXIoKS5saW5lTGVuZ3RoRm9yUm93KHJhbmdlLnN0YXJ0LnJvdyAtIDEpO1xuICAgICAgICAgICAgICAgIG51ZGdlZCA9IFtyYW5nZS5zdGFydC5yb3cgLSAxLCBsYXN0Q29sdW1uXTtcbiAgICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAgIG5ld1Jhbmdlcy5wdXNoKFtzZWxlY3Rpb25SYW5nZS5zdGFydCwgbnVkZ2VkXSk7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIGlmICghcmFuZ2UuZW5kLmlzRXF1YWwoc2VsZWN0aW9uUmFuZ2UuZW5kKSkge1xuICAgICAgICAgICAgICAvLyBJbmNsdWRlIHRoZSBiaXQgZnJvbSB0aGUgcmFuZ2UncyBlbmQgdG8gdGhlIHNlbGVjdGlvbidzIGVuZC5cbiAgICAgICAgICAgICAgbGV0IG51ZGdlZCA9IHJhbmdlLmVuZDtcbiAgICAgICAgICAgICAgY29uc3QgbGFzdENvbHVtbiA9IGVkaXRvci5nZXRCdWZmZXIoKS5saW5lTGVuZ3RoRm9yUm93KHJhbmdlLmVuZC5yb3cpO1xuICAgICAgICAgICAgICBpZiAocmFuZ2UuZW5kLmNvbHVtbiA9PT0gbGFzdENvbHVtbikge1xuICAgICAgICAgICAgICAgIG51ZGdlZCA9IFtyYW5nZS5lbmQucm93ICsgMSwgMF07XG4gICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICBuZXdSYW5nZXMucHVzaChbbnVkZ2VkLCBzZWxlY3Rpb25SYW5nZS5lbmRdKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgaWYgKG5ld1Jhbmdlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAgIHNlbGVjdGlvbi5zZXRCdWZmZXJSYW5nZShuZXdSYW5nZXNbMF0pO1xuICAgICAgICAgICAgICBmb3IgKGNvbnN0IG5ld1JhbmdlIG9mIG5ld1Jhbmdlcy5zbGljZSgxKSkge1xuICAgICAgICAgICAgICAgIGVkaXRvci5hZGRTZWxlY3Rpb25Gb3JCdWZmZXJSYW5nZShuZXdSYW5nZSwge3JldmVyc2VkOiBzZWxlY3Rpb24uaXNSZXZlcnNlZCgpfSk7XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHdpdGhvdXQgPSBzZWxlY3Rpb247XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHdpdGhvdXQgIT09IG51bGwpIHtcbiAgICAgICAgICBjb25zdCByZXBsYWNlbWVudFJhbmdlcyA9IGVkaXRvci5nZXRTZWxlY3Rpb25zKClcbiAgICAgICAgICAgIC5maWx0ZXIoZWFjaCA9PiBlYWNoICE9PSB3aXRob3V0KVxuICAgICAgICAgICAgLm1hcChlYWNoID0+IGVhY2guZ2V0QnVmZmVyUmFuZ2UoKSk7XG4gICAgICAgICAgaWYgKHJlcGxhY2VtZW50UmFuZ2VzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIGVkaXRvci5zZXRTZWxlY3RlZEJ1ZmZlclJhbmdlcyhyZXBsYWNlbWVudFJhbmdlcyk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFpbnRlcnNlY3RzKSB7XG4gICAgICAgICAgLy8gQWRkIHRoaXMgcmFuZ2UgYXMgYSBuZXcsIGRpc3RpbmN0IHNlbGVjdGlvbi5cbiAgICAgICAgICBlZGl0b3IuYWRkU2VsZWN0aW9uRm9yQnVmZmVyUmFuZ2UocmFuZ2UpO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgICB9KTtcbiAgICB9IGVsc2UgaWYgKG9wdGlvbnMuYWRkIHx8IGV2ZW50LnNoaWZ0S2V5KSB7XG4gICAgICAvLyBFeHRlbmQgdGhlIGV4aXN0aW5nIHNlbGVjdGlvbiB0byBlbmNvbXBhc3MgdGhpcyByYW5nZS5cbiAgICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgICBjb25zdCBsYXN0U2VsZWN0aW9uID0gZWRpdG9yLmdldExhc3RTZWxlY3Rpb24oKTtcbiAgICAgICAgY29uc3QgbGFzdFNlbGVjdGlvblJhbmdlID0gbGFzdFNlbGVjdGlvbi5nZXRCdWZmZXJSYW5nZSgpO1xuXG4gICAgICAgIC8vIFlvdSBhcmUgbm93IGVudGVyaW5nIHRoZSB3YWxsIG9mIHRlcm5lcnkgb3BlcmF0b3JzLiBUaGlzIGlzIHlvdXIgbGFzdCBleGl0IGJlZm9yZSB0aGUgdG9sbGJvb3RoXG4gICAgICAgIGNvbnN0IGlzQmVmb3JlID0gcmFuZ2Uuc3RhcnQuaXNMZXNzVGhhbihsYXN0U2VsZWN0aW9uUmFuZ2Uuc3RhcnQpO1xuICAgICAgICBjb25zdCBmYXJFZGdlID0gaXNCZWZvcmUgPyByYW5nZS5zdGFydCA6IHJhbmdlLmVuZDtcbiAgICAgICAgY29uc3QgbmV3UmFuZ2UgPSBpc0JlZm9yZSA/IFtmYXJFZGdlLCBsYXN0U2VsZWN0aW9uUmFuZ2UuZW5kXSA6IFtsYXN0U2VsZWN0aW9uUmFuZ2Uuc3RhcnQsIGZhckVkZ2VdO1xuXG4gICAgICAgIGxhc3RTZWxlY3Rpb24uc2V0QnVmZmVyUmFuZ2UobmV3UmFuZ2UsIHtyZXZlcnNlZDogaXNCZWZvcmV9KTtcbiAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgICB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZShyYW5nZSkpO1xuICAgIH1cblxuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgZGlkQ29uZmlybSgpIHtcbiAgICByZXR1cm4gdGhpcy5wcm9wcy50b2dnbGVSb3dzKHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzLCB0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUpO1xuICB9XG5cbiAgZGlkVG9nZ2xlU2VsZWN0aW9uTW9kZSgpIHtcbiAgICBjb25zdCBzZWxlY3RlZEh1bmtzID0gdGhpcy5nZXRTZWxlY3RlZEh1bmtzKCk7XG4gICAgdGhpcy53aXRoU2VsZWN0aW9uTW9kZSh7XG4gICAgICBsaW5lOiAoKSA9PiB7XG4gICAgICAgIGNvbnN0IGh1bmtSYW5nZXMgPSBzZWxlY3RlZEh1bmtzLm1hcChodW5rID0+IGh1bmsuZ2V0UmFuZ2UoKSk7XG4gICAgICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnaHVuayc7XG4gICAgICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4gZWRpdG9yLnNldFNlbGVjdGVkQnVmZmVyUmFuZ2VzKGh1bmtSYW5nZXMpKTtcbiAgICAgIH0sXG4gICAgICBodW5rOiAoKSA9PiB7XG4gICAgICAgIGxldCBmaXJzdENoYW5nZVJvdyA9IEluZmluaXR5O1xuICAgICAgICBmb3IgKGNvbnN0IGh1bmsgb2Ygc2VsZWN0ZWRIdW5rcykge1xuICAgICAgICAgIGNvbnN0IFtmaXJzdENoYW5nZV0gPSBodW5rLmdldENoYW5nZXMoKTtcbiAgICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgZWxzZSAqL1xuICAgICAgICAgIGlmIChmaXJzdENoYW5nZSAmJiAoIWZpcnN0Q2hhbmdlUm93IHx8IGZpcnN0Q2hhbmdlLmdldFN0YXJ0QnVmZmVyUm93KCkgPCBmaXJzdENoYW5nZVJvdykpIHtcbiAgICAgICAgICAgIGZpcnN0Q2hhbmdlUm93ID0gZmlyc3RDaGFuZ2UuZ2V0U3RhcnRCdWZmZXJSb3coKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2xpbmUnO1xuICAgICAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMoW1tbZmlyc3RDaGFuZ2VSb3csIDBdLCBbZmlyc3RDaGFuZ2VSb3csIEluZmluaXR5XV1dKTtcbiAgICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgICAgfSk7XG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgZGlkVG9nZ2xlTW9kZUNoYW5nZSA9ICgpID0+IHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHRoaXMuZ2V0U2VsZWN0ZWRGaWxlUGF0Y2hlcygpKVxuICAgICAgICAuZmlsdGVyKGZwID0+IGZwLmRpZENoYW5nZUV4ZWN1dGFibGVNb2RlKCkpXG4gICAgICAgIC5tYXAodGhpcy5wcm9wcy50b2dnbGVNb2RlQ2hhbmdlKSxcbiAgICApO1xuICB9XG5cbiAgZGlkVG9nZ2xlU3ltbGlua0NoYW5nZSA9ICgpID0+IHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHRoaXMuZ2V0U2VsZWN0ZWRGaWxlUGF0Y2hlcygpKVxuICAgICAgICAuZmlsdGVyKGZwID0+IGZwLmhhc1R5cGVjaGFuZ2UoKSlcbiAgICAgICAgLm1hcCh0aGlzLnByb3BzLnRvZ2dsZVN5bWxpbmtDaGFuZ2UpLFxuICAgICk7XG4gIH1cblxuICBzZWxlY3ROZXh0SHVuaygpIHtcbiAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IG5leHRIdW5rcyA9IG5ldyBTZXQoXG4gICAgICAgIHRoaXMud2l0aFNlbGVjdGVkSHVua3MoaHVuayA9PiB0aGlzLmdldEh1bmtBZnRlcihodW5rKSB8fCBodW5rKSxcbiAgICAgICk7XG4gICAgICBjb25zdCBuZXh0UmFuZ2VzID0gQXJyYXkuZnJvbShuZXh0SHVua3MsIGh1bmsgPT4gaHVuay5nZXRSYW5nZSgpKTtcbiAgICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnaHVuayc7XG4gICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMobmV4dFJhbmdlcyk7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9KTtcbiAgfVxuXG4gIHNlbGVjdFByZXZpb3VzSHVuaygpIHtcbiAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IG5leHRIdW5rcyA9IG5ldyBTZXQoXG4gICAgICAgIHRoaXMud2l0aFNlbGVjdGVkSHVua3MoaHVuayA9PiB0aGlzLmdldEh1bmtCZWZvcmUoaHVuaykgfHwgaHVuayksXG4gICAgICApO1xuICAgICAgY29uc3QgbmV4dFJhbmdlcyA9IEFycmF5LmZyb20obmV4dEh1bmtzLCBodW5rID0+IGh1bmsuZ2V0UmFuZ2UoKSk7XG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2h1bmsnO1xuICAgICAgZWRpdG9yLnNldFNlbGVjdGVkQnVmZmVyUmFuZ2VzKG5leHRSYW5nZXMpO1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfSk7XG4gIH1cblxuICBkaWRPcGVuRmlsZSh7c2VsZWN0ZWRGaWxlUGF0Y2h9KSB7XG4gICAgY29uc3QgY3Vyc29yc0J5RmlsZVBhdGNoID0gbmV3IE1hcCgpO1xuXG4gICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICBjb25zdCBwbGFjZWRSb3dzID0gbmV3IFNldCgpO1xuXG4gICAgICBmb3IgKGNvbnN0IGN1cnNvciBvZiBlZGl0b3IuZ2V0Q3Vyc29ycygpKSB7XG4gICAgICAgIGNvbnN0IGN1cnNvclJvdyA9IGN1cnNvci5nZXRCdWZmZXJQb3NpdGlvbigpLnJvdztcbiAgICAgICAgY29uc3QgaHVuayA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KGN1cnNvclJvdyk7XG4gICAgICAgIGNvbnN0IGZpbGVQYXRjaCA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0RmlsZVBhdGNoQXQoY3Vyc29yUm93KTtcbiAgICAgICAgLyogaXN0YW5idWwgaWdub3JlIG5leHQgKi9cbiAgICAgICAgaWYgKCFodW5rKSB7XG4gICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cblxuICAgICAgICBsZXQgbmV3Um93ID0gaHVuay5nZXROZXdSb3dBdChjdXJzb3JSb3cpO1xuICAgICAgICBsZXQgbmV3Q29sdW1uID0gY3Vyc29yLmdldEJ1ZmZlclBvc2l0aW9uKCkuY29sdW1uO1xuICAgICAgICBpZiAobmV3Um93ID09PSBudWxsKSB7XG4gICAgICAgICAgbGV0IG5lYXJlc3RSb3cgPSBodW5rLmdldE5ld1N0YXJ0Um93KCk7XG4gICAgICAgICAgZm9yIChjb25zdCByZWdpb24gb2YgaHVuay5nZXRSZWdpb25zKCkpIHtcbiAgICAgICAgICAgIGlmICghcmVnaW9uLmluY2x1ZGVzQnVmZmVyUm93KGN1cnNvclJvdykpIHtcbiAgICAgICAgICAgICAgcmVnaW9uLndoZW4oe1xuICAgICAgICAgICAgICAgIHVuY2hhbmdlZDogKCkgPT4ge1xuICAgICAgICAgICAgICAgICAgbmVhcmVzdFJvdyArPSByZWdpb24uYnVmZmVyUm93Q291bnQoKTtcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIGFkZGl0aW9uOiAoKSA9PiB7XG4gICAgICAgICAgICAgICAgICBuZWFyZXN0Um93ICs9IHJlZ2lvbi5idWZmZXJSb3dDb3VudCgpO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKCFwbGFjZWRSb3dzLmhhcyhuZWFyZXN0Um93KSkge1xuICAgICAgICAgICAgbmV3Um93ID0gbmVhcmVzdFJvdztcbiAgICAgICAgICAgIG5ld0NvbHVtbiA9IDA7XG4gICAgICAgICAgICBwbGFjZWRSb3dzLmFkZChuZWFyZXN0Um93KTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICBpZiAobmV3Um93ICE9PSBudWxsKSB7XG4gICAgICAgICAgLy8gV2h5IGlzIHRoaXMgbmVlZGVkPyBJIF90aGlua18gZXZlcnl0aGluZyBpcyBpbiB0ZXJtcyBvZiBidWZmZXIgcG9zaXRpb25cbiAgICAgICAgICAvLyBzbyB0aGVyZSBzaG91bGRuJ3QgYmUgYW4gb2ZmLWJ5LW9uZSBpc3N1ZVxuICAgICAgICAgIG5ld1JvdyAtPSAxO1xuICAgICAgICAgIGNvbnN0IGN1cnNvcnMgPSBjdXJzb3JzQnlGaWxlUGF0Y2guZ2V0KGZpbGVQYXRjaCk7XG4gICAgICAgICAgaWYgKCFjdXJzb3JzKSB7XG4gICAgICAgICAgICBjdXJzb3JzQnlGaWxlUGF0Y2guc2V0KGZpbGVQYXRjaCwgW1tuZXdSb3csIG5ld0NvbHVtbl1dKTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgY3Vyc29ycy5wdXNoKFtuZXdSb3csIG5ld0NvbHVtbl0pO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9KTtcblxuICAgIGNvbnN0IGZpbGVQYXRjaGVzV2l0aEN1cnNvcnMgPSBuZXcgU2V0KGN1cnNvcnNCeUZpbGVQYXRjaC5rZXlzKCkpO1xuICAgIGlmIChzZWxlY3RlZEZpbGVQYXRjaCAmJiAhZmlsZVBhdGNoZXNXaXRoQ3Vyc29ycy5oYXMoc2VsZWN0ZWRGaWxlUGF0Y2gpKSB7XG4gICAgICBjb25zdCBbZmlyc3RIdW5rXSA9IHNlbGVjdGVkRmlsZVBhdGNoLmdldEh1bmtzKCk7XG4gICAgICBjb25zdCBjdXJzb3JSb3cgPSBmaXJzdEh1bmsgPyBmaXJzdEh1bmsuZ2V0TmV3U3RhcnRSb3coKSAtIDEgOiAwO1xuICAgICAgcmV0dXJuIHRoaXMucHJvcHMub3BlbkZpbGUoc2VsZWN0ZWRGaWxlUGF0Y2gsIFtbY3Vyc29yUm93LCAwXV0sIHRydWUpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBwZW5kaW5nID0gY3Vyc29yc0J5RmlsZVBhdGNoLnNpemUgPT09IDE7XG4gICAgICByZXR1cm4gUHJvbWlzZS5hbGwoQXJyYXkuZnJvbShjdXJzb3JzQnlGaWxlUGF0Y2gsIHZhbHVlID0+IHtcbiAgICAgICAgY29uc3QgW2ZpbGVQYXRjaCwgY3Vyc29yc10gPSB2YWx1ZTtcbiAgICAgICAgcmV0dXJuIHRoaXMucHJvcHMub3BlbkZpbGUoZmlsZVBhdGNoLCBjdXJzb3JzLCBwZW5kaW5nKTtcbiAgICAgIH0pKTtcbiAgICB9XG5cbiAgfVxuXG4gIGdldFNlbGVjdGVkUm93cygpIHtcbiAgICByZXR1cm4gdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICByZXR1cm4gbmV3IFNldChcbiAgICAgICAgZWRpdG9yLmdldFNlbGVjdGlvbnMoKVxuICAgICAgICAgIC5tYXAoc2VsZWN0aW9uID0+IHNlbGVjdGlvbi5nZXRCdWZmZXJSYW5nZSgpKVxuICAgICAgICAgIC5yZWR1Y2UoKGFjYywgcmFuZ2UpID0+IHtcbiAgICAgICAgICAgIGZvciAoY29uc3Qgcm93IG9mIHJhbmdlLmdldFJvd3MoKSkge1xuICAgICAgICAgICAgICBpZiAodGhpcy5pc0NoYW5nZVJvdyhyb3cpKSB7XG4gICAgICAgICAgICAgICAgYWNjLnB1c2gocm93KTtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcmV0dXJuIGFjYztcbiAgICAgICAgICB9LCBbXSksXG4gICAgICApO1xuICAgIH0pLmdldE9yKG5ldyBTZXQoKSk7XG4gIH1cblxuICBkaWRBZGRTZWxlY3Rpb24oKSB7XG4gICAgdGhpcy5kaWRDaGFuZ2VTZWxlY3RlZFJvd3MoKTtcbiAgfVxuXG4gIGRpZENoYW5nZVNlbGVjdGlvblJhbmdlKGV2ZW50KSB7XG4gICAgaWYgKFxuICAgICAgIWV2ZW50IHx8XG4gICAgICBldmVudC5vbGRCdWZmZXJSYW5nZS5zdGFydC5yb3cgIT09IGV2ZW50Lm5ld0J1ZmZlclJhbmdlLnN0YXJ0LnJvdyB8fFxuICAgICAgZXZlbnQub2xkQnVmZmVyUmFuZ2UuZW5kLnJvdyAhPT0gZXZlbnQubmV3QnVmZmVyUmFuZ2UuZW5kLnJvd1xuICAgICkge1xuICAgICAgdGhpcy5kaWRDaGFuZ2VTZWxlY3RlZFJvd3MoKTtcbiAgICB9XG4gIH1cblxuICBkaWREZXN0cm95U2VsZWN0aW9uKCkge1xuICAgIHRoaXMuZGlkQ2hhbmdlU2VsZWN0ZWRSb3dzKCk7XG4gIH1cblxuICBkaWRDaGFuZ2VTZWxlY3RlZFJvd3MoKSB7XG4gICAgaWYgKHRoaXMuc3VwcHJlc3NDaGFuZ2VzKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgbmV4dEN1cnNvclJvd3MgPSB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIHJldHVybiBlZGl0b3IuZ2V0Q3Vyc29yQnVmZmVyUG9zaXRpb25zKCkubWFwKHBvc2l0aW9uID0+IHBvc2l0aW9uLnJvdyk7XG4gICAgfSkuZ2V0T3IoW10pO1xuICAgIGNvbnN0IGhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnMgPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLnNwYW5zTXVsdGlwbGVGaWxlcyhuZXh0Q3Vyc29yUm93cyk7XG5cbiAgICB0aGlzLnByb3BzLnNlbGVjdGVkUm93c0NoYW5nZWQoXG4gICAgICB0aGlzLmdldFNlbGVjdGVkUm93cygpLFxuICAgICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSB8fCAnbGluZScsXG4gICAgICBoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zLFxuICAgICk7XG4gIH1cblxuICBvbGRMaW5lTnVtYmVyTGFiZWwoe2J1ZmZlclJvdywgc29mdFdyYXBwZWR9KSB7XG4gICAgY29uc3QgaHVuayA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KGJ1ZmZlclJvdyk7XG4gICAgaWYgKGh1bmsgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHRoaXMucGFkKCcnKTtcbiAgICB9XG5cbiAgICBjb25zdCBvbGRSb3cgPSBodW5rLmdldE9sZFJvd0F0KGJ1ZmZlclJvdyk7XG4gICAgaWYgKHNvZnRXcmFwcGVkKSB7XG4gICAgICByZXR1cm4gdGhpcy5wYWQob2xkUm93ID09PSBudWxsID8gJycgOiAn4oCiJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMucGFkKG9sZFJvdyk7XG4gIH1cblxuICBuZXdMaW5lTnVtYmVyTGFiZWwoe2J1ZmZlclJvdywgc29mdFdyYXBwZWR9KSB7XG4gICAgY29uc3QgaHVuayA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KGJ1ZmZlclJvdyk7XG4gICAgaWYgKGh1bmsgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHRoaXMucGFkKCcnKTtcbiAgICB9XG5cbiAgICBjb25zdCBuZXdSb3cgPSBodW5rLmdldE5ld1Jvd0F0KGJ1ZmZlclJvdyk7XG4gICAgaWYgKHNvZnRXcmFwcGVkKSB7XG4gICAgICByZXR1cm4gdGhpcy5wYWQobmV3Um93ID09PSBudWxsID8gJycgOiAn4oCiJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnBhZChuZXdSb3cpO1xuICB9XG5cbiAgLypcbiAgICogUmV0dXJuIGEgU2V0IG9mIHRoZSBIdW5rcyB0aGF0IGluY2x1ZGUgYXQgbGVhc3Qgb25lIGVkaXRvciBzZWxlY3Rpb24uIFRoZSBzZWxlY3Rpb24gbmVlZCBub3QgY29udGFpbiBhbiBhY3R1YWxcbiAgICogY2hhbmdlIHJvdy5cbiAgICovXG4gIGdldFNlbGVjdGVkSHVua3MoKSB7XG4gICAgcmV0dXJuIHRoaXMud2l0aFNlbGVjdGVkSHVua3MoZWFjaCA9PiBlYWNoKTtcbiAgfVxuXG4gIHdpdGhTZWxlY3RlZEh1bmtzKGNhbGxiYWNrKSB7XG4gICAgcmV0dXJuIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgY29uc3Qgc2VlbiA9IG5ldyBTZXQoKTtcbiAgICAgIHJldHVybiBlZGl0b3IuZ2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMoKS5yZWR1Y2UoKGFjYywgcmFuZ2UpID0+IHtcbiAgICAgICAgZm9yIChjb25zdCByb3cgb2YgcmFuZ2UuZ2V0Um93cygpKSB7XG4gICAgICAgICAgY29uc3QgaHVuayA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KHJvdyk7XG4gICAgICAgICAgaWYgKCFodW5rIHx8IHNlZW4uaGFzKGh1bmspKSB7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBzZWVuLmFkZChodW5rKTtcbiAgICAgICAgICBhY2MucHVzaChjYWxsYmFjayhodW5rKSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGFjYztcbiAgICAgIH0sIFtdKTtcbiAgICB9KS5nZXRPcihbXSk7XG4gIH1cblxuICAvKlxuICAgKiBSZXR1cm4gYSBTZXQgb2YgRmlsZVBhdGNoZXMgdGhhdCBpbmNsdWRlIGF0IGxlYXN0IG9uZSBlZGl0b3Igc2VsZWN0aW9uLiBUaGUgc2VsZWN0aW9uIG5lZWQgbm90IGNvbnRhaW4gYW4gYWN0dWFsXG4gICAqIGNoYW5nZSByb3cuXG4gICAqL1xuICBnZXRTZWxlY3RlZEZpbGVQYXRjaGVzKCkge1xuICAgIHJldHVybiB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IHBhdGNoZXMgPSBuZXcgU2V0KCk7XG4gICAgICBmb3IgKGNvbnN0IHJhbmdlIG9mIGVkaXRvci5nZXRTZWxlY3RlZEJ1ZmZlclJhbmdlcygpKSB7XG4gICAgICAgIGZvciAoY29uc3Qgcm93IG9mIHJhbmdlLmdldFJvd3MoKSkge1xuICAgICAgICAgIGNvbnN0IHBhdGNoID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hBdChyb3cpO1xuICAgICAgICAgIHBhdGNoZXMuYWRkKHBhdGNoKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgcmV0dXJuIHBhdGNoZXM7XG4gICAgfSkuZ2V0T3IobmV3IFNldCgpKTtcbiAgfVxuXG4gIGdldEh1bmtCZWZvcmUoaHVuaykge1xuICAgIGNvbnN0IHByZXZSb3cgPSBodW5rLmdldFJhbmdlKCkuc3RhcnQucm93IC0gMTtcbiAgICByZXR1cm4gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQocHJldlJvdyk7XG4gIH1cblxuICBnZXRIdW5rQWZ0ZXIoaHVuaykge1xuICAgIGNvbnN0IG5leHRSb3cgPSBodW5rLmdldFJhbmdlKCkuZW5kLnJvdyArIDE7XG4gICAgcmV0dXJuIHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KG5leHRSb3cpO1xuICB9XG5cbiAgaXNDaGFuZ2VSb3coYnVmZmVyUm93KSB7XG4gICAgY29uc3QgY2hhbmdlTGF5ZXJzID0gW3RoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0QWRkaXRpb25MYXllcigpLCB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldERlbGV0aW9uTGF5ZXIoKV07XG4gICAgcmV0dXJuIGNoYW5nZUxheWVycy5zb21lKGxheWVyID0+IGxheWVyLmZpbmRNYXJrZXJzKHtpbnRlcnNlY3RzUm93OiBidWZmZXJSb3d9KS5sZW5ndGggPiAwKTtcbiAgfVxuXG4gIHdpdGhTZWxlY3Rpb25Nb2RlKGNhbGxiYWNrcykge1xuICAgIGNvbnN0IGNhbGxiYWNrID0gY2FsbGJhY2tzW3RoaXMucHJvcHMuc2VsZWN0aW9uTW9kZV07XG4gICAgLyogaXN0YW5idWwgaWdub3JlIGlmICovXG4gICAgaWYgKCFjYWxsYmFjaykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIHNlbGVjdGlvbiBtb2RlOiAke3RoaXMucHJvcHMuc2VsZWN0aW9uTW9kZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIGNhbGxiYWNrKCk7XG4gIH1cblxuICBwYWQobnVtKSB7XG4gICAgY29uc3QgbWF4RGlnaXRzID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRNYXhMaW5lTnVtYmVyV2lkdGgoKTtcbiAgICBpZiAobnVtID09PSBudWxsKSB7XG4gICAgICByZXR1cm4gTkJTUF9DSEFSQUNURVIucmVwZWF0KG1heERpZ2l0cyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOQlNQX0NIQVJBQ1RFUi5yZXBlYXQobWF4RGlnaXRzIC0gbnVtLnRvU3RyaW5nKCkubGVuZ3RoKSArIG51bS50b1N0cmluZygpO1xuICAgIH1cbiAgfVxuXG4gIG1lYXN1cmVQZXJmb3JtYW5jZShhY3Rpb24pIHtcbiAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgZWxzZSAqL1xuICAgIGlmICgoYWN0aW9uID09PSAndXBkYXRlJyB8fCBhY3Rpb24gPT09ICdtb3VudCcpXG4gICAgICAmJiBwZXJmb3JtYW5jZS5nZXRFbnRyaWVzQnlOYW1lKGBNdWx0aUZpbGVQYXRjaFZpZXctJHthY3Rpb259LXN0YXJ0YCkubGVuZ3RoID4gMCkge1xuICAgICAgcGVyZm9ybWFuY2UubWFyayhgTXVsdGlGaWxlUGF0Y2hWaWV3LSR7YWN0aW9ufS1lbmRgKTtcbiAgICAgIHBlcmZvcm1hbmNlLm1lYXN1cmUoXG4gICAgICAgIGBNdWx0aUZpbGVQYXRjaFZpZXctJHthY3Rpb259YCxcbiAgICAgICAgYE11bHRpRmlsZVBhdGNoVmlldy0ke2FjdGlvbn0tc3RhcnRgLFxuICAgICAgICBgTXVsdGlGaWxlUGF0Y2hWaWV3LSR7YWN0aW9ufS1lbmRgKTtcbiAgICAgIGNvbnN0IHBlcmYgPSBwZXJmb3JtYW5jZS5nZXRFbnRyaWVzQnlOYW1lKGBNdWx0aUZpbGVQYXRjaFZpZXctJHthY3Rpb259YClbMF07XG4gICAgICBwZXJmb3JtYW5jZS5jbGVhck1hcmtzKGBNdWx0aUZpbGVQYXRjaFZpZXctJHthY3Rpb259LXN0YXJ0YCk7XG4gICAgICBwZXJmb3JtYW5jZS5jbGVhck1hcmtzKGBNdWx0aUZpbGVQYXRjaFZpZXctJHthY3Rpb259LWVuZGApO1xuICAgICAgcGVyZm9ybWFuY2UuY2xlYXJNZWFzdXJlcyhgTXVsdGlGaWxlUGF0Y2hWaWV3LSR7YWN0aW9ufWApO1xuICAgICAgYWRkRXZlbnQoYE11bHRpRmlsZVBhdGNoVmlldy0ke2FjdGlvbn1gLCB7XG4gICAgICAgIHBhY2thZ2U6ICdnaXRodWInLFxuICAgICAgICBmaWxlUGF0Y2hlc0xpbmVDb3VudHM6IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0RmlsZVBhdGNoZXMoKS5tYXAoXG4gICAgICAgICAgZnAgPT4gZnAuZ2V0UGF0Y2goKS5nZXRDaGFuZ2VkTGluZUNvdW50KCksXG4gICAgICAgICksXG4gICAgICAgIGR1cmF0aW9uOiBwZXJmLmR1cmF0aW9uLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG59XG4iXX0=