"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

/*
 * Register callbacks and construct Promises to wait for the next occurrence of specific events that occur throughout
 * the data refresh and rendering cycle.
 */
class Switchboard {
  constructor() {
    this.promises = new Map();
    this.emitter = new _eventKit.Emitter();
  }
  /*
   * Invoke a callback each time that a desired event is observed. Return a Disposable that can be used to
   * unsubscribe from events.
   *
   * In general, you should use the more specific `onDidXyz` methods.
   */


  onDid(eventName, callback) {
    return this.emitter.on(`did-${eventName}`, callback);
  }
  /*
   * Indicate that a named event has been observed, firing any callbacks and resolving any Promises that were created
   * for this event. Optionally provide a payload with more information.
   *
   * In general, you should prefer the more specific `didXyz()` methods.
   */


  did(eventName, payload) {
    this.emitter.emit(`did-${eventName}`, payload);
  }
  /*
   * Retrieve a Promise that will be resolved the next time a desired event is observed.
   *
   * In general, you should prefer the more specific `getXyzPromise()` methods.
   */


  getPromise(eventName) {
    const existing = this.promises.get(eventName);

    if (existing !== undefined) {
      return existing;
    }

    const created = new Promise((resolve, reject) => {
      const subscription = this.onDid(eventName, payload => {
        subscription.dispose();
        this.promises.delete(eventName);
        resolve(payload);
      });
    });
    this.promises.set(eventName, created);
    return created;
  }

}

exports.default = Switchboard;
['UpdateRepository', 'BeginStageOperation', 'FinishStageOperation', 'ChangePatch', 'ScheduleActiveContextUpdate', 'BeginActiveContextUpdate', 'FinishActiveContextUpdate', 'FinishRender', 'FinishContextChangeRender', 'FinishRepositoryRefresh'].forEach(eventName => {
  Switchboard.prototype[`did${eventName}`] = function (payload) {
    this.did(eventName, payload);
  };

  Switchboard.prototype[`get${eventName}Promise`] = function () {
    return this.getPromise(eventName);
  };

  Switchboard.prototype[`onDid${eventName}`] = function (callback) {
    return this.onDid(eventName, callback);
  };
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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