"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports.default = void 0;

var _util = _interopRequireDefault(require("util"));

var _relayRuntime = require("relay-runtime");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');

  const resetsIn = _moment.default.unix(parseInt(resets, 10)).from(); // eslint-disable-next-line no-console


  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const promise = new Promise((resolve0, reject0) => {
    resolve = () => resolve0({
      data: response
    });

    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    response,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery.delete(operationPattern.name);

  return {
    promise,
    resolve,
    reject,
    disable
  };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (Object.keys(expectation.variables).length !== Object.keys(variables).length) {
          return false;
        }

        for (const key in expectation.variables) {
          if (expectation.variables[key] !== variables[key]) {
            return false;
          }
        }

        return true;
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util.default.inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      if (match.trace) {
        match.promise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`GraphQL query ${operation.name} was:\n` + _util.default.inspect(variables) + '\n' + _util.default.inspect(result, {
            depth: null
          }));
        });
      }

      return match.promise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'content-type': 'application/json',
        'Authorization': `bearer ${currentToken}`,
        'Accept': 'application/vnd.github.graphql-profiling+json'
      },
      body: JSON.stringify({
        query: operation.text,
        variables
      })
    });

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {
      /* do nothing */
    }

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);

    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }

    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);

    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }

    return fetch;
  }

}

exports.default = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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