"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

class ModelObserver {
  constructor({
    fetchData,
    didUpdate
  }) {
    this.fetchData = fetchData || (() => {});

    this.didUpdate = didUpdate || (() => {});

    this.activeModel = null;
    this.activeModelData = null;
    this.activeModelUpdateSubscription = null;
    this.inProgress = false;
    this.pending = false;
  }

  setActiveModel(model) {
    if (model !== this.activeModel) {
      if (this.activeModelUpdateSubscription) {
        this.activeModelUpdateSubscription.dispose();
        this.activeModelUpdateSubscription = null;
      }

      this.activeModel = model;
      this.activeModelData = null;
      this.inProgress = false;
      this.pending = false;
      this.didUpdate(model);

      if (model) {
        this.activeModelUpdateSubscription = model.onDidUpdate(() => this.refreshModelData(model));
        return this.refreshModelData(model);
      }
    }

    return null;
  }

  refreshModelData(model = this.activeModel) {
    if (this.inProgress) {
      this.pending = true;
      return null;
    }

    this.lastModelDataRefreshPromise = this._refreshModelData(model);
    return this.lastModelDataRefreshPromise;
  }

  async _refreshModelData(model) {
    try {
      this.inProgress = true;
      const fetchDataPromise = this.fetchData(model);
      this.lastFetchDataPromise = fetchDataPromise;
      const modelData = await fetchDataPromise; // Since we re-fetch immediately when the model changes,
      // we need to ensure a fetch for an old active model
      // does not trample the newer fetch for the newer active model.

      if (fetchDataPromise === this.lastFetchDataPromise) {
        this.activeModel = model;
        this.activeModelData = modelData;
        this.didUpdate(model);
      }
    } finally {
      this.inProgress = false;

      if (this.pending) {
        this.pending = false;
        this.refreshModelData();
      }
    }
  }

  getActiveModel() {
    return this.activeModel;
  }

  getActiveModelData() {
    return this.activeModelData;
  }

  getLastModelDataRefreshPromise() {
    return this.lastModelDataRefreshPromise;
  }

  destroy() {
    if (this.activeModelUpdateSubscription) {
      this.activeModelUpdateSubscription.dispose();
    }
  }

}

exports.default = ModelObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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