"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NoNewline = exports.Unchanged = exports.Deletion = exports.Addition = void 0;

var _atom = require("atom");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Region {
  constructor(marker) {
    this.marker = marker;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.marker.getRange();
  }

  getStartBufferRow() {
    return this.getRange().start.row;
  }

  getEndBufferRow() {
    return this.getRange().end.row;
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }
  /*
   * intersectRows breaks a Region into runs of rows that are included in
   * rowSet and rows that are not. For example:
   *  @this Region        row 10-20
   *  @param rowSet       row 11, 12, 13, 17, 19
   *  @param includeGaps  true (whether the result will include gaps or not)
   *  @return an array of regions like this:
   *    (10, gap = true) (11, 12, 13, gap = false) (14, 15, 16, gap = true)
   *    (17, gap = false) (18, gap = true) (19, gap = false) (20, gap = true)
   */


  intersectRows(rowSet, includeGaps) {
    const intersections = [];
    let withinIntersection = false;
    let currentRow = this.getRange().start.row;
    let nextStartRow = currentRow;

    const finishRowRange = isGap => {
      if (isGap && !includeGaps) {
        nextStartRow = currentRow;
        return;
      }

      if (currentRow <= this.getRange().start.row) {
        return;
      }

      intersections.push({
        intersection: _atom.Range.fromObject([[nextStartRow, 0], [currentRow - 1, Infinity]]),
        gap: isGap
      });
      nextStartRow = currentRow;
    };

    while (currentRow <= this.getRange().end.row) {
      if (rowSet.has(currentRow) && !withinIntersection) {
        // One row past the end of a gap. Start of intersecting row range.
        finishRowRange(true);
        withinIntersection = true;
      } else if (!rowSet.has(currentRow) && withinIntersection) {
        // One row past the end of intersecting row range. Start of the next gap.
        finishRowRange(false);
        withinIntersection = false;
      }

      currentRow++;
    }

    finishRowRange(!withinIntersection);
    return intersections;
  }

  isAddition() {
    return false;
  }

  isDeletion() {
    return false;
  }

  isUnchanged() {
    return false;
  }

  isNoNewline() {
    return false;
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  when(callbacks) {
    const callback = callbacks[this.constructor.name.toLowerCase()] || callbacks.default || (() => undefined);

    return callback();
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;
  }

  destroyMarkers() {
    this.marker.destroy();
  }

  toStringIn(buffer) {
    const raw = buffer.getTextInRange(this.getRange());
    return this.constructor.origin + raw.replace(/\r?\n/g, '$&' + this.constructor.origin) + buffer.lineEndingForRow(this.getRange().end.row);
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(${this.constructor.name} marker=${this.marker.id})`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    return inspectString + '\n';
  }

  isChange() {
    return true;
  }

}

class Addition extends Region {
  isAddition() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Deletion(nextBuffer.markRange(this.getRange()));
  }

}

exports.Addition = Addition;

_defineProperty(Addition, "origin", '+');

_defineProperty(Addition, "layerName", 'addition');

class Deletion extends Region {
  isDeletion() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Addition(nextBuffer.markRange(this.getRange()));
  }

}

exports.Deletion = Deletion;

_defineProperty(Deletion, "origin", '-');

_defineProperty(Deletion, "layerName", 'deletion');

class Unchanged extends Region {
  isUnchanged() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new Unchanged(nextBuffer.markRange(this.getRange()));
  }

}

exports.Unchanged = Unchanged;

_defineProperty(Unchanged, "origin", ' ');

_defineProperty(Unchanged, "layerName", 'unchanged');

class NoNewline extends Region {
  isNoNewline() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new NoNewline(nextBuffer.markRange(this.getRange()));
  }

}

exports.NoNewline = NoNewline;

_defineProperty(NoNewline, "origin", '\\');

_defineProperty(NoNewline, "layerName", 'nonewline');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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