"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFilePatch = buildFilePatch;
exports.buildMultiFilePatch = buildMultiFilePatch;
exports.DEFAULT_OPTIONS = void 0;

var _patchBuffer = _interopRequireDefault(require("./patch-buffer"));

var _hunk = _interopRequireDefault(require("./hunk"));

var _file = _interopRequireWildcard(require("./file"));

var _patch = _interopRequireWildcard(require("./patch"));

var _region = require("./region");

var _filePatch = _interopRequireDefault(require("./file-patch"));

var _multiFilePatch = _interopRequireDefault(require("./multi-file-patch"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_OPTIONS = {
  // Number of lines after which we consider the diff "large"
  largeDiffThreshold: 800,
  // Map of file path (relative to repository root) to Patch render status (EXPANDED, COLLAPSED, TOO_LARGE)
  renderStatusOverrides: {},
  // Existing patch buffer to render onto
  patchBuffer: null
};
exports.DEFAULT_OPTIONS = DEFAULT_OPTIONS;

function buildFilePatch(diffs, options) {
  const opts = _objectSpread({}, DEFAULT_OPTIONS, options);

  const patchBuffer = new _patchBuffer.default();
  let filePatch;

  if (diffs.length === 0) {
    filePatch = emptyDiffFilePatch();
  } else if (diffs.length === 1) {
    filePatch = singleDiffFilePatch(diffs[0], patchBuffer, opts);
  } else if (diffs.length === 2) {
    filePatch = dualDiffFilePatch(diffs[0], diffs[1], patchBuffer, opts);
  } else {
    throw new Error(`Unexpected number of diffs: ${diffs.length}`);
  } // Delete the trailing newline.


  patchBuffer.deleteLastNewline();
  return new _multiFilePatch.default({
    patchBuffer,
    filePatches: [filePatch]
  });
}

function buildMultiFilePatch(diffs, options) {
  const opts = _objectSpread({}, DEFAULT_OPTIONS, options);

  const patchBuffer = new _patchBuffer.default();
  const byPath = new Map();
  const actions = [];
  let index = 0;

  for (const diff of diffs) {
    const thePath = diff.oldPath || diff.newPath;

    if (diff.status === 'added' || diff.status === 'deleted') {
      // Potential paired diff. Either a symlink deletion + content addition or a symlink addition +
      // content deletion.
      const otherHalf = byPath.get(thePath);

      if (otherHalf) {
        // The second half. Complete the paired diff, or fail if they have unexpected statuses or modes.
        const [otherDiff, otherIndex] = otherHalf;

        actions[otherIndex] = function (_diff, _otherDiff) {
          return () => dualDiffFilePatch(_diff, _otherDiff, patchBuffer, opts);
        }(diff, otherDiff);

        byPath.delete(thePath);
      } else {
        // The first half we've seen.
        byPath.set(thePath, [diff, index]);
        index++;
      }
    } else {
      actions[index] = function (_diff) {
        return () => singleDiffFilePatch(_diff, patchBuffer, opts);
      }(diff);

      index++;
    }
  } // Populate unpaired diffs that looked like they could be part of a pair, but weren't.


  for (const [unpairedDiff, originalIndex] of byPath.values()) {
    actions[originalIndex] = function (_unpairedDiff) {
      return () => singleDiffFilePatch(_unpairedDiff, patchBuffer, opts);
    }(unpairedDiff);
  }

  const filePatches = actions.map(action => action()); // Delete the final trailing newline from the last non-empty patch.

  patchBuffer.deleteLastNewline();
  return new _multiFilePatch.default({
    patchBuffer,
    filePatches
  });
}

function emptyDiffFilePatch() {
  return _filePatch.default.createNull();
}

function singleDiffFilePatch(diff, patchBuffer, opts) {
  const wasSymlink = diff.oldMode === _file.default.modes.SYMLINK;
  const isSymlink = diff.newMode === _file.default.modes.SYMLINK;
  let oldSymlink = null;
  let newSymlink = null;

  if (wasSymlink && !isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (!wasSymlink && isSymlink) {
    newSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (wasSymlink && isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
    newSymlink = diff.hunks[0].lines[2].slice(1);
  }

  const oldFile = diff.oldPath !== null || diff.oldMode !== null ? new _file.default({
    path: diff.oldPath,
    mode: diff.oldMode,
    symlink: oldSymlink
  }) : _file.nullFile;
  const newFile = diff.newPath !== null || diff.newMode !== null ? new _file.default({
    path: diff.newPath,
    mode: diff.newMode,
    symlink: newSymlink
  }) : _file.nullFile;
  const renderStatusOverride = oldFile.isPresent() && opts.renderStatusOverrides[oldFile.getPath()] || newFile.isPresent() && opts.renderStatusOverrides[newFile.getPath()] || undefined;

  const renderStatus = renderStatusOverride || isDiffLarge([diff], opts) && _patch.TOO_LARGE || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch.default.createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer.default();
      const [hunks, nextPatchMarker] = buildHunks(diff, subPatchBuffer);
      const nextPatch = new _patch.default({
        status: diff.status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(diff, patchBuffer);
    const patch = new _patch.default({
      status: diff.status,
      hunks,
      marker: patchMarker
    });
    return new _filePatch.default(oldFile, newFile, patch);
  }
}

function dualDiffFilePatch(diff1, diff2, patchBuffer, opts) {
  let modeChangeDiff, contentChangeDiff;

  if (diff1.oldMode === _file.default.modes.SYMLINK || diff1.newMode === _file.default.modes.SYMLINK) {
    modeChangeDiff = diff1;
    contentChangeDiff = diff2;
  } else {
    modeChangeDiff = diff2;
    contentChangeDiff = diff1;
  }

  const filePath = contentChangeDiff.oldPath || contentChangeDiff.newPath;
  const symlink = modeChangeDiff.hunks[0].lines[0].slice(1);
  let status;
  let oldMode, newMode;
  let oldSymlink = null;
  let newSymlink = null;

  if (modeChangeDiff.status === 'added') {
    // contents were deleted and replaced with symlink
    status = 'deleted';
    oldMode = contentChangeDiff.oldMode;
    newMode = modeChangeDiff.newMode;
    newSymlink = symlink;
  } else if (modeChangeDiff.status === 'deleted') {
    // contents were added after symlink was deleted
    status = 'added';
    oldMode = modeChangeDiff.oldMode;
    oldSymlink = symlink;
    newMode = contentChangeDiff.newMode;
  } else {
    throw new Error(`Invalid mode change diff status: ${modeChangeDiff.status}`);
  }

  const oldFile = new _file.default({
    path: filePath,
    mode: oldMode,
    symlink: oldSymlink
  });
  const newFile = new _file.default({
    path: filePath,
    mode: newMode,
    symlink: newSymlink
  });

  const renderStatus = opts.renderStatusOverrides[filePath] || isDiffLarge([contentChangeDiff], opts) && _patch.TOO_LARGE || _patch.EXPANDED;

  if (!renderStatus.isVisible()) {
    const patchMarker = patchBuffer.markPosition(_patch.default.layerName, patchBuffer.getBuffer().getEndPosition(), {
      invalidate: 'never',
      exclusive: false
    });
    return _filePatch.default.createHiddenFilePatch(oldFile, newFile, patchMarker, renderStatus, () => {
      const subPatchBuffer = new _patchBuffer.default();
      const [hunks, nextPatchMarker] = buildHunks(contentChangeDiff, subPatchBuffer);
      const nextPatch = new _patch.default({
        status,
        hunks,
        marker: nextPatchMarker
      });
      subPatchBuffer.deleteLastNewline();
      return {
        patch: nextPatch,
        patchBuffer: subPatchBuffer
      };
    });
  } else {
    const [hunks, patchMarker] = buildHunks(contentChangeDiff, patchBuffer);
    const patch = new _patch.default({
      status,
      hunks,
      marker: patchMarker
    });
    return new _filePatch.default(oldFile, newFile, patch);
  }
}

const CHANGEKIND = {
  '+': _region.Addition,
  '-': _region.Deletion,
  ' ': _region.Unchanged,
  '\\': _region.NoNewline
};

function buildHunks(diff, patchBuffer) {
  const inserter = patchBuffer.createInserterAtEnd().keepBefore(patchBuffer.findAllMarkers({
    endPosition: patchBuffer.getInsertionPoint()
  }));
  let patchMarker = null;
  let firstHunk = true;
  const hunks = [];
  inserter.markWhile(_patch.default.layerName, () => {
    for (const rawHunk of diff.hunks) {
      let firstRegion = true;
      const regions = []; // Separate hunks with an unmarked newline

      if (firstHunk) {
        firstHunk = false;
      } else {
        inserter.insert('\n');
      }

      inserter.markWhile(_hunk.default.layerName, () => {
        let firstRegionLine = true;
        let currentRegionText = '';
        let CurrentRegionKind = null;

        function finishRegion() {
          if (CurrentRegionKind === null) {
            return;
          } // Separate regions with an unmarked newline


          if (firstRegion) {
            firstRegion = false;
          } else {
            inserter.insert('\n');
          }

          inserter.insertMarked(currentRegionText, CurrentRegionKind.layerName, {
            invalidate: 'never',
            exclusive: false,
            callback: function (_regions, _CurrentRegionKind) {
              return regionMarker => {
                _regions.push(new _CurrentRegionKind(regionMarker));
              };
            }(regions, CurrentRegionKind)
          });
        }

        for (const rawLine of rawHunk.lines) {
          const NextRegionKind = CHANGEKIND[rawLine[0]];

          if (NextRegionKind === undefined) {
            throw new Error(`Unknown diff status character: "${rawLine[0]}"`);
          }

          const nextLine = rawLine.slice(1);
          let separator = '';

          if (firstRegionLine) {
            firstRegionLine = false;
          } else {
            separator = '\n';
          }

          if (NextRegionKind === CurrentRegionKind) {
            currentRegionText += separator + nextLine;
            continue;
          } else {
            finishRegion();
            CurrentRegionKind = NextRegionKind;
            currentRegionText = nextLine;
          }
        }

        finishRegion();
      }, {
        invalidate: 'never',
        exclusive: false,
        callback: function (_hunks, _rawHunk, _regions) {
          return hunkMarker => {
            _hunks.push(new _hunk.default({
              oldStartRow: _rawHunk.oldStartLine,
              newStartRow: _rawHunk.newStartLine,
              oldRowCount: _rawHunk.oldLineCount,
              newRowCount: _rawHunk.newLineCount,
              sectionHeading: _rawHunk.heading,
              marker: hunkMarker,
              regions: _regions
            }));
          };
        }(hunks, rawHunk, regions)
      });
    }
  }, {
    invalidate: 'never',
    exclusive: false,
    callback: marker => {
      patchMarker = marker;
    }
  }); // Separate multiple non-empty patches on the same buffer with an unmarked newline. The newline after the final
  // non-empty patch (if there is one) should be deleted before MultiFilePatch construction.

  if (diff.hunks.length > 0) {
    inserter.insert('\n');
  }

  inserter.apply();
  return [hunks, patchMarker];
}

function isDiffLarge(diffs, opts) {
  const size = diffs.reduce((diffSizeCounter, diff) => {
    return diffSizeCounter + diff.hunks.reduce((hunkSizeCounter, hunk) => {
      return hunkSizeCounter + hunk.lines.length;
    }, 0);
  }, 0);
  return size > opts.largeDiffThreshold;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImJ1aWxkZXIuanMiXSwibmFtZXMiOlsiREVGQVVMVF9PUFRJT05TIiwibGFyZ2VEaWZmVGhyZXNob2xkIiwicmVuZGVyU3RhdHVzT3ZlcnJpZGVzIiwicGF0Y2hCdWZmZXIiLCJidWlsZEZpbGVQYXRjaCIsImRpZmZzIiwib3B0aW9ucyIsIm9wdHMiLCJQYXRjaEJ1ZmZlciIsImZpbGVQYXRjaCIsImxlbmd0aCIsImVtcHR5RGlmZkZpbGVQYXRjaCIsInNpbmdsZURpZmZGaWxlUGF0Y2giLCJkdWFsRGlmZkZpbGVQYXRjaCIsIkVycm9yIiwiZGVsZXRlTGFzdE5ld2xpbmUiLCJNdWx0aUZpbGVQYXRjaCIsImZpbGVQYXRjaGVzIiwiYnVpbGRNdWx0aUZpbGVQYXRjaCIsImJ5UGF0aCIsIk1hcCIsImFjdGlvbnMiLCJpbmRleCIsImRpZmYiLCJ0aGVQYXRoIiwib2xkUGF0aCIsIm5ld1BhdGgiLCJzdGF0dXMiLCJvdGhlckhhbGYiLCJnZXQiLCJvdGhlckRpZmYiLCJvdGhlckluZGV4IiwiX2RpZmYiLCJfb3RoZXJEaWZmIiwiZGVsZXRlIiwic2V0IiwidW5wYWlyZWREaWZmIiwib3JpZ2luYWxJbmRleCIsInZhbHVlcyIsIl91bnBhaXJlZERpZmYiLCJtYXAiLCJhY3Rpb24iLCJGaWxlUGF0Y2giLCJjcmVhdGVOdWxsIiwid2FzU3ltbGluayIsIm9sZE1vZGUiLCJGaWxlIiwibW9kZXMiLCJTWU1MSU5LIiwiaXNTeW1saW5rIiwibmV3TW9kZSIsIm9sZFN5bWxpbmsiLCJuZXdTeW1saW5rIiwiaHVua3MiLCJsaW5lcyIsInNsaWNlIiwib2xkRmlsZSIsInBhdGgiLCJtb2RlIiwic3ltbGluayIsIm51bGxGaWxlIiwibmV3RmlsZSIsInJlbmRlclN0YXR1c092ZXJyaWRlIiwiaXNQcmVzZW50IiwiZ2V0UGF0aCIsInVuZGVmaW5lZCIsInJlbmRlclN0YXR1cyIsImlzRGlmZkxhcmdlIiwiVE9PX0xBUkdFIiwiRVhQQU5ERUQiLCJpc1Zpc2libGUiLCJwYXRjaE1hcmtlciIsIm1hcmtQb3NpdGlvbiIsIlBhdGNoIiwibGF5ZXJOYW1lIiwiZ2V0QnVmZmVyIiwiZ2V0RW5kUG9zaXRpb24iLCJpbnZhbGlkYXRlIiwiZXhjbHVzaXZlIiwiY3JlYXRlSGlkZGVuRmlsZVBhdGNoIiwic3ViUGF0Y2hCdWZmZXIiLCJuZXh0UGF0Y2hNYXJrZXIiLCJidWlsZEh1bmtzIiwibmV4dFBhdGNoIiwibWFya2VyIiwicGF0Y2giLCJkaWZmMSIsImRpZmYyIiwibW9kZUNoYW5nZURpZmYiLCJjb250ZW50Q2hhbmdlRGlmZiIsImZpbGVQYXRoIiwiQ0hBTkdFS0lORCIsIkFkZGl0aW9uIiwiRGVsZXRpb24iLCJVbmNoYW5nZWQiLCJOb05ld2xpbmUiLCJpbnNlcnRlciIsImNyZWF0ZUluc2VydGVyQXRFbmQiLCJrZWVwQmVmb3JlIiwiZmluZEFsbE1hcmtlcnMiLCJlbmRQb3NpdGlvbiIsImdldEluc2VydGlvblBvaW50IiwiZmlyc3RIdW5rIiwibWFya1doaWxlIiwicmF3SHVuayIsImZpcnN0UmVnaW9uIiwicmVnaW9ucyIsImluc2VydCIsIkh1bmsiLCJmaXJzdFJlZ2lvbkxpbmUiLCJjdXJyZW50UmVnaW9uVGV4dCIsIkN1cnJlbnRSZWdpb25LaW5kIiwiZmluaXNoUmVnaW9uIiwiaW5zZXJ0TWFya2VkIiwiY2FsbGJhY2siLCJfcmVnaW9ucyIsIl9DdXJyZW50UmVnaW9uS2luZCIsInJlZ2lvbk1hcmtlciIsInB1c2giLCJyYXdMaW5lIiwiTmV4dFJlZ2lvbktpbmQiLCJuZXh0TGluZSIsInNlcGFyYXRvciIsIl9odW5rcyIsIl9yYXdIdW5rIiwiaHVua01hcmtlciIsIm9sZFN0YXJ0Um93Iiwib2xkU3RhcnRMaW5lIiwibmV3U3RhcnRSb3ciLCJuZXdTdGFydExpbmUiLCJvbGRSb3dDb3VudCIsIm9sZExpbmVDb3VudCIsIm5ld1Jvd0NvdW50IiwibmV3TGluZUNvdW50Iiwic2VjdGlvbkhlYWRpbmciLCJoZWFkaW5nIiwiYXBwbHkiLCJzaXplIiwicmVkdWNlIiwiZGlmZlNpemVDb3VudGVyIiwiaHVua1NpemVDb3VudGVyIiwiaHVuayJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBQ0E7Ozs7Ozs7Ozs7QUFFTyxNQUFNQSxlQUFlLEdBQUc7QUFDN0I7QUFDQUMsRUFBQUEsa0JBQWtCLEVBQUUsR0FGUztBQUk3QjtBQUNBQyxFQUFBQSxxQkFBcUIsRUFBRSxFQUxNO0FBTzdCO0FBQ0FDLEVBQUFBLFdBQVcsRUFBRTtBQVJnQixDQUF4Qjs7O0FBV0EsU0FBU0MsY0FBVCxDQUF3QkMsS0FBeEIsRUFBK0JDLE9BQS9CLEVBQXdDO0FBQzdDLFFBQU1DLElBQUkscUJBQU9QLGVBQVAsRUFBMkJNLE9BQTNCLENBQVY7O0FBQ0EsUUFBTUgsV0FBVyxHQUFHLElBQUlLLG9CQUFKLEVBQXBCO0FBRUEsTUFBSUMsU0FBSjs7QUFDQSxNQUFJSixLQUFLLENBQUNLLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFDdEJELElBQUFBLFNBQVMsR0FBR0Usa0JBQWtCLEVBQTlCO0FBQ0QsR0FGRCxNQUVPLElBQUlOLEtBQUssQ0FBQ0ssTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUM3QkQsSUFBQUEsU0FBUyxHQUFHRyxtQkFBbUIsQ0FBQ1AsS0FBSyxDQUFDLENBQUQsQ0FBTixFQUFXRixXQUFYLEVBQXdCSSxJQUF4QixDQUEvQjtBQUNELEdBRk0sTUFFQSxJQUFJRixLQUFLLENBQUNLLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFDN0JELElBQUFBLFNBQVMsR0FBR0ksaUJBQWlCLENBQUNSLEtBQUssQ0FBQyxDQUFELENBQU4sRUFBV0EsS0FBSyxDQUFDLENBQUQsQ0FBaEIsRUFBcUJGLFdBQXJCLEVBQWtDSSxJQUFsQyxDQUE3QjtBQUNELEdBRk0sTUFFQTtBQUNMLFVBQU0sSUFBSU8sS0FBSixDQUFXLCtCQUE4QlQsS0FBSyxDQUFDSyxNQUFPLEVBQXRELENBQU47QUFDRCxHQWI0QyxDQWU3Qzs7O0FBQ0FQLEVBQUFBLFdBQVcsQ0FBQ1ksaUJBQVo7QUFFQSxTQUFPLElBQUlDLHVCQUFKLENBQW1CO0FBQUNiLElBQUFBLFdBQUQ7QUFBY2MsSUFBQUEsV0FBVyxFQUFFLENBQUNSLFNBQUQ7QUFBM0IsR0FBbkIsQ0FBUDtBQUNEOztBQUVNLFNBQVNTLG1CQUFULENBQTZCYixLQUE3QixFQUFvQ0MsT0FBcEMsRUFBNkM7QUFDbEQsUUFBTUMsSUFBSSxxQkFBT1AsZUFBUCxFQUEyQk0sT0FBM0IsQ0FBVjs7QUFFQSxRQUFNSCxXQUFXLEdBQUcsSUFBSUssb0JBQUosRUFBcEI7QUFFQSxRQUFNVyxNQUFNLEdBQUcsSUFBSUMsR0FBSixFQUFmO0FBQ0EsUUFBTUMsT0FBTyxHQUFHLEVBQWhCO0FBRUEsTUFBSUMsS0FBSyxHQUFHLENBQVo7O0FBQ0EsT0FBSyxNQUFNQyxJQUFYLElBQW1CbEIsS0FBbkIsRUFBMEI7QUFDeEIsVUFBTW1CLE9BQU8sR0FBR0QsSUFBSSxDQUFDRSxPQUFMLElBQWdCRixJQUFJLENBQUNHLE9BQXJDOztBQUVBLFFBQUlILElBQUksQ0FBQ0ksTUFBTCxLQUFnQixPQUFoQixJQUEyQkosSUFBSSxDQUFDSSxNQUFMLEtBQWdCLFNBQS9DLEVBQTBEO0FBQ3hEO0FBQ0E7QUFDQSxZQUFNQyxTQUFTLEdBQUdULE1BQU0sQ0FBQ1UsR0FBUCxDQUFXTCxPQUFYLENBQWxCOztBQUNBLFVBQUlJLFNBQUosRUFBZTtBQUNiO0FBQ0EsY0FBTSxDQUFDRSxTQUFELEVBQVlDLFVBQVosSUFBMEJILFNBQWhDOztBQUNBUCxRQUFBQSxPQUFPLENBQUNVLFVBQUQsQ0FBUCxHQUF1QixVQUFTQyxLQUFULEVBQWdCQyxVQUFoQixFQUE0QjtBQUNqRCxpQkFBTyxNQUFNcEIsaUJBQWlCLENBQUNtQixLQUFELEVBQVFDLFVBQVIsRUFBb0I5QixXQUFwQixFQUFpQ0ksSUFBakMsQ0FBOUI7QUFDRCxTQUZxQixDQUVuQmdCLElBRm1CLEVBRWJPLFNBRmEsQ0FBdEI7O0FBR0FYLFFBQUFBLE1BQU0sQ0FBQ2UsTUFBUCxDQUFjVixPQUFkO0FBQ0QsT0FQRCxNQU9PO0FBQ0w7QUFDQUwsUUFBQUEsTUFBTSxDQUFDZ0IsR0FBUCxDQUFXWCxPQUFYLEVBQW9CLENBQUNELElBQUQsRUFBT0QsS0FBUCxDQUFwQjtBQUNBQSxRQUFBQSxLQUFLO0FBQ047QUFDRixLQWhCRCxNQWdCTztBQUNMRCxNQUFBQSxPQUFPLENBQUNDLEtBQUQsQ0FBUCxHQUFrQixVQUFTVSxLQUFULEVBQWdCO0FBQ2hDLGVBQU8sTUFBTXBCLG1CQUFtQixDQUFDb0IsS0FBRCxFQUFRN0IsV0FBUixFQUFxQkksSUFBckIsQ0FBaEM7QUFDRCxPQUZnQixDQUVkZ0IsSUFGYyxDQUFqQjs7QUFHQUQsTUFBQUEsS0FBSztBQUNOO0FBQ0YsR0FsQ2lELENBb0NsRDs7O0FBQ0EsT0FBSyxNQUFNLENBQUNjLFlBQUQsRUFBZUMsYUFBZixDQUFYLElBQTRDbEIsTUFBTSxDQUFDbUIsTUFBUCxFQUE1QyxFQUE2RDtBQUMzRGpCLElBQUFBLE9BQU8sQ0FBQ2dCLGFBQUQsQ0FBUCxHQUEwQixVQUFTRSxhQUFULEVBQXdCO0FBQ2hELGFBQU8sTUFBTTNCLG1CQUFtQixDQUFDMkIsYUFBRCxFQUFnQnBDLFdBQWhCLEVBQTZCSSxJQUE3QixDQUFoQztBQUNELEtBRndCLENBRXRCNkIsWUFGc0IsQ0FBekI7QUFHRDs7QUFFRCxRQUFNbkIsV0FBVyxHQUFHSSxPQUFPLENBQUNtQixHQUFSLENBQVlDLE1BQU0sSUFBSUEsTUFBTSxFQUE1QixDQUFwQixDQTNDa0QsQ0E2Q2xEOztBQUNBdEMsRUFBQUEsV0FBVyxDQUFDWSxpQkFBWjtBQUVBLFNBQU8sSUFBSUMsdUJBQUosQ0FBbUI7QUFBQ2IsSUFBQUEsV0FBRDtBQUFjYyxJQUFBQTtBQUFkLEdBQW5CLENBQVA7QUFDRDs7QUFFRCxTQUFTTixrQkFBVCxHQUE4QjtBQUM1QixTQUFPK0IsbUJBQVVDLFVBQVYsRUFBUDtBQUNEOztBQUVELFNBQVMvQixtQkFBVCxDQUE2QlcsSUFBN0IsRUFBbUNwQixXQUFuQyxFQUFnREksSUFBaEQsRUFBc0Q7QUFDcEQsUUFBTXFDLFVBQVUsR0FBR3JCLElBQUksQ0FBQ3NCLE9BQUwsS0FBaUJDLGNBQUtDLEtBQUwsQ0FBV0MsT0FBL0M7QUFDQSxRQUFNQyxTQUFTLEdBQUcxQixJQUFJLENBQUMyQixPQUFMLEtBQWlCSixjQUFLQyxLQUFMLENBQVdDLE9BQTlDO0FBRUEsTUFBSUcsVUFBVSxHQUFHLElBQWpCO0FBQ0EsTUFBSUMsVUFBVSxHQUFHLElBQWpCOztBQUNBLE1BQUlSLFVBQVUsSUFBSSxDQUFDSyxTQUFuQixFQUE4QjtBQUM1QkUsSUFBQUEsVUFBVSxHQUFHNUIsSUFBSSxDQUFDOEIsS0FBTCxDQUFXLENBQVgsRUFBY0MsS0FBZCxDQUFvQixDQUFwQixFQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBYjtBQUNELEdBRkQsTUFFTyxJQUFJLENBQUNYLFVBQUQsSUFBZUssU0FBbkIsRUFBOEI7QUFDbkNHLElBQUFBLFVBQVUsR0FBRzdCLElBQUksQ0FBQzhCLEtBQUwsQ0FBVyxDQUFYLEVBQWNDLEtBQWQsQ0FBb0IsQ0FBcEIsRUFBdUJDLEtBQXZCLENBQTZCLENBQTdCLENBQWI7QUFDRCxHQUZNLE1BRUEsSUFBSVgsVUFBVSxJQUFJSyxTQUFsQixFQUE2QjtBQUNsQ0UsSUFBQUEsVUFBVSxHQUFHNUIsSUFBSSxDQUFDOEIsS0FBTCxDQUFXLENBQVgsRUFBY0MsS0FBZCxDQUFvQixDQUFwQixFQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBYjtBQUNBSCxJQUFBQSxVQUFVLEdBQUc3QixJQUFJLENBQUM4QixLQUFMLENBQVcsQ0FBWCxFQUFjQyxLQUFkLENBQW9CLENBQXBCLEVBQXVCQyxLQUF2QixDQUE2QixDQUE3QixDQUFiO0FBQ0Q7O0FBRUQsUUFBTUMsT0FBTyxHQUFHakMsSUFBSSxDQUFDRSxPQUFMLEtBQWlCLElBQWpCLElBQXlCRixJQUFJLENBQUNzQixPQUFMLEtBQWlCLElBQTFDLEdBQ1osSUFBSUMsYUFBSixDQUFTO0FBQUNXLElBQUFBLElBQUksRUFBRWxDLElBQUksQ0FBQ0UsT0FBWjtBQUFxQmlDLElBQUFBLElBQUksRUFBRW5DLElBQUksQ0FBQ3NCLE9BQWhDO0FBQXlDYyxJQUFBQSxPQUFPLEVBQUVSO0FBQWxELEdBQVQsQ0FEWSxHQUVaUyxjQUZKO0FBR0EsUUFBTUMsT0FBTyxHQUFHdEMsSUFBSSxDQUFDRyxPQUFMLEtBQWlCLElBQWpCLElBQXlCSCxJQUFJLENBQUMyQixPQUFMLEtBQWlCLElBQTFDLEdBQ1osSUFBSUosYUFBSixDQUFTO0FBQUNXLElBQUFBLElBQUksRUFBRWxDLElBQUksQ0FBQ0csT0FBWjtBQUFxQmdDLElBQUFBLElBQUksRUFBRW5DLElBQUksQ0FBQzJCLE9BQWhDO0FBQXlDUyxJQUFBQSxPQUFPLEVBQUVQO0FBQWxELEdBQVQsQ0FEWSxHQUVaUSxjQUZKO0FBSUEsUUFBTUUsb0JBQW9CLEdBQ3ZCTixPQUFPLENBQUNPLFNBQVIsTUFBdUJ4RCxJQUFJLENBQUNMLHFCQUFMLENBQTJCc0QsT0FBTyxDQUFDUSxPQUFSLEVBQTNCLENBQXhCLElBQ0NILE9BQU8sQ0FBQ0UsU0FBUixNQUF1QnhELElBQUksQ0FBQ0wscUJBQUwsQ0FBMkIyRCxPQUFPLENBQUNHLE9BQVIsRUFBM0IsQ0FEeEIsSUFFQUMsU0FIRjs7QUFLQSxRQUFNQyxZQUFZLEdBQUdKLG9CQUFvQixJQUN0Q0ssV0FBVyxDQUFDLENBQUM1QyxJQUFELENBQUQsRUFBU2hCLElBQVQsQ0FBWCxJQUE2QjZELGdCQURYLElBRW5CQyxlQUZGOztBQUlBLE1BQUksQ0FBQ0gsWUFBWSxDQUFDSSxTQUFiLEVBQUwsRUFBK0I7QUFDN0IsVUFBTUMsV0FBVyxHQUFHcEUsV0FBVyxDQUFDcUUsWUFBWixDQUNsQkMsZUFBTUMsU0FEWSxFQUVsQnZFLFdBQVcsQ0FBQ3dFLFNBQVosR0FBd0JDLGNBQXhCLEVBRmtCLEVBR2xCO0FBQUNDLE1BQUFBLFVBQVUsRUFBRSxPQUFiO0FBQXNCQyxNQUFBQSxTQUFTLEVBQUU7QUFBakMsS0FIa0IsQ0FBcEI7QUFNQSxXQUFPcEMsbUJBQVVxQyxxQkFBVixDQUNMdkIsT0FESyxFQUNJSyxPQURKLEVBQ2FVLFdBRGIsRUFDMEJMLFlBRDFCLEVBRUwsTUFBTTtBQUNKLFlBQU1jLGNBQWMsR0FBRyxJQUFJeEUsb0JBQUosRUFBdkI7QUFDQSxZQUFNLENBQUM2QyxLQUFELEVBQVE0QixlQUFSLElBQTJCQyxVQUFVLENBQUMzRCxJQUFELEVBQU95RCxjQUFQLENBQTNDO0FBQ0EsWUFBTUcsU0FBUyxHQUFHLElBQUlWLGNBQUosQ0FBVTtBQUFDOUMsUUFBQUEsTUFBTSxFQUFFSixJQUFJLENBQUNJLE1BQWQ7QUFBc0IwQixRQUFBQSxLQUF0QjtBQUE2QitCLFFBQUFBLE1BQU0sRUFBRUg7QUFBckMsT0FBVixDQUFsQjtBQUVBRCxNQUFBQSxjQUFjLENBQUNqRSxpQkFBZjtBQUNBLGFBQU87QUFBQ3NFLFFBQUFBLEtBQUssRUFBRUYsU0FBUjtBQUFtQmhGLFFBQUFBLFdBQVcsRUFBRTZFO0FBQWhDLE9BQVA7QUFDRCxLQVRJLENBQVA7QUFXRCxHQWxCRCxNQWtCTztBQUNMLFVBQU0sQ0FBQzNCLEtBQUQsRUFBUWtCLFdBQVIsSUFBdUJXLFVBQVUsQ0FBQzNELElBQUQsRUFBT3BCLFdBQVAsQ0FBdkM7QUFDQSxVQUFNa0YsS0FBSyxHQUFHLElBQUlaLGNBQUosQ0FBVTtBQUFDOUMsTUFBQUEsTUFBTSxFQUFFSixJQUFJLENBQUNJLE1BQWQ7QUFBc0IwQixNQUFBQSxLQUF0QjtBQUE2QitCLE1BQUFBLE1BQU0sRUFBRWI7QUFBckMsS0FBVixDQUFkO0FBRUEsV0FBTyxJQUFJN0Isa0JBQUosQ0FBY2MsT0FBZCxFQUF1QkssT0FBdkIsRUFBZ0N3QixLQUFoQyxDQUFQO0FBQ0Q7QUFDRjs7QUFFRCxTQUFTeEUsaUJBQVQsQ0FBMkJ5RSxLQUEzQixFQUFrQ0MsS0FBbEMsRUFBeUNwRixXQUF6QyxFQUFzREksSUFBdEQsRUFBNEQ7QUFDMUQsTUFBSWlGLGNBQUosRUFBb0JDLGlCQUFwQjs7QUFDQSxNQUFJSCxLQUFLLENBQUN6QyxPQUFOLEtBQWtCQyxjQUFLQyxLQUFMLENBQVdDLE9BQTdCLElBQXdDc0MsS0FBSyxDQUFDcEMsT0FBTixLQUFrQkosY0FBS0MsS0FBTCxDQUFXQyxPQUF6RSxFQUFrRjtBQUNoRndDLElBQUFBLGNBQWMsR0FBR0YsS0FBakI7QUFDQUcsSUFBQUEsaUJBQWlCLEdBQUdGLEtBQXBCO0FBQ0QsR0FIRCxNQUdPO0FBQ0xDLElBQUFBLGNBQWMsR0FBR0QsS0FBakI7QUFDQUUsSUFBQUEsaUJBQWlCLEdBQUdILEtBQXBCO0FBQ0Q7O0FBRUQsUUFBTUksUUFBUSxHQUFHRCxpQkFBaUIsQ0FBQ2hFLE9BQWxCLElBQTZCZ0UsaUJBQWlCLENBQUMvRCxPQUFoRTtBQUNBLFFBQU1pQyxPQUFPLEdBQUc2QixjQUFjLENBQUNuQyxLQUFmLENBQXFCLENBQXJCLEVBQXdCQyxLQUF4QixDQUE4QixDQUE5QixFQUFpQ0MsS0FBakMsQ0FBdUMsQ0FBdkMsQ0FBaEI7QUFFQSxNQUFJNUIsTUFBSjtBQUNBLE1BQUlrQixPQUFKLEVBQWFLLE9BQWI7QUFDQSxNQUFJQyxVQUFVLEdBQUcsSUFBakI7QUFDQSxNQUFJQyxVQUFVLEdBQUcsSUFBakI7O0FBQ0EsTUFBSW9DLGNBQWMsQ0FBQzdELE1BQWYsS0FBMEIsT0FBOUIsRUFBdUM7QUFDckM7QUFDQUEsSUFBQUEsTUFBTSxHQUFHLFNBQVQ7QUFDQWtCLElBQUFBLE9BQU8sR0FBRzRDLGlCQUFpQixDQUFDNUMsT0FBNUI7QUFDQUssSUFBQUEsT0FBTyxHQUFHc0MsY0FBYyxDQUFDdEMsT0FBekI7QUFDQUUsSUFBQUEsVUFBVSxHQUFHTyxPQUFiO0FBQ0QsR0FORCxNQU1PLElBQUk2QixjQUFjLENBQUM3RCxNQUFmLEtBQTBCLFNBQTlCLEVBQXlDO0FBQzlDO0FBQ0FBLElBQUFBLE1BQU0sR0FBRyxPQUFUO0FBQ0FrQixJQUFBQSxPQUFPLEdBQUcyQyxjQUFjLENBQUMzQyxPQUF6QjtBQUNBTSxJQUFBQSxVQUFVLEdBQUdRLE9BQWI7QUFDQVQsSUFBQUEsT0FBTyxHQUFHdUMsaUJBQWlCLENBQUN2QyxPQUE1QjtBQUNELEdBTk0sTUFNQTtBQUNMLFVBQU0sSUFBSXBDLEtBQUosQ0FBVyxvQ0FBbUMwRSxjQUFjLENBQUM3RCxNQUFPLEVBQXBFLENBQU47QUFDRDs7QUFFRCxRQUFNNkIsT0FBTyxHQUFHLElBQUlWLGFBQUosQ0FBUztBQUFDVyxJQUFBQSxJQUFJLEVBQUVpQyxRQUFQO0FBQWlCaEMsSUFBQUEsSUFBSSxFQUFFYixPQUF2QjtBQUFnQ2MsSUFBQUEsT0FBTyxFQUFFUjtBQUF6QyxHQUFULENBQWhCO0FBQ0EsUUFBTVUsT0FBTyxHQUFHLElBQUlmLGFBQUosQ0FBUztBQUFDVyxJQUFBQSxJQUFJLEVBQUVpQyxRQUFQO0FBQWlCaEMsSUFBQUEsSUFBSSxFQUFFUixPQUF2QjtBQUFnQ1MsSUFBQUEsT0FBTyxFQUFFUDtBQUF6QyxHQUFULENBQWhCOztBQUVBLFFBQU1jLFlBQVksR0FBRzNELElBQUksQ0FBQ0wscUJBQUwsQ0FBMkJ3RixRQUEzQixLQUNsQnZCLFdBQVcsQ0FBQyxDQUFDc0IsaUJBQUQsQ0FBRCxFQUFzQmxGLElBQXRCLENBQVgsSUFBMEM2RCxnQkFEeEIsSUFFbkJDLGVBRkY7O0FBSUEsTUFBSSxDQUFDSCxZQUFZLENBQUNJLFNBQWIsRUFBTCxFQUErQjtBQUM3QixVQUFNQyxXQUFXLEdBQUdwRSxXQUFXLENBQUNxRSxZQUFaLENBQ2xCQyxlQUFNQyxTQURZLEVBRWxCdkUsV0FBVyxDQUFDd0UsU0FBWixHQUF3QkMsY0FBeEIsRUFGa0IsRUFHbEI7QUFBQ0MsTUFBQUEsVUFBVSxFQUFFLE9BQWI7QUFBc0JDLE1BQUFBLFNBQVMsRUFBRTtBQUFqQyxLQUhrQixDQUFwQjtBQU1BLFdBQU9wQyxtQkFBVXFDLHFCQUFWLENBQ0x2QixPQURLLEVBQ0lLLE9BREosRUFDYVUsV0FEYixFQUMwQkwsWUFEMUIsRUFFTCxNQUFNO0FBQ0osWUFBTWMsY0FBYyxHQUFHLElBQUl4RSxvQkFBSixFQUF2QjtBQUNBLFlBQU0sQ0FBQzZDLEtBQUQsRUFBUTRCLGVBQVIsSUFBMkJDLFVBQVUsQ0FBQ08saUJBQUQsRUFBb0JULGNBQXBCLENBQTNDO0FBQ0EsWUFBTUcsU0FBUyxHQUFHLElBQUlWLGNBQUosQ0FBVTtBQUFDOUMsUUFBQUEsTUFBRDtBQUFTMEIsUUFBQUEsS0FBVDtBQUFnQitCLFFBQUFBLE1BQU0sRUFBRUg7QUFBeEIsT0FBVixDQUFsQjtBQUVBRCxNQUFBQSxjQUFjLENBQUNqRSxpQkFBZjtBQUNBLGFBQU87QUFBQ3NFLFFBQUFBLEtBQUssRUFBRUYsU0FBUjtBQUFtQmhGLFFBQUFBLFdBQVcsRUFBRTZFO0FBQWhDLE9BQVA7QUFDRCxLQVRJLENBQVA7QUFXRCxHQWxCRCxNQWtCTztBQUNMLFVBQU0sQ0FBQzNCLEtBQUQsRUFBUWtCLFdBQVIsSUFBdUJXLFVBQVUsQ0FBQ08saUJBQUQsRUFBb0J0RixXQUFwQixDQUF2QztBQUNBLFVBQU1rRixLQUFLLEdBQUcsSUFBSVosY0FBSixDQUFVO0FBQUM5QyxNQUFBQSxNQUFEO0FBQVMwQixNQUFBQSxLQUFUO0FBQWdCK0IsTUFBQUEsTUFBTSxFQUFFYjtBQUF4QixLQUFWLENBQWQ7QUFFQSxXQUFPLElBQUk3QixrQkFBSixDQUFjYyxPQUFkLEVBQXVCSyxPQUF2QixFQUFnQ3dCLEtBQWhDLENBQVA7QUFDRDtBQUNGOztBQUVELE1BQU1NLFVBQVUsR0FBRztBQUNqQixPQUFLQyxnQkFEWTtBQUVqQixPQUFLQyxnQkFGWTtBQUdqQixPQUFLQyxpQkFIWTtBQUlqQixRQUFNQztBQUpXLENBQW5COztBQU9BLFNBQVNiLFVBQVQsQ0FBb0IzRCxJQUFwQixFQUEwQnBCLFdBQTFCLEVBQXVDO0FBQ3JDLFFBQU02RixRQUFRLEdBQUc3RixXQUFXLENBQUM4RixtQkFBWixHQUNkQyxVQURjLENBQ0gvRixXQUFXLENBQUNnRyxjQUFaLENBQTJCO0FBQUNDLElBQUFBLFdBQVcsRUFBRWpHLFdBQVcsQ0FBQ2tHLGlCQUFaO0FBQWQsR0FBM0IsQ0FERyxDQUFqQjtBQUdBLE1BQUk5QixXQUFXLEdBQUcsSUFBbEI7QUFDQSxNQUFJK0IsU0FBUyxHQUFHLElBQWhCO0FBQ0EsUUFBTWpELEtBQUssR0FBRyxFQUFkO0FBRUEyQyxFQUFBQSxRQUFRLENBQUNPLFNBQVQsQ0FBbUI5QixlQUFNQyxTQUF6QixFQUFvQyxNQUFNO0FBQ3hDLFNBQUssTUFBTThCLE9BQVgsSUFBc0JqRixJQUFJLENBQUM4QixLQUEzQixFQUFrQztBQUNoQyxVQUFJb0QsV0FBVyxHQUFHLElBQWxCO0FBQ0EsWUFBTUMsT0FBTyxHQUFHLEVBQWhCLENBRmdDLENBSWhDOztBQUNBLFVBQUlKLFNBQUosRUFBZTtBQUNiQSxRQUFBQSxTQUFTLEdBQUcsS0FBWjtBQUNELE9BRkQsTUFFTztBQUNMTixRQUFBQSxRQUFRLENBQUNXLE1BQVQsQ0FBZ0IsSUFBaEI7QUFDRDs7QUFFRFgsTUFBQUEsUUFBUSxDQUFDTyxTQUFULENBQW1CSyxjQUFLbEMsU0FBeEIsRUFBbUMsTUFBTTtBQUN2QyxZQUFJbUMsZUFBZSxHQUFHLElBQXRCO0FBQ0EsWUFBSUMsaUJBQWlCLEdBQUcsRUFBeEI7QUFDQSxZQUFJQyxpQkFBaUIsR0FBRyxJQUF4Qjs7QUFFQSxpQkFBU0MsWUFBVCxHQUF3QjtBQUN0QixjQUFJRCxpQkFBaUIsS0FBSyxJQUExQixFQUFnQztBQUM5QjtBQUNELFdBSHFCLENBS3RCOzs7QUFDQSxjQUFJTixXQUFKLEVBQWlCO0FBQ2ZBLFlBQUFBLFdBQVcsR0FBRyxLQUFkO0FBQ0QsV0FGRCxNQUVPO0FBQ0xULFlBQUFBLFFBQVEsQ0FBQ1csTUFBVCxDQUFnQixJQUFoQjtBQUNEOztBQUVEWCxVQUFBQSxRQUFRLENBQUNpQixZQUFULENBQXNCSCxpQkFBdEIsRUFBeUNDLGlCQUFpQixDQUFDckMsU0FBM0QsRUFBc0U7QUFDcEVHLFlBQUFBLFVBQVUsRUFBRSxPQUR3RDtBQUVwRUMsWUFBQUEsU0FBUyxFQUFFLEtBRnlEO0FBR3BFb0MsWUFBQUEsUUFBUSxFQUFHLFVBQVNDLFFBQVQsRUFBbUJDLGtCQUFuQixFQUF1QztBQUNoRCxxQkFBT0MsWUFBWSxJQUFJO0FBQUVGLGdCQUFBQSxRQUFRLENBQUNHLElBQVQsQ0FBYyxJQUFJRixrQkFBSixDQUF1QkMsWUFBdkIsQ0FBZDtBQUFzRCxlQUEvRTtBQUNELGFBRlMsQ0FFUFgsT0FGTyxFQUVFSyxpQkFGRjtBQUgwRCxXQUF0RTtBQU9EOztBQUVELGFBQUssTUFBTVEsT0FBWCxJQUFzQmYsT0FBTyxDQUFDbEQsS0FBOUIsRUFBcUM7QUFDbkMsZ0JBQU1rRSxjQUFjLEdBQUc3QixVQUFVLENBQUM0QixPQUFPLENBQUMsQ0FBRCxDQUFSLENBQWpDOztBQUNBLGNBQUlDLGNBQWMsS0FBS3ZELFNBQXZCLEVBQWtDO0FBQ2hDLGtCQUFNLElBQUluRCxLQUFKLENBQVcsbUNBQWtDeUcsT0FBTyxDQUFDLENBQUQsQ0FBSSxHQUF4RCxDQUFOO0FBQ0Q7O0FBQ0QsZ0JBQU1FLFFBQVEsR0FBR0YsT0FBTyxDQUFDaEUsS0FBUixDQUFjLENBQWQsQ0FBakI7QUFFQSxjQUFJbUUsU0FBUyxHQUFHLEVBQWhCOztBQUNBLGNBQUliLGVBQUosRUFBcUI7QUFDbkJBLFlBQUFBLGVBQWUsR0FBRyxLQUFsQjtBQUNELFdBRkQsTUFFTztBQUNMYSxZQUFBQSxTQUFTLEdBQUcsSUFBWjtBQUNEOztBQUVELGNBQUlGLGNBQWMsS0FBS1QsaUJBQXZCLEVBQTBDO0FBQ3hDRCxZQUFBQSxpQkFBaUIsSUFBSVksU0FBUyxHQUFHRCxRQUFqQztBQUVBO0FBQ0QsV0FKRCxNQUlPO0FBQ0xULFlBQUFBLFlBQVk7QUFFWkQsWUFBQUEsaUJBQWlCLEdBQUdTLGNBQXBCO0FBQ0FWLFlBQUFBLGlCQUFpQixHQUFHVyxRQUFwQjtBQUNEO0FBQ0Y7O0FBQ0RULFFBQUFBLFlBQVk7QUFDYixPQXBERCxFQW9ERztBQUNEbkMsUUFBQUEsVUFBVSxFQUFFLE9BRFg7QUFFREMsUUFBQUEsU0FBUyxFQUFFLEtBRlY7QUFHRG9DLFFBQUFBLFFBQVEsRUFBRyxVQUFTUyxNQUFULEVBQWlCQyxRQUFqQixFQUEyQlQsUUFBM0IsRUFBcUM7QUFDOUMsaUJBQU9VLFVBQVUsSUFBSTtBQUNuQkYsWUFBQUEsTUFBTSxDQUFDTCxJQUFQLENBQVksSUFBSVYsYUFBSixDQUFTO0FBQ25Ca0IsY0FBQUEsV0FBVyxFQUFFRixRQUFRLENBQUNHLFlBREg7QUFFbkJDLGNBQUFBLFdBQVcsRUFBRUosUUFBUSxDQUFDSyxZQUZIO0FBR25CQyxjQUFBQSxXQUFXLEVBQUVOLFFBQVEsQ0FBQ08sWUFISDtBQUluQkMsY0FBQUEsV0FBVyxFQUFFUixRQUFRLENBQUNTLFlBSkg7QUFLbkJDLGNBQUFBLGNBQWMsRUFBRVYsUUFBUSxDQUFDVyxPQUxOO0FBTW5CbkQsY0FBQUEsTUFBTSxFQUFFeUMsVUFOVztBQU9uQm5CLGNBQUFBLE9BQU8sRUFBRVM7QUFQVSxhQUFULENBQVo7QUFTRCxXQVZEO0FBV0QsU0FaUyxDQVlQOUQsS0FaTyxFQVlBbUQsT0FaQSxFQVlTRSxPQVpUO0FBSFQsT0FwREg7QUFxRUQ7QUFDRixHQWxGRCxFQWtGRztBQUNEN0IsSUFBQUEsVUFBVSxFQUFFLE9BRFg7QUFFREMsSUFBQUEsU0FBUyxFQUFFLEtBRlY7QUFHRG9DLElBQUFBLFFBQVEsRUFBRTlCLE1BQU0sSUFBSTtBQUFFYixNQUFBQSxXQUFXLEdBQUdhLE1BQWQ7QUFBdUI7QUFINUMsR0FsRkgsRUFScUMsQ0FnR3JDO0FBQ0E7O0FBQ0EsTUFBSTdELElBQUksQ0FBQzhCLEtBQUwsQ0FBVzNDLE1BQVgsR0FBb0IsQ0FBeEIsRUFBMkI7QUFDekJzRixJQUFBQSxRQUFRLENBQUNXLE1BQVQsQ0FBZ0IsSUFBaEI7QUFDRDs7QUFFRFgsRUFBQUEsUUFBUSxDQUFDd0MsS0FBVDtBQUVBLFNBQU8sQ0FBQ25GLEtBQUQsRUFBUWtCLFdBQVIsQ0FBUDtBQUNEOztBQUVELFNBQVNKLFdBQVQsQ0FBcUI5RCxLQUFyQixFQUE0QkUsSUFBNUIsRUFBa0M7QUFDaEMsUUFBTWtJLElBQUksR0FBR3BJLEtBQUssQ0FBQ3FJLE1BQU4sQ0FBYSxDQUFDQyxlQUFELEVBQWtCcEgsSUFBbEIsS0FBMkI7QUFDbkQsV0FBT29ILGVBQWUsR0FBR3BILElBQUksQ0FBQzhCLEtBQUwsQ0FBV3FGLE1BQVgsQ0FBa0IsQ0FBQ0UsZUFBRCxFQUFrQkMsSUFBbEIsS0FBMkI7QUFDcEUsYUFBT0QsZUFBZSxHQUFHQyxJQUFJLENBQUN2RixLQUFMLENBQVc1QyxNQUFwQztBQUNELEtBRndCLEVBRXRCLENBRnNCLENBQXpCO0FBR0QsR0FKWSxFQUlWLENBSlUsQ0FBYjtBQU1BLFNBQU8rSCxJQUFJLEdBQUdsSSxJQUFJLENBQUNOLGtCQUFuQjtBQUNEIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tLTEuMzYuMC9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgUGF0Y2hCdWZmZXIgZnJvbSAnLi9wYXRjaC1idWZmZXInO1xuaW1wb3J0IEh1bmsgZnJvbSAnLi9odW5rJztcbmltcG9ydCBGaWxlLCB7bnVsbEZpbGV9IGZyb20gJy4vZmlsZSc7XG5pbXBvcnQgUGF0Y2gsIHtUT09fTEFSR0UsIEVYUEFOREVEfSBmcm9tICcuL3BhdGNoJztcbmltcG9ydCB7VW5jaGFuZ2VkLCBBZGRpdGlvbiwgRGVsZXRpb24sIE5vTmV3bGluZX0gZnJvbSAnLi9yZWdpb24nO1xuaW1wb3J0IEZpbGVQYXRjaCBmcm9tICcuL2ZpbGUtcGF0Y2gnO1xuaW1wb3J0IE11bHRpRmlsZVBhdGNoIGZyb20gJy4vbXVsdGktZmlsZS1wYXRjaCc7XG5cbmV4cG9ydCBjb25zdCBERUZBVUxUX09QVElPTlMgPSB7XG4gIC8vIE51bWJlciBvZiBsaW5lcyBhZnRlciB3aGljaCB3ZSBjb25zaWRlciB0aGUgZGlmZiBcImxhcmdlXCJcbiAgbGFyZ2VEaWZmVGhyZXNob2xkOiA4MDAsXG5cbiAgLy8gTWFwIG9mIGZpbGUgcGF0aCAocmVsYXRpdmUgdG8gcmVwb3NpdG9yeSByb290KSB0byBQYXRjaCByZW5kZXIgc3RhdHVzIChFWFBBTkRFRCwgQ09MTEFQU0VELCBUT09fTEFSR0UpXG4gIHJlbmRlclN0YXR1c092ZXJyaWRlczoge30sXG5cbiAgLy8gRXhpc3RpbmcgcGF0Y2ggYnVmZmVyIHRvIHJlbmRlciBvbnRvXG4gIHBhdGNoQnVmZmVyOiBudWxsLFxufTtcblxuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkRmlsZVBhdGNoKGRpZmZzLCBvcHRpb25zKSB7XG4gIGNvbnN0IG9wdHMgPSB7Li4uREVGQVVMVF9PUFRJT05TLCAuLi5vcHRpb25zfTtcbiAgY29uc3QgcGF0Y2hCdWZmZXIgPSBuZXcgUGF0Y2hCdWZmZXIoKTtcblxuICBsZXQgZmlsZVBhdGNoO1xuICBpZiAoZGlmZnMubGVuZ3RoID09PSAwKSB7XG4gICAgZmlsZVBhdGNoID0gZW1wdHlEaWZmRmlsZVBhdGNoKCk7XG4gIH0gZWxzZSBpZiAoZGlmZnMubGVuZ3RoID09PSAxKSB7XG4gICAgZmlsZVBhdGNoID0gc2luZ2xlRGlmZkZpbGVQYXRjaChkaWZmc1swXSwgcGF0Y2hCdWZmZXIsIG9wdHMpO1xuICB9IGVsc2UgaWYgKGRpZmZzLmxlbmd0aCA9PT0gMikge1xuICAgIGZpbGVQYXRjaCA9IGR1YWxEaWZmRmlsZVBhdGNoKGRpZmZzWzBdLCBkaWZmc1sxXSwgcGF0Y2hCdWZmZXIsIG9wdHMpO1xuICB9IGVsc2Uge1xuICAgIHRocm93IG5ldyBFcnJvcihgVW5leHBlY3RlZCBudW1iZXIgb2YgZGlmZnM6ICR7ZGlmZnMubGVuZ3RofWApO1xuICB9XG5cbiAgLy8gRGVsZXRlIHRoZSB0cmFpbGluZyBuZXdsaW5lLlxuICBwYXRjaEJ1ZmZlci5kZWxldGVMYXN0TmV3bGluZSgpO1xuXG4gIHJldHVybiBuZXcgTXVsdGlGaWxlUGF0Y2goe3BhdGNoQnVmZmVyLCBmaWxlUGF0Y2hlczogW2ZpbGVQYXRjaF19KTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkTXVsdGlGaWxlUGF0Y2goZGlmZnMsIG9wdGlvbnMpIHtcbiAgY29uc3Qgb3B0cyA9IHsuLi5ERUZBVUxUX09QVElPTlMsIC4uLm9wdGlvbnN9O1xuXG4gIGNvbnN0IHBhdGNoQnVmZmVyID0gbmV3IFBhdGNoQnVmZmVyKCk7XG5cbiAgY29uc3QgYnlQYXRoID0gbmV3IE1hcCgpO1xuICBjb25zdCBhY3Rpb25zID0gW107XG5cbiAgbGV0IGluZGV4ID0gMDtcbiAgZm9yIChjb25zdCBkaWZmIG9mIGRpZmZzKSB7XG4gICAgY29uc3QgdGhlUGF0aCA9IGRpZmYub2xkUGF0aCB8fCBkaWZmLm5ld1BhdGg7XG5cbiAgICBpZiAoZGlmZi5zdGF0dXMgPT09ICdhZGRlZCcgfHwgZGlmZi5zdGF0dXMgPT09ICdkZWxldGVkJykge1xuICAgICAgLy8gUG90ZW50aWFsIHBhaXJlZCBkaWZmLiBFaXRoZXIgYSBzeW1saW5rIGRlbGV0aW9uICsgY29udGVudCBhZGRpdGlvbiBvciBhIHN5bWxpbmsgYWRkaXRpb24gK1xuICAgICAgLy8gY29udGVudCBkZWxldGlvbi5cbiAgICAgIGNvbnN0IG90aGVySGFsZiA9IGJ5UGF0aC5nZXQodGhlUGF0aCk7XG4gICAgICBpZiAob3RoZXJIYWxmKSB7XG4gICAgICAgIC8vIFRoZSBzZWNvbmQgaGFsZi4gQ29tcGxldGUgdGhlIHBhaXJlZCBkaWZmLCBvciBmYWlsIGlmIHRoZXkgaGF2ZSB1bmV4cGVjdGVkIHN0YXR1c2VzIG9yIG1vZGVzLlxuICAgICAgICBjb25zdCBbb3RoZXJEaWZmLCBvdGhlckluZGV4XSA9IG90aGVySGFsZjtcbiAgICAgICAgYWN0aW9uc1tvdGhlckluZGV4XSA9IChmdW5jdGlvbihfZGlmZiwgX290aGVyRGlmZikge1xuICAgICAgICAgIHJldHVybiAoKSA9PiBkdWFsRGlmZkZpbGVQYXRjaChfZGlmZiwgX290aGVyRGlmZiwgcGF0Y2hCdWZmZXIsIG9wdHMpO1xuICAgICAgICB9KShkaWZmLCBvdGhlckRpZmYpO1xuICAgICAgICBieVBhdGguZGVsZXRlKHRoZVBhdGgpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gVGhlIGZpcnN0IGhhbGYgd2UndmUgc2Vlbi5cbiAgICAgICAgYnlQYXRoLnNldCh0aGVQYXRoLCBbZGlmZiwgaW5kZXhdKTtcbiAgICAgICAgaW5kZXgrKztcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgYWN0aW9uc1tpbmRleF0gPSAoZnVuY3Rpb24oX2RpZmYpIHtcbiAgICAgICAgcmV0dXJuICgpID0+IHNpbmdsZURpZmZGaWxlUGF0Y2goX2RpZmYsIHBhdGNoQnVmZmVyLCBvcHRzKTtcbiAgICAgIH0pKGRpZmYpO1xuICAgICAgaW5kZXgrKztcbiAgICB9XG4gIH1cblxuICAvLyBQb3B1bGF0ZSB1bnBhaXJlZCBkaWZmcyB0aGF0IGxvb2tlZCBsaWtlIHRoZXkgY291bGQgYmUgcGFydCBvZiBhIHBhaXIsIGJ1dCB3ZXJlbid0LlxuICBmb3IgKGNvbnN0IFt1bnBhaXJlZERpZmYsIG9yaWdpbmFsSW5kZXhdIG9mIGJ5UGF0aC52YWx1ZXMoKSkge1xuICAgIGFjdGlvbnNbb3JpZ2luYWxJbmRleF0gPSAoZnVuY3Rpb24oX3VucGFpcmVkRGlmZikge1xuICAgICAgcmV0dXJuICgpID0+IHNpbmdsZURpZmZGaWxlUGF0Y2goX3VucGFpcmVkRGlmZiwgcGF0Y2hCdWZmZXIsIG9wdHMpO1xuICAgIH0pKHVucGFpcmVkRGlmZik7XG4gIH1cblxuICBjb25zdCBmaWxlUGF0Y2hlcyA9IGFjdGlvbnMubWFwKGFjdGlvbiA9PiBhY3Rpb24oKSk7XG5cbiAgLy8gRGVsZXRlIHRoZSBmaW5hbCB0cmFpbGluZyBuZXdsaW5lIGZyb20gdGhlIGxhc3Qgbm9uLWVtcHR5IHBhdGNoLlxuICBwYXRjaEJ1ZmZlci5kZWxldGVMYXN0TmV3bGluZSgpO1xuXG4gIHJldHVybiBuZXcgTXVsdGlGaWxlUGF0Y2goe3BhdGNoQnVmZmVyLCBmaWxlUGF0Y2hlc30pO1xufVxuXG5mdW5jdGlvbiBlbXB0eURpZmZGaWxlUGF0Y2goKSB7XG4gIHJldHVybiBGaWxlUGF0Y2guY3JlYXRlTnVsbCgpO1xufVxuXG5mdW5jdGlvbiBzaW5nbGVEaWZmRmlsZVBhdGNoKGRpZmYsIHBhdGNoQnVmZmVyLCBvcHRzKSB7XG4gIGNvbnN0IHdhc1N5bWxpbmsgPSBkaWZmLm9sZE1vZGUgPT09IEZpbGUubW9kZXMuU1lNTElOSztcbiAgY29uc3QgaXNTeW1saW5rID0gZGlmZi5uZXdNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTks7XG5cbiAgbGV0IG9sZFN5bWxpbmsgPSBudWxsO1xuICBsZXQgbmV3U3ltbGluayA9IG51bGw7XG4gIGlmICh3YXNTeW1saW5rICYmICFpc1N5bWxpbmspIHtcbiAgICBvbGRTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1swXS5zbGljZSgxKTtcbiAgfSBlbHNlIGlmICghd2FzU3ltbGluayAmJiBpc1N5bWxpbmspIHtcbiAgICBuZXdTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1swXS5zbGljZSgxKTtcbiAgfSBlbHNlIGlmICh3YXNTeW1saW5rICYmIGlzU3ltbGluaykge1xuICAgIG9sZFN5bWxpbmsgPSBkaWZmLmh1bmtzWzBdLmxpbmVzWzBdLnNsaWNlKDEpO1xuICAgIG5ld1N5bWxpbmsgPSBkaWZmLmh1bmtzWzBdLmxpbmVzWzJdLnNsaWNlKDEpO1xuICB9XG5cbiAgY29uc3Qgb2xkRmlsZSA9IGRpZmYub2xkUGF0aCAhPT0gbnVsbCB8fCBkaWZmLm9sZE1vZGUgIT09IG51bGxcbiAgICA/IG5ldyBGaWxlKHtwYXRoOiBkaWZmLm9sZFBhdGgsIG1vZGU6IGRpZmYub2xkTW9kZSwgc3ltbGluazogb2xkU3ltbGlua30pXG4gICAgOiBudWxsRmlsZTtcbiAgY29uc3QgbmV3RmlsZSA9IGRpZmYubmV3UGF0aCAhPT0gbnVsbCB8fCBkaWZmLm5ld01vZGUgIT09IG51bGxcbiAgICA/IG5ldyBGaWxlKHtwYXRoOiBkaWZmLm5ld1BhdGgsIG1vZGU6IGRpZmYubmV3TW9kZSwgc3ltbGluazogbmV3U3ltbGlua30pXG4gICAgOiBudWxsRmlsZTtcblxuICBjb25zdCByZW5kZXJTdGF0dXNPdmVycmlkZSA9XG4gICAgKG9sZEZpbGUuaXNQcmVzZW50KCkgJiYgb3B0cy5yZW5kZXJTdGF0dXNPdmVycmlkZXNbb2xkRmlsZS5nZXRQYXRoKCldKSB8fFxuICAgIChuZXdGaWxlLmlzUHJlc2VudCgpICYmIG9wdHMucmVuZGVyU3RhdHVzT3ZlcnJpZGVzW25ld0ZpbGUuZ2V0UGF0aCgpXSkgfHxcbiAgICB1bmRlZmluZWQ7XG5cbiAgY29uc3QgcmVuZGVyU3RhdHVzID0gcmVuZGVyU3RhdHVzT3ZlcnJpZGUgfHxcbiAgICAoaXNEaWZmTGFyZ2UoW2RpZmZdLCBvcHRzKSAmJiBUT09fTEFSR0UpIHx8XG4gICAgRVhQQU5ERUQ7XG5cbiAgaWYgKCFyZW5kZXJTdGF0dXMuaXNWaXNpYmxlKCkpIHtcbiAgICBjb25zdCBwYXRjaE1hcmtlciA9IHBhdGNoQnVmZmVyLm1hcmtQb3NpdGlvbihcbiAgICAgIFBhdGNoLmxheWVyTmFtZSxcbiAgICAgIHBhdGNoQnVmZmVyLmdldEJ1ZmZlcigpLmdldEVuZFBvc2l0aW9uKCksXG4gICAgICB7aW52YWxpZGF0ZTogJ25ldmVyJywgZXhjbHVzaXZlOiBmYWxzZX0sXG4gICAgKTtcblxuICAgIHJldHVybiBGaWxlUGF0Y2guY3JlYXRlSGlkZGVuRmlsZVBhdGNoKFxuICAgICAgb2xkRmlsZSwgbmV3RmlsZSwgcGF0Y2hNYXJrZXIsIHJlbmRlclN0YXR1cyxcbiAgICAgICgpID0+IHtcbiAgICAgICAgY29uc3Qgc3ViUGF0Y2hCdWZmZXIgPSBuZXcgUGF0Y2hCdWZmZXIoKTtcbiAgICAgICAgY29uc3QgW2h1bmtzLCBuZXh0UGF0Y2hNYXJrZXJdID0gYnVpbGRIdW5rcyhkaWZmLCBzdWJQYXRjaEJ1ZmZlcik7XG4gICAgICAgIGNvbnN0IG5leHRQYXRjaCA9IG5ldyBQYXRjaCh7c3RhdHVzOiBkaWZmLnN0YXR1cywgaHVua3MsIG1hcmtlcjogbmV4dFBhdGNoTWFya2VyfSk7XG5cbiAgICAgICAgc3ViUGF0Y2hCdWZmZXIuZGVsZXRlTGFzdE5ld2xpbmUoKTtcbiAgICAgICAgcmV0dXJuIHtwYXRjaDogbmV4dFBhdGNoLCBwYXRjaEJ1ZmZlcjogc3ViUGF0Y2hCdWZmZXJ9O1xuICAgICAgfSxcbiAgICApO1xuICB9IGVsc2Uge1xuICAgIGNvbnN0IFtodW5rcywgcGF0Y2hNYXJrZXJdID0gYnVpbGRIdW5rcyhkaWZmLCBwYXRjaEJ1ZmZlcik7XG4gICAgY29uc3QgcGF0Y2ggPSBuZXcgUGF0Y2goe3N0YXR1czogZGlmZi5zdGF0dXMsIGh1bmtzLCBtYXJrZXI6IHBhdGNoTWFya2VyfSk7XG5cbiAgICByZXR1cm4gbmV3IEZpbGVQYXRjaChvbGRGaWxlLCBuZXdGaWxlLCBwYXRjaCk7XG4gIH1cbn1cblxuZnVuY3Rpb24gZHVhbERpZmZGaWxlUGF0Y2goZGlmZjEsIGRpZmYyLCBwYXRjaEJ1ZmZlciwgb3B0cykge1xuICBsZXQgbW9kZUNoYW5nZURpZmYsIGNvbnRlbnRDaGFuZ2VEaWZmO1xuICBpZiAoZGlmZjEub2xkTW9kZSA9PT0gRmlsZS5tb2Rlcy5TWU1MSU5LIHx8IGRpZmYxLm5ld01vZGUgPT09IEZpbGUubW9kZXMuU1lNTElOSykge1xuICAgIG1vZGVDaGFuZ2VEaWZmID0gZGlmZjE7XG4gICAgY29udGVudENoYW5nZURpZmYgPSBkaWZmMjtcbiAgfSBlbHNlIHtcbiAgICBtb2RlQ2hhbmdlRGlmZiA9IGRpZmYyO1xuICAgIGNvbnRlbnRDaGFuZ2VEaWZmID0gZGlmZjE7XG4gIH1cblxuICBjb25zdCBmaWxlUGF0aCA9IGNvbnRlbnRDaGFuZ2VEaWZmLm9sZFBhdGggfHwgY29udGVudENoYW5nZURpZmYubmV3UGF0aDtcbiAgY29uc3Qgc3ltbGluayA9IG1vZGVDaGFuZ2VEaWZmLmh1bmtzWzBdLmxpbmVzWzBdLnNsaWNlKDEpO1xuXG4gIGxldCBzdGF0dXM7XG4gIGxldCBvbGRNb2RlLCBuZXdNb2RlO1xuICBsZXQgb2xkU3ltbGluayA9IG51bGw7XG4gIGxldCBuZXdTeW1saW5rID0gbnVsbDtcbiAgaWYgKG1vZGVDaGFuZ2VEaWZmLnN0YXR1cyA9PT0gJ2FkZGVkJykge1xuICAgIC8vIGNvbnRlbnRzIHdlcmUgZGVsZXRlZCBhbmQgcmVwbGFjZWQgd2l0aCBzeW1saW5rXG4gICAgc3RhdHVzID0gJ2RlbGV0ZWQnO1xuICAgIG9sZE1vZGUgPSBjb250ZW50Q2hhbmdlRGlmZi5vbGRNb2RlO1xuICAgIG5ld01vZGUgPSBtb2RlQ2hhbmdlRGlmZi5uZXdNb2RlO1xuICAgIG5ld1N5bWxpbmsgPSBzeW1saW5rO1xuICB9IGVsc2UgaWYgKG1vZGVDaGFuZ2VEaWZmLnN0YXR1cyA9PT0gJ2RlbGV0ZWQnKSB7XG4gICAgLy8gY29udGVudHMgd2VyZSBhZGRlZCBhZnRlciBzeW1saW5rIHdhcyBkZWxldGVkXG4gICAgc3RhdHVzID0gJ2FkZGVkJztcbiAgICBvbGRNb2RlID0gbW9kZUNoYW5nZURpZmYub2xkTW9kZTtcbiAgICBvbGRTeW1saW5rID0gc3ltbGluaztcbiAgICBuZXdNb2RlID0gY29udGVudENoYW5nZURpZmYubmV3TW9kZTtcbiAgfSBlbHNlIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgbW9kZSBjaGFuZ2UgZGlmZiBzdGF0dXM6ICR7bW9kZUNoYW5nZURpZmYuc3RhdHVzfWApO1xuICB9XG5cbiAgY29uc3Qgb2xkRmlsZSA9IG5ldyBGaWxlKHtwYXRoOiBmaWxlUGF0aCwgbW9kZTogb2xkTW9kZSwgc3ltbGluazogb2xkU3ltbGlua30pO1xuICBjb25zdCBuZXdGaWxlID0gbmV3IEZpbGUoe3BhdGg6IGZpbGVQYXRoLCBtb2RlOiBuZXdNb2RlLCBzeW1saW5rOiBuZXdTeW1saW5rfSk7XG5cbiAgY29uc3QgcmVuZGVyU3RhdHVzID0gb3B0cy5yZW5kZXJTdGF0dXNPdmVycmlkZXNbZmlsZVBhdGhdIHx8XG4gICAgKGlzRGlmZkxhcmdlKFtjb250ZW50Q2hhbmdlRGlmZl0sIG9wdHMpICYmIFRPT19MQVJHRSkgfHxcbiAgICBFWFBBTkRFRDtcblxuICBpZiAoIXJlbmRlclN0YXR1cy5pc1Zpc2libGUoKSkge1xuICAgIGNvbnN0IHBhdGNoTWFya2VyID0gcGF0Y2hCdWZmZXIubWFya1Bvc2l0aW9uKFxuICAgICAgUGF0Y2gubGF5ZXJOYW1lLFxuICAgICAgcGF0Y2hCdWZmZXIuZ2V0QnVmZmVyKCkuZ2V0RW5kUG9zaXRpb24oKSxcbiAgICAgIHtpbnZhbGlkYXRlOiAnbmV2ZXInLCBleGNsdXNpdmU6IGZhbHNlfSxcbiAgICApO1xuXG4gICAgcmV0dXJuIEZpbGVQYXRjaC5jcmVhdGVIaWRkZW5GaWxlUGF0Y2goXG4gICAgICBvbGRGaWxlLCBuZXdGaWxlLCBwYXRjaE1hcmtlciwgcmVuZGVyU3RhdHVzLFxuICAgICAgKCkgPT4ge1xuICAgICAgICBjb25zdCBzdWJQYXRjaEJ1ZmZlciA9IG5ldyBQYXRjaEJ1ZmZlcigpO1xuICAgICAgICBjb25zdCBbaHVua3MsIG5leHRQYXRjaE1hcmtlcl0gPSBidWlsZEh1bmtzKGNvbnRlbnRDaGFuZ2VEaWZmLCBzdWJQYXRjaEJ1ZmZlcik7XG4gICAgICAgIGNvbnN0IG5leHRQYXRjaCA9IG5ldyBQYXRjaCh7c3RhdHVzLCBodW5rcywgbWFya2VyOiBuZXh0UGF0Y2hNYXJrZXJ9KTtcblxuICAgICAgICBzdWJQYXRjaEJ1ZmZlci5kZWxldGVMYXN0TmV3bGluZSgpO1xuICAgICAgICByZXR1cm4ge3BhdGNoOiBuZXh0UGF0Y2gsIHBhdGNoQnVmZmVyOiBzdWJQYXRjaEJ1ZmZlcn07XG4gICAgICB9LFxuICAgICk7XG4gIH0gZWxzZSB7XG4gICAgY29uc3QgW2h1bmtzLCBwYXRjaE1hcmtlcl0gPSBidWlsZEh1bmtzKGNvbnRlbnRDaGFuZ2VEaWZmLCBwYXRjaEJ1ZmZlcik7XG4gICAgY29uc3QgcGF0Y2ggPSBuZXcgUGF0Y2goe3N0YXR1cywgaHVua3MsIG1hcmtlcjogcGF0Y2hNYXJrZXJ9KTtcblxuICAgIHJldHVybiBuZXcgRmlsZVBhdGNoKG9sZEZpbGUsIG5ld0ZpbGUsIHBhdGNoKTtcbiAgfVxufVxuXG5jb25zdCBDSEFOR0VLSU5EID0ge1xuICAnKyc6IEFkZGl0aW9uLFxuICAnLSc6IERlbGV0aW9uLFxuICAnICc6IFVuY2hhbmdlZCxcbiAgJ1xcXFwnOiBOb05ld2xpbmUsXG59O1xuXG5mdW5jdGlvbiBidWlsZEh1bmtzKGRpZmYsIHBhdGNoQnVmZmVyKSB7XG4gIGNvbnN0IGluc2VydGVyID0gcGF0Y2hCdWZmZXIuY3JlYXRlSW5zZXJ0ZXJBdEVuZCgpXG4gICAgLmtlZXBCZWZvcmUocGF0Y2hCdWZmZXIuZmluZEFsbE1hcmtlcnMoe2VuZFBvc2l0aW9uOiBwYXRjaEJ1ZmZlci5nZXRJbnNlcnRpb25Qb2ludCgpfSkpO1xuXG4gIGxldCBwYXRjaE1hcmtlciA9IG51bGw7XG4gIGxldCBmaXJzdEh1bmsgPSB0cnVlO1xuICBjb25zdCBodW5rcyA9IFtdO1xuXG4gIGluc2VydGVyLm1hcmtXaGlsZShQYXRjaC5sYXllck5hbWUsICgpID0+IHtcbiAgICBmb3IgKGNvbnN0IHJhd0h1bmsgb2YgZGlmZi5odW5rcykge1xuICAgICAgbGV0IGZpcnN0UmVnaW9uID0gdHJ1ZTtcbiAgICAgIGNvbnN0IHJlZ2lvbnMgPSBbXTtcblxuICAgICAgLy8gU2VwYXJhdGUgaHVua3Mgd2l0aCBhbiB1bm1hcmtlZCBuZXdsaW5lXG4gICAgICBpZiAoZmlyc3RIdW5rKSB7XG4gICAgICAgIGZpcnN0SHVuayA9IGZhbHNlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgaW5zZXJ0ZXIuaW5zZXJ0KCdcXG4nKTtcbiAgICAgIH1cblxuICAgICAgaW5zZXJ0ZXIubWFya1doaWxlKEh1bmsubGF5ZXJOYW1lLCAoKSA9PiB7XG4gICAgICAgIGxldCBmaXJzdFJlZ2lvbkxpbmUgPSB0cnVlO1xuICAgICAgICBsZXQgY3VycmVudFJlZ2lvblRleHQgPSAnJztcbiAgICAgICAgbGV0IEN1cnJlbnRSZWdpb25LaW5kID0gbnVsbDtcblxuICAgICAgICBmdW5jdGlvbiBmaW5pc2hSZWdpb24oKSB7XG4gICAgICAgICAgaWYgKEN1cnJlbnRSZWdpb25LaW5kID09PSBudWxsKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gU2VwYXJhdGUgcmVnaW9ucyB3aXRoIGFuIHVubWFya2VkIG5ld2xpbmVcbiAgICAgICAgICBpZiAoZmlyc3RSZWdpb24pIHtcbiAgICAgICAgICAgIGZpcnN0UmVnaW9uID0gZmFsc2U7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGluc2VydGVyLmluc2VydCgnXFxuJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaW5zZXJ0ZXIuaW5zZXJ0TWFya2VkKGN1cnJlbnRSZWdpb25UZXh0LCBDdXJyZW50UmVnaW9uS2luZC5sYXllck5hbWUsIHtcbiAgICAgICAgICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgICAgICAgICBleGNsdXNpdmU6IGZhbHNlLFxuICAgICAgICAgICAgY2FsbGJhY2s6IChmdW5jdGlvbihfcmVnaW9ucywgX0N1cnJlbnRSZWdpb25LaW5kKSB7XG4gICAgICAgICAgICAgIHJldHVybiByZWdpb25NYXJrZXIgPT4geyBfcmVnaW9ucy5wdXNoKG5ldyBfQ3VycmVudFJlZ2lvbktpbmQocmVnaW9uTWFya2VyKSk7IH07XG4gICAgICAgICAgICB9KShyZWdpb25zLCBDdXJyZW50UmVnaW9uS2luZCksXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cblxuICAgICAgICBmb3IgKGNvbnN0IHJhd0xpbmUgb2YgcmF3SHVuay5saW5lcykge1xuICAgICAgICAgIGNvbnN0IE5leHRSZWdpb25LaW5kID0gQ0hBTkdFS0lORFtyYXdMaW5lWzBdXTtcbiAgICAgICAgICBpZiAoTmV4dFJlZ2lvbktpbmQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGRpZmYgc3RhdHVzIGNoYXJhY3RlcjogXCIke3Jhd0xpbmVbMF19XCJgKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgY29uc3QgbmV4dExpbmUgPSByYXdMaW5lLnNsaWNlKDEpO1xuXG4gICAgICAgICAgbGV0IHNlcGFyYXRvciA9ICcnO1xuICAgICAgICAgIGlmIChmaXJzdFJlZ2lvbkxpbmUpIHtcbiAgICAgICAgICAgIGZpcnN0UmVnaW9uTGluZSA9IGZhbHNlO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBzZXBhcmF0b3IgPSAnXFxuJztcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBpZiAoTmV4dFJlZ2lvbktpbmQgPT09IEN1cnJlbnRSZWdpb25LaW5kKSB7XG4gICAgICAgICAgICBjdXJyZW50UmVnaW9uVGV4dCArPSBzZXBhcmF0b3IgKyBuZXh0TGluZTtcblxuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGZpbmlzaFJlZ2lvbigpO1xuXG4gICAgICAgICAgICBDdXJyZW50UmVnaW9uS2luZCA9IE5leHRSZWdpb25LaW5kO1xuICAgICAgICAgICAgY3VycmVudFJlZ2lvblRleHQgPSBuZXh0TGluZTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgZmluaXNoUmVnaW9uKCk7XG4gICAgICB9LCB7XG4gICAgICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgICAgIGV4Y2x1c2l2ZTogZmFsc2UsXG4gICAgICAgIGNhbGxiYWNrOiAoZnVuY3Rpb24oX2h1bmtzLCBfcmF3SHVuaywgX3JlZ2lvbnMpIHtcbiAgICAgICAgICByZXR1cm4gaHVua01hcmtlciA9PiB7XG4gICAgICAgICAgICBfaHVua3MucHVzaChuZXcgSHVuayh7XG4gICAgICAgICAgICAgIG9sZFN0YXJ0Um93OiBfcmF3SHVuay5vbGRTdGFydExpbmUsXG4gICAgICAgICAgICAgIG5ld1N0YXJ0Um93OiBfcmF3SHVuay5uZXdTdGFydExpbmUsXG4gICAgICAgICAgICAgIG9sZFJvd0NvdW50OiBfcmF3SHVuay5vbGRMaW5lQ291bnQsXG4gICAgICAgICAgICAgIG5ld1Jvd0NvdW50OiBfcmF3SHVuay5uZXdMaW5lQ291bnQsXG4gICAgICAgICAgICAgIHNlY3Rpb25IZWFkaW5nOiBfcmF3SHVuay5oZWFkaW5nLFxuICAgICAgICAgICAgICBtYXJrZXI6IGh1bmtNYXJrZXIsXG4gICAgICAgICAgICAgIHJlZ2lvbnM6IF9yZWdpb25zLFxuICAgICAgICAgICAgfSkpO1xuICAgICAgICAgIH07XG4gICAgICAgIH0pKGh1bmtzLCByYXdIdW5rLCByZWdpb25zKSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfSwge1xuICAgIGludmFsaWRhdGU6ICduZXZlcicsXG4gICAgZXhjbHVzaXZlOiBmYWxzZSxcbiAgICBjYWxsYmFjazogbWFya2VyID0+IHsgcGF0Y2hNYXJrZXIgPSBtYXJrZXI7IH0sXG4gIH0pO1xuXG4gIC8vIFNlcGFyYXRlIG11bHRpcGxlIG5vbi1lbXB0eSBwYXRjaGVzIG9uIHRoZSBzYW1lIGJ1ZmZlciB3aXRoIGFuIHVubWFya2VkIG5ld2xpbmUuIFRoZSBuZXdsaW5lIGFmdGVyIHRoZSBmaW5hbFxuICAvLyBub24tZW1wdHkgcGF0Y2ggKGlmIHRoZXJlIGlzIG9uZSkgc2hvdWxkIGJlIGRlbGV0ZWQgYmVmb3JlIE11bHRpRmlsZVBhdGNoIGNvbnN0cnVjdGlvbi5cbiAgaWYgKGRpZmYuaHVua3MubGVuZ3RoID4gMCkge1xuICAgIGluc2VydGVyLmluc2VydCgnXFxuJyk7XG4gIH1cblxuICBpbnNlcnRlci5hcHBseSgpO1xuXG4gIHJldHVybiBbaHVua3MsIHBhdGNoTWFya2VyXTtcbn1cblxuZnVuY3Rpb24gaXNEaWZmTGFyZ2UoZGlmZnMsIG9wdHMpIHtcbiAgY29uc3Qgc2l6ZSA9IGRpZmZzLnJlZHVjZSgoZGlmZlNpemVDb3VudGVyLCBkaWZmKSA9PiB7XG4gICAgcmV0dXJuIGRpZmZTaXplQ291bnRlciArIGRpZmYuaHVua3MucmVkdWNlKChodW5rU2l6ZUNvdW50ZXIsIGh1bmspID0+IHtcbiAgICAgIHJldHVybiBodW5rU2l6ZUNvdW50ZXIgKyBodW5rLmxpbmVzLmxlbmd0aDtcbiAgICB9LCAwKTtcbiAgfSwgMCk7XG5cbiAgcmV0dXJuIHNpemUgPiBvcHRzLmxhcmdlRGlmZlRocmVzaG9sZDtcbn1cbiJdfQ==