"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _position = require("./position");

var _source = require("./source");

var _side = _interopRequireDefault(require("./side"));

var _banner = _interopRequireDefault(require("./banner"));

var _separator = _interopRequireDefault(require("./separator"));

var _parser = require("./parser");

var _adapter = require("./parser/adapter");

var _noopVisitor = require("./parser/noop-visitor");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Regular expression that matches the beginning of a potential conflict.
const CONFLICT_START_REGEX = /^<{7} ([^\r\n]+)\r?\n/g;
/*
 * Conflict parser visitor that marks each buffer range and assembles a Conflict from the
 * pieces.
 */

class ConflictVisitor {
  /*
   * editor - [TextEditor] displaying the conflicting text.
   * layer - [DisplayMarkerLayer] to created conflict markers on.
   */
  constructor(editor, layer) {
    this.editor = editor;
    this.layer = layer;
    this.ours = null;
    this.base = null;
    this.separator = null;
    this.theirs = null;
  }
  /*
   * position - [Position] one of TOP or BOTTOM.
   * bannerRow - [Integer] of the buffer row that contains our side's banner.
   * textRowStart - [Integer] of the first buffer row that contain this side's text.
   * textRowEnd - [Integer] of the first buffer row beyond the extend of this side's text.
   */


  visitOurSide(position, bannerRow, textRowStart, textRowEnd) {
    this.ours = this.markSide(position, _source.OURS, bannerRow, textRowStart, textRowEnd);
  }
  /*
   * bannerRow - [Integer] the buffer row that contains our side's banner.
   * textRowStart - [Integer] first buffer row that contain this side's text.
   * textRowEnd - [Integer] first buffer row beyond the extend of this side's text.
   */


  visitBaseSide(bannerRow, textRowStart, textRowEnd) {
    this.base = this.markSide(_position.MIDDLE, _source.BASE, bannerRow, textRowStart, textRowEnd);
  }
  /*
   * sepRowStart - [Integer] buffer row that contains the "=======" separator.
   */


  visitSeparator(sepRowStart) {
    const marker = this.layer.markBufferRange([[sepRowStart, 0], [sepRowStart + 1, 0]], {
      invalidate: 'surround',
      exclusive: true
    });
    this.separator = new _separator.default(this.editor, marker);
  }
  /*
   * position - [Position] alignment within the conflict marker: TOP or BOTTOM.
   * bannerRow - [Integer] the buffer row that contains our side's banner.
   * textRowStart - [Integer] first buffer row that contain this side's text.
   * textRowEnd - [Integer] first buffer row beyond the extent of this side's text.
   */


  visitTheirSide(position, bannerRow, textRowStart, textRowEnd) {
    this.theirs = this.markSide(position, _source.THEIRS, bannerRow, textRowStart, textRowEnd);
  }

  markSide(position, source, bannerRow, textRowStart, textRowEnd) {
    const blockCol = position.when({
      top: () => 0,
      middle: () => 0,
      bottom: () => this.editor.lineTextForBufferRow(bannerRow).length
    });
    const blockRange = [[bannerRow, blockCol], [bannerRow, blockCol]];
    const blockMarker = this.layer.markBufferRange(blockRange, {
      invalidate: 'surround',
      exclusive: true
    });
    const description = this.sideDescription(bannerRow);
    const bannerRange = [[bannerRow, 0], [bannerRow + 1, 0]];
    const bannerMarker = this.layer.markBufferRange(bannerRange, {
      invalidate: 'surround',
      exclusive: true
    });
    const originalBannerText = this.editor.getTextInBufferRange(bannerRange);
    const banner = new _banner.default(this.editor, bannerMarker, description, originalBannerText);
    const textRange = [[textRowStart, 0], [textRowEnd, 0]];
    const sideMarker = this.layer.markBufferRange(textRange, {
      invalidate: 'surround',
      exclusive: false
    });
    const originalText = this.editor.getTextInBufferRange(textRange);
    return new _side.default(this.editor, sideMarker, blockMarker, source, position, banner, originalText);
  }
  /*
   * Parse the banner description for the current side from a banner row.
   *
   * bannerRow - [Integer] buffer row containing the <, |, or > marker
   */


  sideDescription(bannerRow) {
    return this.editor.lineTextForBufferRow(bannerRow).match(/^[<|>]{7} (.*)$/)[1];
  }

  conflict() {
    return new Conflict(this.ours, this.separator, this.base, this.theirs);
  }

}

class Conflict {
  constructor(ours, separator, base, theirs) {
    this.separator = separator;
    this.bySource = {};
    this.byPosition = {};
    [ours, base, theirs].forEach(side => {
      if (!side) {
        return;
      }

      this.bySource[side.getSource().getName()] = side;
      this.byPosition[side.getPosition().getName()] = side;
    });
    this.resolution = null;
  }

  getKey() {
    return this.getSide(_source.OURS).getMarker().id;
  }

  isResolved() {
    return this.resolution !== null;
  }

  resolveAs(source) {
    this.resolution = this.getSide(source);
  }

  getSides() {
    return ['ours', 'base', 'theirs'].map(sourceName => this.bySource[sourceName]).filter(side => side);
  }

  getChosenSide() {
    return this.resolution;
  }

  getUnchosenSides() {
    return this.getSides().filter(side => side !== this.resolution);
  }

  getSide(source) {
    return this.bySource[source.getName()];
  }
  /*
   * Return a `Side` containing a buffer point, or `undefined` if none do.
   */


  getSideContaining(point) {
    return this.getSides().find(side => side.includesPoint(point));
  }
  /*
   * Return a `Range` that encompasses the entire Conflict region.
   */


  getRange() {
    const topRange = this.byPosition.top.getRange();
    const bottomRange = this.byPosition.bottom.getRange();
    return topRange.union(bottomRange);
  }
  /*
   * Determine whether or not a buffer position is contained within this conflict.
   */


  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }
  /*
   * Return the `DisplayMarker` that immediately follows the `Side` in a given `Position`. Return `null` if no such
   * marker exists.
   */


  markerAfter(position) {
    return position.when({
      top: () => this.byPosition.middle ? this.byPosition.middle.getBannerMarker() : this.getSeparator().getMarker(),
      middle: () => this.getSeparator().getMarker(),
      bottom: () => this.byPosition.bottom.getBannerMarker()
    });
  }

  getSeparator() {
    return this.separator;
  }
  /*
   * Parse any conflict markers in a TextEditor's buffer and return a Conflict that contains markers corresponding to
   * each.
   *
   * editor [TextEditor] The editor to search.
   * layer [DisplayMarkerLayer] Marker layer to create markers on.
   * return [Array<Conflict>] A (possibly empty) collection of parsed Conflicts.
   */


  static allFromEditor(editor, layer, isRebase) {
    const conflicts = [];
    let lastRow = -1;
    editor.getBuffer().scan(CONFLICT_START_REGEX, m => {
      const conflictStartRow = m.range.start.row;

      if (conflictStartRow < lastRow) {
        // Match within an already-parsed conflict.
        return;
      }

      const adapter = new _adapter.EditorAdapter(editor, conflictStartRow);
      const visitor = new ConflictVisitor(editor, layer);
      const parser = new _parser.ConflictParser(adapter, visitor, isRebase);

      if (parser.parse().wasSuccessful()) {
        conflicts.push(visitor.conflict());
      }

      lastRow = adapter.getCurrentRow();
    });
    return conflicts;
  }
  /*
   * Return the number of conflict markers present in a streamed file.
   */


  static countFromStream(stream, isRebase) {
    return new Promise((resolve, reject) => {
      let count = 0;
      let lastResult = null;
      let lastPartialMarker = '';
      stream.on('data', chunk => {
        const adapter = new _adapter.ChunkAdapter(lastPartialMarker + chunk);

        if (!lastResult) {
          if (!adapter.advanceTo(CONFLICT_START_REGEX)) {
            lastPartialMarker = adapter.getLastPartialMarker();
            return;
          }
        }

        do {
          const parser = new _parser.ConflictParser(adapter, new _noopVisitor.NoopVisitor(), isRebase);
          const result = lastResult ? parser.continueFrom(lastResult) : parser.parse();

          if (result.wasSuccessful()) {
            count++;
          } else {
            lastResult = result;
          }
        } while (adapter.advanceTo(CONFLICT_START_REGEX));

        lastPartialMarker = adapter.getLastPartialMarker();
      });
      stream.on('error', reject);
      stream.on('end', () => resolve(count));
    });
  }

}

exports.default = Conflict;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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