/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *  strict-local
 * @format
 */
'use strict';

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread"));

var _objectWithoutPropertiesLoose2 = _interopRequireDefault(require("@babel/runtime/helpers/objectWithoutPropertiesLoose"));

var _inheritsLoose2 = _interopRequireDefault(require("@babel/runtime/helpers/inheritsLoose"));

var _assertThisInitialized2 = _interopRequireDefault(require("@babel/runtime/helpers/assertThisInitialized"));

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var React = require("react");

var areEqual = require("fbjs/lib/areEqual");

var buildReactRelayContainer = require("./buildReactRelayContainer");

var _require = require("./ReactRelayContainerProfiler"),
    profileContainer = _require.profileContainer;

var _require2 = require("./ReactRelayContainerUtils"),
    getContainerName = _require2.getContainerName;

var _require3 = require("./RelayContext"),
    assertRelayContext = _require3.assertRelayContext;

var _require4 = require("relay-runtime"),
    RelayProfiler = _require4.RelayProfiler,
    isScalarAndEqual = _require4.isScalarAndEqual;

/**
 * Composes a React component class, returning a new class that intercepts
 * props, resolving them with the provided fragments and subscribing for
 * updates.
 */
function createContainerWithFragments(Component, fragments) {
  var containerName = getContainerName(Component);

  var Container =
  /*#__PURE__*/
  function (_React$Component) {
    (0, _inheritsLoose2["default"])(Container, _React$Component);

    function Container(props) {
      var _this;

      _this = _React$Component.call(this, props) || this;
      (0, _defineProperty2["default"])((0, _assertThisInitialized2["default"])((0, _assertThisInitialized2["default"])(_this)), "_handleFragmentDataUpdate", function () {
        var profiler = RelayProfiler.profile('ReactRelayFragmentContainer.handleFragmentDataUpdate');
        var resolverFromThisUpdate = _this.state.resolver;

        _this.setState(function (updatedState) {
          // If this event belongs to the current data source, update.
          // Otherwise we should ignore it.
          if (resolverFromThisUpdate === updatedState.resolver) {
            return {
              data: updatedState.resolver.resolve(),
              relayProp: getRelayProp(updatedState.relayProp.environment)
            };
          }

          return null;
        }, profiler.stop);
      });
      var relayContext = assertRelayContext(props.__relayContext);
      var createFragmentSpecResolver = relayContext.environment.unstable_internal.createFragmentSpecResolver; // Do not provide a subscription/callback here.
      // It is possible for this render to be interrupted or aborted,
      // In which case the subscription would cause a leak.
      // We will add the subscription in componentDidMount().

      var resolver = createFragmentSpecResolver(relayContext, containerName, fragments, props);
      _this.state = {
        data: resolver.resolve(),
        prevProps: props,
        prevPropsContext: relayContext,
        relayProp: getRelayProp(relayContext.environment),
        resolver: resolver
      };
      return _this;
    }
    /**
     * When new props are received, read data for the new props and subscribe
     * for updates. Props may be the same in which case previous data and
     * subscriptions can be reused.
     */


    Container.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
      // Any props change could impact the query, so we mirror props in state.
      // This is an unusual pattern, but necessary for this container usecase.
      var prevProps = prevState.prevProps;
      var relayContext = assertRelayContext(nextProps.__relayContext);
      var _relayContext$environ = relayContext.environment.unstable_internal,
          createFragmentSpecResolver = _relayContext$environ.createFragmentSpecResolver,
          getDataIDsFromObject = _relayContext$environ.getDataIDsFromObject;
      var prevIDs = getDataIDsFromObject(fragments, prevProps);
      var nextIDs = getDataIDsFromObject(fragments, nextProps);
      var resolver = prevState.resolver; // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.

      if (prevState.prevPropsContext.environment !== relayContext.environment || prevState.prevPropsContext.variables !== relayContext.variables || !areEqual(prevIDs, nextIDs)) {
        // Do not provide a subscription/callback here.
        // It is possible for this render to be interrupted or aborted,
        // In which case the subscription would cause a leak.
        // We will add the subscription in componentDidUpdate().
        resolver = createFragmentSpecResolver(relayContext, containerName, fragments, nextProps);
        return {
          data: resolver.resolve(),
          prevPropsContext: relayContext,
          prevProps: nextProps,
          relayProp: getRelayProp(relayContext.environment),
          resolver: resolver
        };
      } else {
        resolver.setProps(nextProps);
        var data = resolver.resolve();

        if (data !== prevState.data) {
          return {
            data: data,
            prevProps: nextProps,
            prevPropsContext: relayContext,
            relayProp: getRelayProp(relayContext.environment)
          };
        }
      }

      return null;
    };

    var _proto = Container.prototype;

    _proto.componentDidMount = function componentDidMount() {
      this._subscribeToNewResolver();

      this._rerenderIfStoreHasChanged();
    };

    _proto.componentDidUpdate = function componentDidUpdate(prevProps, prevState) {
      if (this.state.resolver !== prevState.resolver) {
        prevState.resolver.dispose();

        this._subscribeToNewResolver();
      }

      this._rerenderIfStoreHasChanged();
    };

    _proto.componentWillUnmount = function componentWillUnmount() {
      this.state.resolver.dispose();
    };

    _proto.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState) {
      // Short-circuit if any Relay-related data has changed
      if (nextState.data !== this.state.data) {
        return true;
      } // Otherwise, for convenience short-circuit if all non-Relay props
      // are scalar and equal


      var keys = Object.keys(nextProps);

      for (var ii = 0; ii < keys.length; ii++) {
        var _key = keys[ii];

        if (_key === '__relayContext') {
          if (nextState.prevPropsContext.environment !== this.state.prevPropsContext.environment || nextState.prevPropsContext.variables !== this.state.prevPropsContext.variables) {
            return true;
          }
        } else {
          if (!fragments.hasOwnProperty(_key) && !isScalarAndEqual(nextProps[_key], this.props[_key])) {
            return true;
          }
        }
      }

      return false;
    };
    /**
     * Render new data for the existing props/context.
     */


    _proto._rerenderIfStoreHasChanged = function _rerenderIfStoreHasChanged() {
      var _this$state = this.state,
          data = _this$state.data,
          resolver = _this$state.resolver; // External values could change between render and commit.
      // Check for this case, even though it requires an extra store read.

      var maybeNewData = resolver.resolve();

      if (data !== maybeNewData) {
        this.setState({
          data: maybeNewData
        });
      }
    };

    _proto._subscribeToNewResolver = function _subscribeToNewResolver() {
      var resolver = this.state.resolver; // Event listeners are only safe to add during the commit phase,
      // So they won't leak if render is interrupted or errors.

      resolver.setCallback(this._handleFragmentDataUpdate);
    };

    _proto.render = function render() {
      var _this$props = this.props,
          componentRef = _this$props.componentRef,
          _ = _this$props.__relayContext,
          props = (0, _objectWithoutPropertiesLoose2["default"])(_this$props, ["componentRef", "__relayContext"]);
      return React.createElement(Component, (0, _objectSpread2["default"])({}, props, this.state.data, {
        ref: componentRef,
        relay: this.state.relayProp
      }));
    };

    return Container;
  }(React.Component);

  (0, _defineProperty2["default"])(Container, "displayName", containerName);
  profileContainer(Container, 'ReactRelayFragmentContainer');
  return Container;
}

function getRelayProp(environment) {
  return {
    environment: environment
  };
}
/**
 * Wrap the basic `createContainer()` function with logic to adapt to the
 * `context.relay.environment` in which it is rendered. Specifically, the
 * extraction of the environment-specific version of fragments in the
 * `fragmentSpec` is memoized once per environment, rather than once per
 * instance of the container constructed/rendered.
 */


function createContainer(Component, fragmentSpec) {
  return buildReactRelayContainer(Component, fragmentSpec, createContainerWithFragments);
}

module.exports = {
  createContainer: createContainer,
  createContainerWithFragments: createContainerWithFragments
};