"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// Measure elapsed durations from specific beginning points.
// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
const MAXIMUM_MARKS = 100; // Flush all non-active DurationSets to disk each time that this many marks have been accumulated.

const PERSIST_INTERVAL = 1000; // A sequence of durations measured from a fixed beginning point.

class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }

  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }

    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({
        eventName,
        duration
      });
    }

    this.markCount++;

    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }

  finish() {
    this.mark('finish');

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }

  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }

  getCount() {
    return this.marks.length;
  }

}

let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();

function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });

  if (!anyActive) {
    return false;
  }

  const mask = new RegExp(atom.config.get('github.performanceMask'));

  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }

  return true;
}

const yardstick = {
  async save() {
    const destDir = atom.config.get('github.performanceToDirectory');

    if (destDir === '' || destDir === undefined || destDir === null) {
      return;
    }

    const fileName = _path.default.join(destDir, `performance-${Date.now()}.json`);

    await new Promise((resolve, reject) => {
      _fsExtra.default.ensureDir(destDir, err => err ? reject(err) : resolve());
    });
    const payload = JSON.stringify(durationSets.map(set => set.serialize()));
    await _fsExtra.default.writeFile(fileName, payload, {
      encoding: 'utf8'
    });

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
    }

    durationSets = [];
  },

  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }

    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },

  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }

      return;
    }

    if (!shouldCapture(seriesName, eventName)) {
      return;
    }

    const ds = activeSets.get(seriesName);

    if (ds === undefined) {
      return;
    }

    ds.mark(eventName);
  },

  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }

      return;
    }

    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }

    const ds = activeSets.get(seriesName);

    if (ds === undefined) {
      return;
    }

    ds.finish();
    durationSets.push(ds);
    activeSets.delete(seriesName);
    totalMarkCount += ds.getCount();

    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },

  async flush() {
    durationSets.push(...activeSets.values());
    activeSets.clear();
    await this.save();
  }

};
var _default = yardstick;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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