"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _actionPipeline = require("../action-pipeline");

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _author = _interopRequireWildcard(require("./author"));

var _branch = _interopRequireDefault(require("./branch"));

var _repositoryStates = require("./repository-states");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m; // Internal option keys used to designate the desired initial state of a Repository.

const initialStateSym = Symbol('initialState');

class Repository {
  constructor(workingDirectoryPath, gitStrategy = null, options = {}) {
    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy.default.create(workingDirectoryPath);
    this.emitter = new _eventKit.Emitter();
    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });
    this.pipelineManager = options.pipelineManager || (0, _actionPipeline.getNullActionPipelineManager)();
    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.Absent
    }, options));
  }

  static loadingGuess(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.LoadingGuess
    }, options));
  }

  static absentGuess(options) {
    return new Repository(null, null, _objectSpread({
      [initialStateSym]: _repositoryStates.AbsentGuess
    }, options));
  } // State management //////////////////////////////////////////////////////////////////////////////////////////////////


  transition(currentState, StateConstructor, ...payload) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;
    this.emitter.emit('did-change-state', {
      from: currentState,
      to: this.state
    });

    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor, ...payload) {
    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }
  /*
   * Use `callback` to request user input from all git strategies.
   */


  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  } // Pipeline


  getPipeline(actionName) {
    const actionKey = this.pipelineManager.actionKeys[actionName];
    return this.pipelineManager.getPipeline(actionKey);
  }

  executePipelineAction(actionName, fn, ...args) {
    const pipeline = this.getPipeline(actionName);
    return pipeline.run(fn, this, ...args);
  } // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////


  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  onPullError(callback) {
    return this.emitter.on('pull-error', callback);
  }

  didPullError() {
    return this.emitter.emit('pull-error');
  } // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.


  async pathHasMergeMarkers(relativePath) {
    try {
      const contents = await _fsExtra.default.readFile(_path.default.join(this.getWorkingDirectoryPath(), relativePath), {
        encoding: 'utf8'
      });
      return MERGE_MARKER_REGEX.test(contents);
    } catch (e) {
      // EISDIR implies this is a submodule
      if (e.code === 'ENOENT' || e.code === 'EISDIR') {
        return false;
      } else {
        throw e;
      }
    }
  }

  async getMergeMessage() {
    try {
      const contents = await _fsExtra.default.readFile(_path.default.join(this.getGitDirectoryPath(), 'MERGE_MSG'), {
        encoding: 'utf8'
      });
      return contents.split(/\n/).filter(line => line.length > 0 && !line.startsWith('#')).join('\n');
    } catch (e) {
      return null;
    }
  } // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////


  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  } // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.


  async getCurrentBranch() {
    const branches = await this.getBranches();
    const head = branches.getHeadBranch();

    if (head.isPresent()) {
      return head;
    }

    const description = await this.getHeadDescription();
    return _branch.default.createDetached(description);
  }

  async getUnstagedChanges() {
    const {
      unstagedFiles
    } = await this.getStatusBundle();
    return Object.keys(unstagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: unstagedFiles[filePath]
      };
    });
  }

  async getStagedChanges() {
    const {
      stagedFiles
    } = await this.getStatusBundle();
    return Object.keys(stagedFiles).sort().map(filePath => {
      return {
        filePath,
        status: stagedFiles[filePath]
      };
    });
  }

  async getMergeConflicts() {
    const {
      mergeConflictFiles
    } = await this.getStatusBundle();
    return Object.keys(mergeConflictFiles).map(filePath => {
      return {
        filePath,
        status: mergeConflictFiles[filePath]
      };
    });
  }

  async isPartiallyStaged(fileName) {
    const {
      unstagedFiles,
      stagedFiles
    } = await this.getStatusBundle();
    const u = unstagedFiles[fileName];
    const s = stagedFiles[fileName];
    return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
  }

  async getRemoteForBranch(branchName) {
    const name = await this.getConfig(`branch.${branchName}.remote`);
    return (await this.getRemotes()).withName(name);
  }

  async saveDiscardHistory() {
    if (this.isDestroyed()) {
      return;
    }

    const historySha = await this.createDiscardHistoryBlob();

    if (this.isDestroyed()) {
      return;
    }

    await this.setConfig('atomGithub.historySha', historySha);
  }

  async getCommitter(options = {}) {
    const output = await this.getConfig(['--get-regexp', '^user.*'], options);
    const committer = {
      name: null,
      email: null
    }; // todo (tt, 4/2018): do we need null byte terminated output here for Windows?

    if (output) {
      output.trim().split('\n').forEach(line => {
        if (line.includes('user.email')) {
          committer.email = line.slice(11);
        } else if (line.includes('user.name')) {
          committer.name = line.slice(10);
        }
      });
    }

    return committer.name !== null && committer.email !== null ? new _author.default(committer.email, committer.name) : _author.nullAuthor;
  }

  async hasGitHubRemote(host, owner, name) {
    const remotes = await this.getRemotes();
    return remotes.matchingGitHubRepository(owner, name).length > 0;
  }

} // The methods named here will be delegated to the current State.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".


exports.default = Repository;
const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isTooLarge', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'updateCommitMessageAfterFileSystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'stageFileModeChange', 'stageFileSymlinkChange', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'undoLastCommit', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getFilePatchForPath', 'getStagedChangesPatch', 'readFileFromIndex', 'getLastCommit', 'getCommit', 'getRecentCommits', 'isCommitPushed', 'getAuthors', 'getBranches', 'getHeadDescription', 'isMerging', 'isRebasing', 'getRemotes', 'addRemote', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots', 'getOperationStates', 'setCommitMessage', 'getCommitMessage', 'fetchCommitMessageTemplate', 'getCache'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function (...args) {
    return this.state[delegate](...args);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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