"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _helpers = require("../helpers");

const COPY = Symbol('copy');

class ListSelection {
  constructor(options = {}) {
    (0, _helpers.autobind)(this, 'isItemSelectable');

    if (options._copy !== COPY) {
      this.options = {
        isItemSelectable: options.isItemSelectable || (item => !!item)
      };
      this.items = options.items || [];
      this.selections = this.items.length > 0 ? [{
        head: 0,
        tail: 0
      }] : [];
    } else {
      this.options = {
        isItemSelectable: options.isItemSelectable
      };
      this.items = options.items;
      this.selections = options.selections;
    }
  }

  copy(options = {}) {
    return new ListSelection({
      _copy: COPY,
      isItemSelectable: options.isItemSelectable || this.options.isItemSelectable,
      items: options.items || this.items,
      selections: options.selections || this.selections
    });
  }

  isItemSelectable(item) {
    return this.options.isItemSelectable(item);
  }

  setItems(items) {
    let newSelectionIndex;

    if (this.selections.length > 0) {
      const [{
        head,
        tail
      }] = this.selections;
      newSelectionIndex = Math.min(head, tail, items.length - 1);
    } else {
      newSelectionIndex = 0;
    }

    const newSelections = items.length > 0 ? [{
      head: newSelectionIndex,
      tail: newSelectionIndex
    }] : [];
    return this.copy({
      items,
      selections: newSelections
    });
  }

  getItems() {
    return this.items;
  }

  getLastItem() {
    return this.items[this.items.length - 1];
  }

  selectFirstItem(preserveTail) {
    for (let i = 0; i < this.items.length; i++) {
      const item = this.items[i];

      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }

    return this;
  }

  selectLastItem(preserveTail) {
    for (let i = this.items.length - 1; i > 0; i--) {
      const item = this.items[i];

      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }

    return this;
  }

  selectAllItems() {
    return this.selectFirstItem().selectLastItem(true);
  }

  selectNextItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectFirstItem();
    }

    let itemIndex = this.selections[0].head;
    let nextItemIndex = itemIndex;

    while (itemIndex < this.items.length - 1) {
      itemIndex++;

      if (this.isItemSelectable(this.items[itemIndex])) {
        nextItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[nextItemIndex], preserveTail);
  }

  selectPreviousItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectLastItem();
    }

    let itemIndex = this.selections[0].head;
    let previousItemIndex = itemIndex;

    while (itemIndex > 0) {
      itemIndex--;

      if (this.isItemSelectable(this.items[itemIndex])) {
        previousItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[previousItemIndex], preserveTail);
  }

  selectItem(item, preserveTail, addOrSubtract) {
    if (addOrSubtract && preserveTail) {
      throw new Error('addOrSubtract and preserveTail cannot both be true at the same time');
    }

    const itemIndex = this.items.indexOf(item);

    if (preserveTail && this.selections[0]) {
      const newSelections = [{
        head: itemIndex,
        tail: this.selections[0].tail,
        negate: this.selections[0].negate
      }, ...this.selections.slice(1)];
      return this.copy({
        selections: newSelections
      });
    } else {
      const selection = {
        head: itemIndex,
        tail: itemIndex
      };

      if (addOrSubtract) {
        if (this.getSelectedItems().has(item)) {
          selection.negate = true;
        }

        return this.copy({
          selections: [selection, ...this.selections]
        });
      } else {
        return this.copy({
          selections: [selection]
        });
      }
    }
  }

  addOrSubtractSelection(item) {
    return this.selectItem(item, false, true);
  }

  coalesce() {
    if (this.selections.length === 0) {
      return this;
    }

    const mostRecent = this.selections[0];
    let mostRecentStart = Math.min(mostRecent.head, mostRecent.tail);
    let mostRecentEnd = Math.max(mostRecent.head, mostRecent.tail);

    while (mostRecentStart > 0 && !this.isItemSelectable(this.items[mostRecentStart - 1])) {
      mostRecentStart--;
    }

    while (mostRecentEnd < this.items.length - 1 && !this.isItemSelectable(this.items[mostRecentEnd + 1])) {
      mostRecentEnd++;
    }

    let changed = false;
    const newSelections = [mostRecent];

    for (let i = 1; i < this.selections.length;) {
      const current = this.selections[i];
      const currentStart = Math.min(current.head, current.tail);
      const currentEnd = Math.max(current.head, current.tail);

      if (mostRecentStart <= currentEnd + 1 && currentStart - 1 <= mostRecentEnd) {
        if (mostRecent.negate) {
          if (current.head > current.tail) {
            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({
                tail: mostRecentEnd + 1,
                head: currentEnd
              });
            }

            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({
                tail: currentStart,
                head: mostRecentStart - 1
              });
            }
          } else {
            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({
                head: currentStart,
                tail: mostRecentStart - 1
              });
            }

            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({
                head: mostRecentEnd + 1,
                tail: currentEnd
              });
            }
          }

          changed = true;
          i++;
        } else {
          mostRecentStart = Math.min(mostRecentStart, currentStart);
          mostRecentEnd = Math.max(mostRecentEnd, currentEnd);

          if (mostRecent.head >= mostRecent.tail) {
            mostRecent.head = mostRecentEnd;
            mostRecent.tail = mostRecentStart;
          } else {
            mostRecent.head = mostRecentStart;
            mostRecent.tail = mostRecentEnd;
          }

          changed = true;
          i++;
        }
      } else {
        newSelections.push(current);
        i++;
      }
    }

    if (mostRecent.negate) {
      changed = true;
      newSelections.shift();
    }

    return changed ? this.copy({
      selections: newSelections
    }) : this;
  }

  getSelectedItems() {
    const selectedItems = new Set();

    for (const _ref of this.selections.slice().reverse()) {
      const {
        head,
        tail,
        negate
      } = _ref;
      const start = Math.min(head, tail);
      const end = Math.max(head, tail);

      for (let i = start; i <= end; i++) {
        const item = this.items[i];

        if (this.isItemSelectable(item)) {
          if (negate) {
            selectedItems.delete(item);
          } else {
            selectedItems.add(item);
          }
        }
      }
    }

    return selectedItems;
  }

  getHeadItem() {
    return this.selections.length > 0 ? this.items[this.selections[0].head] : null;
  }

  getMostRecentSelectionStartIndex() {
    const selection = this.selections[0];
    return Math.min(selection.head, selection.tail);
  }

  getTailIndex() {
    return this.selections[0] ? this.selections[0].tail : null;
  }

}

exports.default = ListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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