"use strict";
/*
 * Wire
 * Copyright (C) 2018 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
const platform = require("platform");
const CommonConfig_1 = require("../config/CommonConfig");
const UNKNOWN_PROPERTY = 'unknown';
var OperatingSystem;
(function (OperatingSystem) {
    OperatingSystem["ANDROID"] = "OperatingSystem.ANDROID";
    OperatingSystem["IOS"] = "OperatingSystem.IOS";
    OperatingSystem["LINUX"] = "OperatingSystem.LINUX";
    OperatingSystem["MAC"] = "OperatingSystem.MAC";
    OperatingSystem["WINDOWS"] = "OperatingSystem.WINDOWS";
})(OperatingSystem || (OperatingSystem = {}));
exports.OperatingSystem = OperatingSystem;
class Runtime {
    static getPlatform() {
        const unsetPlatform = {};
        return platform || unsetPlatform;
    }
    static getOSFamily() {
        const family = Runtime.getOS().family.toLowerCase();
        if (family.includes('windows')) {
            return OperatingSystem.WINDOWS;
        }
        if (family.includes('android')) {
            return OperatingSystem.ANDROID;
        }
        if (family.includes('ios')) {
            return OperatingSystem.IOS;
        }
        if (['os x', 'mac os'].includes(family)) {
            return OperatingSystem.MAC;
        }
        return OperatingSystem.LINUX;
    }
    static getBrowserName() {
        return (Runtime.getPlatform().name || UNKNOWN_PROPERTY).toLowerCase();
    }
    static getBrowserVersion() {
        const [majorVersion, minorVersion] = (Runtime.getPlatform().version || UNKNOWN_PROPERTY).split('.');
        return { major: parseInt(majorVersion, 10), minor: parseInt(minorVersion, 10) };
    }
    static getUserAgent() {
        return (Runtime.getPlatform().ua || UNKNOWN_PROPERTY).toLowerCase();
    }
    static isWebappSupportedBrowser() {
        if (Runtime.isFranz()) {
            return false;
        }
        return Object.entries(CommonConfig_1.WEBAPP_SUPPORTED_BROWSERS).some(([browser, supportedVersion]) => {
            const isBrowserSupported = Runtime.getBrowserName() === browser;
            const currentVersion = Runtime.getBrowserVersion();
            const isSupportedMajorVersion = currentVersion.major >= supportedVersion.major;
            const isHigherMajorVersion = currentVersion.major > supportedVersion.major;
            const isSupportedMinorVersion = isHigherMajorVersion || currentVersion.minor >= supportedVersion.minor;
            return isBrowserSupported && isSupportedMajorVersion && isSupportedMinorVersion;
        });
    }
    static getOS() {
        return Object.assign({ architecture: UNKNOWN_PROPERTY, family: UNKNOWN_PROPERTY, version: UNKNOWN_PROPERTY }, platform.os);
    }
    static isChrome() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.CHROME;
    }
    static isEdge() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.MS_EDGE;
    }
    static isFirefox() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.FIREFOX;
    }
    static isInternetExplorer() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.IE;
    }
    static isOpera() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.OPERA;
    }
    static isSafari() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.SAFARI;
    }
    static isDesktopOS() {
        return Runtime.isMacOS() || Runtime.isWindows() || Runtime.isLinux();
    }
    static isElectron() {
        return Runtime.getBrowserName() === CommonConfig_1.BROWSER.ELECTRON;
    }
    static isDesktopApp() {
        return Runtime.isElectron() && Runtime.getUserAgent().includes('wire');
    }
    static isFranz() {
        return Runtime.isElectron() && Runtime.getUserAgent().includes('franz');
    }
    static isMacOS() {
        return Runtime.getOSFamily() === OperatingSystem.MAC;
    }
    static isWindows() {
        return Runtime.getOSFamily() === OperatingSystem.WINDOWS;
    }
    static isLinux() {
        return Runtime.getOSFamily() === OperatingSystem.LINUX;
    }
    static isMobileOS() {
        return Runtime.isAndroid() || Runtime.isIOS();
    }
    static isAndroid() {
        return Runtime.getOSFamily() === OperatingSystem.ANDROID;
    }
    static isIOS() {
        return Runtime.getOSFamily() === OperatingSystem.IOS;
    }
}
exports.Runtime = Runtime;
//# sourceMappingURL=Runtime.js.map