(function() {
  var CompositeDisposable, Disposable, Emitter, File, MarkdownPreviewView, _, fs, path, ref, renderer;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  fs = require('fs-plus');

  renderer = require('./renderer');

  module.exports = MarkdownPreviewView = (function() {
    MarkdownPreviewView.deserialize = function(params) {
      return new MarkdownPreviewView(params);
    };

    function MarkdownPreviewView(arg) {
      this.editorId = arg.editorId, this.filePath = arg.filePath;
      this.element = document.createElement('div');
      this.element.classList.add('markdown-preview');
      this.element.tabIndex = -1;
      this.emitter = new Emitter;
      this.loaded = false;
      this.disposables = new CompositeDisposable;
      this.registerScrollCommands();
      if (this.editorId != null) {
        this.resolveEditor(this.editorId);
      } else if (atom.packages.hasActivatedInitialPackages()) {
        this.subscribeToFilePath(this.filePath);
      } else {
        this.disposables.add(atom.packages.onDidActivateInitialPackages((function(_this) {
          return function() {
            return _this.subscribeToFilePath(_this.filePath);
          };
        })(this)));
      }
    }

    MarkdownPreviewView.prototype.serialize = function() {
      var ref1;
      return {
        deserializer: 'MarkdownPreviewView',
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath,
        editorId: this.editorId
      };
    };

    MarkdownPreviewView.prototype.copy = function() {
      var ref1;
      return new MarkdownPreviewView({
        editorId: this.editorId,
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath
      });
    };

    MarkdownPreviewView.prototype.destroy = function() {
      this.disposables.dispose();
      return this.element.remove();
    };

    MarkdownPreviewView.prototype.registerScrollCommands = function() {
      this.disposables.add(atom.commands.add(this.element, {
        'core:move-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= document.body.offsetHeight / 20;
          };
        })(this),
        'core:move-down': (function(_this) {
          return function() {
            _this.element.scrollTop += document.body.offsetHeight / 20;
          };
        })(this),
        'core:page-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= _this.element.offsetHeight;
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            _this.element.scrollTop += _this.element.offsetHeight;
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            _this.element.scrollTop = 0;
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            _this.element.scrollTop = _this.element.scrollHeight;
          };
        })(this)
      }));
    };

    MarkdownPreviewView.prototype.onDidChangeTitle = function(callback) {
      return this.emitter.on('did-change-title', callback);
    };

    MarkdownPreviewView.prototype.onDidChangeModified = function(callback) {
      return new Disposable;
    };

    MarkdownPreviewView.prototype.onDidChangeMarkdown = function(callback) {
      return this.emitter.on('did-change-markdown', callback);
    };

    MarkdownPreviewView.prototype.subscribeToFilePath = function(filePath) {
      this.file = new File(filePath);
      this.emitter.emit('did-change-title');
      this.disposables.add(this.file.onDidRename((function(_this) {
        return function() {
          return _this.emitter.emit('did-change-title');
        };
      })(this)));
      this.handleEvents();
      return this.renderMarkdown();
    };

    MarkdownPreviewView.prototype.resolveEditor = function(editorId) {
      var resolve;
      resolve = (function(_this) {
        return function() {
          _this.editor = _this.editorForId(editorId);
          if (_this.editor != null) {
            _this.emitter.emit('did-change-title');
            _this.disposables.add(_this.editor.onDidDestroy(function() {
              return _this.subscribeToFilePath(_this.getPath());
            }));
            _this.handleEvents();
            return _this.renderMarkdown();
          } else {
            return _this.subscribeToFilePath(_this.filePath);
          }
        };
      })(this);
      if (atom.packages.hasActivatedInitialPackages()) {
        return resolve();
      } else {
        return this.disposables.add(atom.packages.onDidActivateInitialPackages(resolve));
      }
    };

    MarkdownPreviewView.prototype.editorForId = function(editorId) {
      var editor, i, len, ref1, ref2;
      ref1 = atom.workspace.getTextEditors();
      for (i = 0, len = ref1.length; i < len; i++) {
        editor = ref1[i];
        if (((ref2 = editor.id) != null ? ref2.toString() : void 0) === editorId.toString()) {
          return editor;
        }
      }
      return null;
    };

    MarkdownPreviewView.prototype.handleEvents = function() {
      var changeHandler, lazyRenderMarkdown;
      lazyRenderMarkdown = _.debounce(((function(_this) {
        return function() {
          return _this.renderMarkdown();
        };
      })(this)), 250);
      this.disposables.add(atom.grammars.onDidAddGrammar(function() {
        return lazyRenderMarkdown();
      }));
      if (typeof atom.grammars.onDidRemoveGrammar === 'function') {
        this.disposables.add(atom.grammars.onDidRemoveGrammar(function() {
          return lazyRenderMarkdown();
        }));
      } else {
        this.disposables.add(atom.grammars.onDidUpdateGrammar(function() {
          return lazyRenderMarkdown();
        }));
      }
      atom.commands.add(this.element, {
        'core:copy': (function(_this) {
          return function(event) {
            event.stopPropagation();
            return _this.copyToClipboard();
          };
        })(this),
        'markdown-preview:select-all': (function(_this) {
          return function() {
            return _this.selectAll();
          };
        })(this),
        'markdown-preview:zoom-in': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel + 0.1;
          };
        })(this),
        'markdown-preview:zoom-out': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel - 0.1;
          };
        })(this),
        'markdown-preview:reset-zoom': (function(_this) {
          return function() {
            return _this.element.style.zoom = 1;
          };
        })(this),
        'markdown-preview:toggle-break-on-single-newline': function() {
          var keyPath;
          keyPath = 'markdown-preview.breakOnSingleNewline';
          return atom.config.set(keyPath, !atom.config.get(keyPath));
        },
        'markdown-preview:toggle-github-style': function() {
          var keyPath;
          keyPath = 'markdown-preview.useGitHubStyle';
          return atom.config.set(keyPath, !atom.config.get(keyPath));
        }
      });
      changeHandler = (function(_this) {
        return function() {
          var pane;
          _this.renderMarkdown();
          pane = atom.workspace.paneForItem(_this);
          if ((pane != null) && pane !== atom.workspace.getActivePane()) {
            return pane.activateItem(_this);
          }
        };
      })(this);
      if (this.file != null) {
        this.disposables.add(this.file.onDidChange(changeHandler));
      } else if (this.editor != null) {
        this.disposables.add(this.editor.getBuffer().onDidStopChanging(function() {
          if (atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.onDidChangePath((function(_this) {
          return function() {
            return _this.emitter.emit('did-change-title');
          };
        })(this)));
        this.disposables.add(this.editor.getBuffer().onDidSave(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.getBuffer().onDidReload(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
      }
      this.disposables.add(atom.config.onDidChange('markdown-preview.breakOnSingleNewline', changeHandler));
      this.disposables.add(atom.config.observe('markdown-preview.useGitHubStyle', (function(_this) {
        return function(useGitHubStyle) {
          if (useGitHubStyle) {
            return _this.element.setAttribute('data-use-github-style', '');
          } else {
            return _this.element.removeAttribute('data-use-github-style');
          }
        };
      })(this)));
      return document.onselectionchange = (function(_this) {
        return function() {
          var selectedNode, selection;
          selection = window.getSelection();
          selectedNode = selection.baseNode;
          if (selectedNode === null || _this.element === selectedNode || _this.element.contains(selectedNode)) {
            if (selection.isCollapsed) {
              return _this.element.classList.remove('has-selection');
            } else {
              return _this.element.classList.add('has-selection');
            }
          }
        };
      })(this);
    };

    MarkdownPreviewView.prototype.renderMarkdown = function() {
      if (!this.loaded) {
        this.showLoading();
      }
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source != null) {
            return _this.renderMarkdownText(source);
          }
        };
      })(this))["catch"]((function(_this) {
        return function(reason) {
          return _this.showError({
            message: reason
          });
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getMarkdownSource = function() {
      var ref1;
      if ((ref1 = this.file) != null ? ref1.getPath() : void 0) {
        return this.file.read().then((function(_this) {
          return function(source) {
            if (source === null) {
              return Promise.reject((_this.file.getBaseName()) + " could not be found");
            } else {
              return Promise.resolve(source);
            }
          };
        })(this))["catch"](function(reason) {
          return Promise.reject(reason);
        });
      } else if (this.editor != null) {
        return Promise.resolve(this.editor.getText());
      } else {
        return Promise.reject();
      }
    };

    MarkdownPreviewView.prototype.getHTML = function(callback) {
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source == null) {
            return;
          }
          return renderer.toHTML(source, _this.getPath(), _this.getGrammar(), callback);
        };
      })(this));
    };

    MarkdownPreviewView.prototype.renderMarkdownText = function(text) {
      var scrollTop;
      scrollTop = this.element.scrollTop;
      return renderer.toDOMFragment(text, this.getPath(), this.getGrammar(), (function(_this) {
        return function(error, domFragment) {
          if (error) {
            return _this.showError(error);
          } else {
            _this.loading = false;
            _this.loaded = true;
            _this.element.textContent = '';
            _this.element.appendChild(domFragment);
            _this.emitter.emit('did-change-markdown');
            return _this.element.scrollTop = scrollTop;
          }
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getTitle = function() {
      if ((this.file != null) && (this.getPath() != null)) {
        return (path.basename(this.getPath())) + " Preview";
      } else if (this.editor != null) {
        return (this.editor.getTitle()) + " Preview";
      } else {
        return "Markdown Preview";
      }
    };

    MarkdownPreviewView.prototype.getIconName = function() {
      return "markdown";
    };

    MarkdownPreviewView.prototype.getURI = function() {
      if (this.file != null) {
        return "markdown-preview://" + (this.getPath());
      } else {
        return "markdown-preview://editor/" + this.editorId;
      }
    };

    MarkdownPreviewView.prototype.getPath = function() {
      if (this.file != null) {
        return this.file.getPath();
      } else if (this.editor != null) {
        return this.editor.getPath();
      }
    };

    MarkdownPreviewView.prototype.getGrammar = function() {
      var ref1;
      return (ref1 = this.editor) != null ? ref1.getGrammar() : void 0;
    };

    MarkdownPreviewView.prototype.getDocumentStyleSheets = function() {
      return document.styleSheets;
    };

    MarkdownPreviewView.prototype.getTextEditorStyles = function() {
      var textEditorStyles;
      textEditorStyles = document.createElement("atom-styles");
      textEditorStyles.initialize(atom.styles);
      textEditorStyles.setAttribute("context", "atom-text-editor");
      document.body.appendChild(textEditorStyles);
      return Array.prototype.slice.apply(textEditorStyles.childNodes).map(function(styleElement) {
        return styleElement.innerText;
      });
    };

    MarkdownPreviewView.prototype.getMarkdownPreviewCSS = function() {
      var cssUrlRegExp, i, j, len, len1, markdownPreviewRules, ref1, ref2, ref3, rule, ruleRegExp, stylesheet;
      markdownPreviewRules = [];
      ruleRegExp = /\.markdown-preview/;
      cssUrlRegExp = /url\(atom:\/\/markdown-preview\/assets\/(.*)\)/;
      ref1 = this.getDocumentStyleSheets();
      for (i = 0, len = ref1.length; i < len; i++) {
        stylesheet = ref1[i];
        if (stylesheet.rules != null) {
          ref2 = stylesheet.rules;
          for (j = 0, len1 = ref2.length; j < len1; j++) {
            rule = ref2[j];
            if (((ref3 = rule.selectorText) != null ? ref3.match(ruleRegExp) : void 0) != null) {
              markdownPreviewRules.push(rule.cssText);
            }
          }
        }
      }
      return markdownPreviewRules.concat(this.getTextEditorStyles()).join('\n').replace(/atom-text-editor/g, 'pre.editor-colors').replace(/:host/g, '.host').replace(cssUrlRegExp, function(match, assetsName, offset, string) {
        var assetPath, base64Data, originalData;
        assetPath = path.join(__dirname, '../assets', assetsName);
        originalData = fs.readFileSync(assetPath, 'binary');
        base64Data = new Buffer(originalData, 'binary').toString('base64');
        return "url('data:image/jpeg;base64," + base64Data + "')";
      });
    };

    MarkdownPreviewView.prototype.showError = function(result) {
      var failureMessage, h2, h3;
      this.element.textContent = '';
      h2 = document.createElement('h2');
      h2.textContent = 'Previewing Markdown Failed';
      this.element.appendChild(h2);
      if (failureMessage = result != null ? result.message : void 0) {
        h3 = document.createElement('h3');
        h3.textContent = failureMessage;
        return this.element.appendChild(h3);
      }
    };

    MarkdownPreviewView.prototype.showLoading = function() {
      var div;
      this.loading = true;
      this.element.textContent = '';
      div = document.createElement('div');
      div.classList.add('markdown-spinner');
      div.textContent = 'Loading Markdown\u2026';
      return this.element.appendChild(div);
    };

    MarkdownPreviewView.prototype.selectAll = function() {
      var range, selection;
      if (this.loading) {
        return;
      }
      selection = window.getSelection();
      selection.removeAllRanges();
      range = document.createRange();
      range.selectNodeContents(this.element);
      return selection.addRange(range);
    };

    MarkdownPreviewView.prototype.copyToClipboard = function() {
      var selectedNode, selectedText, selection;
      if (this.loading) {
        return;
      }
      selection = window.getSelection();
      selectedText = selection.toString();
      selectedNode = selection.baseNode;
      if (selectedText && (selectedNode != null) && (this.element === selectedNode || this.element.contains(selectedNode))) {
        return atom.clipboard.write(selectedText);
      } else {
        return this.getHTML(function(error, html) {
          if (error != null) {
            return atom.notifications.addError('Copying Markdown as HTML failed', {
              dismissable: true,
              detail: error.message
            });
          } else {
            return atom.clipboard.write(html);
          }
        });
      }
    };

    MarkdownPreviewView.prototype.getSaveDialogOptions = function() {
      var defaultPath, projectPath;
      defaultPath = this.getPath();
      if (defaultPath) {
        defaultPath += '.html';
      } else {
        defaultPath = 'untitled.md.html';
        if (projectPath = atom.project.getPaths()[0]) {
          defaultPath = path.join(projectPath, defaultPath);
        }
      }
      return {
        defaultPath: defaultPath
      };
    };

    MarkdownPreviewView.prototype.saveAs = function(htmlFilePath) {
      var filePath, title;
      if (this.loading) {
        atom.notifications.addWarning('Please wait until the Markdown Preview has finished loading before saving');
        return;
      }
      filePath = this.getPath();
      title = 'Markdown to HTML';
      if (filePath) {
        title = path.parse(filePath).name;
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.getHTML(function(error, htmlBody) {
            var html;
            if (error != null) {
              throw error;
            } else {
              html = ("<!DOCTYPE html>\n<html>\n  <head>\n      <meta charset=\"utf-8\" />\n      <title>" + title + "</title>\n      <style>" + (_this.getMarkdownPreviewCSS()) + "</style>\n  </head>\n  <body class='markdown-preview' data-use-github-style>" + htmlBody + "</body>\n</html>") + "\n";
              fs.writeFileSync(htmlFilePath, html);
              return atom.workspace.open(htmlFilePath).then(resolve);
            }
          });
        };
      })(this));
    };

    return MarkdownPreviewView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
