"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _eventKit = require("event-kit");

var _atom = require("atom");

var _path = _interopRequireDefault(require("path"));

var _eventLogger = _interopRequireDefault(require("./event-logger"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class FileSystemChangeObserver {
  constructor(repository) {
    this.emitter = new _eventKit.Emitter();
    this.repository = repository;
    this.logger = new _eventLogger.default('fs watcher');
    this.started = false;
  }

  async start() {
    await this.watchRepository();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.emitter.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchRepository() {
    const allPaths = event => {
      const ps = [event.path];

      if (event.oldPath) {
        ps.push(event.oldPath);
      }

      return ps;
    };

    const isNonGitFile = event => allPaths(event).some(eventPath => !eventPath.split(_path.default.sep).includes('.git'));

    const isWatchedGitFile = event => allPaths(event).some(eventPath => {
      return ['config', 'index', 'HEAD', 'MERGE_HEAD'].includes(_path.default.basename(eventPath)) || _path.default.dirname(eventPath).includes(_path.default.join('.git', 'refs'));
    });

    const handleEvents = events => {
      const filteredEvents = events.filter(e => isNonGitFile(e) || isWatchedGitFile(e));

      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);
        const workdirOrHeadEvent = filteredEvents.find(event => {
          return allPaths(event).every(eventPath => !['config', 'index'].includes(_path.default.basename(eventPath)));
        });

        if (workdirOrHeadEvent) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(this.repository.getWorkingDirectoryPath(), {}, handleEvents);
    this.logger.showStarted(this.repository.getWorkingDirectoryPath(), 'Atom watchPath');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.logger.showStopped();
    }

    return Promise.resolve();
  }

}

exports.default = FileSystemChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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