"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.category = category;
exports.default = exports.BarePrStatusesView = exports.stateToIconAndStyle = void 0;

var _react = _interopRequireDefault(require("react"));

var _reactRelay = require("react-relay");

var _propTypes = _interopRequireDefault(require("prop-types"));

var _helpers = require("../helpers");

var _prStatusContextView = _interopRequireDefault(require("./pr-status-context-view"));

var _octicon = _interopRequireDefault(require("../atom/octicon"));

var _statusDonutChart = _interopRequireDefault(require("./status-donut-chart"));

var _periodicRefresher = _interopRequireDefault(require("../periodic-refresher"));

var _propTypes2 = require("../prop-types");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const stateToIconAndStyle = {
  EXPECTED: {
    category: 'PENDING',
    icon: 'primitive-dot',
    style: 'github-PrStatuses--warning'
  },
  PENDING: {
    category: 'PENDING',
    icon: 'primitive-dot',
    style: 'github-PrStatuses--warning'
  },
  SUCCESS: {
    category: 'SUCCESS',
    icon: 'check',
    style: 'github-PrStatuses--success'
  },
  ERROR: {
    category: 'FAILURE',
    icon: 'alert',
    style: 'github-PrStatuses--error'
  },
  FAILURE: {
    category: 'FAILURE',
    icon: 'x',
    style: 'github-PrStatuses--error'
  }
};
exports.stateToIconAndStyle = stateToIconAndStyle;

function category(state) {
  const info = stateToIconAndStyle[state];

  if (!info) {
    throw new Error(`Unknown state ${state}`);
  }

  return info.category;
}

class BarePrStatusesView extends _react.default.Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'refresh');
  }

  componentDidMount() {
    this.refresher = new _periodicRefresher.default(this.constructor, {
      interval: () => {
        if (this.isPendingResults()) {
          return this.constructor.PENDING_REFRESH_TIMEOUT;
        } else {
          return this.constructor.SUCCESS_REFRESH_TIMEOUT;
        }
      },
      getCurrentId: () => this.props.pullRequest.id,
      refresh: this.refresh,
      minimumIntervalPerId: this.constructor.MINIMUM_REFRESH_INTERVAL
    });
    this.refresher.start();
  }

  componentWillUnmount() {
    this.refresher.destroy();
  }

  refresh() {
    this.props.relay.refetch({
      id: this.props.pullRequest.id
    }, null, null, {
      force: true
    });
  }

  render() {
    const headCommit = this.props.pullRequest.recentCommits.edges[0].node.commit;

    if (!headCommit.status) {
      return null;
    }

    const {
      state,
      contexts
    } = headCommit.status;

    if (this.props.displayType === 'check') {
      const {
        icon,
        style
      } = stateToIconAndStyle[state];
      return _react.default.createElement(_octicon.default, {
        icon: icon,
        className: style
      });
    } else if (this.props.displayType === 'full') {
      return _react.default.createElement("div", {
        className: "github-PrStatuses"
      }, _react.default.createElement("div", {
        className: "github-PrStatuses-header"
      }, _react.default.createElement("div", {
        className: "github-PrStatuses-donut-chart"
      }, this.renderDonutChart(headCommit.status)), _react.default.createElement("div", {
        className: "github-PrStatuses-summary"
      }, this.summarySentence(headCommit.status))), _react.default.createElement("ul", {
        className: "github-PrStatuses-list"
      }, contexts.map(context => _react.default.createElement(_prStatusContextView.default, {
        key: context.id,
        context: context
      }))));
    } else {
      throw new Error(`Invalid \`displayType\` prop value: ${this.props.displayType}`);
    }
  }

  isPendingResults() {
    const headCommit = this.props.pullRequest.recentCommits.edges[0].node.commit;

    if (!headCommit.status) {
      return false;
    }

    const {
      contexts
    } = headCommit.status;
    return contexts.some(c => category(c.state) === 'PENDING');
  }

  renderDonutChart(status) {
    const {
      contexts
    } = status;
    const pendingLen = contexts.filter(c => category(c.state) === 'PENDING').length;
    const failedLen = contexts.filter(c => category(c.state) === 'FAILURE').length;
    const succeededLen = contexts.filter(c => category(c.state) === 'SUCCESS').length;
    return _react.default.createElement(_statusDonutChart.default, {
      pending: pendingLen,
      failure: failedLen,
      success: succeededLen
    });
  }

  summarySentence(status) {
    if (this.isAllSucceeded(status)) {
      return 'All checks succeeded';
    } else if (this.isAllFailed(status)) {
      return 'All checks failed';
    } else {
      const {
        contexts
      } = status;
      const noun = contexts.length === 1 ? 'check' : 'checks';
      const parts = [];
      const pending = contexts.filter(c => category(c.state) === 'PENDING');
      const failing = contexts.filter(c => category(c.state) === 'FAILURE');
      const succeeded = contexts.filter(c => category(c.state) === 'SUCCESS');

      if (pending.length) {
        parts.push(`${pending.length} pending`);
      }

      if (failing.length) {
        parts.push(`${failing.length} failing`);
      }

      if (succeeded.length) {
        parts.push(`${succeeded.length} successful`);
      }

      return (0, _helpers.toSentence)(parts) + ` ${noun}`;
    }
  }

  isAllSucceeded(status) {
    return category(status.state) === 'SUCCESS';
  }

  isAllFailed(status) {
    return status.contexts.every(c => category(c.state) === 'FAILURE');
  }

}

exports.BarePrStatusesView = BarePrStatusesView;

_defineProperty(BarePrStatusesView, "propTypes", {
  relay: _propTypes.default.shape({
    refetch: _propTypes.default.func.isRequired
  }).isRequired,
  displayType: _propTypes.default.oneOf(['check', 'full']),
  pullRequest: _propTypes.default.shape({
    id: _propTypes.default.string.isRequired,
    recentCommits: (0, _propTypes2.RelayConnectionPropType)(_propTypes.default.shape({
      commit: _propTypes.default.shape({
        status: _propTypes.default.shape({
          state: _propTypes.default.string.isRequired,
          contexts: _propTypes.default.arrayOf(_propTypes.default.shape({
            id: _propTypes.default.string.isRequired
          }).isRequired).isRequired
        })
      }).isRequired
    }).isRequired).isRequired
  }).isRequired
});

_defineProperty(BarePrStatusesView, "defaultProps", {
  displayType: 'full'
});

_defineProperty(BarePrStatusesView, "lastRefreshPerPr", new Map());

_defineProperty(BarePrStatusesView, "SUCCESS_REFRESH_TIMEOUT", 3 * 60 * 1000);

_defineProperty(BarePrStatusesView, "PENDING_REFRESH_TIMEOUT", 30 * 1000);

_defineProperty(BarePrStatusesView, "MINIMUM_REFRESH_INTERVAL", 15 * 1000);

var _default = (0, _reactRelay.createRefetchContainer)(BarePrStatusesView, {
  pullRequest: function () {
    const node = require("./__generated__/prStatusesView_pullRequest.graphql");

    if (node.hash && node.hash !== "96a5a2c7004e7526967de3b7b551a54b") {
      console.error("The definition of 'prStatusesView_pullRequest' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prStatusesView_pullRequest.graphql");
  }
}, function () {
  const node = require("./__generated__/prStatusesViewRefetchQuery.graphql");

  if (node.hash && node.hash !== "a7600333b3bc426d899c4d0183095a1f") {
    console.error("The definition of 'prStatusesViewRefetchQuery' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
  }

  return require("./__generated__/prStatusesViewRefetchQuery.graphql");
});

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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