"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.COMMIT_GRAMMAR_SCOPE = void 0;

var _path = _interopRequireDefault(require("path"));

var _atom = require("atom");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _commitView = _interopRequireDefault(require("../views/commit-view"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _propTypes2 = require("../prop-types");

var _watchWorkspaceItem = require("../watch-workspace-item");

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const COMMIT_GRAMMAR_SCOPE = 'text.git-commit';
exports.COMMIT_GRAMMAR_SCOPE = COMMIT_GRAMMAR_SCOPE;

class CommitController extends _react.default.Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "activateCommitPreview", () => {
      const uri = _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath());

      return this.props.workspace.open(uri, {
        searchAllPanes: true,
        pending: true,
        activate: true
      });
    });

    (0, _helpers.autobind)(this, 'commit', 'handleMessageChange', 'toggleExpandedCommitMessageEditor', 'grammarAdded', 'toggleCommitPreview');
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.refCommitView = new _refHolder.default();
    this.commitMessageBuffer = new _atom.TextBuffer({
      text: this.props.repository.getCommitMessage()
    });
    this.subscriptions.add(this.commitMessageBuffer.onDidChange(this.handleMessageChange));
    this.previewWatcher = (0, _watchWorkspaceItem.watchWorkspaceItem)(this.props.workspace, _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath()), this, 'commitPreviewActive');
    this.subscriptions.add(this.previewWatcher);
  }

  componentDidMount() {
    this.subscriptions.add(this.props.workspace.onDidAddTextEditor(({
      textEditor
    }) => {
      if (this.props.repository.isPresent() && textEditor.getPath() === this.getCommitMessagePath()) {
        const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);

        if (grammar) {
          textEditor.setGrammar(grammar);
        }
      }
    }), this.props.workspace.onDidDestroyPaneItem(async ({
      item
    }) => {
      if (this.props.repository.isPresent() && item.getPath && item.getPath() === this.getCommitMessagePath() && this.getCommitMessageEditors().length === 0) {
        // we closed the last editor pointing to the commit message file
        try {
          this.commitMessageBuffer.setText((await _fsExtra.default.readFile(this.getCommitMessagePath(), {
            encoding: 'utf8'
          })));
        } catch (e) {
          if (e.code !== 'ENOENT') {
            throw e;
          }
        }
      }
    }));
  }

  render() {
    const operationStates = this.props.repository.getOperationStates();
    return _react.default.createElement(_commitView.default, {
      ref: this.refCommitView.setter,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      stagedChangesExist: this.props.stagedChangesExist,
      mergeConflictsExist: this.props.mergeConflictsExist,
      prepareToCommit: this.props.prepareToCommit,
      commit: this.commit,
      abortMerge: this.props.abortMerge,
      commandRegistry: this.props.commandRegistry,
      maximumCharacterLimit: 72,
      messageBuffer: this.commitMessageBuffer,
      isMerging: this.props.isMerging,
      isCommitting: operationStates.isCommitInProgress(),
      lastCommit: this.props.lastCommit,
      currentBranch: this.props.currentBranch,
      toggleExpandedCommitMessageEditor: this.toggleExpandedCommitMessageEditor,
      deactivateCommitBox: this.isCommitMessageEditorExpanded(),
      userStore: this.props.userStore,
      selectedCoAuthors: this.props.selectedCoAuthors,
      updateSelectedCoAuthors: this.props.updateSelectedCoAuthors,
      toggleCommitPreview: this.toggleCommitPreview,
      activateCommitPreview: this.activateCommitPreview,
      commitPreviewActive: this.state.commitPreviewActive
    });
  }

  componentDidUpdate(prevProps) {
    this.commitMessageBuffer.setTextViaDiff(this.getCommitMessage());

    if (prevProps.repository !== this.props.repository) {
      this.previewWatcher.setPattern(_commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath()));
    }
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  commit(message, coAuthors = [], amend = false) {
    let msg, verbatim;

    if (this.isCommitMessageEditorExpanded()) {
      msg = this.getCommitMessageEditors()[0].getText();
      verbatim = false;
    } else {
      const wrapMessage = this.props.config.get('github.automaticCommitMessageWrapping');
      msg = wrapMessage ? wrapCommitMessage(message) : message;
      verbatim = true;
    }

    return this.props.commit(msg.trim(), {
      amend,
      coAuthors,
      verbatim
    });
  }

  setCommitMessage(message, options) {
    if (!this.props.repository.isPresent()) {
      return;
    }

    this.props.repository.setCommitMessage(message, options);
  }

  getCommitMessage() {
    return this.props.repository.getCommitMessage();
  }

  getCommitMessagePath() {
    return _path.default.join(this.props.repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
  }

  handleMessageChange() {
    if (!this.props.repository.isPresent()) {
      return;
    }

    this.setCommitMessage(this.commitMessageBuffer.getText(), {
      suppressUpdate: true
    });
  }

  getCommitMessageEditors() {
    if (!this.props.repository.isPresent()) {
      return [];
    }

    return this.props.workspace.getTextEditors().filter(editor => editor.getPath() === this.getCommitMessagePath());
  }

  async toggleExpandedCommitMessageEditor(messageFromBox) {
    if (this.isCommitMessageEditorExpanded()) {
      if (this.commitMessageEditorIsInForeground()) {
        await this.closeAllOpenCommitMessageEditors();
        this.forceUpdate();
      } else {
        this.activateCommitMessageEditor();
      }
    } else {
      await this.openCommitMessageEditor(messageFromBox);
      this.forceUpdate();
    }
  }

  isCommitMessageEditorExpanded() {
    return this.getCommitMessageEditors().length > 0;
  }

  commitMessageEditorIsInForeground() {
    const commitMessageEditorsInForeground = this.props.workspace.getPanes().map(pane => pane.getActiveItem()).filter(item => item && item.getPath && item.getPath() === this.getCommitMessagePath());
    return commitMessageEditorsInForeground.length > 0;
  }

  activateCommitMessageEditor() {
    const panes = this.props.workspace.getPanes();
    let editor;
    const paneWithEditor = panes.find(pane => {
      editor = pane.getItems().find(item => item.getPath && item.getPath() === this.getCommitMessagePath());
      return !!editor;
    });
    paneWithEditor.activate();
    paneWithEditor.activateItem(editor);
  }

  closeAllOpenCommitMessageEditors() {
    return Promise.all(this.props.workspace.getPanes().map(pane => {
      return Promise.all(pane.getItems().map(async item => {
        if (item && item.getPath && item.getPath() === this.getCommitMessagePath()) {
          const destroyed = await pane.destroyItem(item);

          if (!destroyed) {
            pane.activateItem(item);
          }
        }
      }));
    }));
  }

  async openCommitMessageEditor(messageFromBox) {
    await _fsExtra.default.writeFile(this.getCommitMessagePath(), messageFromBox, 'utf8');
    const commitEditor = await this.props.workspace.open(this.getCommitMessagePath());
    (0, _reporterProxy.addEvent)('open-commit-message-editor', {
      package: 'github'
    });
    const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);

    if (grammar) {
      commitEditor.setGrammar(grammar);
    } else {
      this.grammarSubscription = this.props.grammars.onDidAddGrammar(this.grammarAdded);
      this.subscriptions.add(this.grammarSubscription);
    }
  }

  grammarAdded(grammar) {
    if (grammar.scopeName !== COMMIT_GRAMMAR_SCOPE) {
      return;
    }

    this.getCommitMessageEditors().forEach(editor => editor.setGrammar(grammar));
    this.grammarSubscription.dispose();
  }

  getFocus(element) {
    return this.refCommitView.map(view => view.getFocus(element)).getOr(null);
  }

  setFocus(focus) {
    return this.refCommitView.map(view => view.setFocus(focus)).getOr(false);
  }

  advanceFocusFrom(...args) {
    return this.refCommitView.map(view => view.advanceFocusFrom(...args)).getOr(false);
  }

  retreatFocusFrom(...args) {
    return this.refCommitView.map(view => view.retreatFocusFrom(...args)).getOr(false);
  }

  toggleCommitPreview() {
    (0, _reporterProxy.addEvent)('toggle-commit-preview', {
      package: 'github'
    });

    const uri = _commitPreviewItem.default.buildURI(this.props.repository.getWorkingDirectoryPath());

    if (this.props.workspace.hide(uri)) {
      return Promise.resolve();
    } else {
      return this.props.workspace.open(uri, {
        searchAllPanes: true,
        pending: true
      });
    }
  }

}

exports.default = CommitController;

_defineProperty(CommitController, "focus", _objectSpread({}, _commitView.default.focus));

_defineProperty(CommitController, "propTypes", {
  workspace: _propTypes.default.object.isRequired,
  grammars: _propTypes.default.object.isRequired,
  commandRegistry: _propTypes.default.object.isRequired,
  config: _propTypes.default.object.isRequired,
  tooltips: _propTypes.default.object.isRequired,
  repository: _propTypes.default.object.isRequired,
  isMerging: _propTypes.default.bool.isRequired,
  mergeConflictsExist: _propTypes.default.bool.isRequired,
  stagedChangesExist: _propTypes.default.bool.isRequired,
  lastCommit: _propTypes.default.object.isRequired,
  currentBranch: _propTypes.default.object.isRequired,
  userStore: _propTypes2.UserStorePropType.isRequired,
  selectedCoAuthors: _propTypes.default.arrayOf(_propTypes2.AuthorPropType),
  updateSelectedCoAuthors: _propTypes.default.func,
  prepareToCommit: _propTypes.default.func.isRequired,
  commit: _propTypes.default.func.isRequired,
  abortMerge: _propTypes.default.func.isRequired
});

function wrapCommitMessage(message) {
  // hard wrap message (except for first line) at 72 characters
  let results = [];
  message.split('\n').forEach((line, index) => {
    if (line.length <= 72 || index === 0) {
      results.push(line);
    } else {
      const matches = line.match(/.{1,72}(\s|$)|\S+?(\s|$)/g).map(match => {
        return match.endsWith('\n') ? match.substr(0, match.length - 1) : match;
      });
      results = results.concat(matches);
    }
  });
  return results.join('\n');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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