#ifndef _EFL_TEXT_FONT_EO_H_
#define _EFL_TEXT_FONT_EO_H_

#ifndef _EFL_TEXT_FONT_EO_CLASS_TYPE
#define _EFL_TEXT_FONT_EO_CLASS_TYPE

typedef Eo Efl_Text_Font;

#endif

#ifndef _EFL_TEXT_FONT_EO_TYPES
#define _EFL_TEXT_FONT_EO_TYPES

/** Type of font weight
 *
 * @ingroup Efl
 */
typedef enum
{
  EFL_TEXT_FONT_WEIGHT_NORMAL = 0, /**< Normal font weight */
  EFL_TEXT_FONT_WEIGHT_THIN, /**< Thin font weight */
  EFL_TEXT_FONT_WEIGHT_ULTRALIGHT, /**< Ultralight font weight */
  EFL_TEXT_FONT_WEIGHT_EXTRALIGHT, /**< Extralight font weight */
  EFL_TEXT_FONT_WEIGHT_LIGHT, /**< Light font weight */
  EFL_TEXT_FONT_WEIGHT_BOOK, /**< Book font weight */
  EFL_TEXT_FONT_WEIGHT_MEDIUM, /**< Medium font weight */
  EFL_TEXT_FONT_WEIGHT_SEMIBOLD, /**< Semibold font weight */
  EFL_TEXT_FONT_WEIGHT_BOLD, /**< Bold font weight */
  EFL_TEXT_FONT_WEIGHT_ULTRABOLD, /**< Ultrabold font weight */
  EFL_TEXT_FONT_WEIGHT_EXTRABOLD, /**< Extrabold font weight */
  EFL_TEXT_FONT_WEIGHT_BLACK, /**< Black font weight */
  EFL_TEXT_FONT_WEIGHT_EXTRABLACK /**< Extrablack font weight */
} Efl_Text_Font_Weight;

/** Type of font width
 *
 * @ingroup Efl
 */
typedef enum
{
  EFL_TEXT_FONT_WIDTH_NORMAL = 0, /**< Normal font width */
  EFL_TEXT_FONT_WIDTH_ULTRACONDENSED, /**< Ultracondensed font width */
  EFL_TEXT_FONT_WIDTH_EXTRACONDENSED, /**< Extracondensed font width */
  EFL_TEXT_FONT_WIDTH_CONDENSED, /**< Condensed font width */
  EFL_TEXT_FONT_WIDTH_SEMICONDENSED, /**< Semicondensed font width */
  EFL_TEXT_FONT_WIDTH_SEMIEXPANDED, /**< Semiexpanded font width */
  EFL_TEXT_FONT_WIDTH_EXPANDED, /**< Expanded font width */
  EFL_TEXT_FONT_WIDTH_EXTRAEXPANDED, /**< Extraexpanded font width */
  EFL_TEXT_FONT_WIDTH_ULTRAEXPANDED /**< Ultraexpanded font width */
} Efl_Text_Font_Width;

/** Type of font slant
 *
 * @ingroup Efl
 */
typedef enum
{
  EFL_TEXT_FONT_SLANT_NORMAL = 0, /**< Normal font slant */
  EFL_TEXT_FONT_SLANT_OBLIQUE, /**< Oblique font slant */
  EFL_TEXT_FONT_SLANT_ITALIC /**< Italic font slant */
} Efl_Text_Font_Slant;

/** Scalable of bitmap fonts
 *
 * @since 1.21
 *
 * @ingroup Efl
 */
typedef enum
{
  EFL_TEXT_FONT_BITMAP_SCALABLE_NONE = 0, /**< Disable scalable feature for
                                           * bitmap fonts. */
  EFL_TEXT_FONT_BITMAP_SCALABLE_COLOR = 1 /* 1 >> 0 */ /**< Enable scalable
                                                        * feature for color
                                                        * bitmap fonts. */
} Efl_Text_Font_Bitmap_Scalable;


#endif
/** Font settings of the text
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
#define EFL_TEXT_FONT_INTERFACE efl_text_font_interface_get()

EWAPI const Efl_Class *efl_text_font_interface_get(void);

/**
 * @brief Set the font family, filename and size for a given text object.
 *
 * This function allows the font name and size of a text object to be set. The
 * font string has to follow fontconfig's convention for naming fonts, as it's
 * the underlying library used to query system fonts by Evas (see the fc-list
 * command's output, on your system, to get an idea). Alternatively, youe can
 * use the full path to a font file.
 *
 * See also @ref efl_text_font_get, @ref efl_text_font_source_get.
 *
 * @param[in] obj The object.
 * @param[in] font The font family name or filename.
 * @param[in] size The font size, in points.
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_set(Eo *obj, const char *font, Efl_Font_Size size);

/**
 * @brief Retrieve the font family and size in use on a given text object.
 *
 * This function allows the font name and size of a text object to be queried.
 * Remember that the font name string is still owned by Evas and should not
 * have free() called on it by the caller of the function.
 *
 * See also @ref efl_text_font_set.
 *
 * @param[in] obj The object.
 * @param[out] font The font family name or filename.
 * @param[out] size The font size, in points.
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_get(const Eo *obj, const char **font, Efl_Font_Size *size);

/**
 * @brief Set the font (source) file to be used on a given text object.
 *
 * This function allows the font file to be explicitly set for a given text
 * object, overriding system lookup, which will first occur in the given file's
 * contents.
 *
 * See also @ref efl_text_font_set.
 *
 * @param[in] obj The object.
 * @param[in] font_source The font file's path.
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_source_set(Eo *obj, const char *font_source);

/**
 * @brief Get the font file's path which is being used on a given text object.
 *
 * See @ref efl_text_font_get for more details.
 *
 * @param[in] obj The object.
 *
 * @return The font file's path.
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI const char *efl_text_font_source_get(const Eo *obj);

/**
 * @brief Comma-separated list of font fallbacks
 *
 * Will be used in case the primary font isn't available.
 *
 * @param[in] obj The object.
 * @param[in] font_fallbacks Font name fallbacks
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_fallbacks_set(Eo *obj, const char *font_fallbacks);

/**
 * @brief Comma-separated list of font fallbacks
 *
 * Will be used in case the primary font isn't available.
 *
 * @param[in] obj The object.
 *
 * @return Font name fallbacks
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI const char *efl_text_font_fallbacks_get(const Eo *obj);

/**
 * @brief Type of weight of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_WEIGHT_NORMAL\.
 *
 * @param[in] obj The object.
 * @param[in] font_weight Font weight
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_weight_set(Eo *obj, Efl_Text_Font_Weight font_weight);

/**
 * @brief Type of weight of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_WEIGHT_NORMAL\.
 *
 * @param[in] obj The object.
 *
 * @return Font weight
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI Efl_Text_Font_Weight efl_text_font_weight_get(const Eo *obj);

/**
 * @brief Type of slant of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_SLANT_NORMAL\.
 *
 * @param[in] obj The object.
 * @param[in] style Font slant
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_slant_set(Eo *obj, Efl_Text_Font_Slant style);

/**
 * @brief Type of slant of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_SLANT_NORMAL\.
 *
 * @param[in] obj The object.
 *
 * @return Font slant
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI Efl_Text_Font_Slant efl_text_font_slant_get(const Eo *obj);

/**
 * @brief Type of width of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_WIDTH_NORMAL\.
 *
 * @param[in] obj The object.
 * @param[in] width Font width
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_width_set(Eo *obj, Efl_Text_Font_Width width);

/**
 * @brief Type of width of the displayed font
 *
 * Default is @ref EFL_TEXT_FONT_WIDTH_NORMAL\.
 *
 * @param[in] obj The object.
 *
 * @return Font width
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI Efl_Text_Font_Width efl_text_font_width_get(const Eo *obj);

/**
 * @brief Specific language of the displayed font
 *
 * This is used to lookup fonts suitable to the specified language, as well as
 * helping the font shaper backend. The language @c lang can be either a code
 * e.g "en_US", "auto" to use the system locale, or "none".
 *
 * @param[in] obj The object.
 * @param[in] lang Language
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_lang_set(Eo *obj, const char *lang);

/**
 * @brief Specific language of the displayed font
 *
 * This is used to lookup fonts suitable to the specified language, as well as
 * helping the font shaper backend. The language @c lang can be either a code
 * e.g "en_US", "auto" to use the system locale, or "none".
 *
 * @param[in] obj The object.
 *
 * @return Language
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI const char *efl_text_font_lang_get(const Eo *obj);

/**
 * @brief The bitmap fonts have fixed size glyphs for several available sizes.
 * Basically, it is not scalable. But, it needs to be scalable for some use
 * cases. (ex. colorful emoji fonts)
 *
 * Default is @ref EFL_TEXT_FONT_BITMAP_SCALABLE_NONE\.
 *
 * @param[in] obj The object.
 * @param[in] scalable Scalable
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI void efl_text_font_bitmap_scalable_set(Eo *obj, Efl_Text_Font_Bitmap_Scalable scalable);

/**
 * @brief The bitmap fonts have fixed size glyphs for several available sizes.
 * Basically, it is not scalable. But, it needs to be scalable for some use
 * cases. (ex. colorful emoji fonts)
 *
 * Default is @ref EFL_TEXT_FONT_BITMAP_SCALABLE_NONE\.
 *
 * @param[in] obj The object.
 *
 * @return Scalable
 *
 * @since 1.20
 *
 * @ingroup Efl_Text_Font
 */
EOAPI Efl_Text_Font_Bitmap_Scalable efl_text_font_bitmap_scalable_get(const Eo *obj);

#endif
