/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */

'use strict';

var _extends3 = _interopRequireDefault(require('babel-runtime/helpers/extends'));

var _classCallCheck3 = _interopRequireDefault(require('babel-runtime/helpers/classCallCheck'));

var _possibleConstructorReturn3 = _interopRequireDefault(require('babel-runtime/helpers/possibleConstructorReturn'));

var _inherits3 = _interopRequireDefault(require('babel-runtime/helpers/inherits'));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var _require = require('./RelayContainerUtils'),
    getComponentName = _require.getComponentName,
    getReactComponent = _require.getReactComponent;

var _require2 = require('./RelayContext'),
    assertRelayContext = _require2.assertRelayContext;

var _require3 = require('./ReactRelayContainerProfiler'),
    profileContainer = _require3.profileContainer;

var _require4 = require('relay-runtime'),
    Observable = _require4.Observable,
    RelayProfiler = _require4.RelayProfiler,
    isScalarAndEqual = _require4.isScalarAndEqual;

var containerContextTypes = {
  relay: require('./RelayPropTypes').Relay
};

/**
 * Composes a React component class, returning a new class that intercepts
 * props, resolving them with the provided fragments and subscribing for
 * updates.
 */
function createContainerWithFragments(Component, fragments, taggedNode) {
  var ComponentClass = getReactComponent(Component);
  var componentName = getComponentName(Component);
  var containerName = 'Relay(' + componentName + ')';

  var Container = function (_React$Component) {
    (0, _inherits3['default'])(Container, _React$Component);

    function Container(props, context) {
      (0, _classCallCheck3['default'])(this, Container);

      var _this = (0, _possibleConstructorReturn3['default'])(this, _React$Component.call(this, props, context));

      _this._handleFragmentDataUpdate = function () {
        var profiler = RelayProfiler.profile('ReactRelayRefetchContainer.handleFragmentDataUpdate');
        var resolverFromThisUpdate = _this.state.resolver;
        _this.setState(function (updatedState) {
          // If this event belongs to the current data source, update.
          // Otherwise we should ignore it.
          if (resolverFromThisUpdate === updatedState.resolver) {
            return {
              data: updatedState.resolver.resolve()
            };
          }

          return null;
        }, profiler.stop);
      };

      _this._refetch = function (refetchVariables, renderVariables, observerOrCallback, options) {
        var _assertRelayContext = assertRelayContext(_this.context.relay),
            environment = _assertRelayContext.environment,
            rootVariables = _assertRelayContext.variables;

        var fetchVariables = typeof refetchVariables === 'function' ? refetchVariables(_this._getFragmentVariables()) : refetchVariables;
        fetchVariables = (0, _extends3['default'])({}, rootVariables, fetchVariables);
        var fragmentVariables = renderVariables ? (0, _extends3['default'])({}, rootVariables, renderVariables) : fetchVariables;
        var cacheConfig = options ? { force: !!options.force } : undefined;

        var observer = typeof observerOrCallback === 'function' ? {
          // callback is not exectued on complete or unsubscribe
          // for backward compatibility
          next: observerOrCallback,
          error: observerOrCallback
        } : observerOrCallback || {};

        var _this$context$relay$e = _this.context.relay.environment.unstable_internal,
            createOperationSelector = _this$context$relay$e.createOperationSelector,
            getRequest = _this$context$relay$e.getRequest;

        var query = getRequest(taggedNode);
        var operation = createOperationSelector(query, fetchVariables);

        // TODO: T26288752 find a better way
        /* eslint-disable lint/react-state-props-mutation */
        _this.state.localVariables = fetchVariables;
        /* eslint-enable lint/react-state-props-mutation */

        // Cancel any previously running refetch.
        _this._refetchSubscription && _this._refetchSubscription.unsubscribe();

        // Declare refetchSubscription before assigning it in .start(), since
        // synchronous completion may call callbacks .subscribe() returns.
        var refetchSubscription = void 0;
        _this._getQueryFetcher().execute({
          environment: environment,
          operation: operation,
          cacheConfig: cacheConfig,
          // TODO (T26430099): Cleanup old references
          preservePreviousReferences: true
        }).mergeMap(function (response) {
          // Child containers rely on context.relay being mutated (for gDSFP).
          // TODO: T26288752 find a better way
          /* eslint-disable lint/react-state-props-mutation */
          _this.state.relayContext.environment = _this.context.relay.environment;
          _this.state.relayContext.variables = fragmentVariables;
          /* eslint-enable lint/react-state-props-mutation */
          _this.state.resolver.setVariables(fragmentVariables);
          return Observable.create(function (sink) {
            return _this.setState({ data: _this.state.resolver.resolve() }, function () {
              sink.next();
              sink.complete();
            });
          });
        })['finally'](function () {
          // Finalizing a refetch should only clear this._refetchSubscription
          // if the finizing subscription is the most recent call.
          if (_this._refetchSubscription === refetchSubscription) {
            _this._refetchSubscription = null;
          }
        }).subscribe((0, _extends3['default'])({}, observer, {
          start: function start(subscription) {
            _this._refetchSubscription = refetchSubscription = subscription;
            observer.start && observer.start(subscription);
          }
        }));

        return {
          dispose: function dispose() {
            refetchSubscription && refetchSubscription.unsubscribe();
          }
        };
      };

      _this._legacyStringishRef = require('./makeLegacyStringishComponentRef')(_this, componentName);

      var relay = assertRelayContext(context.relay);
      var createFragmentSpecResolver = relay.environment.unstable_internal.createFragmentSpecResolver;

      _this._refetchSubscription = null;
      // Do not provide a subscription/callback here.
      // It is possible for this render to be interrupted or aborted,
      // In which case the subscription would cause a leak.
      // We will add the subscription in componentDidMount().
      var resolver = createFragmentSpecResolver(relay, containerName, fragments, props);
      _this.state = {
        data: resolver.resolve(),
        localVariables: null,
        relay: relay,
        relayContext: {
          environment: relay.environment,
          variables: relay.variables
        },
        relayEnvironment: relay.environment,
        relayProp: {
          environment: relay.environment,
          refetch: _this._refetch
        },
        prevProps: _this.props,
        relayVariables: relay.variables,
        resolver: resolver
      };
      return _this;
    }

    Container.prototype.componentDidMount = function componentDidMount() {
      this._subscribeToNewResolver();
    };

    Container.prototype.componentDidUpdate = function componentDidUpdate(prevProps, prevState) {
      // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.
      if (this.state.resolver !== prevState.resolver) {
        prevState.resolver.dispose();
        this._queryFetcher && this._queryFetcher.dispose();
        this._refetchSubscription && this._refetchSubscription.unsubscribe();

        this._subscribeToNewResolver();
      }
    };

    /**
     * When new props are received, read data for the new props and add it to
     * state. Props may be the same in which case previous data can be reused.
     */


    Container.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
      // Any props change could impact the query, so we mirror props in state.
      // This is an unusual pattern, but necessary for this container usecase.
      var prevProps = prevState.prevProps;

      var relay = assertRelayContext(prevState.relay);

      var _relay$environment$un = relay.environment.unstable_internal,
          createFragmentSpecResolver = _relay$environment$un.createFragmentSpecResolver,
          getDataIDsFromObject = _relay$environment$un.getDataIDsFromObject;

      var prevIDs = getDataIDsFromObject(fragments, prevProps);
      var nextIDs = getDataIDsFromObject(fragments, nextProps);

      var resolver = prevState.resolver;

      // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.
      if (prevState.relayEnvironment !== relay.environment || prevState.relayVariables !== relay.variables || !require('fbjs/lib/areEqual')(prevIDs, nextIDs)) {
        // Child containers rely on context.relay being mutated (for gDSFP).
        var mutatedRelayContext = prevState.relayContext;
        mutatedRelayContext.environment = relay.environment;
        mutatedRelayContext.variables = relay.variables;

        // Do not provide a subscription/callback here.
        // It is possible for this render to be interrupted or aborted,
        // In which case the subscription would cause a leak.
        // We will add the subscription in componentDidUpdate().
        resolver = createFragmentSpecResolver(relay, containerName, fragments, nextProps);
        return {
          data: resolver.resolve(),
          localVariables: null,
          prevProps: nextProps,
          relayContext: mutatedRelayContext,
          relayEnvironment: relay.environment,
          relayProp: {
            environment: relay.environment,
            // refetch should never really change
            refetch: prevState.relayProp.refetch
          },
          relayVariables: relay.variables,
          resolver: resolver
        };
      } else if (!prevState.localVariables) {
        resolver.setProps(nextProps);
      }
      var data = resolver.resolve();
      if (data !== prevState.data) {
        return {
          data: data,
          prevProps: nextProps
        };
      }
      return null;
    };

    Container.prototype.componentWillUnmount = function componentWillUnmount() {
      this.state.resolver.dispose();
      this._queryFetcher && this._queryFetcher.dispose();
      this._refetchSubscription && this._refetchSubscription.unsubscribe();
    };

    Container.prototype.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState, nextContext) {
      // Short-circuit if any Relay-related data has changed
      if (nextState.data !== this.state.data || nextState.relayProp !== this.state.relayProp) {
        return true;
      }
      // Otherwise, for convenience short-circuit if all non-Relay props
      // are scalar and equal
      var keys = Object.keys(nextProps);
      for (var ii = 0; ii < keys.length; ii++) {
        var _key = keys[ii];
        if (_key === 'relay') {
          if (nextState.relayEnvironment !== this.state.relayEnvironment || nextState.relayVariables !== this.state.relayVariables) {
            return true;
          }
        } else {
          if (!fragments.hasOwnProperty(_key) && !isScalarAndEqual(nextProps[_key], this.props[_key])) {
            return true;
          }
        }
      }
      return false;
    };

    Container.prototype._subscribeToNewResolver = function _subscribeToNewResolver() {
      var _state = this.state,
          data = _state.data,
          resolver = _state.resolver;

      // Event listeners are only safe to add during the commit phase,
      // So they won't leak if render is interrupted or errors.

      resolver.setCallback(this._handleFragmentDataUpdate);

      // External values could change between render and commit.
      // Check for this case, even though it requires an extra store read.
      var maybeNewData = resolver.resolve();
      if (data !== maybeNewData) {
        this.setState({ data: maybeNewData });
      }
    };

    /**
     * Render new data for the existing props/context.
     */


    Container.prototype._getFragmentVariables = function _getFragmentVariables() {
      var getVariablesFromObject = this.context.relay.environment.unstable_internal.getVariablesFromObject;

      return getVariablesFromObject(this.context.relay.variables, fragments, this.props);
    };

    Container.prototype._getQueryFetcher = function _getQueryFetcher() {
      if (!this._queryFetcher) {
        this._queryFetcher = new (require('./ReactRelayQueryFetcher'))();
      }
      return this._queryFetcher;
    };

    Container.prototype.getChildContext = function getChildContext() {
      return { relay: this.state.relayContext };
    };

    Container.prototype.render = function render() {
      if (ComponentClass) {
        return require('react').createElement(ComponentClass, (0, _extends3['default'])({}, this.props, this.state.data, {
          // @TODO (T28161354) Remove the string ref fallback
          ref: this.props.componentRef || this._legacyStringishRef,
          relay: this.state.relayProp
        }));
      } else {
        // Stateless functional, doesn't support `ref`
        return require('react').createElement(Component, (0, _extends3['default'])({}, this.props, this.state.data, {
          relay: this.state.relayProp
        }));
      }
    };

    // @TODO (T28161354) Remove this once string ref usage is gone.


    return Container;
  }(require('react').Component);

  Container.displayName = containerName;
  Container.contextTypes = containerContextTypes;

  profileContainer(Container, 'ReactRelayRefetchContainer');

  return Container;
}

/**
 * Wrap the basic `createContainer()` function with logic to adapt to the
 * `context.relay.environment` in which it is rendered. Specifically, the
 * extraction of the environment-specific version of fragments in the
 * `fragmentSpec` is memoized once per environment, rather than once per
 * instance of the container constructed/rendered.
 */
function createContainer(Component, fragmentSpec, taggedNode) {
  var Container = require('./buildReactRelayContainer')(Component, fragmentSpec, function (ComponentClass, fragments) {
    return createContainerWithFragments(ComponentClass, fragments, taggedNode);
  });
  /* $FlowFixMe(>=0.53.0) This comment suppresses an error
   * when upgrading Flow's support for React. Common errors found when
   * upgrading Flow's React support are documented at
   * https://fburl.com/eq7bs81w */
  Container.childContextTypes = containerContextTypes;
  return Container;
}

module.exports = { createContainer: createContainer, createContainerWithFragments: createContainerWithFragments };