'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

exports.buildFilePatch = buildFilePatch;
exports.buildMultiFilePatch = buildMultiFilePatch;

var _atom = require('atom');

var _hunk = require('./hunk');

var _hunk2 = _interopRequireDefault(_hunk);

var _file = require('./file');

var _file2 = _interopRequireDefault(_file);

var _patch = require('./patch');

var _patch2 = _interopRequireDefault(_patch);

var _region = require('./region');

var _filePatch = require('./file-patch');

var _filePatch2 = _interopRequireDefault(_filePatch);

var _multiFilePatch = require('./multi-file-patch');

var _multiFilePatch2 = _interopRequireDefault(_multiFilePatch);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function buildFilePatch(diffs) {
  const layeredBuffer = initializeBuffer();

  let filePatch;
  if (diffs.length === 0) {
    filePatch = emptyDiffFilePatch();
  } else if (diffs.length === 1) {
    filePatch = singleDiffFilePatch(diffs[0], layeredBuffer);
  } else if (diffs.length === 2) {
    filePatch = dualDiffFilePatch(diffs[0], diffs[1], layeredBuffer);
  } else {
    throw new Error(`Unexpected number of diffs: ${diffs.length}`);
  }

  return new _multiFilePatch2.default(_extends({ filePatches: [filePatch] }, layeredBuffer));
}

function buildMultiFilePatch(diffs) {
  const layeredBuffer = initializeBuffer();
  const byPath = new Map();
  const actions = [];

  let index = 0;
  for (const diff of diffs) {
    const thePath = diff.oldPath || diff.newPath;

    if (diff.status === 'added' || diff.status === 'deleted') {
      // Potential paired diff. Either a symlink deletion + content addition or a symlink addition +
      // content deletion.
      const otherHalf = byPath.get(thePath);
      if (otherHalf) {
        // The second half. Complete the paired diff, or fail if they have unexpected statuses or modes.
        const [otherDiff, otherIndex] = otherHalf;
        actions[otherIndex] = () => dualDiffFilePatch(diff, otherDiff, layeredBuffer);
        byPath.delete(thePath);
      } else {
        // The first half we've seen.
        byPath.set(thePath, [diff, index]);
        index++;
      }
    } else {
      actions[index] = () => singleDiffFilePatch(diff, layeredBuffer);
      index++;
    }
  }

  // Populate unpaired diffs that looked like they could be part of a pair, but weren't.
  for (const [unpairedDiff, originalIndex] of byPath.values()) {
    actions[originalIndex] = () => singleDiffFilePatch(unpairedDiff, layeredBuffer);
  }

  const filePatches = actions.map(action => action());

  // Fix markers for patches with no hunks.
  // Head position was moved everytime lines were appended.
  filePatches.forEach(filePatch => {
    if (filePatch.getHunks().length === 0) {
      const marker = filePatch.getMarker();
      marker.setHeadPosition(marker.getTailPosition());
    }
  });

  return new _multiFilePatch2.default(_extends({ filePatches }, layeredBuffer));
}

function emptyDiffFilePatch() {
  return _filePatch2.default.createNull();
}

function singleDiffFilePatch(diff, layeredBuffer) {
  const wasSymlink = diff.oldMode === _file2.default.modes.SYMLINK;
  const isSymlink = diff.newMode === _file2.default.modes.SYMLINK;

  const [hunks, patchMarker] = buildHunks(diff, layeredBuffer);

  let oldSymlink = null;
  let newSymlink = null;
  if (wasSymlink && !isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (!wasSymlink && isSymlink) {
    newSymlink = diff.hunks[0].lines[0].slice(1);
  } else if (wasSymlink && isSymlink) {
    oldSymlink = diff.hunks[0].lines[0].slice(1);
    newSymlink = diff.hunks[0].lines[2].slice(1);
  }

  const oldFile = diff.oldPath !== null || diff.oldMode !== null ? new _file2.default({ path: diff.oldPath, mode: diff.oldMode, symlink: oldSymlink }) : _file.nullFile;
  const newFile = diff.newPath !== null || diff.newMode !== null ? new _file2.default({ path: diff.newPath, mode: diff.newMode, symlink: newSymlink }) : _file.nullFile;
  const patch = new _patch2.default({ status: diff.status, hunks, marker: patchMarker, buffer: layeredBuffer.buffer });

  return new _filePatch2.default(oldFile, newFile, patch);
}

function dualDiffFilePatch(diff1, diff2, layeredBuffer) {
  let modeChangeDiff, contentChangeDiff;
  if (diff1.oldMode === _file2.default.modes.SYMLINK || diff1.newMode === _file2.default.modes.SYMLINK) {
    modeChangeDiff = diff1;
    contentChangeDiff = diff2;
  } else {
    modeChangeDiff = diff2;
    contentChangeDiff = diff1;
  }

  const [hunks, patchMarker] = buildHunks(contentChangeDiff, layeredBuffer);
  const filePath = contentChangeDiff.oldPath || contentChangeDiff.newPath;
  const symlink = modeChangeDiff.hunks[0].lines[0].slice(1);

  let status;
  let oldMode, newMode;
  let oldSymlink = null;
  let newSymlink = null;
  if (modeChangeDiff.status === 'added') {
    // contents were deleted and replaced with symlink
    status = 'deleted';
    oldMode = contentChangeDiff.oldMode;
    newMode = modeChangeDiff.newMode;
    newSymlink = symlink;
  } else if (modeChangeDiff.status === 'deleted') {
    // contents were added after symlink was deleted
    status = 'added';
    oldMode = modeChangeDiff.oldMode;
    oldSymlink = symlink;
    newMode = contentChangeDiff.newMode;
  } else {
    throw new Error(`Invalid mode change diff status: ${modeChangeDiff.status}`);
  }

  const oldFile = new _file2.default({ path: filePath, mode: oldMode, symlink: oldSymlink });
  const newFile = new _file2.default({ path: filePath, mode: newMode, symlink: newSymlink });
  const patch = new _patch2.default({ status, hunks, marker: patchMarker, buffer: layeredBuffer.buffer });

  return new _filePatch2.default(oldFile, newFile, patch);
}

const CHANGEKIND = {
  '+': _region.Addition,
  '-': _region.Deletion,
  ' ': _region.Unchanged,
  '\\': _region.NoNewline
};

function initializeBuffer() {
  const buffer = new _atom.TextBuffer();
  buffer.retain();

  const layers = ['patch', 'hunk', 'unchanged', 'addition', 'deletion', 'noNewline'].reduce((obj, key) => {
    obj[key] = buffer.addMarkerLayer();
    return obj;
  }, {});

  return { buffer, layers };
}

function buildHunks(diff, { buffer, layers }) {
  const layersByKind = new Map([[_region.Unchanged, layers.unchanged], [_region.Addition, layers.addition], [_region.Deletion, layers.deletion], [_region.NoNewline, layers.noNewline]]);
  const hunks = [];

  const patchStartRow = buffer.getLastRow();
  let bufferRow = patchStartRow;
  let nextLineLength = 0;

  if (diff.hunks.length === 0) {
    const patchMarker = layers.patch.markPosition([patchStartRow, 0], { invalidate: 'never', exclusive: false });

    return [hunks, patchMarker];
  }

  for (const hunkData of diff.hunks) {
    const bufferStartRow = bufferRow;

    const regions = [];

    let LastChangeKind = null;
    let currentRangeStart = bufferRow;
    let lastLineLength = 0;

    const finishCurrentRange = () => {
      if (currentRangeStart === bufferRow) {
        return;
      }

      regions.push(new LastChangeKind(layersByKind.get(LastChangeKind).markRange([[currentRangeStart, 0], [bufferRow - 1, lastLineLength]], { invalidate: 'never', exclusive: false })));
      currentRangeStart = bufferRow;
    };

    for (const lineText of hunkData.lines) {
      const bufferLine = lineText.slice(1) + '\n';
      nextLineLength = lineText.length - 1;
      buffer.append(bufferLine);

      const ChangeKind = CHANGEKIND[lineText[0]];
      if (ChangeKind === undefined) {
        throw new Error(`Unknown diff status character: "${lineText[0]}"`);
      }

      if (ChangeKind !== LastChangeKind) {
        finishCurrentRange();
      }

      LastChangeKind = ChangeKind;
      bufferRow++;
      lastLineLength = nextLineLength;
    }
    finishCurrentRange();

    hunks.push(new _hunk2.default({
      oldStartRow: hunkData.oldStartLine,
      newStartRow: hunkData.newStartLine,
      oldRowCount: hunkData.oldLineCount,
      newRowCount: hunkData.newLineCount,
      sectionHeading: hunkData.heading,
      marker: layers.hunk.markRange([[bufferStartRow, 0], [bufferRow - 1, nextLineLength]], { invalidate: 'never', exclusive: false }),
      regions
    }));
  }

  const patchMarker = layers.patch.markRange([[patchStartRow, 0], [bufferRow - 1, nextLineLength]], { invalidate: 'never', exclusive: false });

  return [hunks, patchMarker];
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImJ1aWxkZXIuanMiXSwibmFtZXMiOlsiYnVpbGRGaWxlUGF0Y2giLCJidWlsZE11bHRpRmlsZVBhdGNoIiwiZGlmZnMiLCJsYXllcmVkQnVmZmVyIiwiaW5pdGlhbGl6ZUJ1ZmZlciIsImZpbGVQYXRjaCIsImxlbmd0aCIsImVtcHR5RGlmZkZpbGVQYXRjaCIsInNpbmdsZURpZmZGaWxlUGF0Y2giLCJkdWFsRGlmZkZpbGVQYXRjaCIsIkVycm9yIiwiTXVsdGlGaWxlUGF0Y2giLCJmaWxlUGF0Y2hlcyIsImJ5UGF0aCIsIk1hcCIsImFjdGlvbnMiLCJpbmRleCIsImRpZmYiLCJ0aGVQYXRoIiwib2xkUGF0aCIsIm5ld1BhdGgiLCJzdGF0dXMiLCJvdGhlckhhbGYiLCJnZXQiLCJvdGhlckRpZmYiLCJvdGhlckluZGV4IiwiZGVsZXRlIiwic2V0IiwidW5wYWlyZWREaWZmIiwib3JpZ2luYWxJbmRleCIsInZhbHVlcyIsIm1hcCIsImFjdGlvbiIsImZvckVhY2giLCJnZXRIdW5rcyIsIm1hcmtlciIsImdldE1hcmtlciIsInNldEhlYWRQb3NpdGlvbiIsImdldFRhaWxQb3NpdGlvbiIsIkZpbGVQYXRjaCIsImNyZWF0ZU51bGwiLCJ3YXNTeW1saW5rIiwib2xkTW9kZSIsIkZpbGUiLCJtb2RlcyIsIlNZTUxJTksiLCJpc1N5bWxpbmsiLCJuZXdNb2RlIiwiaHVua3MiLCJwYXRjaE1hcmtlciIsImJ1aWxkSHVua3MiLCJvbGRTeW1saW5rIiwibmV3U3ltbGluayIsImxpbmVzIiwic2xpY2UiLCJvbGRGaWxlIiwicGF0aCIsIm1vZGUiLCJzeW1saW5rIiwibnVsbEZpbGUiLCJuZXdGaWxlIiwicGF0Y2giLCJQYXRjaCIsImJ1ZmZlciIsImRpZmYxIiwiZGlmZjIiLCJtb2RlQ2hhbmdlRGlmZiIsImNvbnRlbnRDaGFuZ2VEaWZmIiwiZmlsZVBhdGgiLCJDSEFOR0VLSU5EIiwiQWRkaXRpb24iLCJEZWxldGlvbiIsIlVuY2hhbmdlZCIsIk5vTmV3bGluZSIsIlRleHRCdWZmZXIiLCJyZXRhaW4iLCJsYXllcnMiLCJyZWR1Y2UiLCJvYmoiLCJrZXkiLCJhZGRNYXJrZXJMYXllciIsImxheWVyc0J5S2luZCIsInVuY2hhbmdlZCIsImFkZGl0aW9uIiwiZGVsZXRpb24iLCJub05ld2xpbmUiLCJwYXRjaFN0YXJ0Um93IiwiZ2V0TGFzdFJvdyIsImJ1ZmZlclJvdyIsIm5leHRMaW5lTGVuZ3RoIiwibWFya1Bvc2l0aW9uIiwiaW52YWxpZGF0ZSIsImV4Y2x1c2l2ZSIsImh1bmtEYXRhIiwiYnVmZmVyU3RhcnRSb3ciLCJyZWdpb25zIiwiTGFzdENoYW5nZUtpbmQiLCJjdXJyZW50UmFuZ2VTdGFydCIsImxhc3RMaW5lTGVuZ3RoIiwiZmluaXNoQ3VycmVudFJhbmdlIiwicHVzaCIsIm1hcmtSYW5nZSIsImxpbmVUZXh0IiwiYnVmZmVyTGluZSIsImFwcGVuZCIsIkNoYW5nZUtpbmQiLCJ1bmRlZmluZWQiLCJIdW5rIiwib2xkU3RhcnRSb3ciLCJvbGRTdGFydExpbmUiLCJuZXdTdGFydFJvdyIsIm5ld1N0YXJ0TGluZSIsIm9sZFJvd0NvdW50Iiwib2xkTGluZUNvdW50IiwibmV3Um93Q291bnQiLCJuZXdMaW5lQ291bnQiLCJzZWN0aW9uSGVhZGluZyIsImhlYWRpbmciLCJodW5rIl0sIm1hcHBpbmdzIjoiOzs7Ozs7OztRQVNnQkEsYyxHQUFBQSxjO1FBaUJBQyxtQixHQUFBQSxtQjs7QUExQmhCOztBQUVBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOztBQUNBOzs7O0FBQ0E7Ozs7OztBQUVPLFNBQVNELGNBQVQsQ0FBd0JFLEtBQXhCLEVBQStCO0FBQ3BDLFFBQU1DLGdCQUFnQkMsa0JBQXRCOztBQUVBLE1BQUlDLFNBQUo7QUFDQSxNQUFJSCxNQUFNSSxNQUFOLEtBQWlCLENBQXJCLEVBQXdCO0FBQ3RCRCxnQkFBWUUsb0JBQVo7QUFDRCxHQUZELE1BRU8sSUFBSUwsTUFBTUksTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUM3QkQsZ0JBQVlHLG9CQUFvQk4sTUFBTSxDQUFOLENBQXBCLEVBQThCQyxhQUE5QixDQUFaO0FBQ0QsR0FGTSxNQUVBLElBQUlELE1BQU1JLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFDN0JELGdCQUFZSSxrQkFBa0JQLE1BQU0sQ0FBTixDQUFsQixFQUE0QkEsTUFBTSxDQUFOLENBQTVCLEVBQXNDQyxhQUF0QyxDQUFaO0FBQ0QsR0FGTSxNQUVBO0FBQ0wsVUFBTSxJQUFJTyxLQUFKLENBQVcsK0JBQThCUixNQUFNSSxNQUFPLEVBQXRELENBQU47QUFDRDs7QUFFRCxTQUFPLElBQUlLLHdCQUFKLFlBQW9CQyxhQUFhLENBQUNQLFNBQUQsQ0FBakMsSUFBaURGLGFBQWpELEVBQVA7QUFDRDs7QUFFTSxTQUFTRixtQkFBVCxDQUE2QkMsS0FBN0IsRUFBb0M7QUFDekMsUUFBTUMsZ0JBQWdCQyxrQkFBdEI7QUFDQSxRQUFNUyxTQUFTLElBQUlDLEdBQUosRUFBZjtBQUNBLFFBQU1DLFVBQVUsRUFBaEI7O0FBRUEsTUFBSUMsUUFBUSxDQUFaO0FBQ0EsT0FBSyxNQUFNQyxJQUFYLElBQW1CZixLQUFuQixFQUEwQjtBQUN4QixVQUFNZ0IsVUFBVUQsS0FBS0UsT0FBTCxJQUFnQkYsS0FBS0csT0FBckM7O0FBRUEsUUFBSUgsS0FBS0ksTUFBTCxLQUFnQixPQUFoQixJQUEyQkosS0FBS0ksTUFBTCxLQUFnQixTQUEvQyxFQUEwRDtBQUN4RDtBQUNBO0FBQ0EsWUFBTUMsWUFBWVQsT0FBT1UsR0FBUCxDQUFXTCxPQUFYLENBQWxCO0FBQ0EsVUFBSUksU0FBSixFQUFlO0FBQ2I7QUFDQSxjQUFNLENBQUNFLFNBQUQsRUFBWUMsVUFBWixJQUEwQkgsU0FBaEM7QUFDQVAsZ0JBQVFVLFVBQVIsSUFBc0IsTUFBTWhCLGtCQUFrQlEsSUFBbEIsRUFBd0JPLFNBQXhCLEVBQW1DckIsYUFBbkMsQ0FBNUI7QUFDQVUsZUFBT2EsTUFBUCxDQUFjUixPQUFkO0FBQ0QsT0FMRCxNQUtPO0FBQ0w7QUFDQUwsZUFBT2MsR0FBUCxDQUFXVCxPQUFYLEVBQW9CLENBQUNELElBQUQsRUFBT0QsS0FBUCxDQUFwQjtBQUNBQTtBQUNEO0FBQ0YsS0FkRCxNQWNPO0FBQ0xELGNBQVFDLEtBQVIsSUFBaUIsTUFBTVIsb0JBQW9CUyxJQUFwQixFQUEwQmQsYUFBMUIsQ0FBdkI7QUFDQWE7QUFDRDtBQUNGOztBQUVEO0FBQ0EsT0FBSyxNQUFNLENBQUNZLFlBQUQsRUFBZUMsYUFBZixDQUFYLElBQTRDaEIsT0FBT2lCLE1BQVAsRUFBNUMsRUFBNkQ7QUFDM0RmLFlBQVFjLGFBQVIsSUFBeUIsTUFBTXJCLG9CQUFvQm9CLFlBQXBCLEVBQWtDekIsYUFBbEMsQ0FBL0I7QUFDRDs7QUFFRCxRQUFNUyxjQUFjRyxRQUFRZ0IsR0FBUixDQUFZQyxVQUFVQSxRQUF0QixDQUFwQjs7QUFFQTtBQUNBO0FBQ0FwQixjQUFZcUIsT0FBWixDQUFvQjVCLGFBQWE7QUFDL0IsUUFBSUEsVUFBVTZCLFFBQVYsR0FBcUI1QixNQUFyQixLQUFnQyxDQUFwQyxFQUF1QztBQUNyQyxZQUFNNkIsU0FBUzlCLFVBQVUrQixTQUFWLEVBQWY7QUFDQUQsYUFBT0UsZUFBUCxDQUF1QkYsT0FBT0csZUFBUCxFQUF2QjtBQUNEO0FBQ0YsR0FMRDs7QUFPQSxTQUFPLElBQUkzQix3QkFBSixZQUFvQkMsV0FBcEIsSUFBb0NULGFBQXBDLEVBQVA7QUFDRDs7QUFFRCxTQUFTSSxrQkFBVCxHQUE4QjtBQUM1QixTQUFPZ0Msb0JBQVVDLFVBQVYsRUFBUDtBQUNEOztBQUVELFNBQVNoQyxtQkFBVCxDQUE2QlMsSUFBN0IsRUFBbUNkLGFBQW5DLEVBQWtEO0FBQ2hELFFBQU1zQyxhQUFheEIsS0FBS3lCLE9BQUwsS0FBaUJDLGVBQUtDLEtBQUwsQ0FBV0MsT0FBL0M7QUFDQSxRQUFNQyxZQUFZN0IsS0FBSzhCLE9BQUwsS0FBaUJKLGVBQUtDLEtBQUwsQ0FBV0MsT0FBOUM7O0FBRUEsUUFBTSxDQUFDRyxLQUFELEVBQVFDLFdBQVIsSUFBdUJDLFdBQVdqQyxJQUFYLEVBQWlCZCxhQUFqQixDQUE3Qjs7QUFFQSxNQUFJZ0QsYUFBYSxJQUFqQjtBQUNBLE1BQUlDLGFBQWEsSUFBakI7QUFDQSxNQUFJWCxjQUFjLENBQUNLLFNBQW5CLEVBQThCO0FBQzVCSyxpQkFBYWxDLEtBQUsrQixLQUFMLENBQVcsQ0FBWCxFQUFjSyxLQUFkLENBQW9CLENBQXBCLEVBQXVCQyxLQUF2QixDQUE2QixDQUE3QixDQUFiO0FBQ0QsR0FGRCxNQUVPLElBQUksQ0FBQ2IsVUFBRCxJQUFlSyxTQUFuQixFQUE4QjtBQUNuQ00saUJBQWFuQyxLQUFLK0IsS0FBTCxDQUFXLENBQVgsRUFBY0ssS0FBZCxDQUFvQixDQUFwQixFQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBYjtBQUNELEdBRk0sTUFFQSxJQUFJYixjQUFjSyxTQUFsQixFQUE2QjtBQUNsQ0ssaUJBQWFsQyxLQUFLK0IsS0FBTCxDQUFXLENBQVgsRUFBY0ssS0FBZCxDQUFvQixDQUFwQixFQUF1QkMsS0FBdkIsQ0FBNkIsQ0FBN0IsQ0FBYjtBQUNBRixpQkFBYW5DLEtBQUsrQixLQUFMLENBQVcsQ0FBWCxFQUFjSyxLQUFkLENBQW9CLENBQXBCLEVBQXVCQyxLQUF2QixDQUE2QixDQUE3QixDQUFiO0FBQ0Q7O0FBRUQsUUFBTUMsVUFBVXRDLEtBQUtFLE9BQUwsS0FBaUIsSUFBakIsSUFBeUJGLEtBQUt5QixPQUFMLEtBQWlCLElBQTFDLEdBQ1osSUFBSUMsY0FBSixDQUFTLEVBQUNhLE1BQU12QyxLQUFLRSxPQUFaLEVBQXFCc0MsTUFBTXhDLEtBQUt5QixPQUFoQyxFQUF5Q2dCLFNBQVNQLFVBQWxELEVBQVQsQ0FEWSxHQUVaUSxjQUZKO0FBR0EsUUFBTUMsVUFBVTNDLEtBQUtHLE9BQUwsS0FBaUIsSUFBakIsSUFBeUJILEtBQUs4QixPQUFMLEtBQWlCLElBQTFDLEdBQ1osSUFBSUosY0FBSixDQUFTLEVBQUNhLE1BQU12QyxLQUFLRyxPQUFaLEVBQXFCcUMsTUFBTXhDLEtBQUs4QixPQUFoQyxFQUF5Q1csU0FBU04sVUFBbEQsRUFBVCxDQURZLEdBRVpPLGNBRko7QUFHQSxRQUFNRSxRQUFRLElBQUlDLGVBQUosQ0FBVSxFQUFDekMsUUFBUUosS0FBS0ksTUFBZCxFQUFzQjJCLEtBQXRCLEVBQTZCYixRQUFRYyxXQUFyQyxFQUFrRGMsUUFBUTVELGNBQWM0RCxNQUF4RSxFQUFWLENBQWQ7O0FBRUEsU0FBTyxJQUFJeEIsbUJBQUosQ0FBY2dCLE9BQWQsRUFBdUJLLE9BQXZCLEVBQWdDQyxLQUFoQyxDQUFQO0FBQ0Q7O0FBRUQsU0FBU3BELGlCQUFULENBQTJCdUQsS0FBM0IsRUFBa0NDLEtBQWxDLEVBQXlDOUQsYUFBekMsRUFBd0Q7QUFDdEQsTUFBSStELGNBQUosRUFBb0JDLGlCQUFwQjtBQUNBLE1BQUlILE1BQU10QixPQUFOLEtBQWtCQyxlQUFLQyxLQUFMLENBQVdDLE9BQTdCLElBQXdDbUIsTUFBTWpCLE9BQU4sS0FBa0JKLGVBQUtDLEtBQUwsQ0FBV0MsT0FBekUsRUFBa0Y7QUFDaEZxQixxQkFBaUJGLEtBQWpCO0FBQ0FHLHdCQUFvQkYsS0FBcEI7QUFDRCxHQUhELE1BR087QUFDTEMscUJBQWlCRCxLQUFqQjtBQUNBRSx3QkFBb0JILEtBQXBCO0FBQ0Q7O0FBRUQsUUFBTSxDQUFDaEIsS0FBRCxFQUFRQyxXQUFSLElBQXVCQyxXQUFXaUIsaUJBQVgsRUFBOEJoRSxhQUE5QixDQUE3QjtBQUNBLFFBQU1pRSxXQUFXRCxrQkFBa0JoRCxPQUFsQixJQUE2QmdELGtCQUFrQi9DLE9BQWhFO0FBQ0EsUUFBTXNDLFVBQVVRLGVBQWVsQixLQUFmLENBQXFCLENBQXJCLEVBQXdCSyxLQUF4QixDQUE4QixDQUE5QixFQUFpQ0MsS0FBakMsQ0FBdUMsQ0FBdkMsQ0FBaEI7O0FBRUEsTUFBSWpDLE1BQUo7QUFDQSxNQUFJcUIsT0FBSixFQUFhSyxPQUFiO0FBQ0EsTUFBSUksYUFBYSxJQUFqQjtBQUNBLE1BQUlDLGFBQWEsSUFBakI7QUFDQSxNQUFJYyxlQUFlN0MsTUFBZixLQUEwQixPQUE5QixFQUF1QztBQUNyQztBQUNBQSxhQUFTLFNBQVQ7QUFDQXFCLGNBQVV5QixrQkFBa0J6QixPQUE1QjtBQUNBSyxjQUFVbUIsZUFBZW5CLE9BQXpCO0FBQ0FLLGlCQUFhTSxPQUFiO0FBQ0QsR0FORCxNQU1PLElBQUlRLGVBQWU3QyxNQUFmLEtBQTBCLFNBQTlCLEVBQXlDO0FBQzlDO0FBQ0FBLGFBQVMsT0FBVDtBQUNBcUIsY0FBVXdCLGVBQWV4QixPQUF6QjtBQUNBUyxpQkFBYU8sT0FBYjtBQUNBWCxjQUFVb0Isa0JBQWtCcEIsT0FBNUI7QUFDRCxHQU5NLE1BTUE7QUFDTCxVQUFNLElBQUlyQyxLQUFKLENBQVcsb0NBQW1Dd0QsZUFBZTdDLE1BQU8sRUFBcEUsQ0FBTjtBQUNEOztBQUVELFFBQU1rQyxVQUFVLElBQUlaLGNBQUosQ0FBUyxFQUFDYSxNQUFNWSxRQUFQLEVBQWlCWCxNQUFNZixPQUF2QixFQUFnQ2dCLFNBQVNQLFVBQXpDLEVBQVQsQ0FBaEI7QUFDQSxRQUFNUyxVQUFVLElBQUlqQixjQUFKLENBQVMsRUFBQ2EsTUFBTVksUUFBUCxFQUFpQlgsTUFBTVYsT0FBdkIsRUFBZ0NXLFNBQVNOLFVBQXpDLEVBQVQsQ0FBaEI7QUFDQSxRQUFNUyxRQUFRLElBQUlDLGVBQUosQ0FBVSxFQUFDekMsTUFBRCxFQUFTMkIsS0FBVCxFQUFnQmIsUUFBUWMsV0FBeEIsRUFBcUNjLFFBQVE1RCxjQUFjNEQsTUFBM0QsRUFBVixDQUFkOztBQUVBLFNBQU8sSUFBSXhCLG1CQUFKLENBQWNnQixPQUFkLEVBQXVCSyxPQUF2QixFQUFnQ0MsS0FBaEMsQ0FBUDtBQUNEOztBQUVELE1BQU1RLGFBQWE7QUFDakIsT0FBS0MsZ0JBRFk7QUFFakIsT0FBS0MsZ0JBRlk7QUFHakIsT0FBS0MsaUJBSFk7QUFJakIsUUFBTUM7QUFKVyxDQUFuQjs7QUFPQSxTQUFTckUsZ0JBQVQsR0FBNEI7QUFDMUIsUUFBTTJELFNBQVMsSUFBSVcsZ0JBQUosRUFBZjtBQUNBWCxTQUFPWSxNQUFQOztBQUVBLFFBQU1DLFNBQVMsQ0FBQyxPQUFELEVBQVUsTUFBVixFQUFrQixXQUFsQixFQUErQixVQUEvQixFQUEyQyxVQUEzQyxFQUF1RCxXQUF2RCxFQUFvRUMsTUFBcEUsQ0FBMkUsQ0FBQ0MsR0FBRCxFQUFNQyxHQUFOLEtBQWM7QUFDdEdELFFBQUlDLEdBQUosSUFBV2hCLE9BQU9pQixjQUFQLEVBQVg7QUFDQSxXQUFPRixHQUFQO0FBQ0QsR0FIYyxFQUdaLEVBSFksQ0FBZjs7QUFLQSxTQUFPLEVBQUNmLE1BQUQsRUFBU2EsTUFBVCxFQUFQO0FBQ0Q7O0FBRUQsU0FBUzFCLFVBQVQsQ0FBb0JqQyxJQUFwQixFQUEwQixFQUFDOEMsTUFBRCxFQUFTYSxNQUFULEVBQTFCLEVBQTRDO0FBQzFDLFFBQU1LLGVBQWUsSUFBSW5FLEdBQUosQ0FBUSxDQUMzQixDQUFDMEQsaUJBQUQsRUFBWUksT0FBT00sU0FBbkIsQ0FEMkIsRUFFM0IsQ0FBQ1osZ0JBQUQsRUFBV00sT0FBT08sUUFBbEIsQ0FGMkIsRUFHM0IsQ0FBQ1osZ0JBQUQsRUFBV0ssT0FBT1EsUUFBbEIsQ0FIMkIsRUFJM0IsQ0FBQ1gsaUJBQUQsRUFBWUcsT0FBT1MsU0FBbkIsQ0FKMkIsQ0FBUixDQUFyQjtBQU1BLFFBQU1yQyxRQUFRLEVBQWQ7O0FBRUEsUUFBTXNDLGdCQUFnQnZCLE9BQU93QixVQUFQLEVBQXRCO0FBQ0EsTUFBSUMsWUFBWUYsYUFBaEI7QUFDQSxNQUFJRyxpQkFBaUIsQ0FBckI7O0FBRUEsTUFBSXhFLEtBQUsrQixLQUFMLENBQVcxQyxNQUFYLEtBQXNCLENBQTFCLEVBQTZCO0FBQzNCLFVBQU0yQyxjQUFjMkIsT0FBT2YsS0FBUCxDQUFhNkIsWUFBYixDQUNsQixDQUFDSixhQUFELEVBQWdCLENBQWhCLENBRGtCLEVBRWxCLEVBQUNLLFlBQVksT0FBYixFQUFzQkMsV0FBVyxLQUFqQyxFQUZrQixDQUFwQjs7QUFLQSxXQUFPLENBQUM1QyxLQUFELEVBQVFDLFdBQVIsQ0FBUDtBQUNEOztBQUVELE9BQUssTUFBTTRDLFFBQVgsSUFBdUI1RSxLQUFLK0IsS0FBNUIsRUFBbUM7QUFDakMsVUFBTThDLGlCQUFpQk4sU0FBdkI7O0FBRUEsVUFBTU8sVUFBVSxFQUFoQjs7QUFFQSxRQUFJQyxpQkFBaUIsSUFBckI7QUFDQSxRQUFJQyxvQkFBb0JULFNBQXhCO0FBQ0EsUUFBSVUsaUJBQWlCLENBQXJCOztBQUVBLFVBQU1DLHFCQUFxQixNQUFNO0FBQy9CLFVBQUlGLHNCQUFzQlQsU0FBMUIsRUFBcUM7QUFDbkM7QUFDRDs7QUFFRE8sY0FBUUssSUFBUixDQUNFLElBQUlKLGNBQUosQ0FDRWYsYUFBYTFELEdBQWIsQ0FBaUJ5RSxjQUFqQixFQUFpQ0ssU0FBakMsQ0FDRSxDQUFDLENBQUNKLGlCQUFELEVBQW9CLENBQXBCLENBQUQsRUFBeUIsQ0FBQ1QsWUFBWSxDQUFiLEVBQWdCVSxjQUFoQixDQUF6QixDQURGLEVBRUUsRUFBQ1AsWUFBWSxPQUFiLEVBQXNCQyxXQUFXLEtBQWpDLEVBRkYsQ0FERixDQURGO0FBUUFLLDBCQUFvQlQsU0FBcEI7QUFDRCxLQWREOztBQWdCQSxTQUFLLE1BQU1jLFFBQVgsSUFBdUJULFNBQVN4QyxLQUFoQyxFQUF1QztBQUNyQyxZQUFNa0QsYUFBYUQsU0FBU2hELEtBQVQsQ0FBZSxDQUFmLElBQW9CLElBQXZDO0FBQ0FtQyx1QkFBaUJhLFNBQVNoRyxNQUFULEdBQWtCLENBQW5DO0FBQ0F5RCxhQUFPeUMsTUFBUCxDQUFjRCxVQUFkOztBQUVBLFlBQU1FLGFBQWFwQyxXQUFXaUMsU0FBUyxDQUFULENBQVgsQ0FBbkI7QUFDQSxVQUFJRyxlQUFlQyxTQUFuQixFQUE4QjtBQUM1QixjQUFNLElBQUloRyxLQUFKLENBQVcsbUNBQWtDNEYsU0FBUyxDQUFULENBQVksR0FBekQsQ0FBTjtBQUNEOztBQUVELFVBQUlHLGVBQWVULGNBQW5CLEVBQW1DO0FBQ2pDRztBQUNEOztBQUVESCx1QkFBaUJTLFVBQWpCO0FBQ0FqQjtBQUNBVSx1QkFBaUJULGNBQWpCO0FBQ0Q7QUFDRFU7O0FBRUFuRCxVQUFNb0QsSUFBTixDQUFXLElBQUlPLGNBQUosQ0FBUztBQUNsQkMsbUJBQWFmLFNBQVNnQixZQURKO0FBRWxCQyxtQkFBYWpCLFNBQVNrQixZQUZKO0FBR2xCQyxtQkFBYW5CLFNBQVNvQixZQUhKO0FBSWxCQyxtQkFBYXJCLFNBQVNzQixZQUpKO0FBS2xCQyxzQkFBZ0J2QixTQUFTd0IsT0FMUDtBQU1sQmxGLGNBQVF5QyxPQUFPMEMsSUFBUCxDQUFZakIsU0FBWixDQUNOLENBQUMsQ0FBQ1AsY0FBRCxFQUFpQixDQUFqQixDQUFELEVBQXNCLENBQUNOLFlBQVksQ0FBYixFQUFnQkMsY0FBaEIsQ0FBdEIsQ0FETSxFQUVOLEVBQUNFLFlBQVksT0FBYixFQUFzQkMsV0FBVyxLQUFqQyxFQUZNLENBTlU7QUFVbEJHO0FBVmtCLEtBQVQsQ0FBWDtBQVlEOztBQUVELFFBQU05QyxjQUFjMkIsT0FBT2YsS0FBUCxDQUFhd0MsU0FBYixDQUNsQixDQUFDLENBQUNmLGFBQUQsRUFBZ0IsQ0FBaEIsQ0FBRCxFQUFxQixDQUFDRSxZQUFZLENBQWIsRUFBZ0JDLGNBQWhCLENBQXJCLENBRGtCLEVBRWxCLEVBQUNFLFlBQVksT0FBYixFQUFzQkMsV0FBVyxLQUFqQyxFQUZrQixDQUFwQjs7QUFLQSxTQUFPLENBQUM1QyxLQUFELEVBQVFDLFdBQVIsQ0FBUDtBQUNEIiwiZmlsZSI6ImJ1aWxkZXIuanMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNS4xL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1Yi9saWIvbW9kZWxzL3BhdGNoIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtUZXh0QnVmZmVyfSBmcm9tICdhdG9tJztcblxuaW1wb3J0IEh1bmsgZnJvbSAnLi9odW5rJztcbmltcG9ydCBGaWxlLCB7bnVsbEZpbGV9IGZyb20gJy4vZmlsZSc7XG5pbXBvcnQgUGF0Y2ggZnJvbSAnLi9wYXRjaCc7XG5pbXBvcnQge1VuY2hhbmdlZCwgQWRkaXRpb24sIERlbGV0aW9uLCBOb05ld2xpbmV9IGZyb20gJy4vcmVnaW9uJztcbmltcG9ydCBGaWxlUGF0Y2ggZnJvbSAnLi9maWxlLXBhdGNoJztcbmltcG9ydCBNdWx0aUZpbGVQYXRjaCBmcm9tICcuL211bHRpLWZpbGUtcGF0Y2gnO1xuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRGaWxlUGF0Y2goZGlmZnMpIHtcbiAgY29uc3QgbGF5ZXJlZEJ1ZmZlciA9IGluaXRpYWxpemVCdWZmZXIoKTtcblxuICBsZXQgZmlsZVBhdGNoO1xuICBpZiAoZGlmZnMubGVuZ3RoID09PSAwKSB7XG4gICAgZmlsZVBhdGNoID0gZW1wdHlEaWZmRmlsZVBhdGNoKCk7XG4gIH0gZWxzZSBpZiAoZGlmZnMubGVuZ3RoID09PSAxKSB7XG4gICAgZmlsZVBhdGNoID0gc2luZ2xlRGlmZkZpbGVQYXRjaChkaWZmc1swXSwgbGF5ZXJlZEJ1ZmZlcik7XG4gIH0gZWxzZSBpZiAoZGlmZnMubGVuZ3RoID09PSAyKSB7XG4gICAgZmlsZVBhdGNoID0gZHVhbERpZmZGaWxlUGF0Y2goZGlmZnNbMF0sIGRpZmZzWzFdLCBsYXllcmVkQnVmZmVyKTtcbiAgfSBlbHNlIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYFVuZXhwZWN0ZWQgbnVtYmVyIG9mIGRpZmZzOiAke2RpZmZzLmxlbmd0aH1gKTtcbiAgfVxuXG4gIHJldHVybiBuZXcgTXVsdGlGaWxlUGF0Y2goe2ZpbGVQYXRjaGVzOiBbZmlsZVBhdGNoXSwgLi4ubGF5ZXJlZEJ1ZmZlcn0pO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRNdWx0aUZpbGVQYXRjaChkaWZmcykge1xuICBjb25zdCBsYXllcmVkQnVmZmVyID0gaW5pdGlhbGl6ZUJ1ZmZlcigpO1xuICBjb25zdCBieVBhdGggPSBuZXcgTWFwKCk7XG4gIGNvbnN0IGFjdGlvbnMgPSBbXTtcblxuICBsZXQgaW5kZXggPSAwO1xuICBmb3IgKGNvbnN0IGRpZmYgb2YgZGlmZnMpIHtcbiAgICBjb25zdCB0aGVQYXRoID0gZGlmZi5vbGRQYXRoIHx8IGRpZmYubmV3UGF0aDtcblxuICAgIGlmIChkaWZmLnN0YXR1cyA9PT0gJ2FkZGVkJyB8fCBkaWZmLnN0YXR1cyA9PT0gJ2RlbGV0ZWQnKSB7XG4gICAgICAvLyBQb3RlbnRpYWwgcGFpcmVkIGRpZmYuIEVpdGhlciBhIHN5bWxpbmsgZGVsZXRpb24gKyBjb250ZW50IGFkZGl0aW9uIG9yIGEgc3ltbGluayBhZGRpdGlvbiArXG4gICAgICAvLyBjb250ZW50IGRlbGV0aW9uLlxuICAgICAgY29uc3Qgb3RoZXJIYWxmID0gYnlQYXRoLmdldCh0aGVQYXRoKTtcbiAgICAgIGlmIChvdGhlckhhbGYpIHtcbiAgICAgICAgLy8gVGhlIHNlY29uZCBoYWxmLiBDb21wbGV0ZSB0aGUgcGFpcmVkIGRpZmYsIG9yIGZhaWwgaWYgdGhleSBoYXZlIHVuZXhwZWN0ZWQgc3RhdHVzZXMgb3IgbW9kZXMuXG4gICAgICAgIGNvbnN0IFtvdGhlckRpZmYsIG90aGVySW5kZXhdID0gb3RoZXJIYWxmO1xuICAgICAgICBhY3Rpb25zW290aGVySW5kZXhdID0gKCkgPT4gZHVhbERpZmZGaWxlUGF0Y2goZGlmZiwgb3RoZXJEaWZmLCBsYXllcmVkQnVmZmVyKTtcbiAgICAgICAgYnlQYXRoLmRlbGV0ZSh0aGVQYXRoKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIFRoZSBmaXJzdCBoYWxmIHdlJ3ZlIHNlZW4uXG4gICAgICAgIGJ5UGF0aC5zZXQodGhlUGF0aCwgW2RpZmYsIGluZGV4XSk7XG4gICAgICAgIGluZGV4Kys7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGFjdGlvbnNbaW5kZXhdID0gKCkgPT4gc2luZ2xlRGlmZkZpbGVQYXRjaChkaWZmLCBsYXllcmVkQnVmZmVyKTtcbiAgICAgIGluZGV4Kys7XG4gICAgfVxuICB9XG5cbiAgLy8gUG9wdWxhdGUgdW5wYWlyZWQgZGlmZnMgdGhhdCBsb29rZWQgbGlrZSB0aGV5IGNvdWxkIGJlIHBhcnQgb2YgYSBwYWlyLCBidXQgd2VyZW4ndC5cbiAgZm9yIChjb25zdCBbdW5wYWlyZWREaWZmLCBvcmlnaW5hbEluZGV4XSBvZiBieVBhdGgudmFsdWVzKCkpIHtcbiAgICBhY3Rpb25zW29yaWdpbmFsSW5kZXhdID0gKCkgPT4gc2luZ2xlRGlmZkZpbGVQYXRjaCh1bnBhaXJlZERpZmYsIGxheWVyZWRCdWZmZXIpO1xuICB9XG5cbiAgY29uc3QgZmlsZVBhdGNoZXMgPSBhY3Rpb25zLm1hcChhY3Rpb24gPT4gYWN0aW9uKCkpO1xuXG4gIC8vIEZpeCBtYXJrZXJzIGZvciBwYXRjaGVzIHdpdGggbm8gaHVua3MuXG4gIC8vIEhlYWQgcG9zaXRpb24gd2FzIG1vdmVkIGV2ZXJ5dGltZSBsaW5lcyB3ZXJlIGFwcGVuZGVkLlxuICBmaWxlUGF0Y2hlcy5mb3JFYWNoKGZpbGVQYXRjaCA9PiB7XG4gICAgaWYgKGZpbGVQYXRjaC5nZXRIdW5rcygpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgY29uc3QgbWFya2VyID0gZmlsZVBhdGNoLmdldE1hcmtlcigpO1xuICAgICAgbWFya2VyLnNldEhlYWRQb3NpdGlvbihtYXJrZXIuZ2V0VGFpbFBvc2l0aW9uKCkpO1xuICAgIH1cbiAgfSk7XG5cbiAgcmV0dXJuIG5ldyBNdWx0aUZpbGVQYXRjaCh7ZmlsZVBhdGNoZXMsIC4uLmxheWVyZWRCdWZmZXJ9KTtcbn1cblxuZnVuY3Rpb24gZW1wdHlEaWZmRmlsZVBhdGNoKCkge1xuICByZXR1cm4gRmlsZVBhdGNoLmNyZWF0ZU51bGwoKTtcbn1cblxuZnVuY3Rpb24gc2luZ2xlRGlmZkZpbGVQYXRjaChkaWZmLCBsYXllcmVkQnVmZmVyKSB7XG4gIGNvbnN0IHdhc1N5bWxpbmsgPSBkaWZmLm9sZE1vZGUgPT09IEZpbGUubW9kZXMuU1lNTElOSztcbiAgY29uc3QgaXNTeW1saW5rID0gZGlmZi5uZXdNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTks7XG5cbiAgY29uc3QgW2h1bmtzLCBwYXRjaE1hcmtlcl0gPSBidWlsZEh1bmtzKGRpZmYsIGxheWVyZWRCdWZmZXIpO1xuXG4gIGxldCBvbGRTeW1saW5rID0gbnVsbDtcbiAgbGV0IG5ld1N5bWxpbmsgPSBudWxsO1xuICBpZiAod2FzU3ltbGluayAmJiAhaXNTeW1saW5rKSB7XG4gICAgb2xkU3ltbGluayA9IGRpZmYuaHVua3NbMF0ubGluZXNbMF0uc2xpY2UoMSk7XG4gIH0gZWxzZSBpZiAoIXdhc1N5bWxpbmsgJiYgaXNTeW1saW5rKSB7XG4gICAgbmV3U3ltbGluayA9IGRpZmYuaHVua3NbMF0ubGluZXNbMF0uc2xpY2UoMSk7XG4gIH0gZWxzZSBpZiAod2FzU3ltbGluayAmJiBpc1N5bWxpbmspIHtcbiAgICBvbGRTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1swXS5zbGljZSgxKTtcbiAgICBuZXdTeW1saW5rID0gZGlmZi5odW5rc1swXS5saW5lc1syXS5zbGljZSgxKTtcbiAgfVxuXG4gIGNvbnN0IG9sZEZpbGUgPSBkaWZmLm9sZFBhdGggIT09IG51bGwgfHwgZGlmZi5vbGRNb2RlICE9PSBudWxsXG4gICAgPyBuZXcgRmlsZSh7cGF0aDogZGlmZi5vbGRQYXRoLCBtb2RlOiBkaWZmLm9sZE1vZGUsIHN5bWxpbms6IG9sZFN5bWxpbmt9KVxuICAgIDogbnVsbEZpbGU7XG4gIGNvbnN0IG5ld0ZpbGUgPSBkaWZmLm5ld1BhdGggIT09IG51bGwgfHwgZGlmZi5uZXdNb2RlICE9PSBudWxsXG4gICAgPyBuZXcgRmlsZSh7cGF0aDogZGlmZi5uZXdQYXRoLCBtb2RlOiBkaWZmLm5ld01vZGUsIHN5bWxpbms6IG5ld1N5bWxpbmt9KVxuICAgIDogbnVsbEZpbGU7XG4gIGNvbnN0IHBhdGNoID0gbmV3IFBhdGNoKHtzdGF0dXM6IGRpZmYuc3RhdHVzLCBodW5rcywgbWFya2VyOiBwYXRjaE1hcmtlciwgYnVmZmVyOiBsYXllcmVkQnVmZmVyLmJ1ZmZlcn0pO1xuXG4gIHJldHVybiBuZXcgRmlsZVBhdGNoKG9sZEZpbGUsIG5ld0ZpbGUsIHBhdGNoKTtcbn1cblxuZnVuY3Rpb24gZHVhbERpZmZGaWxlUGF0Y2goZGlmZjEsIGRpZmYyLCBsYXllcmVkQnVmZmVyKSB7XG4gIGxldCBtb2RlQ2hhbmdlRGlmZiwgY29udGVudENoYW5nZURpZmY7XG4gIGlmIChkaWZmMS5vbGRNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTksgfHwgZGlmZjEubmV3TW9kZSA9PT0gRmlsZS5tb2Rlcy5TWU1MSU5LKSB7XG4gICAgbW9kZUNoYW5nZURpZmYgPSBkaWZmMTtcbiAgICBjb250ZW50Q2hhbmdlRGlmZiA9IGRpZmYyO1xuICB9IGVsc2Uge1xuICAgIG1vZGVDaGFuZ2VEaWZmID0gZGlmZjI7XG4gICAgY29udGVudENoYW5nZURpZmYgPSBkaWZmMTtcbiAgfVxuXG4gIGNvbnN0IFtodW5rcywgcGF0Y2hNYXJrZXJdID0gYnVpbGRIdW5rcyhjb250ZW50Q2hhbmdlRGlmZiwgbGF5ZXJlZEJ1ZmZlcik7XG4gIGNvbnN0IGZpbGVQYXRoID0gY29udGVudENoYW5nZURpZmYub2xkUGF0aCB8fCBjb250ZW50Q2hhbmdlRGlmZi5uZXdQYXRoO1xuICBjb25zdCBzeW1saW5rID0gbW9kZUNoYW5nZURpZmYuaHVua3NbMF0ubGluZXNbMF0uc2xpY2UoMSk7XG5cbiAgbGV0IHN0YXR1cztcbiAgbGV0IG9sZE1vZGUsIG5ld01vZGU7XG4gIGxldCBvbGRTeW1saW5rID0gbnVsbDtcbiAgbGV0IG5ld1N5bWxpbmsgPSBudWxsO1xuICBpZiAobW9kZUNoYW5nZURpZmYuc3RhdHVzID09PSAnYWRkZWQnKSB7XG4gICAgLy8gY29udGVudHMgd2VyZSBkZWxldGVkIGFuZCByZXBsYWNlZCB3aXRoIHN5bWxpbmtcbiAgICBzdGF0dXMgPSAnZGVsZXRlZCc7XG4gICAgb2xkTW9kZSA9IGNvbnRlbnRDaGFuZ2VEaWZmLm9sZE1vZGU7XG4gICAgbmV3TW9kZSA9IG1vZGVDaGFuZ2VEaWZmLm5ld01vZGU7XG4gICAgbmV3U3ltbGluayA9IHN5bWxpbms7XG4gIH0gZWxzZSBpZiAobW9kZUNoYW5nZURpZmYuc3RhdHVzID09PSAnZGVsZXRlZCcpIHtcbiAgICAvLyBjb250ZW50cyB3ZXJlIGFkZGVkIGFmdGVyIHN5bWxpbmsgd2FzIGRlbGV0ZWRcbiAgICBzdGF0dXMgPSAnYWRkZWQnO1xuICAgIG9sZE1vZGUgPSBtb2RlQ2hhbmdlRGlmZi5vbGRNb2RlO1xuICAgIG9sZFN5bWxpbmsgPSBzeW1saW5rO1xuICAgIG5ld01vZGUgPSBjb250ZW50Q2hhbmdlRGlmZi5uZXdNb2RlO1xuICB9IGVsc2Uge1xuICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBtb2RlIGNoYW5nZSBkaWZmIHN0YXR1czogJHttb2RlQ2hhbmdlRGlmZi5zdGF0dXN9YCk7XG4gIH1cblxuICBjb25zdCBvbGRGaWxlID0gbmV3IEZpbGUoe3BhdGg6IGZpbGVQYXRoLCBtb2RlOiBvbGRNb2RlLCBzeW1saW5rOiBvbGRTeW1saW5rfSk7XG4gIGNvbnN0IG5ld0ZpbGUgPSBuZXcgRmlsZSh7cGF0aDogZmlsZVBhdGgsIG1vZGU6IG5ld01vZGUsIHN5bWxpbms6IG5ld1N5bWxpbmt9KTtcbiAgY29uc3QgcGF0Y2ggPSBuZXcgUGF0Y2goe3N0YXR1cywgaHVua3MsIG1hcmtlcjogcGF0Y2hNYXJrZXIsIGJ1ZmZlcjogbGF5ZXJlZEJ1ZmZlci5idWZmZXJ9KTtcblxuICByZXR1cm4gbmV3IEZpbGVQYXRjaChvbGRGaWxlLCBuZXdGaWxlLCBwYXRjaCk7XG59XG5cbmNvbnN0IENIQU5HRUtJTkQgPSB7XG4gICcrJzogQWRkaXRpb24sXG4gICctJzogRGVsZXRpb24sXG4gICcgJzogVW5jaGFuZ2VkLFxuICAnXFxcXCc6IE5vTmV3bGluZSxcbn07XG5cbmZ1bmN0aW9uIGluaXRpYWxpemVCdWZmZXIoKSB7XG4gIGNvbnN0IGJ1ZmZlciA9IG5ldyBUZXh0QnVmZmVyKCk7XG4gIGJ1ZmZlci5yZXRhaW4oKTtcblxuICBjb25zdCBsYXllcnMgPSBbJ3BhdGNoJywgJ2h1bmsnLCAndW5jaGFuZ2VkJywgJ2FkZGl0aW9uJywgJ2RlbGV0aW9uJywgJ25vTmV3bGluZSddLnJlZHVjZSgob2JqLCBrZXkpID0+IHtcbiAgICBvYmpba2V5XSA9IGJ1ZmZlci5hZGRNYXJrZXJMYXllcigpO1xuICAgIHJldHVybiBvYmo7XG4gIH0sIHt9KTtcblxuICByZXR1cm4ge2J1ZmZlciwgbGF5ZXJzfTtcbn1cblxuZnVuY3Rpb24gYnVpbGRIdW5rcyhkaWZmLCB7YnVmZmVyLCBsYXllcnN9KSB7XG4gIGNvbnN0IGxheWVyc0J5S2luZCA9IG5ldyBNYXAoW1xuICAgIFtVbmNoYW5nZWQsIGxheWVycy51bmNoYW5nZWRdLFxuICAgIFtBZGRpdGlvbiwgbGF5ZXJzLmFkZGl0aW9uXSxcbiAgICBbRGVsZXRpb24sIGxheWVycy5kZWxldGlvbl0sXG4gICAgW05vTmV3bGluZSwgbGF5ZXJzLm5vTmV3bGluZV0sXG4gIF0pO1xuICBjb25zdCBodW5rcyA9IFtdO1xuXG4gIGNvbnN0IHBhdGNoU3RhcnRSb3cgPSBidWZmZXIuZ2V0TGFzdFJvdygpO1xuICBsZXQgYnVmZmVyUm93ID0gcGF0Y2hTdGFydFJvdztcbiAgbGV0IG5leHRMaW5lTGVuZ3RoID0gMDtcblxuICBpZiAoZGlmZi5odW5rcy5sZW5ndGggPT09IDApIHtcbiAgICBjb25zdCBwYXRjaE1hcmtlciA9IGxheWVycy5wYXRjaC5tYXJrUG9zaXRpb24oXG4gICAgICBbcGF0Y2hTdGFydFJvdywgMF0sXG4gICAgICB7aW52YWxpZGF0ZTogJ25ldmVyJywgZXhjbHVzaXZlOiBmYWxzZX0sXG4gICAgKTtcblxuICAgIHJldHVybiBbaHVua3MsIHBhdGNoTWFya2VyXTtcbiAgfVxuXG4gIGZvciAoY29uc3QgaHVua0RhdGEgb2YgZGlmZi5odW5rcykge1xuICAgIGNvbnN0IGJ1ZmZlclN0YXJ0Um93ID0gYnVmZmVyUm93O1xuXG4gICAgY29uc3QgcmVnaW9ucyA9IFtdO1xuXG4gICAgbGV0IExhc3RDaGFuZ2VLaW5kID0gbnVsbDtcbiAgICBsZXQgY3VycmVudFJhbmdlU3RhcnQgPSBidWZmZXJSb3c7XG4gICAgbGV0IGxhc3RMaW5lTGVuZ3RoID0gMDtcblxuICAgIGNvbnN0IGZpbmlzaEN1cnJlbnRSYW5nZSA9ICgpID0+IHtcbiAgICAgIGlmIChjdXJyZW50UmFuZ2VTdGFydCA9PT0gYnVmZmVyUm93KSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgcmVnaW9ucy5wdXNoKFxuICAgICAgICBuZXcgTGFzdENoYW5nZUtpbmQoXG4gICAgICAgICAgbGF5ZXJzQnlLaW5kLmdldChMYXN0Q2hhbmdlS2luZCkubWFya1JhbmdlKFxuICAgICAgICAgICAgW1tjdXJyZW50UmFuZ2VTdGFydCwgMF0sIFtidWZmZXJSb3cgLSAxLCBsYXN0TGluZUxlbmd0aF1dLFxuICAgICAgICAgICAge2ludmFsaWRhdGU6ICduZXZlcicsIGV4Y2x1c2l2ZTogZmFsc2V9LFxuICAgICAgICAgICksXG4gICAgICAgICksXG4gICAgICApO1xuICAgICAgY3VycmVudFJhbmdlU3RhcnQgPSBidWZmZXJSb3c7XG4gICAgfTtcblxuICAgIGZvciAoY29uc3QgbGluZVRleHQgb2YgaHVua0RhdGEubGluZXMpIHtcbiAgICAgIGNvbnN0IGJ1ZmZlckxpbmUgPSBsaW5lVGV4dC5zbGljZSgxKSArICdcXG4nO1xuICAgICAgbmV4dExpbmVMZW5ndGggPSBsaW5lVGV4dC5sZW5ndGggLSAxO1xuICAgICAgYnVmZmVyLmFwcGVuZChidWZmZXJMaW5lKTtcblxuICAgICAgY29uc3QgQ2hhbmdlS2luZCA9IENIQU5HRUtJTkRbbGluZVRleHRbMF1dO1xuICAgICAgaWYgKENoYW5nZUtpbmQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gZGlmZiBzdGF0dXMgY2hhcmFjdGVyOiBcIiR7bGluZVRleHRbMF19XCJgKTtcbiAgICAgIH1cblxuICAgICAgaWYgKENoYW5nZUtpbmQgIT09IExhc3RDaGFuZ2VLaW5kKSB7XG4gICAgICAgIGZpbmlzaEN1cnJlbnRSYW5nZSgpO1xuICAgICAgfVxuXG4gICAgICBMYXN0Q2hhbmdlS2luZCA9IENoYW5nZUtpbmQ7XG4gICAgICBidWZmZXJSb3crKztcbiAgICAgIGxhc3RMaW5lTGVuZ3RoID0gbmV4dExpbmVMZW5ndGg7XG4gICAgfVxuICAgIGZpbmlzaEN1cnJlbnRSYW5nZSgpO1xuXG4gICAgaHVua3MucHVzaChuZXcgSHVuayh7XG4gICAgICBvbGRTdGFydFJvdzogaHVua0RhdGEub2xkU3RhcnRMaW5lLFxuICAgICAgbmV3U3RhcnRSb3c6IGh1bmtEYXRhLm5ld1N0YXJ0TGluZSxcbiAgICAgIG9sZFJvd0NvdW50OiBodW5rRGF0YS5vbGRMaW5lQ291bnQsXG4gICAgICBuZXdSb3dDb3VudDogaHVua0RhdGEubmV3TGluZUNvdW50LFxuICAgICAgc2VjdGlvbkhlYWRpbmc6IGh1bmtEYXRhLmhlYWRpbmcsXG4gICAgICBtYXJrZXI6IGxheWVycy5odW5rLm1hcmtSYW5nZShcbiAgICAgICAgW1tidWZmZXJTdGFydFJvdywgMF0sIFtidWZmZXJSb3cgLSAxLCBuZXh0TGluZUxlbmd0aF1dLFxuICAgICAgICB7aW52YWxpZGF0ZTogJ25ldmVyJywgZXhjbHVzaXZlOiBmYWxzZX0sXG4gICAgICApLFxuICAgICAgcmVnaW9ucyxcbiAgICB9KSk7XG4gIH1cblxuICBjb25zdCBwYXRjaE1hcmtlciA9IGxheWVycy5wYXRjaC5tYXJrUmFuZ2UoXG4gICAgW1twYXRjaFN0YXJ0Um93LCAwXSwgW2J1ZmZlclJvdyAtIDEsIG5leHRMaW5lTGVuZ3RoXV0sXG4gICAge2ludmFsaWRhdGU6ICduZXZlcicsIGV4Y2x1c2l2ZTogZmFsc2V9LFxuICApO1xuXG4gIHJldHVybiBbaHVua3MsIHBhdGNoTWFya2VyXTtcbn1cbiJdfQ==