(function() {
  var CSON, ContextMenuItemSet, ContextMenuManager, Disposable, MenuHelpers, calculateSpecificity, fs, path, platformContextMenu, ref, ref1, ref2, remote, sortMenuItems, validateSelector;

  path = require('path');

  CSON = require('season');

  fs = require('fs-plus');

  ref = require('clear-cut'), calculateSpecificity = ref.calculateSpecificity, validateSelector = ref.validateSelector;

  Disposable = require('event-kit').Disposable;

  remote = require('electron').remote;

  MenuHelpers = require('./menu-helpers');

  sortMenuItems = require('./menu-sort-helpers').sortMenuItems;

  platformContextMenu = (ref1 = require('../package.json')) != null ? (ref2 = ref1._atomMenu) != null ? ref2['context-menu'] : void 0 : void 0;

  module.exports = ContextMenuManager = (function() {
    function ContextMenuManager(arg) {
      this.keymapManager = arg.keymapManager;
      this.definitions = {
        '.overlayer': []
      };
      this.clear();
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
    }

    ContextMenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath, this.devMode = arg.devMode;
    };

    ContextMenuManager.prototype.loadPlatformItems = function() {
      var map, menusDirPath, platformMenuPath, ref3;
      if (platformContextMenu != null) {
        return this.add(platformContextMenu, (ref3 = this.devMode) != null ? ref3 : false);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        map = CSON.readFileSync(platformMenuPath);
        return this.add(map['context-menu']);
      }
    };

    ContextMenuManager.prototype.add = function(itemsBySelector, throwOnInvalidSelector) {
      var addedItemSets, itemSet, items, selector;
      if (throwOnInvalidSelector == null) {
        throwOnInvalidSelector = true;
      }
      addedItemSets = [];
      for (selector in itemsBySelector) {
        items = itemsBySelector[selector];
        if (throwOnInvalidSelector) {
          validateSelector(selector);
        }
        itemSet = new ContextMenuItemSet(selector, items);
        addedItemSets.push(itemSet);
        this.itemSets.push(itemSet);
      }
      return new Disposable((function(_this) {
        return function() {
          var i, len;
          for (i = 0, len = addedItemSets.length; i < len; i++) {
            itemSet = addedItemSets[i];
            _this.itemSets.splice(_this.itemSets.indexOf(itemSet), 1);
          }
        };
      })(this));
    };

    ContextMenuManager.prototype.templateForElement = function(target) {
      return this.templateForEvent({
        target: target
      });
    };

    ContextMenuManager.prototype.templateForEvent = function(event) {
      var currentTarget, currentTargetItems, i, item, itemForEvent, itemSet, j, k, len, len1, len2, matchingItemSets, ref3, template;
      template = [];
      currentTarget = event.target;
      while (currentTarget != null) {
        currentTargetItems = [];
        matchingItemSets = this.itemSets.filter(function(itemSet) {
          return currentTarget.webkitMatchesSelector(itemSet.selector);
        });
        for (i = 0, len = matchingItemSets.length; i < len; i++) {
          itemSet = matchingItemSets[i];
          ref3 = itemSet.items;
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            item = ref3[j];
            itemForEvent = this.cloneItemForEvent(item, event);
            if (itemForEvent) {
              MenuHelpers.merge(currentTargetItems, itemForEvent, itemSet.specificity);
            }
          }
        }
        for (k = 0, len2 = currentTargetItems.length; k < len2; k++) {
          item = currentTargetItems[k];
          MenuHelpers.merge(template, item, false);
        }
        currentTarget = currentTarget.parentElement;
      }
      this.pruneRedundantSeparators(template);
      this.addAccelerators(template);
      return this.sortTemplate(template);
    };

    ContextMenuManager.prototype.addAccelerators = function(template) {
      var accelerator, id, item, keymaps, ref3, results;
      results = [];
      for (id in template) {
        item = template[id];
        if (item.command) {
          keymaps = this.keymapManager.findKeyBindings({
            command: item.command,
            target: document.activeElement
          });
          accelerator = MenuHelpers.acceleratorForKeystroke(keymaps != null ? (ref3 = keymaps[0]) != null ? ref3.keystrokes : void 0 : void 0);
          if (accelerator) {
            item.accelerator = accelerator;
          }
        }
        if (Array.isArray(item.submenu)) {
          results.push(this.addAccelerators(item.submenu));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.pruneRedundantSeparators = function(menu) {
      var index, keepNextItemIfSeparator, results;
      keepNextItemIfSeparator = false;
      index = 0;
      results = [];
      while (index < menu.length) {
        if (menu[index].type === 'separator') {
          if (!keepNextItemIfSeparator || index === menu.length - 1) {
            results.push(menu.splice(index, 1));
          } else {
            results.push(index++);
          }
        } else {
          keepNextItemIfSeparator = true;
          results.push(index++);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.sortTemplate = function(template) {
      var id, item;
      template = sortMenuItems(template);
      for (id in template) {
        item = template[id];
        if (Array.isArray(item.submenu)) {
          item.submenu = this.sortTemplate(item.submenu);
        }
      }
      return template;
    };

    ContextMenuManager.prototype.cloneItemForEvent = function(item, event) {
      if (item.devMode && !this.devMode) {
        return null;
      }
      item = Object.create(item);
      if (typeof item.shouldDisplay === 'function') {
        if (!item.shouldDisplay(event)) {
          return null;
        }
      }
      if (typeof item.created === "function") {
        item.created(event);
      }
      if (Array.isArray(item.submenu)) {
        item.submenu = item.submenu.map((function(_this) {
          return function(submenuItem) {
            return _this.cloneItemForEvent(submenuItem, event);
          };
        })(this)).filter(function(submenuItem) {
          return submenuItem !== null;
        });
      }
      return item;
    };

    ContextMenuManager.prototype.showForEvent = function(event) {
      var menuTemplate;
      this.activeElement = event.target;
      menuTemplate = this.templateForEvent(event);
      if (!((menuTemplate != null ? menuTemplate.length : void 0) > 0)) {
        return;
      }
      remote.getCurrentWindow().emit('context-menu', menuTemplate);
    };

    ContextMenuManager.prototype.clear = function() {
      var inspectElement;
      this.activeElement = null;
      this.itemSets = [];
      inspectElement = {
        'atom-workspace': [
          {
            label: 'Inspect Element',
            command: 'application:inspect',
            devMode: true,
            created: function(event) {
              var pageX, pageY;
              pageX = event.pageX, pageY = event.pageY;
              return this.commandDetail = {
                x: pageX,
                y: pageY
              };
            }
          }
        ]
      };
      return this.add(inspectElement, false);
    };

    return ContextMenuManager;

  })();

  ContextMenuItemSet = (function() {
    function ContextMenuItemSet(selector1, items1) {
      this.selector = selector1;
      this.items = items1;
      this.specificity = calculateSpecificity(this.selector);
    }

    return ContextMenuItemSet;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
