'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _eventKit = require('event-kit');

/*
 * Allow child components to operate on refs captured by a parent component.
 *
 * React does not guarantee that refs are available until the component has finished mounting (before
 * componentDidMount() is called), but a component does not finish mounting until all of its children are mounted. This
 * causes problems when a child needs to consume a DOM node from its parent to interact with the Atom API, like we do in
 * the `Tooltip` and `Commands` components.
 *
 * To pass a ref to a child, capture it in a RefHolder in the parent, and pass the RefHolder to the child:
 *
 *   class Parent extends React.Component {
 *     constructor() {
 *       this.theRef = new RefHolder();
 *     }
 *
 *     render() {
 *       return (
 *         <div ref={this.theRef.setter}>
 *           <Child theRef={this.theRef} />
 *         </div>
 *       )
 *     }
 *   }
 *
 * In the child, use the `observe()` method to defer operations that need the DOM node to proceed:
 *
 *   class Child extends React.Component {
 *
 *     componentDidMount() {
 *       this.props.theRef.observe(domNode => this.register(domNode))
 *     }
 *
 *     render() {
 *       return null;
 *     }
 *
 *     register(domNode) {
 *       console.log('Hey look I have a real DOM node', domNode);
 *     }
 *   }
 */
class RefHolder {
  constructor() {
    this.setter = value => {
      const oldValue = this.value;
      this.value = value;
      if (value !== oldValue && value !== null && value !== undefined) {
        this.emitter.emit('did-update', value);
      }
    };

    this.emitter = new _eventKit.Emitter();
    this.value = undefined;
  }

  isEmpty() {
    return this.value === undefined || this.value === null;
  }

  get() {
    if (this.isEmpty()) {
      throw new Error('RefHolder is empty');
    }
    return this.value;
  }

  getOr(def) {
    if (this.isEmpty()) {
      return def;
    }
    return this.value;
  }

  getPromise() {
    if (this.isEmpty()) {
      return new Promise(resolve => {
        const sub = this.observe(value => {
          resolve(value);
          sub.dispose();
        });
      });
    }

    return Promise.resolve(this.get());
  }

  map(present, absent = () => this) {
    return RefHolder.on(this.isEmpty() ? absent() : present(this.get()));
  }

  observe(callback) {
    if (!this.isEmpty()) {
      callback(this.value);
    }
    return this.emitter.on('did-update', callback);
  }

  static on(valueOrHolder) {
    if (valueOrHolder instanceof this) {
      return valueOrHolder;
    } else {
      const holder = new this();
      holder.setter(valueOrHolder);
      return holder;
    }
  }
}
exports.default = RefHolder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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