'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullRemote = undefined;

var _endpoint = require('./endpoint');

class Remote {
  constructor(name, url) {
    this.name = name;
    this.url = url;

    const { isGithubRepo, domain, protocol, owner, repo } = githubInfoFromRemote(url);
    this.githubRepo = isGithubRepo;
    this.domain = domain;
    this.protocol = protocol;
    this.owner = owner;
    this.repo = repo;
  }

  getName() {
    return this.name;
  }

  getUrl() {
    return this.url;
  }

  isGithubRepo() {
    return this.githubRepo;
  }

  getProtocol() {
    return this.protocol;
  }

  getDomain() {
    return this.domain;
  }

  getOwner() {
    return this.owner;
  }

  getRepo() {
    return this.repo;
  }

  getNameOr() {
    return this.getName();
  }

  getSlug() {
    if (this.owner === null || this.repo === null) {
      return null;
    }

    return `${this.owner}/${this.repo}`;
  }

  getEndpoint() {
    return this.domain === null ? null : (0, _endpoint.getEndpoint)(this.domain);
  }

  isPresent() {
    return true;
  }
}

exports.default = Remote;
function githubInfoFromRemote(remoteUrl) {
  if (!remoteUrl) {
    return {
      isGithubRepo: false,
      domain: null,
      owner: null,
      repo: null
    };
  }

  //             proto         login   domain           owner    repo
  const regex = /(?:(.+):\/\/)?(?:.+@)?(github\.com)[:/]([^/]+)\/(.+)/;
  const match = remoteUrl.match(regex);
  if (match) {
    return {
      isGithubRepo: true,
      protocol: match[1] || 'ssh',
      domain: match[2],
      owner: match[3],
      repo: match[4].replace(/\.git$/, '')
    };
  } else {
    return {
      isGithubRepo: false,
      protocol: null,
      domain: null,
      owner: null,
      repo: null
    };
  }
}

const nullRemote = exports.nullRemote = {
  getName() {
    return '';
  },

  getUrl() {
    return '';
  },

  isGithubRepo() {
    return false;
  },

  getDomain() {
    return null;
  },

  getProtocol() {
    return null;
  },

  getOwner() {
    return null;
  },

  getRepo() {
    return null;
  },

  getNameOr(fallback) {
    return fallback;
  },

  getSlug() {
    return null;
  },

  getEndpoint() {
    return null;
  },

  isPresent() {
    return false;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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