'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LargeRepoError = exports.GitError = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _child_process = require('child_process');

var _child_process2 = _interopRequireDefault(_child_process);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _util = require('util');

var _util2 = _interopRequireDefault(_util);

var _electron = require('electron');

var _eventKit = require('event-kit');

var _dugite = require('dugite');

var _whatTheDiff = require('what-the-diff');

var _whatTheStatus = require('what-the-status');

var _gitPromptServer = require('./git-prompt-server');

var _gitPromptServer2 = _interopRequireDefault(_gitPromptServer);

var _gitTempDir = require('./git-temp-dir');

var _gitTempDir2 = _interopRequireDefault(_gitTempDir);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _reporterProxy = require('./reporter-proxy');

var _helpers = require('./helpers');

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _file = require('./models/patch/file');

var _file2 = _interopRequireDefault(_file);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;

let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
}

exports.GitError = GitError;
class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
}

exports.LargeRepoError = LargeRepoError; // ignored for the purposes of usage metrics tracking because they're noisy

const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];

const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);

/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^/]*)/ captures non-forwardslash characters before first slash
 */
const EXPAND_TILDE_REGEX = new RegExp('^~([^/]*)/');

class GitShellOutStrategy {

  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;
    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os2.default.cpus().length);
      this.commandQueue = new _asyncQueue2.default({ parallelism });
    }

    this.prompt = options.prompt || (query => Promise.reject());
    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */
  setPromptCallback(prompt) {
    this.prompt = prompt;
  }

  // Execute a command and read the output using the embedded Git environment
  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const { stdin, useGitPromptServer, useGpgWrapper, useGpgAtomPrompt, writeOperation } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');

    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;
    const timingMarker = _gitTimingsView2.default.generateMarker(`git ${args.join(' ')}`);
    timingMarker.mark('queued');

    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise((resolve, reject) => {
        _child_process2.default.exec('git --exec-path', (error, stdout, stderr) => {
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }
    const execPath = await execPathPromise;

    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;

      const pathParts = [];
      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }
      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _extends({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path2.default.delimiter)
      });

      const gitTempDir = new _gitTempDir2.default();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer2.default(gitTempDir);
        await gitPromptServer.start(this.prompt);

        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getSocketPath());

        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy');

        // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.
        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';

        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }

      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = { env };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }

      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }
      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({ args, opts });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }
        const { promise, cancel } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;
        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({ handlerPid }) => {
            expectCancel = true;
            await cancel();

            // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.
            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const { stdout, stderr, exitCode, signal, timing } = await promise.catch(err => {
          if (err.signal) {
            return { signal: err.signal };
          }
          reject(err);
          return {};
        });

        if (timing) {
          const { execTime, spawnTime, ipcTime } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }
        timingMarker.finalize();
        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }
        if (gitPromptServer) {
          gitPromptServer.terminate();
        }
        subscriptions.dispose();

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;
            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }
            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }
            summary += 'stdout:';
            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }
            summary += 'stderr:';
            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';

            console.groupCollapsed(`git:${formattedArgs}`);
            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }
            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util2.default.inspect(args, { breakLength: Infinity }));
            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }
            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }
        resolve(stdout);
      });
    }, { parallel: !writeOperation });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _extends({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _extends({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager2.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');

      let childPid;
      options.processCallback = child => {
        childPid = child.pid;

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);
      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager2.default.getInstance();
      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra2.default.stat(this.workingDir); // fails if folder doesn't exist
      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path2.default.join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      if (_path2.default.isAbsolute(dotGitDir)) {
        return (0, _helpers.toNativePathSep)(dotGitDir);
      } else {
        return (0, _helpers.toNativePathSep)(_path2.default.resolve(_path2.default.join(this.workingDir, dotGitDir)));
      }
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }

  /**
   * Staging/Unstaging files and patches and committing
   */
  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');
    if (!templatePath) {
      return null;
    }

    const homeDir = _os2.default.homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path2.default.join(_path2.default.dirname(homeDir), user) : homeDir}/`;
    });

    if (!_path2.default.isAbsolute(templatePath)) {
      templatePath = _path2.default.join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }
    return await _fsExtra2.default.readFile(templatePath, { encoding: 'utf8' });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({ args, opts }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], { writeOperation: true });
  }

  applyPatch(patch, { index } = {}) {
    const args = ['apply', '-'];
    if (index) {
      args.splice(1, 0, '--cached');
    }
    return this.exec(args, { stdin: patch, writeOperation: true });
  }

  async commit(rawMessage, { allowEmpty, amend, coAuthors, verbatim } = {}) {
    const args = ['commit'];
    let msg;

    // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.
    if (amend && rawMessage.length === 0) {
      const { unbornRef, messageBody, messageSubject } = await this.getHeadCommit();
      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    }

    // Determine the cleanup mode.
    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    }

    // add co-author commit trailers if necessary
    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }
    if (allowEmpty) {
      args.push('--allow-empty');
    }
    return this.gpgExec(args, { writeOperation: true });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    });

    // Ensure that message ends with newline for git-interpret trailers to work
    const msg = `${message.trim()}\n`;

    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }

  /**
   * File Status and Diffs
   */
  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);
    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }
      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];
    if (options.staged) {
      args.push('--staged');
    }
    if (options.target) {
      args.push(options.target);
    }
    const output = await this.exec(args);

    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };

    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });
    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }
    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);
    if (output.trim() === '') {
      return [];
    }
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, { staged, baseCommit } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];
    if (staged) {
      args.push('--staged');
    }
    if (baseCommit) {
      args.push(baseCommit);
    }
    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);

    let rawDiffs = [];
    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];
        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }
        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path2.default.join(this.workingDir, filePath);
      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra2.default.readFile(absPath, { encoding: 'utf8' });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;
      if (executable) {
        mode = _file2.default.modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file2.default.modes.SYMLINK;
        realpath = await _fsExtra2.default.realpath(absPath);
      } else {
        mode = _file2.default.modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }
    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }
    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);
    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }
      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }
    return diffs;
  }

  /**
   * Miscellaneous getters
   */
  async getCommit(ref) {
    const [commit] = await this.getCommits({ max: 1, ref, includeUnborn: true });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({ max: 1, ref: 'HEAD', includeUnborn: true });
    return headCommit;
  }

  async getCommits(options = {}) {
    const { max, ref, includeUnborn, includePatch } = _extends({
      max: 1,
      ref: 'HEAD',
      includeUnborn: false,
      includePatch: false
    }, options);

    // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body
    const args = ['log', '--pretty=format:%H%x00%ae%x00%at%x00%s%x00%b%x00', '--no-abbrev-commit', '--no-prefix', '--no-ext-diff', '--no-renames', '-z', '-n', max, ref];

    if (includePatch) {
      args.push('--patch', '-m', '--first-parent');
    }

    const output = await this.exec(args.concat('--')).catch(err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{ sha: '', message: '', unbornRef: true }] : [];
    }

    const fields = output.trim().split('\0');

    const commits = [];
    for (let i = 0; i < fields.length; i += 6) {
      const body = fields[i + 4].trim();
      let patch = [];
      if (includePatch) {
        const diffs = fields[i + 5];
        patch = (0, _whatTheDiff.parse)(diffs.trim());
      }

      const { message: messageBody, coAuthors } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);

      commits.push({
        sha: fields[i] && fields[i].trim(),
        authorEmail: fields[i + 1] && fields[i + 1].trim(),
        authorDate: parseInt(fields[i + 2], 10),
        messageSubject: fields[i + 3],
        messageBody,
        coAuthors,
        unbornRef: false,
        patch
      });
    }
    return commits;
  }

  async getAuthors(options = {}) {
    const { max, ref } = _extends({ max: 1, ref: 'HEAD' }, options);

    // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).

    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);

      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });

        acc[ae] = an;
        acc[ce] = cn;

        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers, unfold) {
    const args = ['interpret-trailers'];
    if (unfold) {
      args.push('--unfold');
    }
    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }
    return this.exec(args, { stdin: commitMessage });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }

  /**
   * Merge
   */
  merge(branchName) {
    return this.gpgExec(['merge', branchName], { writeOperation: true });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'MERGE_HEAD')).catch(() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], { writeOperation: true });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }

  /**
   * Rebase
   */
  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }

  /**
   * Remote interactions
   */
  clone(remoteUrl, options = {}) {
    const args = ['clone'];
    if (options.noLocal) {
      args.push('--no-local');
    }
    if (options.bare) {
      args.push('--bare');
    }
    if (options.recursive) {
      args.push('--recursive');
    }
    args.push(remoteUrl, this.workingDir);

    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];
    if (options.ffOnly) {
      args.push('--ff-only');
    }
    return this.gpgExec(args, { useGitPromptServer: true, writeOperation: true });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];
    if (options.setUpstream) {
      args.push('--set-upstream');
    }
    if (options.force) {
      args.push('--force');
    }
    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  /**
   * Undo Operations
   */
  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];
    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }
    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }

  /**
   * Branches
   */
  checkout(branchName, options = {}) {
    const args = ['checkout'];
    if (options.createNew) {
      args.push('-b');
    }
    args.push(branchName);
    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }
      args.push(options.startPoint);
    }

    return this.exec(args, { writeOperation: true });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');

    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');

      const branch = { name, sha, head: head === '*' };
      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }
      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }
      return branch;
    });
  }

  async getBranchesWithCommit(sha, option = {}) {
    const args = ['branch', '--format=%(refname)', '--contains', sha];
    if (option.showLocal && option.showRemote) {
      args.splice(1, 0, '--all');
    } else if (option.showRemote) {
      args.splice(1, 0, '--remotes');
    }
    return (await this.exec(args)).trim().split(_helpers.LINE_ENDING_REGEX);
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }
    const args = ['checkout'];
    if (revision) {
      args.push(revision);
    }
    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), { writeOperation: true });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, { local } = {}) {
    let output;
    try {
      let args = ['config'];
      if (local || atom.inSpecMode()) {
        args.push('--local');
      }
      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1) {
        // No matching config found
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, { replaceAll } = {}) {
    let args = ['config'];
    if (replaceAll) {
      args.push('--replace-all');
    }
    args = args.concat(option, value);
    return this.exec(args, { writeOperation: true });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], { writeOperation: true });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], { local: true });
    if (output) {
      output = output.trim();
      if (!output.length) {
        return [];
      }
      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({ filePath, stdin } = {}) {
    let output;
    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], { writeOperation: true })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], { stdin, writeOperation: true })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }
    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra2.default.writeFile(absFilePath, output, { encoding: 'utf8' });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;
    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    }

    // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.
    const resolvedResultPath = _path2.default.resolve(this.workingDir, resultPath);
    await _fsExtra2.default.writeFile(resolvedResultPath, output, { encoding: 'utf8' });

    return { filePath: oursPath, resultPath, conflict };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;
    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }
    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }
    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }
    return this.exec(['update-index', '--index-info'], { stdin: indexInfo, writeOperation: true });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);
    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path2.default.join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path2.default.join(this.workingDir, filePath));
      if (symlink) {
        return _file2.default.modes.SYMLINK;
      } else if (executable) {
        return _file2.default.modes.EXECUTABLE;
      } else {
        return _file2.default.modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }
}

exports.default = GitShellOutStrategy;
GitShellOutStrategy.defaultExecArgs = {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
};
function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];
  if (contents) {
    let noNewLine;
    let lines;
    if (mode === _file2.default.modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }
    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }
    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }
  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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