'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _eventKit = require('event-kit');

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _reactDom = require('react-dom');

var _reactDom2 = _interopRequireDefault(_reactDom);

var _helpers = require('./helpers');

var _workdirCache = require('./models/workdir-cache');

var _workdirCache2 = _interopRequireDefault(_workdirCache);

var _workdirContext = require('./models/workdir-context');

var _workdirContext2 = _interopRequireDefault(_workdirContext);

var _workdirContextPool = require('./models/workdir-context-pool');

var _workdirContextPool2 = _interopRequireDefault(_workdirContextPool);

var _repository = require('./models/repository');

var _repository2 = _interopRequireDefault(_repository);

var _styleCalculator = require('./models/style-calculator');

var _styleCalculator2 = _interopRequireDefault(_styleCalculator);

var _githubLoginModel = require('./models/github-login-model');

var _githubLoginModel2 = _interopRequireDefault(_githubLoginModel);

var _rootController = require('./controllers/root-controller');

var _rootController2 = _interopRequireDefault(_rootController);

var _stubItem = require('./items/stub-item');

var _stubItem2 = _interopRequireDefault(_stubItem);

var _switchboard = require('./switchboard');

var _switchboard2 = _interopRequireDefault(_switchboard);

var _yardstick = require('./yardstick');

var _yardstick2 = _interopRequireDefault(_yardstick);

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _contextMenuInterceptor = require('./context-menu-interceptor');

var _contextMenuInterceptor2 = _interopRequireDefault(_contextMenuInterceptor);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

var _getRepoPipelineManager = require('./get-repo-pipeline-manager');

var _getRepoPipelineManager2 = _interopRequireDefault(_getRepoPipelineManager);

var _reporterProxy = require('./reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const defaultState = {
  newProject: true
};

class GithubPackage {
  constructor({
    workspace, project, commandRegistry, notificationManager, tooltips, styles, grammars,
    keymaps, config, deserializers,
    confirm, getLoadSettings,
    configDirPath,
    renderFn, loginModel
  }) {
    (0, _helpers.autobind)(this, 'consumeStatusBar', 'createGitTimingsView', 'createIssueishPaneItemStub', 'createDockItemStub', 'createFilePatchControllerStub', 'destroyGitTabItem', 'destroyGithubTabItem', 'createRepositoryForProjectPath', 'cloneRepositoryForProjectPath', 'getRepositoryForWorkdir', 'scheduleActiveContextUpdate');

    this.workspace = workspace;
    this.project = project;
    this.commandRegistry = commandRegistry;
    this.deserializers = deserializers;
    this.notificationManager = notificationManager;
    this.tooltips = tooltips;
    this.config = config;
    this.styles = styles;
    this.grammars = grammars;
    this.keymaps = keymaps;
    this.configPath = _path2.default.join(configDirPath, 'github.cson');

    this.styleCalculator = new _styleCalculator2.default(this.styles, this.config);
    this.confirm = confirm;
    this.startOpen = false;
    this.activated = false;

    const criteria = {
      projectPathCount: this.project.getPaths().length,
      initPathCount: (getLoadSettings().initialPaths || []).length
    };

    this.pipelineManager = (0, _getRepoPipelineManager2.default)({ confirm, notificationManager, workspace });

    this.activeContextQueue = new _asyncQueue2.default();
    this.guessedContext = _workdirContext2.default.guess(criteria, this.pipelineManager);
    this.activeContext = this.guessedContext;
    this.workdirCache = new _workdirCache2.default();
    this.contextPool = new _workdirContextPool2.default({
      window,
      workspace,
      promptCallback: query => this.controller.promptForCredentials(query),
      pipelineManager: this.pipelineManager
    });

    this.switchboard = new _switchboard2.default();

    this.loginModel = loginModel || new _githubLoginModel2.default();
    this.renderFn = renderFn || ((component, node, callback) => {
      return _reactDom2.default.render(component, node, callback);
    });

    // Handle events from all resident contexts.
    this.subscriptions = new _eventKit.CompositeDisposable(this.contextPool.onDidChangeWorkdirOrHead(context => {
      this.refreshAtomGitRepository(context.getWorkingDirectory());
    }), this.contextPool.onDidUpdateRepository(context => {
      this.switchboard.didUpdateRepository(context.getRepository());
    }), this.contextPool.onDidDestroyRepository(context => {
      if (context === this.activeContext) {
        this.setActiveContext(_workdirContext2.default.absent({ pipelineManager: this.pipelineManager }));
      }
    }), _contextMenuInterceptor2.default);

    this.setupYardstick();
  }

  setupYardstick() {
    const stagingSeries = ['stageLine', 'stageHunk', 'unstageLine', 'unstageHunk'];

    this.subscriptions.add(
    // Staging and unstaging operations
    this.switchboard.onDidBeginStageOperation(payload => {
      if (payload.stage && payload.line) {
        _yardstick2.default.begin('stageLine');
      } else if (payload.stage && payload.hunk) {
        _yardstick2.default.begin('stageHunk');
      } else if (payload.stage && payload.file) {
        _yardstick2.default.begin('stageFile');
      } else if (payload.stage && payload.mode) {
        _yardstick2.default.begin('stageMode');
      } else if (payload.stage && payload.symlink) {
        _yardstick2.default.begin('stageSymlink');
      } else if (payload.unstage && payload.line) {
        _yardstick2.default.begin('unstageLine');
      } else if (payload.unstage && payload.hunk) {
        _yardstick2.default.begin('unstageHunk');
      } else if (payload.unstage && payload.file) {
        _yardstick2.default.begin('unstageFile');
      } else if (payload.unstage && payload.mode) {
        _yardstick2.default.begin('unstageMode');
      } else if (payload.unstage && payload.symlink) {
        _yardstick2.default.begin('unstageSymlink');
      }
    }), this.switchboard.onDidUpdateRepository(() => {
      _yardstick2.default.mark(stagingSeries, 'update-repository');
    }), this.switchboard.onDidFinishRender(context => {
      if (context === 'RootController.showFilePatchForPath') {
        _yardstick2.default.finish(stagingSeries);
      }
    }),

    // Active context changes
    this.switchboard.onDidScheduleActiveContextUpdate(() => {
      _yardstick2.default.begin('activeContextChange');
    }), this.switchboard.onDidBeginActiveContextUpdate(() => {
      _yardstick2.default.mark('activeContextChange', 'queue-wait');
    }), this.switchboard.onDidFinishContextChangeRender(() => {
      _yardstick2.default.mark('activeContextChange', 'render');
    }), this.switchboard.onDidFinishActiveContextUpdate(() => {
      _yardstick2.default.finish('activeContextChange');
    }));
  }

  async activate(state = {}) {
    this.savedState = _extends({}, defaultState, state);

    const firstRun = !(await (0, _helpers.fileExists)(this.configPath));
    const newProject = this.savedState.firstRun !== undefined ? this.savedState.firstRun : this.savedState.newProject;

    this.startOpen = firstRun || newProject;
    this.startRevealed = firstRun && !this.config.get('welcome.showOnStartup');

    if (firstRun) {
      await _fsExtra2.default.writeFile(this.configPath, '# Store non-visible GitHub package state.\n', { encoding: 'utf8' });
    }

    const hasSelectedFiles = event => {
      return !!event.target.closest('.github-FilePatchListView').querySelector('.is-selected');
    };

    this.subscriptions.add(this.project.onDidChangePaths(this.scheduleActiveContextUpdate), this.workspace.getCenter().onDidChangeActivePaneItem(this.scheduleActiveContextUpdate), this.styleCalculator.startWatching('github-package-styles', ['editor.fontSize', 'editor.fontFamily', 'editor.lineHeight', 'editor.tabLength'], config => `
          .github-HunkView-line {
            font-family: ${config.get('editor.fontFamily')};
            line-height: ${config.get('editor.lineHeight')};
            tab-size: ${config.get('editor.tabLength')}
          }
        `), atom.contextMenu.add({
      '.github-UnstagedChanges .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Discard Changes',
        command: 'github:discard-changes-in-selected-files',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-StagedChanges .github-FilePatchListView': [{
        label: 'Unstage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-MergeConflictPaths .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Ours',
        command: 'github:resolve-file-as-ours',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Theirs',
        command: 'github:resolve-file-as-theirs',
        shouldDisplay: hasSelectedFiles
      }]
    }));

    this.activated = true;
    this.scheduleActiveContextUpdate(this.savedState);
    this.rerender();
  }

  serialize() {
    const activeRepository = this.getActiveRepository();
    const activeRepositoryPath = activeRepository ? activeRepository.getWorkingDirectoryPath() : null;

    return {
      activeRepositoryPath,
      newProject: false
    };
  }

  rerender(callback) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    if (!this.activated) {
      return;
    }

    if (!this.element) {
      this.element = document.createElement('div');
      this.subscriptions.add(new _eventKit.Disposable(() => {
        _reactDom2.default.unmountComponentAtNode(this.element);
        delete this.element;
      }));
    }

    this.renderFn(_react2.default.createElement(_rootController2.default, {
      ref: c => {
        this.controller = c;
      },
      workspace: this.workspace,
      deserializers: this.deserializers,
      commandRegistry: this.commandRegistry,
      notificationManager: this.notificationManager,
      tooltips: this.tooltips,
      grammars: this.grammars,
      keymaps: this.keymaps,
      config: this.config,
      project: this.project,
      confirm: this.confirm,
      workdirContextPool: this.contextPool,
      loginModel: this.loginModel,
      repository: this.getActiveRepository(),
      resolutionProgress: this.getActiveResolutionProgress(),
      statusBar: this.statusBar,
      createRepositoryForProjectPath: this.createRepositoryForProjectPath,
      cloneRepositoryForProjectPath: this.cloneRepositoryForProjectPath,
      switchboard: this.switchboard,
      startOpen: this.startOpen,
      startRevealed: this.startRevealed,
      removeFilePatchItem: this.removeFilePatchItem
    }), this.element, callback);
  }

  async deactivate() {
    this.subscriptions.dispose();
    this.contextPool.clear();
    _workerManager2.default.reset(false);
    if (this.guessedContext) {
      this.guessedContext.destroy();
      this.guessedContext = null;
    }
    await _yardstick2.default.flush();
  }

  consumeStatusBar(statusBar) {
    this.statusBar = statusBar;
    this.rerender();
  }

  consumeReporter(reporter) {
    _reporterProxy.reporterProxy.setReporter(reporter);
  }

  createGitTimingsView() {
    return _stubItem2.default.create('git-timings-view', {
      title: 'GitHub Package Timings View'
    }, _gitTimingsView2.default.buildURI());
  }

  createIssueishPaneItemStub({ uri }) {
    return _stubItem2.default.create('issueish-detail-item', {
      title: 'Issueish'
    }, uri);
  }

  createDockItemStub({ uri }) {
    let item;
    switch (uri) {
      // always return an empty stub
      // but only set it as the active item for a tab type
      // if it doesn't already exist
      case 'atom-github://dock-item/git':
        item = this.createGitStub(uri);
        this.gitTabStubItem = this.gitTabStubItem || item;
        break;
      case 'atom-github://dock-item/github':
        item = this.createGitHubStub(uri);
        this.githubTabStubItem = this.githubTabStubItem || item;
        break;
      default:
        throw new Error(`Invalid DockItem stub URI: ${uri}`);
    }

    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  createGitStub(uri) {
    return _stubItem2.default.create('git', {
      title: 'Git'
    }, uri);
  }

  createGitHubStub(uri) {
    return _stubItem2.default.create('github', {
      title: 'GitHub'
    }, uri);
  }

  createFilePatchControllerStub({ uri } = {}) {
    const item = _stubItem2.default.create('git-file-patch-controller', {
      title: 'Diff'
    }, uri);
    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  createCommitPreviewStub({ uri }) {
    const item = _stubItem2.default.create('git-commit-preview', {
      title: 'Commit preview'
    }, uri);
    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  createCommitDetailStub({ uri }) {
    const item = _stubItem2.default.create('git-commit-detail', {
      title: 'Commit'
    }, uri);
    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  destroyGitTabItem() {
    if (this.gitTabStubItem) {
      this.gitTabStubItem.destroy();
      this.gitTabStubItem = null;
      if (this.controller) {
        this.rerender();
      }
    }
  }

  destroyGithubTabItem() {
    if (this.githubTabStubItem) {
      this.githubTabStubItem.destroy();
      this.githubTabStubItem = null;
      if (this.controller) {
        this.rerender();
      }
    }
  }

  async createRepositoryForProjectPath(projectPath) {
    await _fsExtra2.default.mkdirs(projectPath);

    const repository = this.contextPool.add(projectPath).getRepository();
    await repository.init();
    this.workdirCache.invalidate();

    if (!this.project.contains(projectPath)) {
      this.project.addPath(projectPath);
    }

    await this.scheduleActiveContextUpdate();
  }

  async cloneRepositoryForProjectPath(remoteUrl, projectPath) {
    const context = this.contextPool.getContext(projectPath);
    let repository;
    if (context.isPresent()) {
      repository = context.getRepository();
      await repository.clone(remoteUrl);
      repository.destroy();
    } else {
      repository = new _repository2.default(projectPath, null, { pipelineManager: this.pipelineManager });
      await repository.clone(remoteUrl);
    }

    this.workdirCache.invalidate();

    this.project.addPath(projectPath);

    await this.scheduleActiveContextUpdate();
  }

  getRepositoryForWorkdir(projectPath) {
    const loadingGuessRepo = _repository2.default.loadingGuess({ pipelineManager: this.pipelineManager });
    return this.guessedContext ? loadingGuessRepo : this.contextPool.getContext(projectPath).getRepository();
  }

  getActiveWorkdir() {
    return this.activeContext.getWorkingDirectory();
  }

  getActiveRepository() {
    return this.activeContext.getRepository();
  }

  getActiveResolutionProgress() {
    return this.activeContext.getResolutionProgress();
  }

  getContextPool() {
    return this.contextPool;
  }

  getSwitchboard() {
    return this.switchboard;
  }

  async scheduleActiveContextUpdate(savedState = {}) {
    this.switchboard.didScheduleActiveContextUpdate();
    await this.activeContextQueue.push(this.updateActiveContext.bind(this, savedState), { parallel: false });
  }

  /**
   * Derive the git working directory context that should be used for the package's git operations based on the current
   * state of the Atom workspace. In priority, this prefers:
   *
   * - A git working directory that contains the active pane item in the workspace's center.
   * - A git working directory corresponding to a single Project.
   * - When initially activating the package, the working directory that was active when the package was last
   *   serialized.
   * - The current context, unchanged, which may be a `NullWorkdirContext`.
   *
   * First updates the pool of resident contexts to match all git working directories that correspond to open
   * projects and pane items.
   */
  async getNextContext(savedState) {
    const workdirs = new Set((await Promise.all(this.project.getPaths().map(async projectPath => {
      const workdir = await this.workdirCache.find(projectPath);
      return workdir || projectPath;
    }))));

    const fromPaneItem = async maybeItem => {
      const itemPath = pathForPaneItem(maybeItem);

      if (!itemPath) {
        return {};
      }

      const itemWorkdir = await this.workdirCache.find(itemPath);

      if (itemWorkdir && !this.project.contains(itemPath)) {
        workdirs.add(itemWorkdir);
      }

      return { itemPath, itemWorkdir };
    };

    const active = await fromPaneItem(this.workspace.getCenter().getActivePaneItem());

    this.contextPool.set(workdirs, savedState);

    if (active.itemPath) {
      // Prefer an active item
      return this.contextPool.getContext(active.itemWorkdir || active.itemPath);
    }

    if (this.project.getPaths().length === 1) {
      // Single project
      const projectPath = this.project.getPaths()[0];
      const activeWorkingDir = await this.workdirCache.find(projectPath);
      return this.contextPool.getContext(activeWorkingDir || projectPath);
    }

    if (this.project.getPaths().length === 0 && !this.activeContext.getRepository().isUndetermined()) {
      // No projects. Revert to the absent context unless we've guessed that more projects are on the way.
      return _workdirContext2.default.absent({ pipelineManager: this.pipelineManager });
    }

    // Restore models from saved state. Will return a NullWorkdirContext if this path is not presently
    // resident in the pool.
    const savedWorkingDir = savedState.activeRepositoryPath;
    if (savedWorkingDir) {
      return this.contextPool.getContext(savedWorkingDir);
    }

    return this.activeContext;
  }

  setActiveContext(nextActiveContext) {
    if (nextActiveContext !== this.activeContext) {
      if (this.activeContext === this.guessedContext) {
        this.guessedContext.destroy();
        this.guessedContext = null;
      }
      this.activeContext = nextActiveContext;
      this.rerender(() => {
        this.switchboard.didFinishContextChangeRender();
        this.switchboard.didFinishActiveContextUpdate();
      });
    } else {
      this.switchboard.didFinishActiveContextUpdate();
    }
  }

  async updateActiveContext(savedState = {}) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    this.switchboard.didBeginActiveContextUpdate();

    const nextActiveContext = await this.getNextContext(savedState);
    this.setActiveContext(nextActiveContext);
  }

  refreshAtomGitRepository(workdir) {
    const atomGitRepo = this.project.getRepositories().find(repo => {
      return repo && _path2.default.normalize(repo.getWorkingDirectory()) === workdir;
    });
    return atomGitRepo ? atomGitRepo.refreshStatus() : Promise.resolve();
  }
}

exports.default = GithubPackage;
function pathForPaneItem(paneItem) {
  if (!paneItem) {
    return null;
  }

  // Likely GitHub package provided pane item
  if (typeof paneItem.getWorkingDirectory === 'function') {
    return paneItem.getWorkingDirectory();
  }

  // TextEditor-like
  if (typeof paneItem.getPath === 'function') {
    return paneItem.getPath();
  }

  // Oh well
  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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