"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
class Task {
  constructor(fn, parallel = true) {
    this.fn = fn;
    this.parallel = parallel;
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });
  }

  async execute() {
    try {
      const value = await this.fn.call(undefined);
      this.resolve(value);
    } catch (err) {
      this.reject(err);
    }
  }

  runsInParallel() {
    return this.parallel;
  }

  runsInSerial() {
    return !this.parallel;
  }

  getPromise() {
    return this.promise;
  }
}

class AsyncQueue {
  constructor(options = {}) {
    this.parallelism = options.parallelism || 1;
    this.nonParallelizableOperation = false;
    this.tasksInProgress = 0;
    this.queue = [];
  }

  push(fn, { parallel } = { parallel: true }) {
    const task = new Task(fn, parallel);
    this.queue.push(task);
    this.processQueue();
    return task.getPromise();
  }

  processQueue() {
    if (!this.queue.length || this.nonParallelizableOperation || this.disposed) {
      return;
    }

    const task = this.queue[0];
    const canRunParallelOp = task.runsInParallel() && this.tasksInProgress < this.parallelism;
    const canRunSerialOp = task.runsInSerial() && this.tasksInProgress === 0;
    if (canRunSerialOp || canRunParallelOp) {
      this.processTask(task, task.runsInParallel());
      this.queue.shift();
      this.processQueue();
    }
  }

  async processTask(task, runsInParallel) {
    if (this.disposed) {
      return;
    }

    this.tasksInProgress++;
    if (!runsInParallel) {
      this.nonParallelizableOperation = true;
    }

    try {
      await task.execute();
    } finally {
      this.tasksInProgress--;
      this.nonParallelizableOperation = false;
      this.processQueue();
    }
  }

  dispose() {
    this.queue = [];
    this.disposed = true;
  }
}
exports.default = AsyncQueue;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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