---
title: "Owl Lisp v0.2 manual"
author: Aki Helin
date: 2018-05-17
geometry: "left=3cm,right=3cm,top=2cm,bottom=2cm"
output: pdf_document
---

Owl Lisp is a simple purely functional programming language. This
document describes the language, its implementation and available
libraries. The language is essentially R7RS Scheme, apart from having
only immmutable data structures and relying on multithreading for
some core operations.

# History of Lisp

Lisp (or LISP) is short for LISt Processor. It was initially a
mathematical formalism intended for reasoning about computations. Lisp
was invented by John McCarthy at MIT in 1958. The initial description of
the language contained a definition of the semantics of the language in
terms of itself. This evaluation function was soon implemented on a
computer, turning Lisp from theory to practice.

Since then different dialects of Lisp have evolved into many different directions.
One of the main features connecting the dialects has been the syntax
   - or to be more precise, lack of it.
Most data in Lisp can be displayed textually in a simple format.
Instead of having a syntax for the programming language and one for representing this data,
   Lisp uses the same for both.
Lisp programs are just lists.

Common Lisp is one of the major current dialects.
It is used by several commercial applications.
Scheme is another modern version of Lisp.
It attempts to define a small core of constructs
   out of which many kinds of programming primitives can be built.

Owl is essentially a small step from the multi-paradigm world of Scheme
   into a purely functional one.
Scheme programs are usually mainly functional
   with some mutations sprinkled sparingly where needed.

## Why Another Lisp?

These days compiler construction and parsing are usually taught towards
the end of computer science curriculum, if at all. This is probably
due to the complexity of modern programming languages and environments.
Parsers and compilers, the typical building blocks of programming language
implementations, seem like dark magic only a few select pupils and
devout lone programmers are fit to approach.

This has not always been the case. In a few programming language families it
has been customary to start, not end, studies by building a small version
of the language you are studying. This approach favored
languages which had a small core of features, on top of which you could build the rest
of the system. Forth and Lisp are common examples of such languages.

The goal of Owl Lisp has not at any point been to become an ultimate Lisp and
take over the world. Ïn fact, this has been an anti-goal. The goal has
been to remain simple while incrementally growing only features required
to enable building the kinds of programs it is actually used for. While
this is a somewhat circular definition, it has worked surprisingly well.
Owl is shaped by minimalism and practical applications, not by what seem
like cool and powerful language features.


## Owl vs Scheme

Scheme is a modern lexically scoped multi-paradigm language. One of the
original goals was to also study the actor model of computation. The
actors were eventually removed, because in single threaded operation they
ended up being effectively equivalent with lambda-defined functions.

Owl takes a step back towards the actor model by allowing concurrent
execution of functions and passing messages between them. The operation
is mainly modeled after Erlang.

Another difference is in the multi-paradigm area. Owl does not try to
be able to support also imperative programming. All variable bindings
are made by lambdas, all bindings are permanent, and no data structure
can ever change. The core language is therefore closer to λ-calculus.


## Introduction

Languages such as Latin and English are called natural languages. They
have developed and evolve organically without strict rules or meanings.
It would be impossible to pin down all rules how they operate. There are
also artificial languages which do operate according to fixed rules.
The rules specify what can be considered to be a valid expression in the
language, and usually also what can be done to it without altering
the meaning. Such languages are called *formal languages*. Programming
languages belong to the latter category.

Lisp is a particular family of programming languages. A key feature of
programming languages is that you can write a program to compute anything.
Such programming languages are called *universal*. It is not difficult
to make a universal language - in fact it's quite hard not to!

The definition of a programming language can be thought to consist of two parts, *syntax* and
*semantics*. Since we typically want to write programs as text, we need
some rules to define how sequences of letters are to be interpreted as
something in the programming language. Once we are in the world of the
programming language, and not just reading a sequence of letters, we need
to attach some meaning and action to what we just read. This is the semantics
part.

The Lisp family of programming languages has a peculiar feature not typically
seen in programming languages: it is homoiconic. This means, that the syntax
of the programs is the same as the syntax used to represent data elsewhere.
This makes it extremely easy to write programs which themselves modify or
create programs.

Lisp programs can be developed either by typing the program into one or
more files and running, or by interactively working with a read-eval-print
loop (REPL). A Lisp REPL will repeatedly read one expression from the user,
evaluate expression and finally print out the textual representation of
the result.


## Simplified Core


## Common Data Types

...

## Macros

...

## Multithreading

...

## Modules


# Implementation

...

## Requirements

You should have make and gcc or clang installed.


## Installation

To install system-wide to /usr
```
   $ make
   $ sudo make install
```

Alternatively you can try it out with
```
   $ make
   $ cp bin/ol /somewhere/convenient
   $ /somewhere/convenient/ol
   You see a prompt
   >
```


## Files

   bin/ol      - the owl interpreter/compiler
   c/ovm.c     - the virtual machine / shared owl lisp runtime
   owl/*.scm   - implementation of owl repl and compiler
   fasl/*.fasl - bytecode images for bin/vm used during boot
   bin/vm      - plain VM used during boot
   c/ol.c      - combined VM and REPL heap image


## Usage

Owl can be used either interactively, or to interpret code from files,
or compile programs to fasl-images or c-files. The difference between
an owl program and a plain script is that the program should just have
a function of one argument as the last value, which will be called with
the command line argument list when the program is executed.

In addition to being a regular interpreter, owl also tries to make it
easy to compile programs for different platforms. Owl programs can be
compiled with ol to C-files, which can be compiled to standalone binaries
without needing any owl-specific support files or libraries. The C files
also work on 32- and 64-bit systems, and compile as such at least on
Linux, OpenBSD, and macOS or can be cross-compiled to Windows executables
with MinGW.

For example, to build a hello world program:
```
  $ echo '(lambda (args) (print "Hello, world!"))' > hello.scm
  $ ol -o hello.c hello.scm
  $ gcc -o hello hello.c
  $ ./hello
  Hello, world!
```

Or simply:
```
  $ echo '(λ (args) (print "Hello, world!"))' \
     | ol -x c | gcc -x c -o hello - && ./hello
```

Parts of the compiled programs can be translated to C, instead of being
simply fasl-encoded, to increase speed. Using the --native flag compiles
most of the bytecode to C, and --usual-suspects compiles typically used
functions. To make programs run faster, one can use for example:

```
  $ ol -O2 -o test.c test.scm && gcc -O2 -o test test.c
```

# Libraries

Libraries are named by lists of symbols. For example `(owl lazy)` is
a library name. `ol` comes preloaded with many libraries, some of which
are loaded by default to REPL. If you want to use exported definitions from a
builtin library `(owl lazy)`, you can do so by issuing `(import (owl lazy))`.

Notice that `import` is not a function. It is one of the special forms used
by the REPL to handle management of visible definitions in your sessions. The
syntax is the same as in imports within a library.

If you try to import a library which is not currently loaded, say `(my test)`,
Owl would try to look for library definition from the file "my/test.scm". If
it is available and contains definition of a library called `(my test)`, it will
be loaded and added to your REPL.

You can get a listing of the currently loaded libraries with `,libraries` command.
Many of them are mainly needed for the implementation of the underlying system.
Some of the likely useful ones are documented below.
## (owl codec)

The codec library contains some simple content encoding transformations.

```
  (hex-encode "slartibartfast") → "736c617274696261727466617374"
  (hex-decode "736c617274696261727466617374") → "slartibartfast"
  (hex-decode "7") → #false
  (hex-encode "λx.x") → "cebb782e78"
  (hex-decode "cebb782e78") → "λx.x"
```


## (owl date)

This library attempts to implement date processing functions. Dates
are typically represented as seconds or milliseconds since UNIX Epoch 1.1.1970.
These are generally a good way to work with time, apart from assuming
that an absolute time exists at all. Sometimes it is however necessary
to convert time stamps to human readable form, which means splitting
the time according to various more and less sensible rules.

```
   (time)                → current time in seconds since epoch
   (time-ms)             → current time in milliseconds since epoch
   (date-str 0)          → "00:00:00 1.1.1970 UTC+00:00"
   (date-str (time) 2.5) → "20:49:08 19.3.2018 UTC+02:30"
   (date 0)              → 1 1 1970 0 0 0 ; as multiple values
   (leap-year? 2018)     → #false
```


## (owl digest)

The digest library provides functions for computing cryptographic signatures.
Currently SHA1 and SHA256 digests and corresponding message authentication codes
are supported.

The hash functions also have `hasher-raw` and `hasher-bytes` -variants, which
return the state words and raw signature bytes correspondingly.

```
  (sha1 data)   → hash-string
  (sha256 data) → hash-string
  (hmac-sha1   key message) → hash-string
  (hmac-sha256 key message) → hash-string
```


## (owl eval)

This library exports some read-eval-print-loop functions, such as evaluate.
It is typically called through eval. The `*toplevel*` variable is updated
after each definition, so it can be used to evaluate a term in the corresponding
environment.

```
  (eval (list '+ 1 2) *toplevel*) → 3
  (eval '(/ 1 0) *toplevel*) → #false
```


## (owl fasl)

This library implements serialization of objects to byte
lists, and parsing of the byte lists to corresponding
objects. The format is used internally for storing memory
images to disk. Files with .fasl suffix used in booting
up Owl are just fasl-encoded functions.

```
  (fasl-encode 42) → '(0 0 42)
  (fasl-encode 42) → '(0 0 42)
  (fasl-encode 1/4+i) → '(1 42 2 0 0 1 0 0 4 1 43 2 1 0 0 1 0)
  (fasl-encode (lambda (x) x)) →  '(2 16 7 34 2 0 2 24 4 17 0)
  (fasl-decode '(0 0 0 0) 'bad) → 'bad
  ((fasl-decode (fasl-encode prime?) 'bad) 13337) → #true
  (eq? 'foo (fasl-decode (fasl-encode 'foo) #false)) → #true
```


## (owl ff)

A typical way to make data structures for holding key-value in
Lisp systems is to make an association list. An association list
is a list of pairs, where the car holds the key, and cdr holds the
value. While easy to define and use, they have the downside of slowing
down linearly as the size of the association list grows.

Owl has finite functions, or ffs, which behave like association
lists, but they slow down only logarithmically as they get more keys.
They are internally represented as red-black trees.

`#empty` or `@()` can be used to refer to an empty finite function.
`put` adds or rewrites the value of a key, `get` fetches the value
or returns the third argument if the key is not found. `del` removes
a key from a ff.

```
  > (define f (put (put #empty 'foo 100) 'bar 42))
  > f
  @(foo 100 bar 42)
  > (ff? f)
  #true
  > (get f 'foo #f)
  100
  > (get f 'x #f)
  #f
  > (get (del f 'foo) 'foo #f)
  #f
```
A finite function maps keys to values. As the name implies, a ff
can also be called to do just that. If one argument is given and it
is defined, the value is returned. In case of an undefined value, either
an error is signaled or the second default argument is returned, if
it is specified.

```
  > (f 'foo)
  100
  > (f 'x 'not-there)
  'not-there
  > (map f '(foo bar))
  '(100 42)
```

Many list functions have corresponding functions for ffs, where
usually a function receiving the list element just receives two
arguments, being a particular key and value pair. The name of the
function is typically prefixed with ff-.

```
  (get @(a 1) 'a #f) → 1

  (get @(a 1) 'x #f) → #f

  (put @(a 1 b 2) 'c 3) → @(a 1 b 2 c 3)

  (del @(a 1 b 2) 'a) → @(b 2)

  (fupd ff key value) → ff', like put, but for an existing key

  (keys @(foo 1 bar 2)) → '(foo bar)

  (ff-union @(a 100 b 200) @(b 2 c 3) +) → @(a 100 b 202 c 3)

  (ff-diff @(a 1 b 2 c 3) @(a 10 b 20)) → @(c 3)

  (ff-fold (λ (o k v) (cons (cons k v) o)) #n @(foo 1 bar 2) →
     '((bar . 2) (foo . 1))

  (ff-foldr (λ (o k v) (cons (cons k v) o)) #n @(foo 1 bar 2) →
     '((foo . 1) (bar . 2))
  (ff-map @(a 1 b 2 c 3) (λ (k v) (square v))) → @(a 1 b 4 c 9)

  (ff-iter ff) → a lazy list of key-value pairs

  (list->ff '((a . 1) (b . 2))) → @(a 1 b 2)

  (ff->list @(a 1 b 2)) → '((a . 1) (b . 2))

```


## (owl gensym)

It is sometimes useful to get symbols, which do not occur elsewhere.
This is typically needed in the compiler, but it may also be needed
elsewhere. Gensyms in Owl are just regular symbols, which do not
occur in a given expression. This requires walking through the whole
expression. To avoid having to walk the original expression in many
cases when gensyms are needed, they work in a way that ensures that
the gensym of the gensym of an expression also does not occur in the
original expression.

```
  (gensym '(lambda (x) x)) → g1
  (gensym 'g1) → g2
  (gensym 'g100000) → 'g100001
```


## (owl lazy)

Lazy lists (or streams) are like lists, but they are computed only as far as needed.
You can for example define a lazy list of all integers below a million, and then
proceed to run computations on them, without worrying whether you have enough memory
to hold a million numbers. Lazy lists are for example useful in computations, where
you know how something is constructed, but don't yet know how many of them will be
needed, or know that you only need them one at a time and don't want to waste memory.

A lazy list is either #null, a pair of a value and rest of the lazy list, or a
function of zero arguments (a thunk) which when called will return the rest of the
lazy list. Therefore, since normal lists are a subset of lazy lists, all lazy list
functions can also take normal lists as arguments.

`Scheme warning`: recall that Owl does not have mutable data structures, so lazy
lists do not cache their results.

```
  (pair head exp) → ll, lazy equivalent of (cons head exp), but exp is not evaluated yet
  (force-ll ll) → list, turn a lazy list into a regular one
```


## (owl random)

Randomness is an interesting thing to work with in a purely
functional setting. Owl builds randomness around streams of
typically deterministically generated 24-bit fixnums. These
are usually called rands in the code.

A function involving randomness typically receives a rand
stream, and also returns it after possibly consuming some
rands. Behavior like this would be easy to hide using macros
or monadic code, but Owl generally strives to be explicit and
simple, so the rand streams are handled just like any other
value.

```
  > (define rs (seed->rands 9))
  > (rand rs 10000)
  '(values #<function> 3942) ;; the new rand stream and 3942
  > (lets ((rs a (rand rs 10000))) a)
  3942
  > (lets ((rs elem (rand-elem rs '(a b c d e f g)))) elem)
  'c
  > (lets ((rs sub (rand-subset rs '(a b c d e f g)))) sub)
  '(b e f)
  > (lets ((rs perm (random-permutation rs '(a b c d e f g)))) perm)
  '(g e c b d a f)
  > (lets ((rs ns (random-numbers rs 100 10))) ns)
  '(95 39 69 99 2 98 56 85 77 39)
```


## (owl sys)

Owl sys library exports various operating system calls and helper
functions for using them.


